/*
 *   cddb - CD Database Management Library
 *
 *   Copyright (C) 1993-2000  Ti Kan
 *   E-mail: ti@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef LINT
static char *_cddb_c_ident_ = "@(#)cddb_ext.c	6.17 00/01/26";
#endif

#ifdef __VMS
typedef char *	caddr_t;
#endif

#define _CDDB_INTERN	/* Expose internal function protos in cddb.h */

#include "common_d/appenv.h"
#include "common_d/patchlevel.h"
#include "common_d/util.h"
#include "cddb_d/cddb.h"


#define MAX_ENV_LEN	(STR_BUF_SZ * 16)

extern appdata_t	app_data;
extern FILE		*errfp;
extern cddb_path_t	*cddb_pathhead;
extern cddb_linkopts_t	*cddb_linkhead;
extern cddb_proxy_t	cddb_proxy;
extern char		*auth_buf,
			http_hellostr[MAXHOSTNAMELEN + STR_BUF_SZ],
			http_extinfo[MAXHOSTNAMELEN + STR_BUF_SZ];
extern cddb_client_t	*cddb_clinfo;		/* Client info */

STATIC pid_t		child_pid;		/* Child pid */
STATIC bool_t		ischild;		/* Is a child process */
STATIC char		curfile[FILE_PATH_SZ];	/* Current disc info file */


/***********************
 *   public routines   *
 ***********************/


/*
 * cddb_init
 *	Initialize CD database management services
 *
 * Args:
 *	progname - The client program name string
 *	username - The client user login name string
 *
 * Return:
 *	Nothing.
 */
void
cddb_init(cddb_client_t *clp)
{
	char		*cp,
			*path;

	DBGPRN(errfp, "\nlibcddb: %s %s %s\n",
#ifdef SYNCHRONOUS
		"SYNCHRONOUS",
#else
		"ASYNCHRONOUS",
#endif
#ifdef NOREMOTE
		"NOREMOTE",
		""
#else
		"REMOTE",
#ifdef SOCKS
		"SOCKS"
#else
		""
#endif
#endif
	);

	cddb_clinfo = clp;

	/* Sanity check */
	if (clp->prog[0] == '\0')
		(void) strcpy(clp->prog, "unknown");

	/* Hard wire file modes in case these are missing from the
	 * common.cfg file
	 */
	if (app_data.cddb_filemode == NULL)
		app_data.cddb_filemode = "0664";
	if (app_data.hist_filemode == NULL)
		app_data.hist_filemode = "0644";

#ifndef NOREMOTE
	SOCKSINIT(clp->prog);
#endif

	/* Load XMCD_CDDBPATH environment variable, if specified */
	if ((cp = (char *) getenv("XMCD_CDDBPATH")) != NULL) {
		if ((int) strlen(cp) >= MAX_ENV_LEN) {
			CDDB_FATAL(app_data.str_longpatherr);
			return;
		}
		app_data.cddb_path = (char *) MEM_ALLOC(
			"app_data.cddb_path",
			strlen(cp) + 1
		);
		if (app_data.cddb_path == NULL) {
			CDDB_FATAL(app_data.str_nomemory);
			return;
		}

		(void) strcpy(app_data.cddb_path, cp);
	}

	if (app_data.cddb_path == NULL || app_data.cddb_path[0] == '\0')
		return;

	/* Create the CDDB path list */
	path = app_data.cddb_path;
	while ((cp = strchr(path, CDDBPATH_SEPCHAR)) != NULL) {
		*cp = '\0';

		if (!cddb_add_pathent(path))
			return;

		*cp = CDDBPATH_SEPCHAR;
		path = cp + 1;
	}
	(void) cddb_add_pathent(path);

#ifndef NOREMOTE
	if (app_data.http_proxy) {
		if (app_data.proxy_server == NULL ||
		    app_data.proxy_server[0] == '\0') {
			CDDB_FATAL(app_data.str_proxyerr);
			return;
		}
		if ((cp = strchr(app_data.proxy_server, ':')) != NULL)
			*cp = '\0';

		(void) strncpy(cddb_proxy.host, app_data.proxy_server,
			       MAXHOSTNAMELEN-1);
		cddb_proxy.host[MAXHOSTNAMELEN-1] = '\0';

		if (cp != NULL) {
			cddb_proxy.port = atoi(cp+1);
			*cp = ':';
		}
		else
			cddb_proxy.port = HTTP_DFLT_PORT;
	}

	(void) sprintf(http_hellostr,
		"hello=unknown+localhost+%s+v%s%sPL%d",
		cddb_clinfo->prog, VERSION, VERSION_EXT, PATCHLEVEL);
	(void) sprintf(http_extinfo,
#ifdef NO_MASQUERADE
		"User-Agent: %s/%s%sPL%d\r\nAccept: %s\r\n",
#else
		/* Believe it or not, this is how MS Internet Explorer
		 * does it, so don't blame me...
		 */
		"User-Agent: %s (compatible; %s %s%sPL%d)\r\nAccept: %s\r\n",
		"Mozilla/4.7",
#endif
		cddb_clinfo->prog, VERSION, VERSION_EXT, PATCHLEVEL,
		"text/plain");
#endif
}


/*
 * cddb_halt
 *	Shut down cddb subsystem.
 *
 * Args:
 *	s - Pointer to the curstat_t structure
 *
 * Return:
 *	Nothing
 */
void
cddb_halt(curstat_t *s)
{
	if (curfile[0] != '\0' && s->devlocked)
		(void) UNLINK(curfile);
}


/*
 * cddb_load
 *	Load CD database entry for the currently inserted CD.
 *	A search will be performed on all database paths in the
 *	cddb path list.
 *
 * Args:
 *	dbp - Pointer to the cddb_incore_t structure
 *	s - Pointer to the curstat_t structure
 *
 * Return:
 *	return code as defined by cddb_ret_t
 */
cddb_ret_t
cddb_load(cddb_incore_t *dbp, curstat_t *s)
{
#ifdef __VMS
	cddb_file_t	*fp;

	/* Clear flag */
	dbp->flags &= ~(CDDB_INEXACT | CDDB_AUTHFAIL);

	/* Get a CDDB file handle */
	if ((fp = cddb_open(dbp, s, O_RDONLY)) == NULL) {
		/* Database entry does not exist */
		return CDDB_SET_CODE(OPEN_ERR, 0);
	}

	/* Read contents of CD database entry and update incore structure */
	if (!cddb_read0(fp, dbp, s)) {
		/* Invalid CD database contents */
		(void) cddb_close(fp);
		return CDDB_SET_CODE(READ_ERR, 0);
	}

	/* Close file */
	(void) cddb_close(fp);

	if (dbp->flags & CDDB_INEXACT) {
		/* An inexact CDDB match was found */
		return CDDB_SET_CODE(MATCH_ERR, 0);
	}
	else if (dbp->flags & CDDB_AUTHFAIL) {
		/* Proxy authorization failed */
		return CDDB_SET_CODE(AUTH_ERR, 0);
	}

	switch (dbp->type) {
	case CDDB_REMOTE_CDDBP:
	case CDDB_REMOTE_HTTP:
		if (app_data.cddb_rmtautosave > 0)
			/* Auto-save */
			cddb_autosave(dbp, s);
		break;

	default:
		break;
	}

	return 0;

#else	/* __VMS */
	int		pfd[2],
			ret,
			retcode;
	pid_t		cpid;
	waitret_t	stat_val;
	cddb_file_t	*fp,
			*wfp,
			*rfp;

	/* Clear flag */
	dbp->flags &= ~(CDDB_INEXACT | CDDB_AUTHFAIL);

	/* Set up a pipe */
	if (PIPE(pfd) < 0)
		return CDDB_SET_CODE(OPEN_ERR, errno);

	/* Fork child to performs actual I/O */
	switch (cpid = FORK()) {
	case 0:
		/* Child process */
		ischild = TRUE;
		break;

	case -1:
		(void) close(pfd[0]);
		(void) close(pfd[1]);
		return CDDB_SET_CODE(FORK_ERR, errno);

	default:
		/* Parent process */
		child_pid = cpid;

		/* Close un-needed pipe descriptor */
		(void) close(pfd[1]);

		/* Allocate cddb_file_t structures for read pipe */
		rfp = (cddb_file_t *)(void *) MEM_ALLOC(
			"cddb_file_t",
			sizeof(cddb_file_t)
		);
		if (rfp == NULL) {
			CDDB_FATAL(app_data.str_nomemory);
			return CDDB_SET_CODE(OPEN_ERR, 0);
		}

		/* Allocate read cache */
		rfp->cache = (unsigned char *) MEM_ALLOC(
			"rfp->cache",
			CDDB_CACHESZ
		);
		if (rfp->cache == NULL) {
			CDDB_FATAL(app_data.str_nomemory);
			return CDDB_SET_CODE(OPEN_ERR, 0);
		}

		rfp->magic = CDDB_FMAGIC;
		rfp->fd = pfd[0];
		rfp->rw = O_RDONLY;
		rfp->pos = 0;
		rfp->cnt = 0;
		rfp->type = CDDB_LOCAL;	/* It's really a pipe, but that's ok */

		/* Read contents of CD database entry from pipe into
		 * in-core structure
		 */
		(void) cddb_read2(rfp, dbp, s);

		/* Close read pipe */
		(void) cddb_close(rfp);

		if (dbp->flags & CDDB_INEXACT)
			retcode = CDDB_SET_CODE(MATCH_ERR, 0);
		else if (dbp->flags & CDDB_AUTHFAIL)
			retcode = CDDB_SET_CODE(AUTH_ERR, 0);
		else
			retcode = 0;

		/* Parent process: wait for child to exit */
		while ((ret = WAITPID(cpid, &stat_val, 0)) != cpid) {
			if (ret < 0) {
				child_pid = 0;
				stat_val = 0;
				break;
			}

			/* Do some work */
			if (cddb_clinfo->workproc != NULL)
				cddb_clinfo->workproc(cddb_clinfo->arg);
		}

		child_pid = 0;

		if (WIFEXITED(stat_val)) {
			if (WEXITSTATUS(stat_val) == 0) {
				switch (dbp->type) {
				case CDDB_REMOTE_CDDBP:
				case CDDB_REMOTE_HTTP:
					if (app_data.cddb_rmtautosave > 0)
						/* Auto-save */
						cddb_autosave(dbp, s);
					break;

				default:
					break;
				}

				return (retcode);
			}
			else
				return CDDB_SET_CODE(WEXITSTATUS(stat_val), 0);
		}
		else if (WIFSIGNALED(stat_val))
			return CDDB_SET_CODE(KILLED_ERR, WTERMSIG(stat_val));
		else
			return 0;
	}

	/* Exit on receipt of SIGTERM */
	(void) signal(SIGTERM, onterm);

	/* Force uid and gid to original setting */
	if (!util_set_ougid())
		exit(SETUID_ERR);

	/* Close un-needed pipe descriptor */
	(void) close(pfd[0]);

	/* Allocate cddb_file_t structures for write pipe */
	wfp = (cddb_file_t *)(void *) MEM_ALLOC(
		"cddb_file_t",
		sizeof(cddb_file_t)
	);
	if (wfp == NULL)
		exit(MEM_ERR);

	/* Allocate write cache */
	wfp->cache = (unsigned char *) MEM_ALLOC("wfp->cache", CDDB_CACHESZ);
	if (wfp->cache == NULL)
		exit(MEM_ERR);

	wfp->magic = CDDB_FMAGIC;
	wfp->fd = pfd[1];
	wfp->rw = O_WRONLY;
	wfp->pos = 0;
	wfp->cnt = 0;
	wfp->type = CDDB_LOCAL;	/* It's really a pipe, but that's ok */

	/* Get a CDDB file handle */
	if ((fp = cddb_open(dbp, s, O_RDONLY)) == NULL) {
		/* Database entry does not exist */
		(void) cddb_close(wfp);
		exit(OPEN_ERR);
	}

	/* Read contents of CD database entry and write into pipe */
	if (!cddb_read1(fp, wfp, dbp)) {
		/* Invalid CD database contents */
		(void) cddb_close(fp);
		(void) cddb_close(wfp);
		exit(READ_ERR);
	}

	/* Close file */
	(void) cddb_close(fp);
	(void) cddb_close(wfp);

	/* Child exits here */
	exit(0);

	/*NOTREACHED*/
#endif	/* __VMS */
}


/*
 * cddb_save
 *	Save current in-core CD database data into file
 *
 * Args:
 *	dbp - Pointer to the cddb_incore_t structure
 *	s - Pointer to the curstat_t structure
 *
 * Return:
 *	return code as defined by cddb_ret_t
 */
cddb_ret_t
cddb_save(cddb_incore_t *dbp, curstat_t *s)
{
	cddb_file_t	*fp;
	struct stat	stbuf;
	int		ret;
#ifndef __VMS
	cddb_ret_t	retcode;
	pid_t		cpid;
	waitret_t	stat_val;
	char		dbfile[FILE_PATH_SZ];
#endif

	/* Increment revision if in-core entry has been edited */
	if (dbp->flags & CDDB_CHANGED)
		dbp->revision++;

#ifndef __VMS
	/* Fork child to perform actual I/O */
	switch (cpid = FORK()) {
	case 0:
		/* Child process */
		ischild = TRUE;
		break;

	case -1:
		return CDDB_SET_CODE(FORK_ERR, errno);

	default:
		/* Parent process: wait for child to exit */
		while ((ret = WAITPID(cpid, &stat_val, 0)) != cpid) {
			if (ret < 0)
				return 0;

			/* Do some work */
			if (cddb_clinfo->workproc != NULL)
				cddb_clinfo->workproc(cddb_clinfo->arg);
		}

		retcode = 0;
		if (WIFEXITED(stat_val))
			retcode = CDDB_SET_CODE(WEXITSTATUS(stat_val), 0);
		else if (WIFSIGNALED(stat_val))
			retcode = CDDB_SET_CODE(KILLED_ERR, WTERMSIG(stat_val));

		/* If discid != queryid, perform a link */
		if (retcode == 0 && dbp->discid != dbp->queryid) {
			(void) sprintf(dbfile, CDDBFILE_PATH,
					util_dirname(dbp->dbfile),
					dbp->discid);
			(void) cddb_link(dbp, dbp->dbfile, dbfile);
			(void) strcpy(dbp->dbfile, dbfile);
		}

		return (retcode);
	}

	/* Force uid and gid to original setting */
	if (!util_set_ougid())
		exit(SETUID_ERR);
#endif	/* __VMS */

	DBGPRN(errfp, "Writing CD database file %s\n", dbp->dbfile);

	/* Get a CDDB file handle */
	if ((fp = cddb_open(dbp, s, O_WRONLY | O_TRUNC | O_CREAT)) == NULL)
		CH_RET(OPEN_ERR);

	/* Set CD database file permissions */
	util_setperm(dbp->dbfile, app_data.cddb_filemode);

	ret = 0;

	/* Write to CD database */
	if (!cddb_write(fp, dbp, s))
		ret = WRITE_ERR;

	/* Close file */
	if (!cddb_close(fp))
		ret = CLOSE_ERR;

	/* Check saved file */
	if (stat(dbp->dbfile, &stbuf) < 0)
		ret = WRITE_ERR;
	else if (!S_ISREG(stbuf.st_mode) || stbuf.st_size == 0)
		ret = WRITE_ERR;

	/* Remove any unsuccessfully written entry */
	if (ret != 0)
		(void) UNLINK(dbp->dbfile);

	/* Child exits here */
	CH_RET(ret);
	/*NOTREACHED*/
}


/*
 * cddb_link
 *	Create a link of an existing local CD database entry to another.
 *
 * Args:
 *	origpath - Source directory path string
 *	newpath - target directory path string
 *
 * Return:
 *	return code as defined by cddb_ret_t
 */
cddb_ret_t
cddb_link(cddb_incore_t *dbp, char *origpath, char *newpath)
{
	int		i,
			bufsz = STR_BUF_SZ * 2;
	FILE		*rfp,
			*wfp;
	cddb_linkopts_t	*r;
	char		*buf,
			origid[9],
			newid[9];
	bool_t		process_idlist;
#ifndef __VMS
	int		ret;
	pid_t		cpid;
	waitret_t	stat_val;
	cddb_ret_t	retcode;


	/* Fork child to perform actual I/O */
	switch (cpid = FORK()) {
	case 0:
		/* Child process */
		ischild = TRUE;
		break;

	case -1:
		return CDDB_SET_CODE(FORK_ERR, errno);

	default:
		/* Parent process: wait for child to exit */
		while ((ret = WAITPID(cpid, &stat_val, 0)) != cpid) {
			if (ret < 0)
				return 0;

			/* Do some work */
			if (cddb_clinfo->workproc != NULL)
				cddb_clinfo->workproc(cddb_clinfo->arg);
		}

		retcode = 0;
		if (WIFEXITED(stat_val))
			retcode = CDDB_SET_CODE(WEXITSTATUS(stat_val), 0);
		else if (WIFSIGNALED(stat_val))
			retcode = CDDB_SET_CODE(KILLED_ERR, WTERMSIG(stat_val));

		if (retcode == 0)
			/* Set the query ID back to the disc ID */
			dbp->queryid = dbp->discid;

		return (retcode);
	}

	/* Force uid and gid to original setting */
	if (!util_set_ougid())
		exit(SETUID_ERR);
#endif	/* __VMS */

	DBGPRN(errfp, "Linking %s -> %s\n", origpath, newpath);

	(void) strncpy(origid, util_basename(origpath), 8);
	(void) strncpy(newid, util_basename(newpath), 8);
	origid[8] = newid[8] = '\0';

	/*
	 * Copy original file to new file, modify the DISCID line
	 */

	(void) UNLINK(newpath);

	/* Open original cddb file for reading */
	if ((rfp = fopen(origpath, "r")) == NULL)
		CH_RET(OPEN_ERR);
	/* Open new file for writing */
	if ((wfp = fopen(newpath, "w")) == NULL)
		CH_RET(OPEN_ERR);

	/*
	 * Set file permissions
	 */
	util_setperm(newpath, app_data.cddb_filemode);

	if ((buf = (char *) MEM_ALLOC("link_buf", bufsz)) == NULL)
		CH_RET(MEM_ERR);

	process_idlist = FALSE;
	cddb_free_idlist(dbp);
	while (fgets(buf, bufsz, rfp) != NULL) {
		if (strncmp(buf, "DISCID=", 7) == 0) {
			process_idlist = TRUE;

			/* Zap newline */
			buf[strlen(buf)-1] = '\0';

			/* Build disc id list */
			(void) cddb_add_idlist(dbp, buf + 7);
		}
		else {
			if (process_idlist) {
				process_idlist = FALSE;

				/* Check if the original file's ID is in the
				 * idlist.  If not, add it.
				 */
				for (r = dbp->idlist; r != NULL; r = r->next) {
					if (strcmp(origid, r->idstr) == 0)
						break;
				}
				if (r == NULL)
					cddb_add_ident(dbp, origid);

				/* Pump out the list of disc IDs */
				(void) fprintf(wfp, "DISCID=%s", newid);
				i = 1;
				for (r = dbp->idlist; r != NULL; r = r->next) {
					if (strcmp(newid, r->idstr) == 0)
						continue;

					if (i == 0)
						(void) fprintf(wfp,
							       "DISCID=%s",
							       r->idstr);
					else
						(void) fprintf(wfp, "%c%s",
							       CDDBID_SEPCHAR,
							       r->idstr);

					i++;
					if (i == 8) {
						(void) fprintf(wfp, "\n");
						i = 0;
					}
				}
				if (i != 0)
					(void) fprintf(wfp, "\n");
			}

			(void) fprintf(wfp, "%s", buf);
		}
	}

	(void) fclose(rfp);
	(void) fclose(wfp);

	/*
	 * Write contents of new file back to original file
	 */

	/* Open destination file for reading */
	if ((rfp = fopen(newpath, "r")) == NULL) {
		MEM_FREE(buf);
		CH_RET(OPEN_ERR);
	}
	/* Open original cddb file for writing */
	if ((wfp = fopen(origpath, "w")) == NULL) {
		MEM_FREE(buf);
		CH_RET(OPEN_ERR);
	}

	/*
	 * Set file permissions
	 */
	util_setperm(origpath, app_data.cddb_filemode);

	while (fgets(buf, bufsz, rfp) != NULL)
		(void) fprintf(wfp, "%s", buf);

	(void) fclose(rfp);
	(void) fclose(wfp);
	MEM_FREE(buf);

	/*
	 * Remove new file
	 */
	if (UNLINK(newpath) < 0)
		CH_RET(LINK_ERR);

	/*
	 * Link original file to new file
	 */
	if (LINK(origpath, newpath) < 0)
		CH_RET(LINK_ERR);

	/* Child exits here. */
	CH_RET(0);
	/*NOTREACHED*/
}


/*
 * cddb_send
 *	Send current CD database entry to archive server via e-mail.
 *
 * Args:
 *	dbp - Pointer to the cddb_incore_t structure
 *	s - Pointer to the curstat_t structure
 *
 * Return:
 *	return code as defined by cddb_ret_t
 */
/*ARGSUSED*/
cddb_ret_t
cddb_send(cddb_incore_t *dbp, curstat_t *s)
{
	char		*p,
			*q,
			*cmd,
			*subject;
	int		ret,
			err;
#ifndef __VMS
	pid_t		cpid;
	waitret_t	stat_val;

	/* Fork child to perform actual I/O */
	switch (cpid = FORK()) {
	case 0:
		/* Child process */
		ischild = TRUE;
		break;

	case -1:
		return CDDB_SET_CODE(FORK_ERR, errno);

	default:
		/* Parent process: wait for child to exit */
		while ((ret = WAITPID(cpid, &stat_val, 0)) != cpid) {
			if (ret < 0)
				return 0;

			/* Do some work */
			if (cddb_clinfo->workproc != NULL)
				cddb_clinfo->workproc(cddb_clinfo->arg);
		}

		if (WIFEXITED(stat_val))
			return CDDB_SET_CODE(WEXITSTATUS(stat_val), 0);
		else if (WIFSIGNALED(stat_val))
			return CDDB_SET_CODE(KILLED_ERR, WTERMSIG(stat_val));
		else
			return 0;
	}

	/* Force uid and gid to original setting */
	if (!util_set_ougid())
		exit(SETUID_ERR);
#endif	/* __VMS */

	/* Check CD database entry for sanity */
	if (!cddb_check_entry(dbp))
		CH_RET(INCMPL_ERR);

	/* Allocate command buffer */
	cmd = (char *) MEM_ALLOC(
		"send_cmdbuf",
		strlen(app_data.cddb_mailcmd) +
		strlen(app_data.cddb_mailsite) +
		strlen(dbp->category) + STR_BUF_SZ
	);
	subject = (char *) MEM_ALLOC(
		"subject",
		strlen(dbp->category) + STR_BUF_SZ
	);
	if (cmd == NULL || subject == NULL)
		CH_RET(MEM_ERR);

	/* Mail command */
	for (p = cmd, q = app_data.cddb_mailcmd; *q != '\0'; p++, q++) {
		if (*q ==  '%') {
			/* Support the special meanings of %S, %A and %F */
			switch (*(q+1)) {
			case 'S':
				/* Mail subject */
				(void) sprintf(subject, "cddb %s %08x",
					       dbp->category[0] == '\0' ?
					       "unknown" : dbp->category,
					       dbp->discid);
				(void) strcpy(p, subject);
				p += strlen(subject) - 1;
				q++;
				break;

			case 'A':
				/* Mail address */
				(void) strcpy(p, app_data.cddb_mailsite);
				p += strlen(app_data.cddb_mailsite) - 1;
				q++;
				break;

			case 'F':
				/* CD database file path */
				(void) strcpy(p, dbp->dbfile);
				p += strlen(dbp->dbfile) - 1;
				q++;
				break;

			default:
				*p++ = *q++;	/* '%' */
				*p = *q;
				break;
			}
		}
		else {
			*p = *q;
		}
	}
	*p = '\0';

	DBGPRN(errfp, "Send CDDB: [%s]\n", cmd);

	/* Send the mail */
	if (cddb_clinfo->isdemo != NULL && cddb_clinfo->isdemo()) {
		/* Don't send mail if in demo mode */
		(void) fprintf(errfp, "DEMO mode: mail not sent.\n");
		ret = 0;
	}
	else
		ret = system(cmd);

	MEM_FREE(subject);
	MEM_FREE(cmd);

#ifdef __VMS
	err = (ret == 0 || ret == 1) ? 0 : CMD_ERR;
#else
	err = (ret == 0) ? 0 : CMD_ERR;
#endif

	/* Child exits here */
	CH_RET(err);
	/*NOTREACHED*/
}


/*
 * cddb_clear
 *	Clear the in-core cddb_incore_t structure
 *
 * Args:
 *	dbp - Pointer to the cddb_incore_t structure
 *	s - Pointer to the curstat_t structure
 *	reload - Whether this operation is due to a reload of the CDDB
 *		(We don't want to clear a play sequence in this case).
 *
 * Return:
 *	Nothing
 */
/*ARGSUSED*/
void
cddb_clear(cddb_incore_t *dbp, curstat_t *s, bool_t reload)
{
	int		i;
	cddb_match_t	*p,
			*q;

	/* Cancel pending queries */
	cddb_load_cancel();

	dbp->category[0] = '\0';

	if (dbp->dbfile != NULL) {
		MEM_FREE(dbp->dbfile);
		dbp->dbfile = NULL;
	}

	if (dbp->dtitle != NULL) {
		MEM_FREE(dbp->dtitle);
		dbp->dtitle = NULL;
	}

	if (dbp->extd != NULL) {
		MEM_FREE(dbp->extd);
		dbp->extd = NULL;
	}

	for (i = MAXTRACK-1; i >= 0; i--) {
		if (dbp->trklist[i] != NULL) {
			MEM_FREE(dbp->trklist[i]);
			dbp->trklist[i] = NULL;
		}

		if (dbp->extt[i] != NULL) {
			MEM_FREE(dbp->extt[i]);
			dbp->extt[i] = NULL;
		}

		if (dbp->sav_extt[i] != NULL) {
			MEM_FREE(dbp->sav_extt[i]);
			dbp->sav_extt[i] = NULL;
		}
	}

	if (!reload && dbp->playorder != NULL) {
		MEM_FREE(dbp->playorder);
		dbp->playorder = NULL;
	}

	dbp->discid = 0;
	dbp->queryid = 0;
	dbp->revision = 0;
	dbp->type = CDDB_INVALID;
	dbp->flags = 0;

	for (p = q = dbp->matchlist; p != NULL; p = q) {
		q = p->next;
		MEM_FREE(p->dtitle);
		MEM_FREE(p);
	}
	dbp->matchlist = dbp->match_cur = NULL;

	cddb_free_idlist(dbp);
}


/*
 * cddb_load_cancel
 *	Cancel asynchronous CDDB load operation, if active.
 *
 * Args:
 *	None.
 *
 * Return:
 *	nothing.
 */
void
cddb_load_cancel(void)
{
	if (child_pid > 0) {
		/* Kill child process */
		(void) kill(child_pid, SIGTERM);
		child_pid = 0;
	}
}


/*
 * cddb_discid
 *	Compute a magic disc ID based on the number of tracks,
 *	the length of each track, and a checksum of the string
 *	that represents the offset of each track.
 *
 * Args:
 *	s - Pointer to the curstat_t structure.
 *
 * Return:
 *	The integer disc ID.
 */
word32_t
cddb_discid(curstat_t *s)
{
	int	i,
		a,
		t = 0,
		n = 0;

	/* For backward compatibility this algorithm must not change */

	a = (int) s->tot_trks;

	for (i = 0; i < a; i++)
		n += cddb_sum((s->trkinfo[i].min * 60) + s->trkinfo[i].sec);

	t = ((s->trkinfo[a].min * 60) + s->trkinfo[a].sec) -
	     ((s->trkinfo[0].min * 60) + s->trkinfo[0].sec);

	return ((n % 0xff) << 24 | t << 8 | s->tot_trks);
}


/*
 * cddb_category
 *	Extract the category from a CDDB file path
 *
 * Args:
 *	path - The path name to a CDDB file.
 *
 * Return:
 *	The category string.
 */
char *
cddb_category(char *path)
{
#ifdef __VMS
	char		*p,
			*q;
	static char	retbuf[FILE_BASE_SZ];

	retbuf[0] = '\0';

	if ((p = strrchr(path, '.')) == NULL ||
	    (q = strrchr(path, ']')) == NULL)
		return (path);

	*q = '\0';
	(void) strncpy(retbuf, p+1, FILE_BASE_SZ - 1);
	retbuf[FILE_BASE_SZ - 1] = '\0';
	*q = ']';

	return (retbuf);
#else
	return (util_basename(path));
#endif
}


/*
 * cddb_set_auth
 *	Set proxy authorization user name and password.
 *
 * Args:
 *	name - Proxy user name
 *	passwd - Proxy password
 *
 * Return:
 *	Nothing.
 */
void
cddb_set_auth(char *name, char *passwd)
{
	int	i,
		j,
		n1,
		n2;
	char	*buf;

	if (name == NULL || passwd == NULL)
		return;

	i = strlen(name);
	j = strlen(passwd);
	n1 = i + j + 2;
	n2 = (n1 * 4 / 3) + 8;

	if ((buf = (char *) MEM_ALLOC("set_auth_buf", n1)) == NULL) {
		CDDB_FATAL(app_data.str_nomemory);
		return;
	}

	(void) sprintf(buf, "%s:%s", name, passwd);

	(void) memset(name, 0, i);
	(void) memset(passwd, 0, j);

	if (auth_buf != NULL)
		MEM_FREE(auth_buf);

	if ((auth_buf = (char *) MEM_ALLOC("set_auth_authbuf", n2)) == NULL) {
		CDDB_FATAL(app_data.str_nomemory);
		return;
	}

	/* Base64 encode the name/password pair */
	util_b64encode(
		(byte_t *) buf,
		strlen(buf),
		(byte_t *) auth_buf,
		FALSE
	);

	(void) memset(buf, 0, n1);
	MEM_FREE(buf);
}


/*
 * cddb_pathlist
 *	Return a pointer to the CDDB path list.
 *
 * Args:
 *	None.
 *
 * Return:
 *	A pointer to the path list head, or NULL if the list is empty.
 */
cddb_path_t *
cddb_pathlist(void)
{
	return (cddb_pathhead);
}


/*
 * cddb_init_linkopts
 *	Build a sorted linked list of track titles which are to be
 *	used to present to the user for database search-link.
 *
 * Args:
 *	s - Pointer to the curstat_t structure.
 *
 * Return:
 *	TRUE - success
 *	FALSE - failure
 */
bool_t
cddb_init_linkopts(cddb_incore_t *dbp, curstat_t *s)
{
	int		i,
			n,
			ntrk,
			bufsz = STR_BUF_SZ * 2,
			trk_off[MAXTRACK];
	word32_t	offset,
			trkaddr;
	char		*dbdir,
			*bname,
			*buf,
			tmppath[FILE_PATH_SZ];
	FILE		*fp;
	DIR		*dp;
	struct dirent	*de;
	bool_t		found;

	/* Warning: This code is SYSV-ish.  Porting to other
	 * environment may require some modification here.
	 */

	if (dbp->dbfile == NULL)
		/* Error */
		return FALSE;

	dbdir = util_dirname(dbp->dbfile);
	bname = util_basename(dbp->dbfile);

	if ((dp = OPENDIR(dbdir)) == NULL)
		return FALSE;

	if ((buf = (char *) MEM_ALLOC("init_linkopts", bufsz)) == NULL) {
		CDDB_FATAL(app_data.str_nomemory);
		return FALSE;
	}

	for (n = 0; (de = READDIR(dp)) != NULL; n++) {
		/* Handle some events to avoid GUI freeze-up
		 * (once every 50 loops)
		 */
		if (cddb_clinfo->workproc != NULL && (n % 50) == 0)
			cddb_clinfo->workproc(cddb_clinfo->arg);

		if ((int) strlen(de->d_name) != 8)
			continue;

		/* Find all entries in this directory with the same
		 * number of tracks as this disc.
		 */
		if (strncmp(de->d_name + 6, bname + 6, 2) != 0)
			continue;

		(void) sprintf(tmppath, CONCAT_PATH, dbdir, de->d_name);
		if ((fp = fopen(tmppath, "r")) == NULL)
			continue;
		
		/* Read first line of database file */
		if (fgets(buf, bufsz, fp) == NULL) {
			(void) fclose(fp);
			continue;
		}

		/* Database file signature check */
		if (strncmp(buf, "# xmcd", 6) != 0) {
			/* Not a supported database file */
			(void) fclose(fp);
			continue;
		}

		(void) memset(trk_off, 0, sizeof(trk_off));
		ntrk = 0;
		found = FALSE;

		while (fgets(buf, bufsz, fp) != NULL) {
			/* Look for track addresses of possible links */
			if (strncmp(buf, "# Track frame offsets", 21) == 0) {
				found = TRUE;
				continue;
			}
			else if (strncmp(buf, "# Disc length", 13) == 0) {
				i = sscanf(buf, "# Disc length: %u seconds\n",
					   &trkaddr);
				if (i > 0) {
					trk_off[ntrk] =
						(trkaddr * FRAME_PER_SEC) -
						s->trkinfo[ntrk].addr -
						MSF_OFFSET;
				}
				else {
					/* File format error */
					ntrk = 0;
				}
				found = FALSE;
				continue;
			}

			if (found &&
			    (i = sscanf(buf, "# %u\n", &trkaddr)) > 0) {
				trk_off[ntrk] =
					trkaddr - s->trkinfo[ntrk].addr -
					MSF_OFFSET;
				ntrk++;
			}

			/* Look for disk title */
			if (strncmp(buf, "DTITLE=", 7) == 0) {
				/* Eat newline */
				i = strlen(buf) - 1;
				if (buf[i] == '\n')
					buf[i] = '\0';
				
				/* Check whether a valid offset can be
				 * calculated (compare # of tracks).
				 */
				if (ntrk == (int) s->tot_trks) {
					/* Compute the average block
					 * number difference per track.
					 */
					offset = 0;
					for (i = 0; i <= ntrk; i++) {
						if (trk_off[i] < 0)
						    trk_off[i] = -trk_off[i];
						offset += trk_off[i];
					}
					offset /= ntrk;
				}
				else {
					/* Track offsets not specified or
					 * not valid in database file.
					 */
					offset = (word32_t) OFFSET_UNKN;
				}

				/* Add to list in sorted order */
				(void) cddb_add_linkent(buf + 7, de->d_name,
							offset);
				break;
			}
		}

		(void) fclose(fp);
	}
	(void) CLOSEDIR(dp);

	MEM_FREE(buf);
	return TRUE;
}


/*
 * cddb_free_linkopts
 *	Dismantle the sorted linked list of track titles created by
 *	cddb_init_linkopts.
 *
 * Args:
 *	Nothing.
 *
 * Return:
 *	Nothing.
 */
void
cddb_free_linkopts(void)
{
	cddb_linkopts_t	*p,
			*q;

	for (p = q = cddb_linkhead; p != NULL; p = q) {
		q = p->next;
		if (p->dtitle != NULL)
			MEM_FREE(p->dtitle);
		MEM_FREE((char *) p);
	}
	cddb_linkhead = NULL;
}


/*
 * cddb_linkopts
 *	Return a pointer to the CDDB linkopts list.
 *
 * Args:
 *	None.
 *
 * Return:
 *	A pointer to the linkopts list head, or NULL if the list is empty.
 */
cddb_linkopts_t *
cddb_linkopts(void)
{
	return (cddb_linkhead);
}


/*
 * cddb_rmt_support
 *	Returns information whether this cddb library supports remote
 *	CDDB servers.
 *
 * Args:
 *	None.
 *
 * Return:
 *	TRUE  - Remote CDDB server supported
 *	FALSE - Remote CDDB server not supported
 */
bool_t
cddb_rmt_support(void)
{
	cddb_path_t	*pp;

	if (app_data.cddb_rmtdsbl)
		return FALSE;

	for (pp = cddb_pathhead; pp != NULL; pp = pp->next) {
		if (pp->type == CDDB_REMOTE_CDDBP ||
		    pp->type == CDDB_REMOTE_HTTP)
			return TRUE;
	}
	return FALSE;
}


/*
 * cddb_curfileupd
 *	Update the current disc info file.
 *
 * Args:
 *	dbp - Pointer to the cddb_incore_t structure.
 *
 * Return:
 *	Nothing
 */
void
cddb_curfileupd(cddb_incore_t *dbp)
{
#ifndef __VMS
	FILE		*fp;
	struct stat	stbuf;
	char		*dtitle,
			*ttitle,
			str[FILE_PATH_SZ * 3];
	static char	prev[FILE_PATH_SZ * 3],
			prev_dtitle[STR_BUF_SZ],
			prev_ttitle[STR_BUF_SZ];
	curstat_t	*s;
	bool_t		playing,
			changed;

	if (!app_data.write_curfile ||			/* disabled */
	    strncmp(app_data.device, "(sim", 4) == 0)	/* demo */
		return;

	if (cddb_clinfo->curstat_addr == NULL)
		return;

	s = cddb_clinfo->curstat_addr();

	if (!s->devlocked)
		/* Don't write curfile if we don't own the device */
		return;

	playing = FALSE;
	dtitle = ttitle = NULL;

	if (s->mode != MOD_NODISC) {
		char	modestr[24];

		dtitle = dbp->dtitle;

		switch (s->mode) {
		case MOD_PLAY:
		case MOD_PAUSE:
		case MOD_A:
		case MOD_AB:
		case MOD_SAMPLE:
			playing = TRUE;

			(void) sprintf(modestr, "Playing track %d",
				       s->cur_trk);

			if (s->cur_trk > 0) {
				int	n;

				n = (int) s->cur_trk - 1;

				if (s->trkinfo[n].trkno != s->cur_trk) {
					for (n = 0; n < MAXTRACK; n++) {
						if (s->trkinfo[n].trkno ==
						    s->cur_trk)
							break;
					}
				}

				ttitle = dbp->trklist[n];
			}
			break;

		case MOD_STOP:
			(void) strcpy(modestr, "Stopped");
			break;

		default:
			modestr[0] = '\0';
			break;
		}

		(void) sprintf(str,
			       "%s\t\t%s\n%s\t%s\n%s\t\t%08x\n%s\t\t%s\n",
			       "Device:", s->curdev == NULL ? "-" : s->curdev,
			       "Category:",
			       dbp->category[0] == '\0' ?
					"<unknown>" : dbp->category,
			       "DiscID:", dbp->discid,
			       "Status:", modestr);
	}
	else {
		(void) sprintf(str,
			       "%s\t\t%s\n%s\t%s\n%s\t\t%s\n%s\t\t%s\n",
			       "Device:", s->curdev == NULL ? "-" : s->curdev,
			       "Category:", "-",
			       "DiscID:", "-",
			       "Status:", "No_Disc");
	}

	changed = FALSE;

	if (playing) {
		if (ttitle == NULL && prev_ttitle[0] != '\0')
			changed = TRUE;
		else if (ttitle != NULL &&
			 strncmp(ttitle, prev_ttitle, STR_BUF_SZ - 1) != 0)
			changed = TRUE;
	}

	if (s->mode != MOD_NODISC) {
		if (dtitle == NULL && prev_dtitle[0] != '\0')
			changed = TRUE;
		else if (dtitle != NULL && 
			 strncmp(dtitle, prev_dtitle, STR_BUF_SZ - 1) != 0)
			changed = TRUE;
	}

	if (strcmp(str, prev) != 0)
		changed = TRUE;

	if (!changed)
		return;

	/* Write to file */
	if (curfile[0] == '\0') {
		if (stat(app_data.device, &stbuf) < 0) {
			DBGPRN(errfp, "\nCannot stat %s\n", app_data.device);
			return;
		}

		(void) sprintf(curfile, "%s/curr.%x",
			       TEMP_DIR, (int) stbuf.st_rdev);
	}

	/* Remove original file */
	if (UNLINK(curfile) < 0 && errno != ENOENT) {
		DBGPRN(errfp, "\nCannot unlink old %s\n", curfile);
		return;
	}

	/* Write new file */
	if ((fp = fopen(curfile, "w")) == NULL) {
		DBGPRN(errfp, "\nCannot open %s for writing\n", curfile);
		return;
	}

	DBGPRN(errfp, "\nWriting current disc info file: %s\n", curfile);

	(void) fprintf(fp, "#\n# Xmcd current CD information\n#\n%s", str);

	if (s->mode != MOD_NODISC) {
		(void) fprintf(fp, "Disc:\t\t%s\n",
			       dtitle == NULL ?
			       "<unknown disc title>" : dtitle);
	}
	else {
		(void) fprintf(fp, "Disc:\t\t-\n");
	}

	if (playing) {
		(void) fprintf(fp, "Track:\t\t%s\n",
			       ttitle == NULL ?
			       "<unknown track title>" : ttitle);
	}
	else {
		(void) fprintf(fp, "Track:\t\t-\n");
	}

	(void) fclose(fp);
	(void) chmod(curfile, 0644);
	
	if (ttitle == NULL)
		prev_ttitle[0] = '\0';
	else {
		(void) strncpy(prev_ttitle, ttitle, STR_BUF_SZ - 1);
		prev_ttitle[STR_BUF_SZ - 1] = '\0';
	}

	if (dtitle == NULL)
		prev_dtitle[0] = '\0';
	else {
		(void) strncpy(prev_dtitle, dtitle, STR_BUF_SZ - 1);
		prev_dtitle[STR_BUF_SZ - 1] = '\0';
	}

	(void) strcpy(prev, str);
#endif	/* __VMS */
}


/*
 * cddb_issync
 *	Returns whether the CDDB subsystem is running in SYNCHRONOUS mode.
 *
 * Args:
 *	None.
 *
 * Return:
 *	TRUE if SYNCHRONOUS
 *	FALSE if not SYNCHRONOUS
 */
bool_t
cddb_issync(void)
{
#ifdef SYNCHRONOUS
	return TRUE;
#else
	return FALSE;
#endif
}


/*
 * cddb_dump_incore
 *	Displays the contents of the cddb_incore_t structure.
 *
 * Args:
 *	dbp - Pointer to the cddb_incore_t structure.
 *	s - Pointer to the curstat_t structure.
 *
 * Returns:
 *	Nothing.
 */
void
cddb_dump_incore(cddb_incore_t *dbp, curstat_t *s)
{
	int		i;
	cddb_match_t	*p;
	cddb_linkopts_t	*q;

	(void) fprintf(errfp,
		       "\nDumping the cddb_incore_t structure at 0x%lx:\n",
		       (unsigned long) dbp);
	(void) fprintf(errfp, "discid=%08x queryid=%08x type=%d flags=0x%x\n",
		       dbp->discid, dbp->queryid, dbp->type, dbp->flags);
	(void) fprintf(errfp, "revision=%d dbfile=%s category=%s\n",
		       dbp->revision,
		       dbp->dbfile == NULL ? "NULL" : dbp->dbfile,
		       dbp->category);
	(void) fprintf(errfp, "dtitle=%s\n",
		       dbp->dtitle == NULL ? "NULL" : dbp->dtitle);

	for (i = 0; i < (int) s->tot_trks; i++) {
		(void) fprintf(errfp, "trklist[%d]=%s\n", i,
			       dbp->trklist[i] == NULL ?
				    "NULL" : dbp->trklist[i]);
	}

	if (dbp->extd == NULL)
		(void) fprintf(errfp, "extd=NULL\n");
	else
		(void) fprintf(errfp, "extd=\n%s\n", dbp->extd);

	for (i = 0; i < (int) s->tot_trks; i++) {
		if (dbp->extt[i] == NULL)
			(void) fprintf(errfp, "extt[%d]=NULL\n", i);
		else
			(void) fprintf(errfp, "extt[%d]=\n%s\n",
				       i, dbp->extt[i]);
	}

	for (i = 0; i < (int) s->tot_trks; i++) {
		if (dbp->sav_extt[i] == NULL)
			(void) fprintf(errfp, "sav_extt[%d]=NULL\n", i);
		else
			(void) fprintf(errfp, "sav_extt[%d]=\n%s\n", i,
				dbp->sav_extt[i]);
	}

	(void) fprintf(errfp, "playorder=%s\n",
		       dbp->playorder == NULL ? "NULL" : dbp->playorder);

	if (dbp->idlist == NULL)
		(void) fprintf(errfp, "idlist=NULL\n");
	else {
		for (q = dbp->idlist, i = 0; q != NULL; q = q->next, i++) {
			(void) fprintf(errfp, "idlist[%d] ptr=0x%lx id=%s\n",
				       i, (unsigned long) q, q->idstr);
		}
	}

	if (dbp->matchlist == NULL)
		(void) fprintf(errfp, "matchlist=NULL\n");
	else {
		for (p = dbp->matchlist, i = 0; p != NULL; p = p->next, i++) {
			(void) fprintf(errfp,
				       "matchlist[%d] ptr=0x%lx categ=%s ",
				       i, (unsigned long) p, p->category);
			(void) fprintf(errfp, "discid=%08x dtitle=%s\n",
				       p->discid, p->dtitle);
		}
	}
	if (dbp->match_cur == NULL)
		(void) fprintf(errfp, "match_cur=NULL\n");
	else
		(void) fprintf(errfp, "match_cur=0x%lx\n",
			       (unsigned long) dbp->match_cur);
}


