/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*-  vi:set ts=8 sts=4 sw=4: */

#include "soundcardselect.h"

#include <QString>
#include <QDialog>
#include <QFrame>
#include <QLabel>
#include <QApplication>
#include <QProcess>
#include <QTextStream>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QGroupBox>
#include <QRadioButton>
#include <QCheckBox>
#include <QComboBox>
#include <QSpacerItem>
#include <QFont>
#include <QPushButton>
#include <QToolButton>
#include <QSettings>
#include <QIntValidator>

#include <iostream>


std::vector<int>
SoundcardSelector::m_sampleRates;

int
SoundcardSelector::m_defaultSampleRate = 48000;

SoundcardSelector::SoundcardSelector(QWidget *parent, bool startup, bool showOptions) :
    QDialog(parent, Qt::Dialog | Qt::WindowStaysOnTopHint),
    m_audioRow(0),
    m_midiRow(0),
    m_havePreviousPreference(false),
    m_previousPreferenceValid(false),
    m_jackOption(JackAuto),
    m_latencyOption(LatencyAuto),
    m_sampleRate(m_defaultSampleRate),
    m_useSecondary(true),
    m_startSynth(true)
{ 
    setWindowTitle(tr("Studio to Go!"));

    if (m_sampleRates.empty()) {
        m_sampleRates.push_back(44100);
        m_sampleRates.push_back(48000);
        m_sampleRates.push_back(88200);
        m_sampleRates.push_back(96000);
        m_sampleRates.push_back(192000);
    }

    QSettings settings;
    settings.beginGroup("Options");
    m_rememberedCard = settings.value("selectedcard", "").toString();
    m_havePreviousPreference = (m_rememberedCard != "");
    m_jackOption = (JackOption)(settings.value("jackserver", 0).toInt());
    m_initialJackOption = m_jackOption;
    m_latencyOption = (LatencyOption)(settings.value("latencypreference", 0).toInt());
    m_initialLatencyOption = m_latencyOption;
    m_sampleRate = settings.value("samplerate", m_defaultSampleRate).toInt();
    m_initialSampleRate = m_sampleRate;
    m_useSecondary = settings.value("secondary", true).toBool();
    m_initialUseSecondary = m_useSecondary;
    m_startSynth = settings.value("startsynth", true).toBool();
    m_initialStartSynth = m_startSynth;

    QVBoxLayout *vbox = new QVBoxLayout;
    setLayout(vbox);

    QGroupBox *groupBox = new QGroupBox(tr("Audio cards"));
    vbox->addWidget(groupBox);
    m_audioLayout = new QGridLayout;
    groupBox->setLayout(m_audioLayout);
    m_audioLayout->setColumnStretch(2, 10);

    m_advanced = new QFrame;
    QVBoxLayout *advancedLayout = new QVBoxLayout;
    advancedLayout->setMargin(0);
    m_advanced->setLayout(advancedLayout);
    vbox->addWidget(m_advanced);

    groupBox = new QGroupBox(tr("MIDI interfaces"));
    advancedLayout->addWidget(groupBox);
    m_midiLayout = new QGridLayout;
    groupBox->setLayout(m_midiLayout);
    m_midiLayout->setColumnStretch(1, 10);

    if (showOptions) {

        groupBox = new QGroupBox(tr("Advanced options"));
        advancedLayout->addWidget(groupBox);
        QGridLayout *optionLayout = new QGridLayout;
        groupBox->setLayout(optionLayout);

        optionLayout->addWidget(new QLabel(tr("JACK audio server:")), 0, 0);
        QComboBox *jackOption = new QComboBox;
        jackOption->addItem(tr("(auto)"));
        jackOption->addItem(tr("jackd"));
        jackOption->addItem(tr("jackdmp"));
        jackOption->setCurrentIndex((int)m_jackOption + 1);
        optionLayout->addWidget(jackOption, 0, 1);
        connect(jackOption, SIGNAL(currentIndexChanged(int)),
                this, SLOT(jackOptionChanged(int)));
        jackOption->setCurrentIndex((int)m_jackOption);

        optionLayout->addWidget(new QLabel(tr("Sample rate:")), 2, 0);
        QComboBox *srOption = new QComboBox;
        bool found = false;
        srOption->addItem(tr("(auto)"));
        for (int i = 0; i < m_sampleRates.size(); ++i) {
            srOption->addItem(QString("%1 Hz").arg(m_sampleRates[i]));
            if (m_sampleRates[i] == m_sampleRate) {
                if (m_sampleRates[i] == m_defaultSampleRate) {
                    srOption->setCurrentIndex(0);
                } else {
                    srOption->setCurrentIndex(i);
                }
                found = true;
            }
        }
        if (!found) {
            srOption->addItem(QString("%1 Hz").arg(m_sampleRate));
            srOption->setCurrentIndex(m_sampleRates.size() + 1);
        }

        srOption->setEditable(true);
        srOption->setValidator(new QIntValidator(8000, 1024000, srOption));
        connect(srOption, SIGNAL(activated(QString)),
                this, SLOT(sampleRateChanged(QString)));
        optionLayout->addWidget(srOption, 2, 1);

        QCheckBox *useSecondary = new QCheckBox
            (tr("Make additional cards appear to JACK as secondary cards"));
        optionLayout->addWidget(useSecondary, 3, 0, 1, 3);
        useSecondary->setCheckState(m_useSecondary ? Qt::Unchecked : Qt::Checked);
        connect(useSecondary, SIGNAL(stateChanged(int)),
                this, SLOT(useSecondaryChanged(int)));
        useSecondary->setCheckState(m_useSecondary ? Qt::Checked : Qt::Unchecked);

        optionLayout->addWidget(new QLabel(tr("Latency preference:")), 1, 0);
        QComboBox *latencyOption = new QComboBox;
        latencyOption->addItem(tr("(auto)"));
        latencyOption->addItem(tr("Lower"));
        latencyOption->addItem(tr("Higher"));
        latencyOption->setCurrentIndex((int)m_latencyOption + 1);
        optionLayout->addWidget(latencyOption, 1, 1);
        connect(latencyOption, SIGNAL(currentIndexChanged(int)),
                this, SLOT(latencyOptionChanged(int)));
        latencyOption->setCurrentIndex((int)m_latencyOption);

        QCheckBox *startSynth = new QCheckBox
            (tr("Start a software MIDI synth"));
        optionLayout->addWidget(startSynth, 4, 0, 1, 3);
        startSynth->setCheckState(m_startSynth ? Qt::Unchecked : Qt::Checked);
        connect(startSynth, SIGNAL(stateChanged(int)),
                this, SLOT(startSynthChanged(int)));
        startSynth->setCheckState(m_startSynth ? Qt::Checked : Qt::Unchecked);
    }

    m_audioLayout->setSpacing(2);
    m_audioLayout->setMargin(8);
    m_midiLayout->setSpacing(2);
    m_midiLayout->setMargin(8);

    m_audioLayout->addItem(new QSpacerItem(1, 4), m_audioRow, 0);
    ++m_audioRow;

    m_audioIntro = new QLabel;
    m_audioLayout->addWidget(m_audioIntro, m_audioRow, 0, 1, 3);
    ++m_audioRow;

    m_midiIntro = new QLabel;
    m_midiLayout->addWidget(m_midiIntro, m_midiRow, 0, 1, 3);
    ++m_midiRow;

    QHBoxLayout *hbox = new QHBoxLayout;
    vbox->addLayout(hbox);

    m_advancedVisible = false;

    m_advancedButton = new QPushButton(tr("More >>"));
    m_advancedButton->setCheckable(true);
    connect(m_advancedButton, SIGNAL(clicked()), this, SLOT(advancedToggled()));
        
//    m_advanced->setVisible(false);

    hbox->addWidget(m_advancedButton);
//    m_advancedButton->hide();

    hbox->addStretch(10);

    QPushButton *help = new QPushButton(tr("Help"));
    hbox->addWidget(help);
    connect(help, SIGNAL(clicked()), this, SLOT(help()));

    QPushButton *ok = new QPushButton(tr("OK"));
    hbox->addWidget(ok);
    connect(ok, SIGNAL(clicked()), this, SLOT(accept()));
    m_okButton = ok;

    if (!startup) {
	QPushButton *cancel = new QPushButton(tr("Cancel"));
	hbox->addWidget(cancel);
	connect(cancel, SIGNAL(clicked()), this, SLOT(reject()));
    }

    ok->setDefault(true);
    ok->setFocus(Qt::OtherFocusReason);
}

SoundcardSelector::~SoundcardSelector()
{
}

std::vector<SoundcardData>
SoundcardSelector::getSecondarySoundcards() const
{
    std::vector<SoundcardData> cards;
    for (size_t i = 0; i < m_audioCards.size(); ++i) {
        if (m_audioCards[i].driverClass == m_selectedCard.driverClass &&
            m_audioCards[i].id == m_selectedCard.id) continue;
        if (m_audioCards[i].driverClass == "freebob") continue;
        cards.push_back(m_audioCards[i]);
    }
    return cards;
}

void
SoundcardSelector::addSoundcard(SoundcardData data)
{
    QString cardClass = data.cardClass;
    if (cardClass == "pro") cardClass = tr("Professional or recording soundcard");
    else if (cardClass == "integrated") cardClass = tr("Integrated sound chip");
    else if (cardClass == "consumer") cardClass = tr("Enhanced consumer soundcard");
    else if (cardClass == "usb") cardClass = tr("USB soundcard");
    else if (cardClass == "firewire") cardClass = tr("FireWire soundcard");
    else if (cardClass == "unknown") cardClass = tr("Soundcard");
    else {
        cardClass = QString("%1 soundcard")
            .arg(cardClass.left(1).toUpper() + 
                 cardClass.right(cardClass.length() - 1));
    }

    QString objName = QString("%1:%2").arg(data.driverClass).arg(data.id);

    QLabel *label = new QLabel;
    QToolButton *button = new QToolButton;

    button->setCheckable(true);

    if (m_audioCards.empty()) {
        m_audioLayout->addItem(new QSpacerItem(1, 6), m_audioRow, 0);
        m_audioRow++;
        m_selectedCard = data;
        if (m_rememberedCard == "") {
            // so we can compare to see whether anything has changed later
            m_rememberedCard = data.name;
        } else if (m_rememberedCard == data.name) {
            m_previousPreferenceValid = true;
        }
    } else if (m_rememberedCard != "" &&
               m_rememberedCard == data.name) {
        m_selectedCard = data;
        m_previousPreferenceValid = true;
    }

    QLabel *separator = new QLabel;
    separator->setFixedHeight(1);
    separator->setScaledContents(true);
    QPixmap map(1, 1);
    map.fill(separator->palette().color(QPalette::Dark));
    separator->setPixmap(map);
    m_audioLayout->addWidget(separator, m_audioRow, 0, 1, 3);
    m_audioRow++;
    m_audioLayout->addItem(new QSpacerItem(1, 6), m_audioRow, 0);
    m_audioRow++;

    button->setObjectName(objName);
    connect(button, SIGNAL(clicked()), this, SLOT(audioCardSelected()));

//    button->setAlignment(Qt::AlignCenter);
    button->setFixedWidth(53);

    m_audioCardButtons.push_back(button);

    m_audioLayout->addWidget(button, m_audioRow, 0, 2, 1);

//    QToolButton *button = new QToolButton;
    button = new QToolButton;

    button->setObjectName(objName);
    connect(button, SIGNAL(clicked()), this, SLOT(audioCardSelected()));

    button->setText(data.name);
    QFont fn = button->font();
    fn.setBold(true);
    button->setFont(fn);

    button->setToolButtonStyle(Qt::ToolButtonTextOnly);
    button->setAutoRaise(true);

    m_audioLayout->addWidget(button, m_audioRow, 1, 1, 2);

    m_toResize.push_back(button);

    ++m_audioRow;

    m_audioLayout->addItem(new QSpacerItem(8, 1), m_audioRow, 1);

    label = new QLabel;
    label->setText(cardClass);
//    label->setAlignment(Qt::AlignHCenter | Qt::AlignTop);
    m_audioLayout->addWidget(label, m_audioRow, 2);

    ++m_audioRow;

    m_audioLayout->addItem(new QSpacerItem(1, 6), m_audioRow, 0);
    ++m_audioRow;
    
    m_audioCards.push_back(data);
}

void
SoundcardSelector::done()
{
    updateButtons();

    m_audioLayout->addItem(new QSpacerItem(1, 10), m_audioRow, 0);
    m_audioRow++;

    if (m_audioCards.empty()) {
        m_audioIntro->setText(tr("<b>No audio cards have been detected.</b><br><br>Either no compatible audio hardware is installed,<br>or autodetection or driver initialisation failed.<br><br>Please consult the Studio to Go! documentation<br>for information about supported audio hardware,<br>or contact Fervent Software for support."));
    } else if (m_audioCards.size() > 1) {
        m_audioIntro->setText(tr("The following audio cards have been found.\nPlease select your preferred primary audio card:"));
    } else {
        m_audioIntro->setText(tr("The following audio card will be configured:"));
    }   

    if (m_midiCards.empty()) {
        m_midiIntro->setText(tr("No compatible MIDI interfaces have been detected."));
        m_midiLayout->addItem(new QSpacerItem(1, 10), m_audioRow, 0);
        m_midiRow++;
    } else if (m_midiCards.size() > 1) {
        m_midiIntro->setText(tr("The following MIDI interfaces have been configured:"));
    } else {
        m_midiIntro->setText(tr("The following MIDI interface has been configured:"));
    }        

    setAdvancedVisible(m_advancedVisible);
}    

void
SoundcardSelector::audioCardSelected()
{
    QObject *obj = sender();
    if (!obj) return;

    QString name = obj->objectName();
    QStringList bits = name.split(':');

    if (bits.size() < 2) return;

    QString driverClass = bits[0];

    bool ok = false;
    int id = bits[1].toInt(&ok);
    if (!ok) return;

//    std::cerr << "driver class " << driverClass.toStdString() << " id " << id << " selected" << std::endl;

    for (size_t i = 0; i < m_audioCards.size(); ++i) {
        if (m_audioCards[i].driverClass == driverClass &&
            m_audioCards[i].id == id) {
            m_selectedCard = m_audioCards[i];
            break;
        }
    }

    updateButtons();
}

bool
SoundcardSelector::havePreviousPreference() const
{
    return m_havePreviousPreference && m_previousPreferenceValid;
}

bool
SoundcardSelector::changed() const
{
    if (m_selectedCard.name != m_rememberedCard) return true;
    if (m_jackOption != m_initialJackOption) return true;
    if (m_latencyOption != m_initialLatencyOption) return true;
    if (m_useSecondary != m_initialUseSecondary) return true;
    if (m_startSynth != m_initialStartSynth) return true;
    if (m_sampleRate != m_initialSampleRate) return true;
    return false;
}

SoundcardSelector::JackOption
SoundcardSelector::getJackOption() const
{
    return m_jackOption;
}

SoundcardSelector::LatencyOption
SoundcardSelector::getLatencyOption() const
{
    return m_latencyOption;
}

int
SoundcardSelector::getSampleRate() const
{
    return m_sampleRate;
}

bool
SoundcardSelector::getUseSecondaryCards() const
{
    return m_useSecondary;
}

bool
SoundcardSelector::getStartSynth() const
{
    return m_startSynth;
}

void
SoundcardSelector::jackOptionChanged(int opt)
{
    QObject *s = sender();
    QWidget *w = dynamic_cast<QWidget *>(s);

    if ((JackOption)opt != m_jackOption) {
        m_jackOption = (JackOption)opt;
        updateButtons();
    }
    
    if (!w) return;

    switch (m_jackOption) {

    case JackAuto:
        w->setToolTip(tr("The JACK audio server to use will be chosen automatically by Studio to Go!.\nThe jackd server will normally be used with FireWire cards, and the jackdmp\nserver with most other cards."));
        break;

    case Jackd:
        w->setToolTip(tr("The official reference JACK audio server will be used."));
        break;
       
    case Jackdmp:
        w->setToolTip(tr("The jackdmp multiprocessor-optimised JACK audio server will be used."));
        break;
    }
}

void
SoundcardSelector::sampleRateChanged(QString str)
{
    int rate = 0;
    if (str == tr("(auto)")) rate = m_defaultSampleRate;
    else rate = str.toInt();

    if (rate == 0) return;

    m_sampleRate = rate;
}

void
SoundcardSelector::latencyOptionChanged(int opt)
{
    QObject *s = sender();
    QWidget *w = dynamic_cast<QWidget *>(s);

    if ((LatencyOption)opt != m_latencyOption) {
        m_latencyOption = (LatencyOption)opt;
    }
    
    if (!w) return;

    switch (m_latencyOption) {

    case LatencyAuto:
        w->setToolTip(tr("JACK buffer settings will be chosen by Studio to Go!.\nLower latency settings will normally be used for pro audio cards or when running\nfrom hard disc, and longer buffers for consumer cards or when running from CD-ROM."));
        break;

    case LowLatency:
        w->setToolTip(tr("Buffer settings will be chosen for low latency operation."));
        break;
       
    case ConservativeLatency:
        w->setToolTip(tr("Buffer settings will be chosen for the most reliable operation, at the\nexpense of possibly higher latency."));
        break;
    }
}

void
SoundcardSelector::useSecondaryChanged(int opt)
{
    m_useSecondary = (opt == Qt::Checked);
    updateButtons();
}

void
SoundcardSelector::startSynthChanged(int opt)
{
    m_startSynth = (opt == Qt::Checked);
}

void
SoundcardSelector::updateButtons()
{
    for (size_t i = 0; i < m_audioCards.size(); ++i) {

        QPixmap pixmap;
        QToolButton *button = m_audioCardButtons[i];

        if (m_audioCards[i].driverClass == m_selectedCard.driverClass &&
            m_audioCards[i].id == m_selectedCard.id) {

            if (m_jackOption == Jackdmp && 
                m_audioCards[i].driverClass == "freebob") {

                pixmap = QPixmap(":/audio-cross.png");
                button->setToolTip
                    (tr("This card will not be available for use.\nFireWire cards cannot be used with the jackdmp server.\nPlease change the advanced option for the JACK server to \"Auto\" to use this card."));
                button->setChecked(false);

            } else {

                pixmap = QPixmap(":/audio-tick.png");
                button->setToolTip
                    (tr("This card will be your primary soundcard for audio applications."));
                button->setChecked(true);
            }

        } else {

            if (m_useSecondary) {

                if (m_audioCards[i].driverClass == "freebob") {
                    pixmap = QPixmap(":/audio-cross.png");
                    button->setToolTip
                        (tr("This card will not be available for use, unless you\nmake it your primary soundcard.\nFireWire cards cannot be used as secondary soundcards."));
                } else {
                    pixmap = QPixmap(":/audio-small.png");
                    button->setToolTip
                        (tr("This card will be available as a secondary soundcard."));
                }

            } else {

                pixmap = QPixmap(":/audio-cross.png");

                if (m_audioCards[i].driverClass == "freebob") {
                    button->setToolTip
                        (tr("This card will not be available for use."));
                } else {
                    button->setToolTip
                        (tr("This card will not be available for use by JACK audio applications.\nPlease enable secondary soundcards in the advanced options\nif you wish to make this card available as a secondary card."));
                }
            }

            button->setChecked(false);
        }

        button->setIcon(pixmap);
        button->setIconSize(pixmap.size());
        button->setFixedSize(pixmap.size() + QSize(20, 20));
    }
}

void
SoundcardSelector::setAdvancedVisible(bool visible)
{
    m_advanced->setVisible(visible);

    if (visible) {
        m_advancedButton->setText(tr("More <<"));
        m_advancedButton->setChecked(true);
    } else {
        m_advancedButton->setText(tr("More >>"));
        m_advancedButton->setChecked(false);
    }

//    setMinimumHeight(sizeHint().height());
    setFixedSize(sizeHint());
    adjustSize();

    m_advancedVisible = visible;
    m_okButton->setDefault(true);
    m_okButton->setFocus(Qt::OtherFocusReason);
}

void
SoundcardSelector::advancedToggled()
{
    setAdvancedVisible(!m_advancedVisible);
}

void
SoundcardSelector::help()
{
    QString url(tr("file:///usr/share/doc/stg/SoundcardSelector.html"));

    QProcess *process = new QProcess(this);
    connect(process, SIGNAL(finished(int)), process, SLOT(deleteLater()));

    QStringList args;

    if (!qgetenv("KDE_FULL_SESSION").isEmpty()) {
        args.append("exec");
        args.append(url);
        process->start("kfmclient", args);
    } else if (!qgetenv("BROWSER").isEmpty()) {
        args.append(url);
        process->start(qgetenv("BROWSER"), args);
    } else {
        args.append(url);
        process->start("firefox", args);
    }
}    

void
SoundcardSelector::addMIDIInterface(SoundcardData data)
{
    QLabel *label;

    m_midiLayout->setSpacing(6);
    m_midiLayout->setMargin(12);

    if (m_midiCards.empty()) {
        QLabel *separator = new QLabel;
        separator->setFixedHeight(1);
        separator->setScaledContents(true);
        QPixmap map(1, 1);
        map.fill(separator->palette().color(QPalette::Dark));
        separator->setPixmap(map);
        m_midiLayout->addWidget(separator, m_midiRow, 0, 1, 2);
        m_midiRow++;
        m_midiLayout->addItem(new QSpacerItem(1, 6), m_midiRow, 0);
        m_midiRow++;
    }
        
    label = new QLabel;
    label->setPixmap(QPixmap(":/midi-small.png"));
    label->setAlignment(Qt::AlignCenter);
    label->setFixedWidth(60);
    label->setToolTip(tr("This MIDI interface will be enabled and available for use."));
    m_midiLayout->addWidget(label, m_midiRow, 0);

    label = new QLabel(data.name);
//    label->setAlignment(Qt::AlignHCenter | Qt::AlignTop);
    QFont fn = label->font();
    fn.setBold(true);
    label->setFont(fn);
    m_midiLayout->addWidget(label, m_midiRow, 1);

    ++m_midiRow;

    m_midiCards.push_back(data);
}


int main(int argc, char **argv)
{
    QApplication application(argc, argv);
 
    bool startup = false;
    for (int i = 1; i < argc; ++i) {
	if (!strcmp(argv[i], "--startup") || !strcmp(argv[i], "-s")) {
	    startup = true;
	    break;
	}
    }

    QApplication::setOrganizationName("Fervent Software");
    QApplication::setOrganizationDomain("studio-to-go.com");
    QApplication::setApplicationName("Soundcard Selector");

    QApplication::setWindowIcon(QIcon(":/audio-small.png"));

    QFont fn = application.font();
    fn.setPointSize(fn.pointSize() + 3);
    application.setFont(fn);

    SoundcardSelector selector(0, startup, true);

    QProcess identifier;
    identifier.start("identifysoundcards");

    if (!identifier.waitForStarted()) {
        std::cerr << "ERROR: SoundcardSelector: identifysoundcards failed to start" << std::endl;
        return 1;
    }
    
    if (!identifier.waitForFinished()) {
        std::cerr << "ERROR: SoundcardSelector: identifysoundcards failed" << std::endl;
        return 1;
    }

    QTextStream cardstream(&identifier);
    while (!cardstream.atEnd()) {
        QString line = cardstream.readLine();
//        std::cerr << "read line: \"" << line.toStdString() << "\"" << std::endl;
        QStringList list = line.split(':');
        if (list.size() < 7) continue;
        QString audioTag = list[3];
        QString midiTag = list[4];
        SoundcardData data(list[0], list[1].toInt(), list[2], list[5], list[6]);
        if (data.cardClass == "modem") continue;
        if (audioTag == "audio") {
            selector.addSoundcard(data);
        }
        if (midiTag == "midi" && (data.name != "")) {
            selector.addMIDIInterface(data);
        }
    }

    selector.done();

    if (!startup || !selector.havePreviousPreference()) {
        
        if (selector.exec() != QDialog::Accepted) {
            return 1;
        }

        if (!selector.haveSoundcard()) {
            std::cout << "server:none"  << std::endl;
            return 0;
        }
    }

    SoundcardData selectedCard = selector.getSelectedSoundcard();

    QSettings settings;
    settings.beginGroup("Options");
    settings.setValue("selectedcard", selectedCard.name);
    settings.setValue("startsynth", selector.getStartSynth());
    settings.setValue("secondary", selector.getUseSecondaryCards());
    settings.setValue("jackserver", (int)selector.getJackOption());
    settings.setValue("latencypreference", (int)selector.getLatencyOption());
    settings.setValue("samplerate", selector.getSampleRate());

    std::cout << "changed:" << (selector.changed() ? "true" : "false") << std::endl;

    switch (selector.getJackOption()) {

    case SoundcardSelector::JackAuto:
        if (selectedCard.driverClass == "freebob") {
            std::cout << "server:jackd" << std::endl;
        } else {
            std::cout << "server:jackdmp" << std::endl;
        }
        break;

    case SoundcardSelector::Jackd:
        std::cout << "server:jackd" << std::endl;
        break;

    case SoundcardSelector::Jackdmp:
        std::cout << "server:jackdmp" << std::endl;
        break;
    }

    switch (selector.getLatencyOption()) {
        
    case SoundcardSelector::LatencyAuto:
        std::cout << "latency:auto" << std::endl;
        break;
        
    case SoundcardSelector::LowLatency:
        std::cout << "latency:low" << std::endl;
        break;
        
    case SoundcardSelector::ConservativeLatency:
        std::cout << "latency:conservative" << std::endl;
        break;
    }

    std::cout << "samplerate:" << selector.getSampleRate() << std::endl;

    if (selector.getStartSynth()) {
        std::cout << "startup:synth" << std::endl;
    }

    std::cout << "primary:" << selectedCard.driverClass.toStdString()
              << ":" << selectedCard.id << std::endl;

    if (selector.getUseSecondaryCards()) {

        std::vector<SoundcardData> secondaryCards = selector.getSecondarySoundcards();
        for (size_t i = 0; i < secondaryCards.size(); ++i) {
            std::cout << "secondary:" << secondaryCards[i].driverClass.toStdString()
                      << ":" << secondaryCards[i].id << std::endl;
        }
    }

    return 0;
}


