/*	$NetBSD: rdate.c,v 1.12 2001/02/19 23:22:47 cgd Exp $	*/

/*
 * Copyright (c) 1994 Christos Zoulas
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Christos Zoulas.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * rdate.c: Set the date from the specified host
 * 
 * 	Uses the rfc868 time protocol at socket 37.
 *	Time is returned as the number of seconds since
 *	midnight January 1st 1900.
 */
#include <sys/cdefs.h>
#ifndef lint
__RCSID("$NetBSD: rdate.c,v 1.12 2001/02/19 23:22:47 cgd Exp $");
#endif/* lint */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>

#include <netinet/in.h>

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <langinfo.h>
#include <locale.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <utmp.h>

/* seconds from midnight Jan 1900 - 1970 */
#if __STDC__
#define DIFFERENCE 2208988800UL
#else
#define DIFFERENCE 2208988800
#endif

#define progname program_invocation_short_name

	int	main __P((int, char **));
static	void	usage __P((void));

static void
usage()
{
	(void) fprintf(stderr, "Usage: %s [-psau] host [port] \n", progname);
	(void) fprintf(stderr, "  -p: just print, don't set\n");
	(void) fprintf(stderr, "  -s: just set, don't print\n");
	(void) fprintf(stderr, "  -a: use adjtime instead of instant change\n");
	(void) fprintf(stderr, "  -u: use UDP instead of TCP as transport\n");
}

int
main(argc, argv)
	int             argc;
	char           *argv[];
{
	int             pr = 0, silent = 0, s;
	int		slidetime = 0;
	int		useudp = 0;
	int		adjustment;
	time_t          tim;
	int32_t		ttim;
	char           *hname, *emsg;
	struct addrinfo	hints, *res, *res0;
	int             c;
	int		error;
	const char     *service;

	setlocale(LC_ALL, "");

	adjustment = 0;
	service = "time";
	while ((c = getopt(argc, argv, "psau")) != -1)
		switch (c) {
		case 'p':
			pr++;
			break;

		case 's':
			silent++;
			break;

		case 'a':
			slidetime++;
			break;

		case 'u':
			useudp++;
			break;

		default:
			usage();
			return 1;
		}

	if (argc <= optind) {
		usage();
		return 1;
	}
	hname = argv[optind];
	if (argc > ++optind) {
		if (argc > optind + 1) {
			usage();
			return 1;
		}
		service = argv[optind];
	}

	memset(&hints, 0, sizeof (hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = useudp ? SOCK_DGRAM : SOCK_STREAM;
	hints.ai_flags = AI_CANONNAME;
	error = getaddrinfo(hname, service, &hints, &res0);
	if (error)
		errx(1, "%s: %s", gai_strerror(error), hname);

	res = res0;
	do {
		s = socket(res->ai_family, res->ai_socktype, res->ai_protocol);
		if (s < 0) {
			emsg = "socket";
			continue;
		}

		if (connect(s, res->ai_addr, res->ai_addrlen)) {
			close(s);
			s = -1;
			emsg = "connect";
			continue;
		}
		
		break;
	} while ((res = res->ai_next));
	if (s < 0)
		err(1, "%s", emsg);

	/* UDP requires us to send an empty datagram first */
	if (useudp)
		send(s, NULL, 0, 0);

	if (read(s, &ttim, sizeof(ttim)) != sizeof(ttim))
		err(1, "Could not read data");

	(void) close(s);
	tim = ntohl(ttim) - DIFFERENCE;

	if (!pr) {
	    struct timeval  tv;
	    if (!slidetime) {
		    logwtmp("|", "date", "");
		    tv.tv_sec = tim;
		    tv.tv_usec = 0;
		    if (settimeofday(&tv, NULL) == -1)
			    err(1, "Could not set time of day");
		    logwtmp("{", "date", "");
	    } else {
		    struct timeval tv_current;
		    if (gettimeofday(&tv_current, NULL) == -1)
			    err(1, "Could not get local time of day");
		    adjustment = tv.tv_sec = tim - tv_current.tv_sec;
		    tv.tv_usec = 0;
		    if (adjtime(&tv, NULL) == -1)
			    err(1, "Could not adjust time of day");
	    }
	}

	if (!silent) {
		char buf[256];
		struct tm *ltim;

		ltim = localtime(&tim);
		strftime(buf, sizeof(buf), nl_langinfo(D_T_FMT), ltim);
		(void) puts(buf);
		if (slidetime)
		    (void) fprintf(stdout, 
				   "%s: adjust local clock by %d seconds\n",
				   progname, adjustment);
	}
	return 0;
}
