#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <malloc.h>
#include <unistd.h>
#include <syslog.h>
#include <string.h>
#include <signal.h>

#include "module.h"
#include "version.h"
#include "util.h"
#include "modstat.h"

int sleep_interval = 300;
volatile int selfdestruct = 0;

void usage() {
    printf("Usage: modclean [-i seconds] [-V] [-h]\n"
	   "\n"
	   " -i sec       change sleep interval to sec seconds (default is 300)\n"
	   " -V           show version and exit\n"
	   " -h           show this text and exit\n");
}


void sighandler(int sig) {
    selfdestruct=1;
}


int main(int argc, char** argv) {
    size_t	modules_count;
    char*	module_names = NULL;
    int		i;

    while ((i=getopt(argc, argv, "i:V"))!=-1)
	switch (i) {
	    case 'i':
		{
		    char* end;
		    sleep_interval=strtol(optarg, &end, 10);
		    if (end[0]!=0) {
			fprintf(stderr, "invalid parameter for -i");
			return 1;
		    }
		}
		break;

	    case 'V':
		fprintf(stderr, "modclean version " MODUTILS_VERSION "\n");
		return 0;

	    default:
		fprintf(stderr, "Invalid option: %c", i);
		usage();
		return 1;
	}

    if (optind!=argc) {
	fprintf(stderr, "No parameters allowed\n");
	usage();
	return 1;
    }

    /* Check if the kernel supports modules */
    if ((query_module(NULL, QM_MODULES, NULL, 0, &modules_count)==-1) && (errno!=ENOSPC)) {
	printf("kernel does not support modules, not starting");
	return 0;
    }

    /* Register a signal handler for TERM */
    if (signal(SIGTERM, sighandler)==SIG_ERR) {
	perror("error registering signal handler for SIGTERM: ");
	return 1;
    }

    /* Do the usual daemon startup */
    openlog("modclean", LOG_PID, LOG_DAEMON);
    if (daemon(0, 0)==-1) {
	perror("failed to initialize daemon");
	return 1;
    }

    while (!selfdestruct) {
	/* Get the list of loaded kernel modules */
	get_kernel_info(0);
	modules_count=n_module_stat;
	module_names=xmalloc(l_module_name_list);
	memcpy(module_names, module_name_list, l_module_name_list);

	/* Try to delete modules */
	if (delete_module(NULL)) {
	    syslog(LOG_ERR, "delete_module called failed: %s", strerror(errno));
	    break;
	}
	/* Make a snapshot if necessary */
	snap_shot(module_names, modules_count);

	/* And clean up again */
	free(module_names);
	module_names=NULL;

	sleep(sleep_interval);
    }

    if (module_names!=NULL)
    	free(module_names);
    closelog();

    return (errno==0);
}

/* vim: ts=8 sw=4 ai si cindent
 */
