/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Classes for handling Udp sockets.
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         Mike Butler,    mgb@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *
 * $Id$
 */

#ifndef __UdpSocket_h
#define __UdpSocket_h

#include <UtReport.h>
#include <UtString.h>

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <strstream>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <net/ethernet.h>
#include <unistd.h>
#include <netdb.h>

/* A facade class to facilitate converting between 'sockaddr' and 'sockaddr_in' structs
 * Note: Assumes addresses in sockaddr and sockaddr_in are in NETWORK order.
 */
class InetAddr {
  friend ostream &operator<<(ostream &os, const InetAddr &rhs) {
    unsigned long tmp = ntohl(rhs.cAddr.sin_addr.s_addr);

    os << (0xff &(tmp>>24)) << "." 
       << (0xff &(tmp>>16)) << "." 
       << (0xff &(tmp>> 8)) << "." 
       << (0xff &(tmp>> 0)) << ":"
       << ntohs(rhs.cAddr.sin_port);
    return(os);
  }
private:
  sockaddr_in cAddr;
public:
  enum { BCAST = 0xffffffff, ANY=0x0 };
  InetAddr(const String &s ) { SetAddr(s); }
  InetAddr(const sockaddr& sa)  { cAddr = (*((sockaddr_in*)&sa)); }
  InetAddr(const sockaddr_in &sa) { cAddr = sa; }
  // Assumes addresses are in HOST order!
  InetAddr(unsigned int addr = ANY, unsigned short port = 0) {
    memset(&cAddr,0,sizeof(cAddr));
    cAddr.sin_family = AF_INET;
    cAddr.sin_port   = htons(port);
    cAddr.sin_addr.s_addr = htonl(addr);
  }
  operator String() const { 
    std::ostrstream tmp;
    tmp << (*this) << ends;
    char *s = tmp.str();
    String result(s);
    delete [] s;
    return(result);
  }
  unsigned int Addr() const {
    return(ntohl(cAddr.sin_addr.s_addr));
  }
  int Port() const { 
    return(ntohs(cAddr.sin_port));
  }
  operator sockaddr() const { 
    return(*((sockaddr *)&cAddr));
  };
  operator sockaddr_in() const { return(cAddr); }
  const size_t Size() const { return(sizeof(cAddr)); }
  bool SetAddr(const String &s) {
    int a, b, c, d, p;
    sockaddr_in *sa = (sockaddr_in *)&cAddr;
    memset(sa, 0, sizeof(cAddr));
    sa->sin_family = AF_INET;
    sa->sin_addr.s_addr = htonl(INADDR_ANY);

    /* Empty address is ok... */
    if(!s.length()) return(true);

    /* Just port spec? */
    if(1 == sscanf(s.c_str(), ":%d", &p)) {
      sa->sin_port = htons(p);
      return(true);
    }

    /* Host and port? Snarf and look for host ID too... */
    if(1 == sscanf(s.c_str(), "%*[^:]:%d", &p)) {
      sa->sin_port = htons(p);
    }

    /* Host as IP address? */
    if(4 == sscanf(s.c_str(), "%d.%d.%d.%d", &a, &b, &c, &d)) {
      sa->sin_addr.s_addr = htonl((((((a<<8)|b)<<8)|c)<<8)|d);
      return(true);
    }

    /* Just host name? Attempt to resolve to IP... */
    char name[s.length()+1];
    if(2 != sscanf(s.c_str(), "%[^:]:%d", name, &p)) s.copy(name);
    struct hostent *he = gethostbyname(name);

    /* Can't figure it out, looks bad... */
    if(!he) return(false);
    sa->sin_family = he->h_addrtype;
    memcpy(&sa->sin_addr.s_addr, he->h_addr_list[0], he->h_length);
    return(true);
  }
};

class UdpDatagram {
  friend ostream &operator<<(ostream &os, const UdpDatagram &rhs) {
    os << "(" << rhs.cAddr << ") \"" << rhs.cData << "\"";
    return(os);
  }
private:
  InetAddr cAddr;
  String cData;
public:
  UdpDatagram(const String &s, InetAddr addr) : cAddr(addr), cData(s) {}
  UdpDatagram(const String &s = "") : cData(s) {}
  const String &GetData() const { return(cData); }
  const String &SetData(const String &d) { return(cData = d); }
  const InetAddr &GetAddr() const { return(cAddr); }
  void  SetAddr(const InetAddr &addr) { cAddr = addr; }
  size_t Length() const { return(cData.length()); }
  
};

class UdpSocket {
  int cFd;
  InetAddr cListenAddr;
  
public:
  UdpSocket(const InetAddr& listen, int broadcast = 1, int mcLoop = 0) {
    if ((cFd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
      Report::Error("UdpSocket::UdpSocket() - socket() failed.");
    }

    cListenAddr = listen;
    sockaddr addr = cListenAddr;
    
    // If this is multicast address, prep data structure...
    bool mcast = IN_MULTICAST(ntohl(sockaddr_in(cListenAddr).sin_addr.s_addr)) ? 1 : 0; 
    if(mcast) {
      struct ip_mreq mreq;
      memset(&mreq, 0, sizeof(mreq));
      mreq.imr_multiaddr = sockaddr_in(cListenAddr).sin_addr;
      mreq.imr_interface.s_addr = INADDR_ANY;
    
      if(setsockopt(cFd, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq))) {
	Report::Error(String("UdpSocket::UdpSocket() - setsockopt(IP_ADD_MEMBERSHIP) failed: ") +
		      strerror(errno));
      }
      if(setsockopt(cFd, IPPROTO_IP, IP_MULTICAST_LOOP, &mcLoop, sizeof(mcLoop))) {
	Report::Error(String("UdpSocket::UdpSocket() - setsockopt(IP_MULTICAST_LOOP) failed: ") +
		      strerror(errno));
      }
    }

    // Allow us to reuse this address in case we restart
    int reuse = 1;
    if(setsockopt(cFd, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse)) < 0) {
      close(cFd);
      char buf[1024];
      sprintf(buf,"UdpSocket::UdpSocket() - setsockopt(SO_REUSEADDR) failed: %s",strerror(errno));
      Report::Error(buf);
    }
    
    // Allow us to broadcast if desired
    if(broadcast) {
      if(setsockopt(cFd, SOL_SOCKET, SO_BROADCAST, &broadcast, sizeof(broadcast)) < 0) {
	close(cFd);
	char buf[1024];
	sprintf(buf,"UdpSocket::UdpSocket() - setsockopt(SO_BROADCAST) failed: %s",strerror(errno));
	Report::Error(buf);
      }
    }

    /* If receive port given, bind that port... */
    if(cListenAddr.Port()) {
      if(bind(cFd, &addr, sizeof(addr))) {
	close(cFd);
	char buf[1024];
	sprintf(buf,"UdpSocket::UdpSocket() - bind() failed: %s",strerror(errno));
	Report::Error(buf);
      }
    }

  };
  
  ~UdpSocket() { close(cFd); };

  int Fd() { return cFd; }

  /* Binds the UdpSocket to a device interface
   * User may supply an optional filter to be attached to the interface
   */
  void BindDevice(const String& ifname) {
    struct ifreq ifr;
    memset(&ifr.ifr_name,0,IFNAMSIZ);
    int len = ((ifname.length() > IFNAMSIZ) ? IFNAMSIZ : ifname.length());
    strncpy(ifr.ifr_name, ifname.c_str(), len);
    if (setsockopt(cFd, SOL_SOCKET, SO_BINDTODEVICE, (char *)&ifr, sizeof(ifr)) < 0) {
      char buf[1024];
      sprintf(buf,"UdpSocket::BindDevice() - setsockopt(SO_BINDTODEVICE) on '%s' failed.",ifname.c_str());
      Report::Error(buf);
    }
  };

  UdpDatagram Recv() {
    char buf[2048];
    sockaddr addr = cListenAddr;
    
    int len;
    socklen_t aLen = sizeof(addr);
    len = recvfrom(cFd, buf, sizeof(buf), 0, &addr, &aLen);
    if(len < 0) {
      char buf[1024];
      sprintf(buf,"UdpSocket::Recv() - recvfrom() failed: %s",strerror(errno));
      Report::Error(buf);
    }
    return(UdpDatagram(String(buf, 0, len), addr));
  };

  bool Send(const UdpDatagram &data) {
    int len;
    sockaddr addr = data.GetAddr();
    
    len = sendto(cFd, (void *)data.GetData().c_str(), data.Length(), 
		 0, &addr, sizeof(addr));
    
    if(len <= 0) {
      char buf[1024];
      sprintf(buf,"UdpSocket::Send() - sendto() failed: %s",strerror(errno));
      Report::Error(buf);
    }
    return(true);
  };

};
  

#endif
