/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Author: Mike Butler, mgb@mitre.org
 *
 * $Id: UtSequence.h,v 1.2 1999/09/05 20:39:28 mgb Exp $
 *
 * Breaks an input line into a "Sequence" of line elements.
 *
 * ToDo:
 *   [ ] Make Sequence smart about quoting the break character
 *   [ ] Make it also handle quoted strings...
 */
#ifndef UtSequence_h
#define UtSequence_h

#include <vector>
#include <iterator>
#include <UtString.h> /* Our string class */
#include <stdio.h>

// class Sequence
class Sequence {
friend istream& operator>>(istream &is, Sequence&s) {
  /* Read 'til something retrieved... */
  do {
    String line;
    getline(is, line); 
    s.cLine++;
    // ****Causes problems for last line...
    // if(!is || is.eof()) continue;
    s.Split(line);    
    /* Blanks significant? */
    if((!s.cBlankLines) && (!s)) return(is);
  } while((is) && (!is.eof()) && (!s));
  return(is);
}
friend ostream& operator<<(ostream &os, Sequence&s) {
  std::vector<String>::iterator it;
  for(it = s.cArgv.begin(); it != s.cArgv.end(); it++) {
    os << (*it) << s.cFldSep;
  }
  return(os);
}
private:
  String cFldSep;               /* Field separator String */
  String cWhite;                /* Field White space */
  String cComment;              /* Comment string */
  bool cBlankLines;             /* Read over blank lines... */
  int cLine;                    /* Input line number */
  bool cTrailingSep;		/* True if split saw trailing seperator */
  std::vector<String> cArgv;
public:
  Sequence() : cLine(0) { cBlankLines = true; cWhite = " \t"; cComment = ";"; cFldSep = "#"; cTrailingSep = false; }
  Sequence(const char **argv) { while(*argv) Append(*argv++); }
  void ArgvZ(const char **argv) { for(size_t i = 0; i < Size(); i++) *argv++ = cArgv[i].c_str(); *argv++ = 0; }
  bool   &BlankLines()  { return(cBlankLines); }
  String &FieldSep()    { return(cFldSep); };
  String &Comment()     { return(cComment); };
  String &White()       { return(cWhite); };
  
  Sequence(const Sequence &s) { *this = s; }
  Sequence &operator=(const Sequence &s) {
    cLine = 0;
    cWhite = s.cWhite;
    cFldSep = s.cFldSep;
    cComment = s.cComment;
    cTrailingSep = s.cTrailingSep;
    cArgv = s.cArgv;
    return(*this);
  };

  /* Returns the last element shifted off... */
  String Shift(int i = 1) { String tmp;
			  while(i--) { 
			    if(cArgv.empty()) return("");
			    tmp = cArgv[0];
			    cArgv.erase(cArgv.begin());
			  } 
			  return(tmp); 
			}
  bool Append(const String &s) { cArgv.push_back(s); return(true); }
  bool Prepend(const String &s) { cArgv.insert(cArgv.begin(), s); return(true); }
  virtual ~Sequence() {};
  size_t ArgCnt() const { return(cArgv.size()); }
  size_t Size() const { return(ArgCnt()); }
  bool Empty() const { return(ArgCnt() == 0); }
  size_t WhatLine() const { return(cLine); }
  operator long() const { return(ArgCnt()); }
  String &operator [](size_t index) { while(index > Size()) Append(""); return(cArgv[index]); }
  String operator [](size_t index) const { if(index > Size()) return(""); return(cArgv[index]); }
  Sequence operator +(const String &s) { Sequence seq(*this); seq.Append(s); return(seq); }
  Sequence operator +(const Sequence &s) { Sequence seq(*this); 
                                           for(size_t i = 0; i < s.Size(); i++) 
					     seq.Append(s[i]); 
					   return(seq); 
                                         }
  Sequence &operator +=(const Sequence &s) { (*this) = (*this) + s; return(*this); }
  Sequence &operator +=(const String &s) { (*this) = (*this) + s; return(*this); }
  String Join() const { return(Join(cFldSep)); }
  String Join(const String &sep) const { String s;
					 std::vector<String>::const_iterator it;
					 /* Funnyness for trailing separator correctness... */
					 for(it = cArgv.begin(); it != cArgv.end(); it++) 
					   if(it == cArgv.begin()) s += (*it);
					   else s += sep + (*it);
					 if(cTrailingSep) s += sep;
					 return(s);
				       }
  /* Split off first N pieces of string.  By default, split all */
  bool Split(const String &in, size_t n = (size_t)-1) { 
    String s(in);
    cArgv = std::vector<String>();
    if(s.find_first_not_of(cWhite) == String::npos) return(false);
    s = s.substr(s.find_first_not_of(cWhite));
    if(s.find(cComment) == 0) return(false);
    while(s.length() && n--) {
      cTrailingSep = true;
      size_t brk = s.find(cFldSep);
      cArgv.push_back(s.substr(0, brk));
      if(brk != String::npos) {
	s = s.substr(brk + cFldSep.length());
      } else {
	cTrailingSep = false;
	s = "";
      }
      s = s.substr(s.find_first_not_of(cWhite));
    }
    /* Didn't split last?  Can't have seen trailing sep yet! */
    if(s.length()) {
      cArgv.push_back(s);
      cTrailingSep = false;
      return(true);
    }
    /* Get all the pieces desired? */
    if(!n) return(true);
    if((n + cArgv.size()) == (size_t)-1) return(true);
    return(false);
  }
  bool Get(size_t indx, String &s) const { assert(indx < cArgv.size()) ; s = cArgv[indx]; return(true); }
  bool Get(size_t index, long &l)  const { return(sscanf(GetStr(index).c_str(), "%ld", &l)); }
  bool Get(size_t index, float &f) const { return(sscanf(GetStr(index).c_str(), "%f", &f)); }
  const String &Head() const { return(GetStr(0)); }
  const String &Tail() const { return(GetStr(cArgv.size()-1)); }
  const String &GetStr(size_t index = 0) const { assert(index < cArgv.size()); return(cArgv[index]); }
  long  GetInt(size_t index = 0) const { long tmp; Get(index, tmp); return(tmp); }
  float GetFloat(size_t index = 0) const { float tmp; Get(index, tmp); return(tmp); }
  const std::vector<String> &GetAllArgs() const { return(cArgv); }

};

#ifdef Never
// Debugging cruft...
TripTest()
{
  /* Trip test for sequence class... */
  Sequence seq;
  seq.Split("Now#is#the#time#for#all#good#men", 1);
  cout << "'Now' = '" << seq.Head() << "'" << endl;
  cout << "'is#the#time#for#all#good#men' = '" << seq.Tail() << "'" << endl;
  seq.Split(seq.Tail(), 1);
  cout << "'is' = '" << seq.Shift() << "'" << endl;
  cout << "'the#time#for#all#good#men' = '" << seq.Tail() << "'" << endl;
  seq.Split(seq.Head(), 1);
  cout << "'the' = '" << seq.Shift() << "'" << endl;
  cout << "'time#for#all#good#men' = '" << seq.Head() << "'" << endl;
  
  seq.Split("");
  cout << "'' = '" << seq.Join() << "'" << endl;
  seq.Split("a#b#c#d#");
  cout << "'a#b#c#d#' = '" << seq.Join() << "'" << endl;
  seq.Shift();
  cout << "'b#c#d#' = '" << seq.Join() << "'" << endl;
  seq.Shift(2);
  cout << "'d#' = '" << seq.Join() << "'" << endl;
  seq.Split("a#b");
  cout << "'a#b' = '" << seq.Join() << "'" << endl;
  seq.Split("a#");
  cout << "'a#' = '" << seq.Join() << "'" << endl;
  seq.Split("a");
  cout << "'a' = '" << seq.Join() << "'" << endl;
  seq.Split("#");
  cout << "'#' = '" << seq.Join() << "'" << endl;
  seq.Split("#  ");
  cout << "'#  ' = '" << seq.Join() << "'" << endl;
  seq.Split(" # a # ");
  cout << "' # a # ' = '" << seq.Join() << "'" << endl;
}
#endif



#endif // UtSequence_h
