/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/* Author: Mike Butler, mgb@mitre.org
 *
 * $Id: UtDebug.C,v 1.4 1999/08/23 19:46:02 mgb Exp $
 */

#include <UtDebug.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fstream.h>
#include <strstream>
#include <iterator>
#include <set>
#include <map>

/* Making this a static member function *really* slows compilation...
 * Thanks, STL...
 */
static std::set<String, std::less<String> > cKeys;
static std::set<String, std::less<String> > cAllKeys;
static std::map<String, time_t, std::less<String> > cFiles;
//Load()
/// Load debug flags from a file...
bool Debug::Load(const String &file)
{
  struct stat fs;
  if(stat(file.c_str(), &fs)) return(false);
  cFiles[file] = fs.st_mtime;	/* time of last modification */

  ifstream is(file.c_str());
  if(!is) return(false);
  is >> Debug();
  return(true);
}
// Update()
/// Check if any loaded files have changed and maybe reload...
bool Debug::Update()
{
  struct stat fs;
  bool reload = false;
  std::map<String, time_t, std::less<String> >::const_iterator it;
  for(it = cFiles.begin(); it != cFiles.end(); it++) {
    if(stat((*it).first.c_str(), &fs)) continue;
    if((*it).second != fs.st_mtime) {
      Load((*it).first);
      reload = true;
    }
  }
  return(reload);
}
// operator<<()
/// Insert a list of debug flags...
ostream &operator<<(ostream &os, const Debug &) 
{
  std::ostream_iterator<String> oi(os, ", ");
  copy(cKeys.begin(), cKeys.end(), oi);
  return(os << endl);
}
//operator>>()
/** Read flags from an istream... 
 *  One tag per line, comments begin with ';'
 *  Keys to turn on are prefixed with a '+' or nothing,
 *  keys to turn off are prefixed with a '-' 
 *  keys to leave alone are prefixed with a ' ' or nothing.
 *  Case and blanks ARE significant
 */
istream &operator>>(istream &is, const Debug &) 
{
  String key;
  
  for(;;) {
    key ="";
    getline(is, key);
    if((!is) || (is.eof())) break;
    if(!key.length()) continue;
    switch(key[0]) {
    case '+':
      cKeys.insert(key.substr(1));    
      cAllKeys.insert(key.substr(1));
      break;
    case '-':
      cKeys.erase(key.substr(1));    
      cAllKeys.insert(key.substr(1));
      break;
    case ';':
      continue;
    case ' ':
      cAllKeys.insert(key.substr(1));
      continue;
    default:
      cAllKeys.insert(key);
      continue;
    }
  }
  return(is);
}
// Dump()
/// Print  in a human readable form...
String Debug::Dump() {
  std::ostrstream os;
  os << Debug() << ends;
  char *tmp = os.str();
  String result = tmp;
  delete [] tmp;
  return(result);
}

// DumpAll()
/// Print all keys and values in a form for reading with Load...
ostream &Debug::DumpAll (ostream &os) {
  std::set<String, std::less<String> >::const_iterator it;
  for(it = cAllKeys.begin(); it != cAllKeys.end(); it++) {
    if(cKeys.find(*it) == cKeys.end()) 
      os << "-" << *it << endl;
    else
      os << "+" << *it << endl;
  }
  return(os);
}

Debug::operator bool() {
  return(cKeys.find(cKey) != cKeys.end());
}
// Debug()
/// Set or clear a debug flag...
Debug::Debug(const String &key, const bool &value) : cKey(key) {
  int match = 0;
  std::set<String, std::less<String> >::const_iterator it;
  for(it = cAllKeys.begin(); it != cAllKeys.end(); it++) {
    // Search for matching keys (using globbing)
    if((*it).Match(key)) {
      match++;
      if(value) cKeys.insert(*it);
      else cKeys.erase(*it);
    }
  }
  // No match?  Make this new key (even if it's a wild card!)
  if(!match) {
    if(value) cKeys.insert(key);
    else cKeys.erase(*it);
    cAllKeys.insert(key);
  }
}
Debug::Debug() : cKey(String("")) { }
Debug::Debug(const String &key = "") : cKey(key) { }

/// Free all memory we've used...
void Debug::Clear() {
  cKeys.clear();
  cAllKeys.clear();
  cFiles.clear();
}

