/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * A class for discovering links and reporting them to clients.  A
 * LinkDiscover object is bound to a named IP interface.  The
 * LinkDiscover object is responsible for discovering when links come
 * up and when links go down and reporting these events.  By
 * convention, a LinkDiscover object listens for client commands on a
 * well known Unix socket whose name is based upon the name of the IP
 * interface. For example, a LinkDiscover bound to interface "eth0"
 * will listen for client messages on a Unix socket named "ln-eth0".
 *
 * The LinkDiscover object does not have prior knowledge of potential
 * clients. Clients simply send a message asking for the current list
 * of links. This allows the LinkDiscover object to dynamically learn
 * who its clients are. Then, when a link goes up or down, it sends
 * the event to all clients who have previously requested a list of
 * links. A failure when sending an event to a client implies that
 * the client has stopped listening and thus the LinkDiscover object
 * removes the client from its dynamically managed set of clients.
 *
 * The LinkDiscover object is responsible for determing when links go
 * up and down. This class may be derived so that hardware based
 * mechanisms for link discovery can be utilized. This base class
 * provides a rather simplistic approach to link
 * discovery. Periodically, it broadcasts a Hello message on a Udp
 * socket bound to the IP interface. The LinkDiscover object
 * can be configured to report either uni-directional or only
 * bi-directional links. If the LinkDiscover object is
 * configured to report uni-directional links, then the Hello message
 * simply contains the address of the sending interface. If the
 * LinkDiscover object is configured to report only bi-directional
 * links, then the Hello message also includes a list of the interface
 * addresses of the senders of Hello messages that have been heard by
 * the interface in the last "DeadInterval". 
 * 
 * If the LinkDiscover reports uni-directional links, then whenever
 * it receives a Hello from another node's interface that it does not
 * have a link from, it reports the new link. If the LinkDiscover
 * only reports bi-directional links, then it checks a neighbor list
 * within the received Hello message for the address of the interface
 * that heard the Hello. Note, if the LinkDiscover is configured to
 * report bi-directional links, then it must report that a link went
 * down if it changes from bi-directional to uni-directional;
 * similarly, it reports that a link came up if it changes from
 * uni-directional to bi-directional. If a Hello message for a link
 * which is declared to be up is not received within a "DeadInterval",
 * the LinkDiscover reports that the link is down.
 * 
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *         
 *
 * $Id$
 *    
 */
#ifndef __LnDiscover_h
#define __LnDiscover_h

#include <fstream.h>
#include <list.h>
#include <map.h>
#include <set.h>

#include <MmMsg.h>
#include <UtDebug.h>
#include <UtInterface.h>
#include <UtUdpSocket.h>
#include <UtUnixSocket.h>
#include <UtReport.h>
#include <UtString.h>
#include <UtTime.h>


class LinkDiscover {
protected:
  UnixSocket *cReportSocket;  // Place where we report info about links 
  UdpSocket  *cIfSocket;      // Where we send Hello messages to our neighbors
  Interface  cInterface;      // The interface we are bound to

  typedef set<String, less<String> > ReportSet;
  ReportSet cReportSet;       // Client paths who want to hear link up/down reports

  fd_set cFds;
  int cFdLimit;
  InetAddr   cBroadcastAddr;

  bool cAllowUni;             // True if we report unidirectional links, 
                              //   false if we only report bidirectional links
  unsigned int   cMetric;
  unsigned short cPort;
  unsigned int cHelloPeriod;  // Seconds between sending Hello messages
  unsigned int cDeadInterval; // Seconds after hearing a Hello that we declare a link down
  unsigned int cPrunePeriod;  // How frequently we look for expired Hello's

  // Keys are source IP addr's of Hello's that we've heard, values are the Hello's
  typedef map<unsigned int, Hello, less<unsigned int> > HelloMap;
  HelloMap cHelloMap;

  // Keys are source IP addr's of Hello's that we've heard, values are when we heard the Hello
  typedef map<unsigned int, Time,  less<unsigned int> > HelloTimeMap;
  HelloTimeMap cHelloTimeMap;

  /* Attempts to send the given string to all clients who previously
   *  expressed interest in hearing link info. We assume that
   *  any client who sent us a list command would like to hear future
   *  adds and deletes.
   */
  void ReportString(const String& s) {
    // Send the link info to all known clients
    ReportSet::iterator it;
    for(it = cReportSet.begin(); it != cReportSet.end();) {
      UnixDatagram reply(s,(*it));
      if(cReportSocket && !cReportSocket->Send(reply)) {
	// The client must have gone away...remove him
	cReportSet.erase(it++);
      }
      else {
	it++;
      }
    }
  }

  /* Sends the given link info to all clients who previously
   *  expressed interest in hearing link info.
   */
  enum ReportType { tUp = 'a', tDown = 'd', tList = 'l' };
  void ReportLink(unsigned int addr, ReportType t) {
    String s;
    s += String(char(t)) + " ";
    s += DumpAddr(addr) + " "; 
    if(t == tUp || t == tList)
      s += String((int)cMetric);
    s += " \n";
    ReportString(s);
  }

  /* Iterates across the Hello Time map and looks for Hellos that were heard longer
   * than a Dead period ago. Any of these Hellos are removed from the Hello map and
   * the Hello time map. A report indicating the link is down is made to clients...
   * if we only report bidirectional links then we make sure the link was up before
   * reporting it down
   */
  void PruneOldHellos() {
    Time now = Time::Now();
    HelloTimeMap::iterator it;
    HelloTimeMap& htm = cHelloTimeMap;
    for(it = htm.begin(); it != htm.end(); ) {
      unsigned int addr = (*it).first;
      Time&   t = (*it).second;
      if(double(now) > double(t + (double)cDeadInterval)) {
	if(Debug("PruneOldHellos")) {
	  Report::Debug(String("PruneOldHellos() - Hello from ") + DumpAddr(addr) +
		       " expired...deleting.");
	}
	if(cAllowUni || (!cAllowUni && IsNeighbor(cHelloMap[addr],cInterface.Addr()))) {
	  ReportLink(addr,tDown);
	}
	HelloMap& hm = cHelloMap;
	hm.erase(addr);
	htm.erase(it++);
      }
      else {
	it++;
	if(Debug("PruneOldHellos")) {
	  Report::Debug(String("PruneOldHellos() - Hello from ") + DumpAddr(addr) +
			" arrived at " + t.Str());
	}
      }
    }
  }

  /* Broadcasts a Hello out the interface...each Hello is tailored to the interface 
   * since the Hello contains a list of neighbors who the interface has heard a hello
   * from in the last "DeadInterval"
   */
  void BroadcastHello() {
    list<unsigned int> neighbors;
    if(!cAllowUni) {
      HelloMap::iterator it;
      for(it = cHelloMap.begin(); it != cHelloMap.end(); it++) {
	neighbors.push_back((*it).first);
      }
    }
    
    Hello h(neighbors);
    if(Debug("BroadcastHello"))
      Report::Debug(String("BroadcastHello() - sending ") + h.Dump());
    UdpDatagram dg(h.Encode(), cBroadcastAddr);
    cIfSocket->Send(dg);
  }

  /* Returns true if the Hello message lists the given address as a neighbor
   */
  bool IsNeighbor(const Hello& h, unsigned int addr) {
    list<unsigned int>::iterator j;
    list<unsigned int> nl = h.NeighborList();
    for(j = nl.begin(); j != nl.end(); j++) {
      if((*j) == cInterface.Addr()) {
	return(true);
      }
    }
    return(false);
  }

  /* Stores the Hello...
   * If we didn't have a Hello from this neighbor, report that a link went up.
   * If we only report bidirectional links, we make sure our addresss appears
   * in the list of neighbors.
   */
  void HandleHello(const Hello& h, const InetAddr& src) {

    if(Debug("HandleHello")) {
      String s = String("HandleHello() - From ") +  
	String(src) + " on " + DumpAddr(cInterface.Addr()) + " : " + h.Dump();
      Report::Debug(s);
    }

    bool haveIt = (cHelloMap.find(src.Addr()) != cHelloMap.end());
    unsigned int me = cInterface.Addr();

    if(!haveIt) {
      if ( cAllowUni || (!cAllowUni && IsNeighbor(h,me) ) ) {
	ReportLink(src.Addr(),tUp);
      }
    }
    else {
      // We have a previous Hello
      if( !cAllowUni ) {
	bool wasBi = IsNeighbor(cHelloMap[src.Addr()],me);
	bool isBi  = IsNeighbor(h,me);

	if(wasBi && !isBi) {
	  // A bidirectional link just become uni-directional
	  ReportLink(src.Addr(),tDown);
	}
	else if(!wasBi && isBi) {
	  // A uni-directional link just became bidirectional
	  ReportLink(src.Addr(),tUp);
	}
      }
    }
    cHelloMap[src.Addr()] = h;
    cHelloTimeMap[src.Addr()] = Time::Now();
  }

  /* Determine what type of message this is, decode it, and dispatch it */
  void HandleReceive(UdpDatagram& dg) {
    String d = dg.GetData();
    InetAddr src = dg.GetAddr();
    
    if(d.length() < 2) {
      char buf[1024];
      sprintf(buf,"LinkDiscover::HandleReceive() - Invalid message from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }

    // Make sure we're not hearing one of our own!
    if(cInterface.Addr() == src.Addr()) {
      return;
    }

    unsigned char version = d[0];
    if(version != MsgTypes::Version) {
      char buf[1024];
      sprintf(buf,"LinkDiscover::HandleReceive() - Invalid version from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }

    unsigned char type = d[1];
    switch(type) {
    case MsgTypes::tHello:
      {
	Hello h;
	if(h.Decode(d))
	  HandleHello(h,src);
	break;
      }
    default:
      {
	char buf[1024];
	sprintf(buf,"LinkDiscover::HandleReceive() - Invalid message type from %s, discarding.",
		String(src).c_str());
	Report::Warn(buf);
      }
    }
  }

  /* We provide output to clients to inform them of the links we have discovered
   * or links that have disappeared.
   *
   *  There following commands are supported,
   *
   *  List all known links:
   *   "l"
   *  This command will return a new line delimited list of:
   *    "l <addr> <metric>"...
   *
   *  Ping request:
   *   "q"
   *  The ping request command will return:
   *   "r"
   *
   *
   *  The format of link output information is as follows,
   *
   *  The following link is now up:
   *   "a <addr> <metric>"
   *
   *  The following link is now down:
   *   "d <addr>"
   *
   */
  void HandleReport(UnixDatagram& dg) {
    String d = dg.GetData();
    UnixAddr src = dg.GetAddr();

    if(Debug("HandleReport")) {
      Report::Debug(String("HandleReport() - From: ") + 
		    String(dg.GetAddr()) + "  '" + 
		    dg.GetData() + "' ");
    }

    if(d.length() == 0) {
      char buf[1024];
      sprintf(buf,"Router::HandleReport() - Invalid message from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }

    unsigned char cmd = d[0];
    
    String err("Router::HandleReport() - Invalid command format.");

    // Consume cmd
    d.Token(&d);
    switch(cmd) {
    case 'l':
      {
	cReportSet.insert(dg.GetAddr()); // Remember who wants to hear reports

	// List all links and return them
	HelloMap::iterator it;
	for(it = cHelloMap.begin(); it != cHelloMap.end(); it++) {
	  const Hello& h = (*it).second;
	  if(cAllowUni || (!cAllowUni && IsNeighbor(h,cInterface.Addr()))) {
	    ReportLink((*it).first,tList);
	  }
	}
	break;
      }
    case 'q':
      {
	cReportSet.insert(dg.GetAddr()); // Remember who wants to hear reports

	// Ping request...send a ping reply
	String s = "r\n";
	UnixDatagram reply(s,dg.GetAddr());
	cReportSocket->Send(reply);
	break;
      }
    default:
      {
	Report::Warn(err);
      }
    }
  }

  /* Reads configuration information from the stream
   */
  virtual void ReadConfiguration(istream& is) {
 
    String err("Invalid line in config file: ");
    String s;

    while(s.Getline(is)) {
      String line = s;
      String type     = s.Token(&s);
      if(!type.length()) {
	continue; // Ignore blank lines
      }
      if(type[0] == '#') {
	continue; // Ignore lines that begin with a '#' comment
      }
      else if(type == "metric") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	cMetric = v.Int();
      }
      else if(type == "port") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	cPort = v.Int();
      }
      else if(type == "helloPeriod") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	cHelloPeriod = v.Int();
      }
      else if(type == "deadInterval") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	cDeadInterval = v.Int();
      }
      else if(type == "allowUni") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	cAllowUni = (bool)v.Int();
      }
      else if(type != "") {
	Report::Error(err + line);
      }
    }
  }

public:
  LinkDiscover(const String& ifname, const String& configFile, const String& debugFile) :
    cAllowUni(true), cMetric(1), cPort(20470), cHelloPeriod(5), 
    cDeadInterval(30), cPrunePeriod(1) {
    
    // Make sure we were given a valid interface name!
    list<Interface> sysList = Interface::Discover();
    bool found = false;
    list<Interface>::iterator it;
    for(it = sysList.begin(); it != sysList.end(); it++) {
      if((*it).Name() == ifname) {
	found = true;
	cInterface = (*it);
	break;
      }
    }
    if(!found) {
      Report::Error(String("Invalid interface name: ") + 
		    ifname.c_str() + 
		    ",  make sure interface is up with 'ifconfig'. ");
    }

    // Here if the interface name is valid
    if(debugFile.length())
      Debug::Load(debugFile);

    ifstream is(configFile.c_str());
    if(!is) {
      Report::Error(String("Missing configuration file: ") + configFile);
    }
    ReadConfiguration(is);

    // Turn these off since we aren't in a pretty wired world
    cInterface.RpFilter(false);
    cInterface.Redirects(false);

    FD_ZERO(&cFds);
    cFdLimit = 0;
    
    // Build a well-known Unix socket to report link status to clients (eg. router)
    cReportSocket = new UnixSocket(String("ln-") + cInterface.Name());
    int fd = cReportSocket->Fd();
    FD_SET(fd,&cFds);
    if(fd > cFdLimit) cFdLimit = fd;
    
    // Build a Udp socket for sending Hello messages to our neighbors
    cIfSocket = new UdpSocket(InetAddr(InetAddr::ANY,cPort),1);
    cIfSocket->BindDevice(ifname);
    fd = cIfSocket->Fd();
    FD_SET(fd,&cFds);
    if(fd > cFdLimit) cFdLimit = fd;
    
    cFdLimit += 1; // Sheesh, vagaries of select()
    cBroadcastAddr = InetAddr(InetAddr::BCAST,cPort);
  }
  
  virtual ~LinkDiscover() {
    ReportString("e \n");  // Let clients know we are exiting
    delete(cReportSocket);
    cReportSocket = 0;
    delete(cIfSocket);
    cIfSocket = 0;
  }
  

  /* Main event loop for the Link Discover...
   *
   */
  void Run() {

    Time now = Time::Now();
    Time helloTime = now + (double)cHelloPeriod;
    Time pruneTime = now + (double)cPrunePeriod + Time(0.5);  // prune when we are not busy...

    while(1) {
      Time earliest = Min(helloTime,pruneTime);
      Time timeleft = earliest - Time::Now();


      fd_set fds = cFds;      
      timeval tv = timeval(timeleft);
      int n;

      if(Debug("Run")) {
	Report::Debug(String("Run() - timeleft=") + timeleft.Str());
      }

      if((n=select(cFdLimit, &fds, 0, 0, &tv)) == -1) {
	char buf[1024];
	sprintf(buf,"LinkDiscover::Run() - select failed: %s",strerror(errno));
	Report::Error(buf);
      }

      if(n > 0) {
	if(FD_ISSET(cIfSocket->Fd(),&fds)) {
	  // Handle receive
	  UdpDatagram dg = cIfSocket->Recv();
	  if(Debug("Run")) {
	    Report::Debug(String("Run() - From ") + String(dg.GetAddr()) + 
			  " on " + cInterface.Name());
	  }
	  HandleReceive(dg);
	}
	if(FD_ISSET(cReportSocket->Fd(),&fds)) {
	  UnixDatagram dg = cReportSocket->Recv();
	  HandleReport(dg);
	}
      }

      Time now = Time::Now();
      if(double(earliest) <= double(now)) {
	// Time to do something!

	// Should we prune oldies?
	if(double(pruneTime) <= double(now)) {
	  pruneTime = pruneTime + (double)cPrunePeriod;
	  PruneOldHellos();
	}

	// Should we send a Hello?
	if(double(helloTime) <= double(now)) {
	  helloTime = helloTime + (double)cHelloPeriod;
	  PruneOldHellos();
	  BroadcastHello();
	  Debug::Update();  // Convenient place to check if debug flags have changed
	}
      }
    }
  }
  
  
};


#endif
