#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <SDL.h>
#include <SDL_image.h>
#include <SDL_thread.h>
#include <jack/jack.h>
#include <getopt.h>

#include "config.h"
#include "main.h"
#include "envelopes.h"
#include "find_image.h"

#include "vu_meters.h"
#include "ppm_meters.h"
#include "dpm_meters.h"
#include "jf_meters.h"
#include "scope.h"

#define MET_VU  1 // The meter display type
#define MET_PPM 2
#define MET_DPM 3
#define MET_JF  4
#define MET_SCO 5

void make_channel(jack_client_t *client, int i, char *port_name);
void connect_channel(jack_client_t *client, int i, char *port_name);
void connect_channel_monitoring(jack_client_t *client, int i, char *port_name, int startup);
int graph_order_changed(void *);
void cleanup(void);
int store(char *path, void *data);

SDL_Thread *gt;
SDL_Surface *screen;
SDL_Surface *background_image, *meter, *meter_buf;
SDL_Rect win, dest[MAX_METERS];

int num_meters = 0;
int num_scopes = 0;
int meter_freeze = 0;
float env[MAX_METERS];

char *meter_name = "ppm";
char client_name[256];

float bias = 1.0f; // To allow for ref level changes

jack_port_t *input_ports[MAX_METERS];
jack_port_t *output_ports[MAX_METERS];
jack_client_t *client;

static int monitor = 0;
static char **requested_ports;
static char **master_bridges = NULL;
static int *master_expired = NULL;
static char **known_names = NULL;
static unsigned int known_name_count = 0;
static void handle_sdl_event(SDL_Event *event);
static void port_registration(jack_port_id_t, int, void *);

int main(int argc, char *argv[])
{
	SDL_Event event;
	unsigned int i;
	int opt;
	float ref_lev;
	int port_base = 1;
	unsigned int meter_type = MET_PPM;
	int columns = MAX_METERS;
	int x = 0, y = 0;
	char window_name[256];
	char *us_client_name = NULL;

	num_meters = argc;
	while ((opt = getopt(argc, argv, "mt:r:c:n:h")) != -1) {
		switch (opt) {
			case 'r':
				ref_lev = atof(optarg);
				printf("Reference level: %.1fdB\n", ref_lev);
				bias = powf(10.0f, ref_lev * -0.05f);
				port_base += 2;
				break;
			case 't':
				if (!strcmp(optarg, "vu")) {
					meter_type = MET_VU;
				} else if (!strcmp(optarg, "ppm")) {
					meter_type = MET_PPM;
				} else if (!strcmp(optarg, "dpm")) {
					meter_type = MET_DPM;
				} else if (!strcmp(optarg, "jf")) {
					meter_type = MET_JF;
				} else if (!strcmp(optarg, "sco")) {
					meter_type = MET_SCO;
				} else {
					fprintf(stderr, "Unknown meter type: %s\n", optarg);
					exit(1);
				}
				meter_name = optarg;
				port_base += 2;
				break;
			case 'c':
				columns = atoi(optarg);
				port_base += 2;
				break;
		        case 'm':
			        monitor = 1;
				port_base += 1;
				break;
			case 'n':
				us_client_name = strdup(optarg);
				port_base += 2;
				break;
			case 'h':
				/* Force help to be shown */
				num_meters = 0;
				break;
			default:
				num_meters = 0;
				break;
		}
	}
	num_meters -= port_base;

	if (num_meters > MAX_METERS) {
		num_meters = MAX_METERS;
	}

	if (meter_type == MET_JF) {
		if (num_meters % 2 != 0) {
			fprintf(stderr, "WARNING: Jellyfish meters require an even number of inputs.\n");
		}
		num_scopes = num_meters / 2;
	} else {
		num_scopes = num_meters;
	}

	if (monitor) {
		if (us_client_name != NULL) {
			num_meters = 0;
		}
	}

	if (num_meters < 1) {
		fprintf(stderr, "Meterbridge version %s - http://plugin.org.uk/meterbridge/\n\n", VERSION);
		fprintf(stderr, "Usage %s: [-r ref-level] [-c columns] [-n jack-name] [-m] [-t type] <port>+\n\n", argv[0]);
		fprintf(stderr, "where ref-level is the reference signal level for 0dB on the meter\n");
		fprintf(stderr, "   type is the meter type, one of:\n");
		fprintf(stderr, "     'vu'  - classic moving needle VU meter\n");
		fprintf(stderr, "     'ppm' - PPM meter\n");
		fprintf(stderr, "     'dpm' - Digital peak meter\n");
		fprintf(stderr, "     'jf'  - 'Jellyfish' phase meter\n");
		fprintf(stderr, "     'sco' - Oscilloscope meter\n");
		fprintf(stderr, "\nWhen metering an input port, meterbridge measures the output from all ports\nconnected to the input.  With the -m flag, meterbridge will continue to monitor\nfor new port connections after startup; otherwise it will always meter from the\nset of ports connected when it first started up.  -m cannot be used with -n.\n\n");
		exit(1);
	}
	if (SDL_Init(SDL_INIT_VIDEO) < 0) {
		fprintf(stderr, "Unable to init SDL: %s\n", SDL_GetError());
		exit(1);
	}
	//atexit(SDL_Quit);
	//atexit(cleanup);

	for (i=0; i<num_meters; i++) {
		env[i] = 0.0f;
	}

	switch (meter_type) {
		case MET_VU:
			load_graphics_vu();
			break;
		case MET_PPM:
			load_graphics_ppm();
			break;
		case MET_DPM:
			load_graphics_dpm();
			break;
		case MET_JF:
			load_graphics_jf();
			break;
		case MET_SCO:
			load_graphics_scope();
			break;
	}
		
	if (columns > num_scopes) {
		columns = num_scopes;
	}

	/* Calculate window size */
	win.x = 0;
	win.y = 0;
	win.w = meter->w * columns;
	win.h = meter->h * (((num_scopes - 1) / columns) + 1);

	screen = SDL_SetVideoMode(win.w, win.h, 32, SDL_SWSURFACE);
	if ( screen == NULL ) {
		fprintf(stderr, "Unable to set %dx%d video: %s\n", win.w, win.h, SDL_GetError());
		exit(1);
	}

	background_image = find_image("brushed-steel.png");

	/* Draw background image */
	dest[0].w = background_image->w;
	dest[0].h = background_image->h;
	for (x=0; x < win.w; x += background_image->w) {
		for (y=0; y < win.h; y += background_image->h) {
			dest[0].x = x;
			dest[0].y = y;
			SDL_BlitSurface(background_image, NULL, screen, dest);
		}
	}

	/* Draw meter frames */
	for (i=0, x=0, y=0; i<num_scopes; i++, x++) {
		dest[i].x = meter->w * x;
		dest[i].y = meter->h * y;
		dest[i].w = meter->w;
		dest[i].h = meter->h;
		SDL_BlitSurface(meter, NULL, screen, dest+i);
		if (i % columns == columns - 1) {
			x = -1;
			y++;
		}
	}

	SDL_UpdateRects(screen, 1, &win);

	/* Register with jack */
	if (us_client_name) {
		strncpy(client_name, us_client_name, 31);
	} else {
		snprintf(client_name, 255, "bridge-%d", getpid());
	}
	if ((client = jack_client_new(client_name)) == 0) {
		fprintf(stderr, "jack server not running?\n");
		exit(1);
	}
	printf("Registering as %s\n", client_name);
	snprintf(window_name, 255, "%s %s", meter_name, client_name);
	SDL_WM_SetCaption(window_name, client_name);

	/* Start the graphics thread */
	switch (meter_type) {
		case MET_VU:
			gt = SDL_CreateThread(gfx_thread_vu, NULL);
			break;
		case MET_PPM:
			gt = SDL_CreateThread(gfx_thread_ppm, NULL);
			break;
		case MET_DPM:
			gt = SDL_CreateThread(gfx_thread_dpm, NULL);
			break;
		case MET_JF:
			gt = SDL_CreateThread(gfx_thread_jf, NULL);
			break;
		case MET_SCO:
			gt = SDL_CreateThread(gfx_thread_scope, NULL);
			break;
	}

	/* Pick the jack process method */
	if (meter_type == MET_VU) {
		init_buffers_rms();
		jack_set_process_callback(client, process_rms, 0);
	} else if (meter_type == MET_JF || meter_type == MET_SCO) {
		jack_set_process_callback(client, process_ring, 0);
	} else {
		jack_set_process_callback(client, process_peak, 0);
	}
	
	// This is used to maually save the session, but the auto stuff
	// does a better job anyway. Just here for testing
	//jack_set_store_callback(client, store, NULL);
	
	if (monitor) {
		jack_set_port_registration_callback(client, port_registration, 0);
	}

	if (jack_activate(client)) {
		fprintf (stderr, "cannot activate client");
		exit(1);
	}

	if (monitor) {
		master_bridges = (char **)malloc(num_meters * sizeof(char *));
		master_expired = (int *)malloc(num_meters * sizeof(int));
	}

	/* Create and connect the jack ports */

	requested_ports = (char **)malloc(num_meters * sizeof(char *));

	for (i = 0; i < num_meters; i++) {

		if (monitor) {
			master_bridges[i] = NULL;
			master_expired[i] = 0;
		}

		requested_ports[i] = strdup(argv[port_base + i]);
		make_channel(client, i, requested_ports[i]);

		if (monitor) {
			connect_channel_monitoring(client, i, requested_ports[i], 1);
		} else {
			connect_channel(client, i, requested_ports[i]);
		}
	}

	if (monitor) {
		int count = 0;

		while (1) {
			while (SDL_PollEvent(&event)) {
				handle_sdl_event(&event);
			}
		
			if (++count == 20) {
				for (i = 0; i < num_meters; i++) {
					connect_channel_monitoring
						(client, i, requested_ports[i], 0);
				}
				count = 0;
			}
			
			usleep(50000);
		}
	} else {
		while (SDL_WaitEvent(&event)) {
			handle_sdl_event(&event);
		}
	}

	cleanup();

	/* We can't ever get here, but it keeps gcc quiet */
	return 0;
}

void handle_sdl_event(SDL_Event *event)
{
	if (!event) return;

	switch (event->type) {

	case SDL_QUIT:
		cleanup();
		break;
	case 5:
		meter_freeze = 1;
		break;
	case 6:
		meter_freeze = 0;
		break;
	default:
		//printf("%d\n", event->type);
		break;
	}
}

/* make_channel: An input port for the meter and an output port for monitoring */

void make_channel(jack_client_t *client, int i, char *port_name)
{
	char in_name[256], out_name[256];
	
	snprintf(in_name, 255, "meter_%d", i+1);
	if (!(input_ports[i] = jack_port_register(client, in_name, JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0))) {
		fprintf(stderr, "Cannot register '%s'\n", in_name);
		cleanup();
	}

	snprintf(out_name, 255, "monitor_%d", i+1);
	if (!(output_ports[i] = jack_port_register(client, out_name, JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0))) {
		fprintf(stderr, "Cannot register '%s'\n", out_name);
		cleanup();
	}
}

void port_registration(jack_port_id_t id, int registered, void *d)
{
	jack_port_t *port;
	const char *port_name;
	int j;

	if (!monitor || registered) return;

	port = jack_port_by_id(client, id);
	if (!port) return;
	
	port_name = jack_port_name(port);
	if (!port_name) return;

	for (j = 0; j < num_meters; ++j) {

		if (master_bridges[j] && !master_expired[j] &&
		    !strcmp(port_name, master_bridges[j])) {
			master_expired[j] = 1;
		}
	}
}

void connect_channel(jack_client_t *client, int i, char *port_name)
{
	char in_name[256], out_name[256];
	jack_port_t *port;
	int flags;

	snprintf(in_name, 255, "%s:meter_%d", client_name, i+1);
	snprintf(out_name, 255, "%s:monitor_%d", client_name, i+1);

//	fprintf(stderr, "connecting '%s' to '%s'...\n", in_name, port_name);
	port = jack_port_by_name(client, port_name);
	if (port == NULL) {
		fprintf(stderr, "Can't find port '%s'\n", port_name);
		return;
	}
	flags = jack_port_flags(port);

	if (flags & JackPortIsInput) {
		const char **connections;
		unsigned int j;

		connections = jack_port_get_all_connections(client, port);
		for (j=0; connections && connections[j]; j++) {
			if (jack_disconnect(client, connections[j], port_name) == 0) {
				//fprintf(stderr, "Broke '%s' to '%s'\n", possible_ports[j], port_name);
				if (jack_connect(client, connections[j], in_name)) {
					fprintf(stderr, "Could not connect '%s' to '%s'\n", connections[j], in_name);
					cleanup();
				}
				//fprintf(stderr, "Connected '%s' to '%s'\n", possible_ports[j], in_name);
			} else {
				fprintf(stderr, "Could not disconnect %s\n", connections[j]);
			}
		}
		free(connections);
		jack_connect(client, out_name, port_name);
		//fprintf(stderr, "Connected '%s' to '%s'\n", out_name, port_name);
	} else if (flags & JackPortIsOutput && jack_connect(client, port_name, jack_port_name(input_ports[i]))) {
		fprintf(stderr, "Cannot connect port '%s' to '%s'\n", port_name, in_name);
		cleanup();
	}
}


void connect_channel_monitoring(jack_client_t *client, int i, char *port_name,
				int startup)
{
	char in_name[256], out_name[256], out_prefix[256];
	jack_port_t *port;
	int flags;

	const char **connections;
	unsigned int j, k;

	snprintf(in_name, 255, "%s:meter_%d", client_name, i+1);
	snprintf(out_name, 255, "%s:monitor_%d", client_name, i+1);
	snprintf(out_prefix, 255, "%s:monitor_", client_name);

	port = jack_port_by_name(client, port_name);
	if (port == NULL) {
		return;
	}
	flags = jack_port_flags(port);

	if (flags & JackPortIsOutput) {
		if (jack_connect(client, port_name, jack_port_name(input_ports[i]))) {
			fprintf(stderr, "Cannot connect port '%s' to '%s'\n",
				port_name, in_name);
			cleanup();
		}
		return;
	}
	
	// known to be an input port

	if (master_expired[i]) {
		char *n = master_bridges[i];
		master_expired[i] = 0;
		master_bridges[i] = 0;
		if (n) free(n);
	}
	
	if (!startup && !master_bridges[i]) {
		// Look for any meter bridges that have been
		// (manually or otherwise) connected to us,
		// and if we find one, call it the master.
		jack_port_t *in_port = jack_port_by_name(client, in_name);
		if (in_port) {
			connections = jack_port_get_all_connections
				(client, in_port);
			for (j = 0; connections && connections[j]; ++j) {
				if (!strncmp(connections[j], client_name, 7)) {
					// starts with "bridge-"
					master_bridges[i] = strdup
						(connections[j]);
					break;
				}
			}
			if (connections) free(connections);
		}
	}

	connections = jack_port_get_all_connections(client, port);
	
	if (startup && !master_bridges[i]) {
		for (j = 0; connections && connections[j]; j++) {
			// main() should have enforced that client_name
			// begins with "bridge-" when monitoring
			if (!strncmp(connections[j], client_name, 7)) {
				master_bridges[i] = strdup(connections[j]);
				break;
			}
		}
	}

	for (j = 0; connections && connections[j]; j++) {
		
		if (!strncmp(connections[j], out_prefix, strlen(out_prefix))) {
			continue;
		}
		
		// If, on startup, we see another meterbridge
		// connected to the same output as we're interested
		// in, then we should take _only_ that meterbridge's
		// output and leave everything else alone (because
		// that meterbridge will be handling it).  Same goes
		// for the situation where another bridge is connected
		// to us (by hand or because of the exit of a prior
		// bridge in the graph) while running.
		
		// Conversely, if we see no other meterbridge on
		// startup, we should refuse to steal the output of
		// any new meterbridges that appear during use
		// (because they should be using our output as above).

		if (master_bridges[i]) {
			if (!strcmp(connections[j], master_bridges[i])) {
				if (!startup) continue;
			} else {
				continue;
			}
		}
		
		for (k = 0; k < known_name_count; ++k) {
			if (!strcmp(connections[j], known_names[k])) {
				break;
			}
		}
		
		if (k < known_name_count) continue;
		
		known_names = (char **)realloc
			(known_names,
			 (known_name_count + 1) * sizeof(char *));
		known_names[known_name_count++] = strdup(connections[j]);
		
		if (!strncmp(connections[j], client_name, 7)) {
			// a bridge
			if (!master_bridges[i]) {
				// we might be master, so don't take this
				continue;
			}
		}

		/* Connect the prospective client to us first.  If
		   this fails, that might just mean it's already
		   connected, so we shouldn't worry about it. */

		if (jack_connect(client, connections[j], in_name)) {
//			fprintf(stderr, "Could not connect '%s' to '%s'\n",
//				connections[j], in_name);
		} else if (jack_disconnect(client, connections[j], port_name) == 0) {
//			fprintf(stderr, "Broke '%s' to '%s'\n",
//				connections[j], port_name);
		} else {
			fprintf(stderr, "Could not disconnect %s\n",
				connections[j]);
		}
	}

	if (connections) free(connections);
	
	if (startup) {
		jack_connect(client, out_name, port_name);
//		fprintf(stderr, "Connected '%s' to '%s'\n", out_name, port_name);
	}
}

void cleanup()
{
	unsigned int i, j, l;
	const char **all_iports, **all_oports;

	//printf("cleanup()\n");

	for (i=0; i<num_meters; i++) {
		if (input_ports[i] == NULL || output_ports[i] == NULL) {
			continue;
		}

		all_iports = jack_port_get_all_connections(client,
				output_ports[i]);
		all_oports = jack_port_get_all_connections(client,
				input_ports[i]);

		for (j=0; all_oports && all_oports[j]; j++) {
			jack_disconnect(client, all_oports[j], jack_port_name(input_ports[i]));
		}

		for (j=0; all_iports && all_iports[j]; j++) {
			jack_disconnect(client, jack_port_name(output_ports[i]), all_iports[j]);
			for (l=0; all_oports && all_oports[l]; l++) {
				jack_connect(client, all_oports[l], all_iports[j]);
			}
		}
	}

	/* Leave the jack graph */
	jack_client_close(client);

	/* Get rid of the GUI stuff */
	SDL_KillThread(gt);
	SDL_Quit();

	/* And were done */
	exit(0);
}

int store(char *path, void *data)
{
	char cmdpath[256];
	int i;
	FILE *cmd;

	printf("Store request received: %s\n", path);
	snprintf(cmdpath, 255, "%s/start", path);
	if (!(cmd = fopen(cmdpath, "w"))) {
		printf("Cannot store state in '%s'\n", cmdpath);
		return 0;
	}
	fprintf(cmd, "meterbridge -n '%s' -t %s ", client_name, meter_name);
	for (i=0; i<num_meters; i++) {
		fprintf(cmd, " -");
	}
	fprintf(cmd, "\n");
	fclose(cmd);

	return 0;
}
