/*
** ident-tester.c           A small daemon that can be used to test Ident
**                          servers
**
** Author: Peter Eriksson <pen@lysator.liu.se>, 10 Aug 1992
*/

#ifdef NeXT3
#  include <libc.h>
#endif

#include <stdio.h>
#include <netdb.h>
#include <errno.h>
#include <syslog.h>

#ifdef HAVE_ANSIHEADERS
#  include <stdlib.h>
#  include <unistd.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>

#define IN_LIBIDENT_SRC
#include "ident.h"

#include <arpa/inet.h>

/*
** Return the name of the connecting host, or the IP number as a string.
*/
#ifdef INET6
char *gethost __P1(struct in6_addr *, addr)
{
  static char str[512];
  struct hostent *hp;
#ifndef linux
  int h_error;
#endif

#ifdef linux
  hp = gethostbyaddr((char *) addr, sizeof(struct in6_addr), AF_INET6);
#else
  hp = getipnodebyaddr((char *) addr, sizeof(struct in6_addr), AF_INET6,
		       &h_error);
#endif
  if (hp)
    return (char *) hp->h_name;
  else
    return inet_ntop(AF_INET6, addr, str, sizeof str);
#else
char *gethost __P1(struct in_addr *, addr)
{
  struct hostent *hp;

  
  hp = gethostbyaddr((char *) addr, sizeof(struct in_addr), AF_INET);
  if (hp)
    return (char *) hp->h_name;
  else
    return inet_ntoa(*addr);
#endif
}

int
main __P2(int, argc,
	  char **, argv)
{
#ifdef INET6
  char str[512];
  struct sockaddr_in6 laddr, faddr;
#else
  struct sockaddr_in laddr, faddr;
#endif
  int len, res, lport, fport;
  ident_t *id;
  char *identifier, *opsys, *charset;
  

  puts("Welcome to the IDENT server tester, version 1.9\r\n");
  printf("(Linked with %s)\r\n\n", _id_version);
  
  fflush(stdout);
  
  len = sizeof(faddr);
  getpeername(0, (struct sockaddr *) &faddr, &len);

  len = sizeof(laddr);
  getsockname(0, (struct sockaddr *) &laddr, &len);

#ifdef INET6
  printf("Connecting to Ident server at %s...\r\n",
	 inet_ntop(AF_INET6, &faddr.sin6_addr, str, sizeof str));
#else
  printf("Connecting to Ident server at %s...\r\n", inet_ntoa(faddr.sin_addr));
#endif
  fflush(stdout);

#ifdef LOG_LOCAL3
  openlog("tidentd", 0, LOG_LOCAL3);
#else
  openlog("tidentd", 0);
#endif
  
#ifdef INET6
  id = id_open(&laddr.sin6_addr, &faddr.sin6_addr, NULL);
#else
  id = id_open(&laddr.sin_addr, &faddr.sin_addr, NULL);
#endif
  if (!id)
  {
      if (errno)
      {
	  int saved_errno = errno;
	  char *hs;
	  
	  perror("Connection denied");
	  fflush(stderr);

#ifdef INET6
	  hs = gethost(&faddr.sin6_addr);
#else
	  hs = gethost(&faddr.sin_addr);
#endif
	  errno = saved_errno;
	  syslog(LOG_DEBUG, "Error: id_open(): host=%s, error=%m", hs);
      }
      else
	  puts("Connection denied.");
      return 0;
  }

  printf("Querying for lport %d, fport %d....\r\n",
#ifdef INET6
	 (int) ntohs(faddr.sin6_port),
	 (int) ntohs(laddr.sin6_port));
#else
	 (int) ntohs(faddr.sin_port),
	 (int) ntohs(laddr.sin_port));
#endif
  fflush(stdout);

  errno = 0;
#ifdef INET6
  if (id_query(id, ntohs(faddr.sin6_port), ntohs(laddr.sin6_port), 0) < 0)
#else
  if (id_query(id, ntohs(faddr.sin_port), ntohs(laddr.sin_port), 0) < 0)
#endif
  {
      if (errno)
      {
	  int saved_errno = errno;
	  char *hs;
	  
	  perror("id_query()");
	  fflush(stderr);

#ifdef INET6
	  hs = gethost(&faddr.sin6_addr);
#else
	  hs = gethost(&faddr.sin_addr);
#endif
	  errno = saved_errno;
	  syslog(LOG_DEBUG, "Error: id_query(): host=%s, error=%m", hs);
		 
      }
      else
      {
	  puts("Query failed.\n");
      }
      
    return 0;
  }

  printf("Reading response data...\r\n");
  fflush(stdout);

  res = id_parse(id, NULL,
		   &lport, &fport,
		   &identifier,
		   &opsys,
		   &charset);

  switch (res)
  {
    default:
      if (errno)
      {
	  int saved_errno = errno;
	  char *hs;
	  
	  perror("id_parse()");
#ifdef INET6
	  hs = gethost(&faddr.sin6_addr);
#else
	  hs = gethost(&faddr.sin_addr);
#endif
	  errno = saved_errno;
	  syslog(LOG_DEBUG, "Error: id_parse(): host=%s, error=%m", hs);
      }
      else
	  puts("Error: Invalid response (empty response?).\n");
      
      break;

    case -2:
	{
	    int saved_errno = errno;
#ifdef INET6
	    char *hs = gethost(&faddr.sin6_addr);
#else
	    char *hs = gethost(&faddr.sin_addr);
#endif

	    errno = saved_errno;
	    syslog(LOG_DEBUG, "Error: id_parse(): host=%s, Parse Error: %s",
		   hs,
		   identifier ? identifier : "<no information available>");
	    if (identifier)
		printf("Parse error on reply:\n  \"%s\"\n", identifier);
	    else
		printf("Unidentifiable parse error on reply.\n");
	}
        break;

    case -3:
	{
	    int saved_errno = errno;
#ifdef INET6
	    char *hs = gethost(&faddr.sin6_addr);
#else
	    char *hs = gethost(&faddr.sin_addr);
#endif

	    errno = saved_errno;
	    
	    syslog(LOG_DEBUG,
		   "Error: id_parse(): host=%s, Illegal reply type: %s",
		   hs,
		   identifier);

	    printf("Parse error in reply: Illegal reply type: %s\n",
		   identifier);
	}
        break;
	     
    case 0:
	{
	    int saved_errno = errno;
#ifdef INET6
	    char *hs = gethost(&faddr.sin6_addr);
#else
	    char *hs = gethost(&faddr.sin_addr);
#endif

	    errno = saved_errno;
	    
	    syslog(LOG_DEBUG, "Error: id_parse(): host=%s, NotReady",
		   hs);
	    puts("Not ready. This should not happen...\r");
	}
      break;

    case 2:
	{
	    int saved_errno = errno;
#ifdef INET6
	    char *hs = gethost(&faddr.sin6_addr);
#else
	    char *hs = gethost(&faddr.sin_addr);
#endif

	    errno = saved_errno;
	    
	    syslog(LOG_DEBUG, "Reply: Error: host=%s, error=%s",
		   hs, identifier);
	}
      
      printf("Error response is:\r\n");
      printf("   Lport........ %d\r\n", lport);
      printf("   Fport........ %d\r\n", fport);
      printf("   Error........ %s\r\n", identifier);
      break;

    case 1:
      if (charset)
	syslog(LOG_INFO,
	       "Reply: Userid: host=%s, opsys=%s, charset=%s, userid=%s",
#ifdef INET6
	       gethost(&faddr.sin6_addr), opsys, charset, identifier);
#else
	       gethost(&faddr.sin_addr), opsys, charset, identifier);
#endif
      else
	syslog(LOG_INFO, "Reply: Userid: host=%s, opsys=%s, userid=%s",
#ifdef INET6
	       gethost(&faddr.sin6_addr), opsys, identifier);
#else
	       gethost(&faddr.sin_addr), opsys, identifier);
#endif
      
      printf("Userid response is:\r\n");
      printf("   Lport........ %d\r\n", lport);
      printf("   Fport........ %d\r\n", fport);
      printf("   Opsys........ %s\r\n", opsys);
      printf("   Charset...... %s\r\n", charset ? charset : "<not specified>");
      printf("   Identifier... %s\r\n", identifier);

#ifdef INET6
      if (id_query(id, ntohs(faddr.sin6_port), ntohs(laddr.sin6_port), 0) >= 0)
#else
      if (id_query(id, ntohs(faddr.sin_port), ntohs(laddr.sin_port), 0) >= 0)
#endif
      {
	if (id_parse(id, NULL,
		     &lport, &fport,
		     &identifier,
		     &opsys,
		     &charset) == 1)
	  printf("   Multiquery... Enabled\r\n");
      }
  }

  fflush(stdout);
  sleep(1);
  return 0;
}

