/*
    Copyright (C) 2004 Florian Schmidt
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2.1 of the License, or
    (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.
    
    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software 
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "convolve.h"

#include <stdlib.h>
#include <unistd.h>
#include <iostream>
#include <jack/jack.h>
#include <sstream>
#include <vector>
#include <signal.h>
#include <cmath>
#include <time.h>

#ifdef HAVE_LASH
#include <lash/lash.h>
#endif

/* 
	can be overridden by cmdline
*/
std::string jack_name = "jack_convolve";

jack_client_t *client;

#ifdef HAVE_LASH
lash_client_t *lash_client = 0;
#endif

/*
	one convolution_t for every response if 
	not in mix_outputs_mode.
	If so, we only use one with multiple 
	responses.
*/

std::vector<convolution_t*> convolutions;

/*
	ports and buffers..

	input ports and buffers:
		in pseudo_multi_mode we need as many input ports
		as the responses have channels (total number).
	
			organized as:
		
			response0 input0
			response0 input1
			response0 inputN_0
			.
			.
			.
			responseM inputN_M - 1
			responseM inputN_M			

		when not in pseudo-multi_mode we only need
		as many inputs as there are responses (one
		input per response).

	output portsand buffers:
		in mix_outputs_mode we need only as many ports
		as our responses have channels (not total channel 
		count, as all outputs of all responses get
		mixed together). This also implies equal channel
		count for all responses.

		when not int mix_outputs_mode we need as many
		output channels as our responses  (total number).
*/

jack_port_t **iports;
jack_port_t **oports;

/*	
	 the port buffers for input
*/
float **in_data;

/*
	# of inputs
*/
int in_data_size;

/*
	and output
*/
float **out_data;

/*	
	# of outputs
*/
int out_data_size;

std::vector<std::string> response_file_names;
response_t **responses;

bool fake_mode = false;
unsigned int  pseudo_multi_mode = 0;
unsigned int  mix_outputs_mode = 0;

unsigned int partitionsize = 0;

float gain = 1.0;
int min_bin = -1;
int max_bin = -1;

bool quit = false;

void extract_response_file_names_from_cl (int argc, char *argv[]);

void print_usage();

void extract_options_from_cl (int argc, char *argv[]);

void signalled(int sig);

int process(jack_nframes_t frames, void* arg) {

	for (int i = 0; i < in_data_size; ++i) 
		in_data[i] = (float*)jack_port_get_buffer(iports[i], frames);

	for (int i = 0; i < out_data_size; ++i) 
		out_data[i] = (float*)jack_port_get_buffer(oports[i], frames);

	if (mix_outputs_mode) {
		convolution_process(convolutions[0], in_data, out_data, gain, min_bin, max_bin);
	}
	else {
		int channel = 0;
		for (int i = 0; i < convolutions.size(); ++i) {
			if (pseudo_multi_mode) {
				convolution_process(convolutions[i], in_data+channel, out_data+channel, gain, min_bin, max_bin);

			} else {
				convolution_process(convolutions[i], in_data+i, out_data+channel, gain, min_bin, max_bin);
				
			}
			channel += responses[i]->channels;
		}
	}

	return 0;
}

int main(int argc, char *argv[]) {
#ifdef HAVE_LASH
	lash_event_t* lash_event;
	lash_config_t* lash_config;
#endif

	std::cout << "Jack_convolve (C) 2004 Florian Schmidt - protected by GPL" << std::endl;

	/*	
		hook up signal handler for ctrl-c
	*/
	signal(SIGINT, signalled);
	
	libconvolve_init();

#ifdef HAVE_LASH
	lash_client = lash_init(lash_extract_args(&argc, &argv), "jack_convolve",
	                       LASH_Terminal, LASH_PROTOCOL(2, 0));
	if (lash_client)
		std::cout << "- Connected to LASH server" << std::endl;
	else
		std::cout << "- Failed to connect to LASH server" << std::endl;
#endif

	extract_options_from_cl(argc, argv);

	extract_response_file_names_from_cl(argc, argv);

	if (response_file_names.size() == 0) {
		print_usage();
		exit (0);
	}

	/*
		only a single convolution_t in mix_output_mode
		[the convolution_process() does the mixdown for us then]
	*/
	if (mix_outputs_mode) 
		convolutions.push_back(new convolution_t);
	else {
		for (int i = 0; i < response_file_names.size(); ++i) {
			convolutions.push_back(new convolution_t);
		}
	}

	/* 
		register jack client
	*/
	client = jack_client_new(jack_name.c_str());
	if (!client) {
		std::cout << "- Error: couldn't become jack client - server running? name already taken?" << std::endl;
		exit(0);
	}


	/*	
		find out the samplerate of jack
	*/
	int jack_sample_rate = jack_get_sample_rate(client);

	/*
		allocate array for pointers to the responses
	*/
	responses = (response_t**)malloc(sizeof(response_t*) * response_file_names.size());

	/* 
		channels in the response files.
	*/
	int response_channels = 0;

	/* 
		ok, let's load the responses
	*/
	for (unsigned int index = 0; index < response_file_names.size(); ++index) {
		responses[index] = (convolution_response*)malloc(sizeof(response_t));
		std::cout << "- Loading response file: " << response_file_names[index] << std::endl;
		if (!load_response(responses[index], (char*)response_file_names[index].c_str(), jack_sample_rate)) {
			std::cout << "- Error loading response file: " << response_file_names[index].c_str() << ".. Exiting" << std::endl;
			exit(0);
		}
		std::cout << "Responselength: " << responses[index]->length << " frames, " << (float)responses[index]->length/jack_sample_rate << " sec. " << std::endl;
		if (index == 0) 
			response_channels = responses[index]->channels;
		else {
			/*
				and check for channel count mismatch if in mix_outputs_mode
			*/
			if ((responses[index]->channels != response_channels) && mix_outputs_mode) {
				std::cout << "- Channel count mismatch in response files. exiting." << std::endl;
				exit(0);
			}
		}
	}	

	if (pseudo_multi_mode) {
		int channels = 0;
		for (int i = 0; i < response_file_names.size(); ++i) 
			channels += responses[i]->channels;

		in_data = (float**)malloc(sizeof(float*) * channels);
		in_data_size = channels;
	}
	else {
		in_data = (float**)malloc(sizeof(float*) * response_file_names.size());
		in_data_size = response_file_names.size();
	}

	if (mix_outputs_mode) {
		out_data = (float**)malloc(sizeof(float*) * response_channels);
		out_data_size = response_channels;
	}
	else {
		out_data_size = 0;
		for (int i = 0; i < response_file_names.size(); ++i ) 
			out_data_size += responses[i]->channels;

		out_data = (float**)malloc(sizeof(float*) * out_data_size);
	}

	std::cout << "- Registering ports:" << std::endl;

	/*
		allocate iports array
	*/
	if (pseudo_multi_mode) {
		iports = (jack_port_t**)malloc(sizeof(jack_port_t*) * in_data_size);
	}
	else
		iports = (jack_port_t**)malloc(sizeof(jack_port_t*) * response_file_names.size());

	/*
		register ports
	*/
	for (unsigned int i = 0; i < response_file_names.size(); ++i) {

		if (pseudo_multi_mode) {
			for (unsigned int j = 0; j < responses[i]->channels; ++j) {
				std::stringstream stream_in;
				stream_in << "Response" << i << " in" << j; 
				std::cout << stream_in.str() << std::endl;;
				iports[i*response_channels + j] = jack_port_register(client, stream_in.str().c_str(), JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);
			}
		} else {
			std::stringstream stream_in;

			stream_in << "Response" << i << " in";
			std::cout << stream_in.str() << std::endl;
			iports[i] = jack_port_register(client, stream_in.str().c_str(), JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);
		}
	}

	if (mix_outputs_mode) {
		/* 
			allocate oports array
		*/
		oports = (jack_port_t**)malloc(sizeof(jack_port_t*) * response_channels);

		/*
			register ports
		*/
		for (int i = 0; i < response_channels; ++i) {

			std::stringstream stream_out;
			stream_out << "out" << i;
			std::cout << stream_out.str() << std::endl;
			oports[i] = jack_port_register(client, stream_out.str().c_str(), JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
		} 
	}
	else {
		oports = (jack_port_t**)malloc(sizeof(jack_port_t*) * out_data_size);
		int channel = 0;
		for (int i = 0; i < response_file_names.size(); ++i) {
			for (int j = 0; j < responses[i]->channels; ++j) {
				std::stringstream stream_out;
				stream_out << "Response" << i <<" out" << j;
				std::cout << stream_out.str() << std::endl;
				oports[channel] = jack_port_register(client, stream_out.str().c_str(), JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
				channel++;
			}
		}
	}
	
	std::cout << "- Initializing convolution(s)...";

	if (mix_outputs_mode)
		convolution_init(convolutions[0], response_file_names.size(), response_channels, responses, jack_get_buffer_size(client), pseudo_multi_mode);
	else {
		for (int i = 0; i < response_file_names.size(); ++i) {
			convolution_init(convolutions[i], 1, responses[i]->channels, &responses[i], jack_get_buffer_size(client), pseudo_multi_mode);		
		}
	}
	
	std::cout << "done." << std::endl;

	jack_set_process_callback(client, process, 0);
	jack_activate(client);

#ifdef HAVE_LASH
	/*
		tell LASH who we be
	*/
	if (lash_client) 
		lash_jack_client_name(lash_client, (const char *)jack_name.c_str());

	if (lash_client) {
		lash_event = lash_event_new_with_type(LASH_Client_Name);
		lash_event_set_string(lash_event, jack_name.c_str());
		lash_send_event(lash_client, lash_event);
	}

	if (lash_client)
		std::cout << "- Running (and obeying to LASH).." << std::endl;	
	else
#endif
		std::cout << "- Running (press ctrl-c to quit).." << std::endl;


	while (!quit) {
#ifdef HAVE_LASH
	/*
		we only check for the Quit event, everything else
		doesn't matter to us, as we have all state specified
		on the commandline which LASH restores for us
	*/
	if (lash_client) {
			while (lash_event = lash_get_event(lash_client)) {
				switch (lash_event_get_type(lash_event)) {
				    case LASH_Quit:
						printf("- LASH asked us to quit!\n");
						quit = 1;
						break;
					
				    default:
						 printf("- Got unhandled LASH event - ignoring\n");
				}
				lash_event_destroy(lash_event);
			}
			while (lash_config = lash_get_config(lash_client)) {
				lash_config_destroy(lash_config);
			}
		}
#endif 
		usleep(10000);
	}	
	
	std::cout << "- Exiting.." << std::endl;

	jack_deactivate(client);
	jack_client_close(client);

	convolution_destroy(convolutions[0]);
}


void extract_response_file_names_from_cl (int argc, char *argv[]) {

	/*
		the first one is the name of the executable.
		the following are possible filenames
	*/
	for (int i = 1; i < argc; ++i) {

		if(!(std::string(argv[i]).substr(0,2) == "--")) {

			/*
				found a param which is not an option,
				so it's a filename
			*/
			response_file_names.push_back(argv[i]);
		}
	}
}

void print_usage() {

	std::cout << "Usage: jack_convolve <options> <responsefiles>" << std::endl;
	std::cout << "Where <options> are: " << std::endl;
	std::cout << "--name=name        set the client name used when registering with jack" << std::endl;
	std::cout << "--gain=factor      apply gain factor (default 1.0)" << std::endl;
	std::cout << "--min_bin=bin_no   which bin to start from (default 0)" << std::endl;
	std::cout << "--max_bin=bin_no   which bin to end with (default buffersize + 1" << std::endl;
	std::cout << "--partitionsize=#  (TODO) enable extra buffering and set to buffersize # frames" << std::endl;
	std::cout << "--pseudo-multi     enable pseudo multichannel mode (each channel has an input)" << std::endl;
	std::cout << "--mix-outputs      mixes the outputs of all convolutions (slight optimization saving some IFFT's)" << std::endl;
	std::cout << "--help             show this help" << std::endl;
	std::cout << "Where <responsefiles> are soundfiles with an equal channels count when in mix-outputs mode." << std::endl;
}

void extract_options_from_cl (int argc, char *argv[]) {

	for (int i = 1; i < argc; ++i) {

		if(std::string(argv[i]).substr(0,std::string("--help").length()) == "--help") {

			print_usage();
			exit (0);
		}

		if(std::string(argv[i]).substr(0,std::string("--fake-mode").length()) == "--fake-mode") {

			fake_mode = true;
		}

		if(std::string(argv[i]).substr(0,std::string("--pseudo-multi").length()) == "--pseudo-multi") {

			pseudo_multi_mode = 1;
		}

		if(std::string(argv[i]).substr(0,std::string("--mix-outputs").length()) == "--mix-outputs") {

			mix_outputs_mode = 1;
		}

		if(std::string(argv[i]).substr(0,std::string("--version").length()) == "--version") {

			std::cout << "version 0.0.9" << std::endl;
			exit (0);
		}

		if(std::string(argv[i]).substr(0,std::string("--gain=").length()) == "--gain=") {

			std::stringstream stream;
			stream << (std::string(argv[i]).substr(std::string("--gain=").length(),
			                                      std::string(argv[i]).length() 
			                                       - std::string("--gain=").length()));

			stream >> gain;
		}

		if(std::string(argv[i]).substr(0,std::string("--partitionsize=").length()) == "--partitionsize=") {

			std::stringstream stream;
			stream << (std::string(argv[i]).substr(std::string("--partitionsize=").length(),
			                                      std::string(argv[i]).length() 
			                                       - std::string("--partitionsize=").length()));

			stream >> partitionsize;
		}

		if(std::string(argv[i]).substr(0,std::string("--min_bin=").length()) == "--min_bin=") {

			std::stringstream stream;
			stream << (std::string(argv[i]).substr(std::string("--min_bin=").length(),
			                                      std::string(argv[i]).length() 
			                                       - std::string("--min_bin=").length()));

			stream >> min_bin;
		}

		if(std::string(argv[i]).substr(0,std::string("--max_bin=").length()) == "--max_bin=") {

			std::stringstream stream;
			stream << (std::string(argv[i]).substr(std::string("--max_bin=").length(),
			                                      std::string(argv[i]).length() 
			                                       - std::string("--max_bin=").length()));

			stream >> max_bin;
		}

		if(std::string(argv[i]).substr(0,std::string("--name=").length()) == "--name=") {

			std::stringstream stream;
			stream << (std::string(argv[i]).substr(std::string("--name=").length(),
			                                      std::string(argv[i]).length() 
			                                       - std::string("--name=").length()));
			
			stream >> jack_name;
			std::cout << "- Using jack name: \"" << jack_name << "\"" << std::endl;
		}
	}
}


void signalled(int sig) {
	quit = true;
}

