#!/bin/bash
# jack.ctl - (c) steve harris, 2003 and rohan drape, 2003-2004

# A Bourne shell script to control a jack system.  Derived from the
# jackctl script written by Steve Harris and submitted to the
# jackit-devel list.  Extended to use the exact long option names used
# by jackd(1) and to manage an arbitrary set of secondary jack daemons.

# Returns 1 iff the argument is 'yes', 'on' or 'true'.

function is_on {
  if [[ xx$1 == "xxyes" || xx$1 == "xxon" ||  xx$1 == "xxtrue" ]]; then
    return 0;
  fi
  return 1;
}

# Returns 1 iff the argument is not the empty string.

function is_set {
  if [ xx$1 == "xx" ]; then
    return 1;
  fi
  return 0;
}

# Parse operating mode.

start="off"
stop="off"
case $1 in
  start) start="on";;
  stop) stop="on";;
  restart) stop="on"; restart="on";;
  *) echo "Usage: jack.ctl (start|stop|restart)"; exit 1;;
esac

# This script writes lock and log files to a temporary directory.

tmpdir=/tmp
pidfile="$tmpdir/jack.lock"
errfile="$tmpdir/jack.errlog"

# Stop executing processes and halt script if requested.

if is_on $stop; then
  if [ -r $pidfile ]; then
    kill $(cat $pidfile) 2>/dev/null;
    rm -f $pidfile $errfile;
  else
    echo "jack.ctl: lock file does not exist ($pidfile)";
    exit 1;
  fi
  if (is_on $restart); then
      echo "jack.ctl: preparing to restart"
      sleep 1 
      jack.ctl start
  fi
  exit 0
fi

# Initialize all argument strings.

jackd=""
jackh=""
asio_a=""
realtime_a=""
timeout_a=""
verbose_a=""
silent_a=""
driver_a=""
capture_a=""
playback_a=""
device_a=""
rate_a=""
period_a=""
nperiods_a=""
hwmon_a=""
hwmeter_a=""
softmode_a=""
monitor_a=""
dither_a=""
inchannels_a=""
outchannnels_a=""
shorts_a=""

# Read system wide and user specific run control files.

if [ -r /etc/jackrc ]; then
    echo "jack.ctl: Reading /etc/jackrc" ;
    source /etc/jackrc ;
fi
if [ -r $HOME/.jackrc ]; then
    echo "jack.ctl: Reading $HOME/.jackrc" ;
  source $HOME/.jackrc;
fi

# Warn if the lock file exists, or if required options are not set.

if [ -r "$pidfile" ]; then
  echo "jack.ctl:  lock file exists ($pidfile)";
  exit 1;
fi
if (! is_set $jackd || ! is_set $driver); then
  echo "jack.ctl: jackd or driver not set"
  exit 1;
fi

# Set argument strings.

if is_on $asio; then
  asio_a=" --asio";
fi
if is_on $realtime; then
  realtime_a=" --realtime";
fi
if is_set $timeout; then
  timeout_a=" --timeout=$timeout";
fi
if is_on $verbose; then
  verbose_a=" --verbose";
fi
if is_on $silent; then
  silebt_a=" --silent";
fi
if is_set $driver; then
  driver_a=" -d $driver";
fi
if is_set $capture; then
  capture_a=" --capture=$capture";
fi
if is_set $playback; then
  playback_a=" --playback=$playback";
fi
if is_set $device; then
  device_a=" --device=$device";
fi
if is_set $rate; then
  rate_a=" --rate=$rate";
fi
if is_set $period; then
  period_a=" --period=$period";
fi
if is_set $nperiods; then
  nperiods_a=" --nperiods=$nperiods";
fi
if is_on $hwmon; then
  hwmon_a=" --hwmon";
fi
if is_on $hwmeter; then
  hwmeter_a=" --hwmeter";
fi
if is_on $softmode; then
  softmode_a=" --softmode";
fi
if is_on $monitor; then
  monitor_a=" --monitor";
fi
if is_set $dither; then
  dither_a=" --dither=$dither";
fi
if is_set $inchannels; then
  inchannels_a=" --inchannels=$inchannels";
fi
if is_set $inchannels; then
  outchannels_a=" --outchannels=$outchannels";
fi
if is_on $shorts; then
  shorts_a=" --shorts";
fi

# Construct and print command string.

jcmd="$jackd$asio_a$realtime_a$timeout_a$verbose_a$silent_a$driver_a$capture_a$playback_a$device_a$rate_a$period_a$nperiods_a$hwmon_a$hwmeter_a$softmode_a$monitor_a$dither_a$inchannels_a$outchannels_a$shorts_a";
echo $jcmd

# Start processes.  The pause after starting the JACK daemon is
# required for any secondary daemons to be able to make a connection.
# All error messages are sent to the same log file.

$jcmd 2>$errfile &
sleep 1
if is_set $jackh ; then
    for d in $jackh ; do
	echo $d
	$d 2>>$errfile &
    done
fi

# Store all PID numbers to lock file.

/sbin/pidof $jackd > $pidfile
if is_set $jackh ; then
    for d in $jackh ; do
	/sbin/pidof $d >> $pidfile
    done
fi
