/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*-  vi:set ts=8 sts=4 sw=4: */

#include "jack-loader.h"

#include <dlfcn.h>
#include <iostream>

#include <map>

static std::map<const char *, void *> dynamic_jack_symbols;
static void *dynamic_jack_handle = 0;
static bool dynamic_jack_open_attempted = false;

void *dynamic_jack_symbol(const char *name)
{
    if (dynamic_jack_symbols.find(name) != dynamic_jack_symbols.end()) {
        return dynamic_jack_symbols[name];
    }
    if (!dynamic_jack_handle) {
        if (!dynamic_jack_open_attempted) {
            void *handle = ::dlopen("libjack.so.1", RTLD_NOW);
            if (!handle) handle = ::dlopen("libjack.so.0", RTLD_NOW);
            if (!handle) handle = ::dlopen("libjack.so", RTLD_NOW);
            if (!handle) {
                std::cerr << "WARNING: jack-loader: Failed to load JACK library: "
                          << ::dlerror() << " (tried .so, .so.0, .so.1)"
                          << std::endl;
            } else {
                dynamic_jack_handle = handle;
            }
            dynamic_jack_open_attempted = true;
        }
        if (!dynamic_jack_handle) return 0;
    }
    void *symbol = ::dlsym(dynamic_jack_handle, name);
    if (!symbol) {
        std::cerr << "WARNING: jack-loader: Failed to locate symbol "
                  << name << ": " << ::dlerror() << std::endl;
    }
    dynamic_jack_symbols[name] = symbol;
    return symbol;
}

int dynamic_jack_set_process_callback(jack_client_t *client,
                                      JackProcessCallback process_callback,
                                      void *arg)
{
    typedef int (*func)(jack_client_t *client,
                        JackProcessCallback process_callback,
                        void *arg);
    void *s = dynamic_jack_symbol("jack_set_process_callback");
    if (!s) return 1;
    func f = (func)s;
    return f(client, process_callback, arg);
}

int dynamic_jack_set_xrun_callback(jack_client_t *client,
                                   JackXRunCallback xrun_callback,
                                   void *arg)
{
    typedef int (*func)(jack_client_t *client,
                        JackXRunCallback xrun_callback,
                        void *arg);
    void *s = dynamic_jack_symbol("jack_set_xrun_callback");
    if (!s) return 1;
    func f = (func)s;
    return f(client, xrun_callback, arg);
}

const char **dynamic_jack_get_ports(jack_client_t *client, 
                                    const char *port_name_pattern, 
                                    const char *type_name_pattern, 
                                    unsigned long flags)
{
    typedef const char **(*func)(jack_client_t *client, 
                                 const char *port_name_pattern, 
                                 const char *type_name_pattern, 
                                 unsigned long flags);
    void *s = dynamic_jack_symbol("jack_get_ports");
    if (!s) return 0;
    func f = (func)s;
    return f(client, port_name_pattern, type_name_pattern, flags);
}

jack_port_t *dynamic_jack_port_register(jack_client_t *client,
                                        const char *port_name,
                                        const char *port_type,
                                        unsigned long flags,
                                        unsigned long buffer_size)
{
    typedef jack_port_t *(*func)(jack_client_t *client,
                                 const char *port_name,
                                 const char *port_type,
                                 unsigned long flags,
                                 unsigned long buffer_size);
    void *s = dynamic_jack_symbol("jack_port_register");
    if (!s) return 0;
    func f = (func)s;
    return f(client, port_name, port_type, flags, buffer_size);
}

int dynamic_jack_connect(jack_client_t *client,
                         const char *source,
                         const char *dest)
{
    typedef int (*func)(jack_client_t *client,
                        const char *source,
                        const char *dest);
    void *s = dynamic_jack_symbol("jack_connect");
    if (!s) return 1;
    func f = (func)s;
    return f(client, source, dest);
}

void *dynamic_jack_port_get_buffer(jack_port_t *port,
                                   jack_nframes_t sz)
{
    typedef void *(*func)(jack_port_t *, jack_nframes_t);
    void *s = dynamic_jack_symbol("jack_port_get_buffer");
    if (!s) return 0;
    func f = (func)s;
    return f(port, sz);
}

int dynamic_jack_port_unregister(jack_client_t *client,
                                 jack_port_t *port)
{
    typedef int (*func)(jack_client_t *, jack_port_t *);
    void *s = dynamic_jack_symbol("jack_port_unregister");
    if (!s) return 0;
    func f = (func)s;
    return f(client, port);
}

void dynamic_jack_on_shutdown(jack_client_t *client,
                              void (*function)(void *),
                              void *arg)
{
    typedef int (*func)(jack_client_t *, void (*)(void *), void *);
    void *s = dynamic_jack_symbol("jack_on_shutdown");
    if (!s) return;
    func f = (func)s;
    f(client, function, arg);
}

#define dynamic1(rv, name, argtype, failval)    \
    rv dynamic_##name(argtype arg) {            \
        typedef rv (*func) (argtype);           \
        void *s = dynamic_jack_symbol(#name);   \
        if (!s) return failval;                 \
        func f = (func) s;                      \
        return f(arg);                          \
    }

dynamic1(jack_client_t *, jack_client_new, const char *, 0);
dynamic1(jack_nframes_t, jack_get_buffer_size, jack_client_t *, 0);
dynamic1(jack_nframes_t, jack_get_sample_rate, jack_client_t *, 0);
dynamic1(int, jack_activate, jack_client_t *, 1);
dynamic1(int, jack_deactivate, jack_client_t *, 1);
dynamic1(int, jack_client_close, jack_client_t *, 1);
dynamic1(jack_nframes_t, jack_port_get_latency, jack_port_t *, 0);
dynamic1(const char *, jack_port_name, const jack_port_t *, 0);
dynamic1(int, jack_port_connected, const jack_port_t *, 0);

int dynamic_jack_unload()
{
    if (!dynamic_jack_handle) return 0;
    if (::dlclose(dynamic_jack_handle)) {
        std::cerr << "WARNING: jack-loader: dlclose failed!" << std::endl;
        perror("dlclose");
        return 1;
    }
    dynamic_jack_handle = 0;
    dynamic_jack_symbols.clear();
    dynamic_jack_open_attempted = false;
    std::cerr << "INFO: jack-loader: unloaded" << std::endl;
    return 0;
}

