/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*-  vi:set ts=8 sts=4 sw=4: */

#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include <alloca.h>
#include <math.h>

#include "jack-loader.h"

#define ALSA_PCM_OLD_HW_PARAMS_API
#define ALSA_PCM_OLD_SW_PARAMS_API
#include "alsa/asoundlib.h"

#include <samplerate.h>

#include <vector>

typedef signed short ALSASAMPLE;

typedef std::vector<jack_port_t *> PortList;
typedef std::vector<SRC_STATE *> SRCList;

PortList capture_ports;
SRCList  capture_srcs;
PortList playback_ports;
SRCList  playback_srcs;

int processing = 0;

jack_client_t *client;

// TODO: make the sample format configurable soon...
snd_pcm_format_t format = SND_PCM_FORMAT_S16;	 /* sample format */

snd_pcm_t *alsa_playback_handle;
snd_pcm_t *alsa_capture_handle;

int jack_sample_rate;

char *jack_name = "alsa_aux";

double current_resample_factor = 1.0;

// Command line parameters

int sample_rate = 0;				 /* stream rate */
int num_channels = 2;				 /* count of channels */
int period_size = 1024;
int num_periods = 2;

int target_delay = 0; /* the delay which the program should try to approach */
int max_diff = 0;     /* the diff value when a hard readpointer skip should occur */
int catch_factor = 1000;

int capture_only = 0;
int playback_only = 0;

jack_nframes_t last_playback_frame = 0;
jack_nframes_t last_playback_nonzero_frame = 0;
jack_nframes_t last_capture_frame = 0;
jack_nframes_t last_capture_nonzero_frame = 0;
float noise_floor = 0.000000001;

// Debug stuff:

int print_counter = 10;
int debug_mode = 0;

// Alsa stuff:

static int xrun_recovery(snd_pcm_t *handle, int err) {
    if (err == -EPIPE) {	/* under-run */
        if (handle == alsa_playback_handle) {
            printf("jack_alsa_bridge: ALSA buffer underrun in playback for %s\n", jack_name);
        } else {
            printf("jack_alsa_bridge: ALSA buffer underrun in capture for %s\n", jack_name);
        }            
        err = snd_pcm_prepare(handle);
        if (err < 0) {
            fprintf(stderr, "Can't recover from underrun, prepare failed: %s\n",
                   snd_strerror(err));
        }
        return 0;
    } else if (err == -ESTRPIPE) {
        if (handle == alsa_playback_handle) {
            printf("jack_alsa_bridge: ALSA buffer suspend in playback for %s\n", jack_name);
        } else {
            printf("jack_alsa_bridge: ALSA buffer suspend in capture for %s\n", jack_name);
        }            
        while ((err = snd_pcm_resume(handle)) == -EAGAIN) {
            sleep(1);	/* wait until the suspend flag is released */
        }
        if (err < 0) {
            err = snd_pcm_prepare(handle);
            if (err < 0) {
                fprintf(stderr, "Can't recover from suspend, prepare failed: %s\n",
                       snd_strerror(err));
            }
        }
        return 0;
    }
    return err;
}

static int set_hwparams(snd_pcm_t *handle, snd_pcm_hw_params_t *params, snd_pcm_access_t access, int rate, int channels, int period, int nperiods ) {
    int err, dir=0;

    /* choose all parameters */
    err = snd_pcm_hw_params_any(handle, params);
    if (err < 0) {
        fprintf(stderr, "Broken configuration: no configurations available: %s\n", snd_strerror(err));
        return err;
    }
    /* set the interleaved read/write format */
    err = snd_pcm_hw_params_set_access(handle, params, access);
    if (err < 0) {
        fprintf(stderr, "Access type not available: %s\n", snd_strerror(err));
        return err;
    }
    /* set the sample format */
    err = snd_pcm_hw_params_set_format(handle, params, format);
    if (err < 0) {
        fprintf(stderr, "Sample format not available: %s\n", snd_strerror(err));
        return err;
    }
    /* set the count of channels */
    err = snd_pcm_hw_params_set_channels(handle, params, channels);
    if (err < 0) {
        fprintf(stderr, "Channel count (%i) not available: %s\n", channels, snd_strerror(err));
        return err;
    }
    /* set the stream rate */
    err = snd_pcm_hw_params_set_rate_near(handle, params, rate, 0);
    if (err < 0) {
        fprintf(stderr, "Rate %iHz not available: %s\n", rate, snd_strerror(err));
        return err;
    }
    if (err != rate) {
        fprintf(stderr, "Rate doesn't match (requested %iHz, got %iHz)\n", rate, err);
        return -EINVAL;
    }
    /* set the buffer time */
    err = snd_pcm_hw_params_set_buffer_time_near(handle, params, 1000000*period*nperiods/rate, &dir);
    if (err < 0) {
        fprintf(stderr, "Unable to set buffer time %i: %s\n",  1000000*period*nperiods/rate, snd_strerror(err));
        return err;
    }
    if( snd_pcm_hw_params_get_buffer_size(params) != nperiods * period ) {
        fprintf(stderr,  "WARNING: buffer size does not match: (requested %d, got %d)\n", nperiods * period, (int) snd_pcm_hw_params_get_buffer_size(params) );
    }
    /* set the period time */
    err = snd_pcm_hw_params_set_period_time_near(handle, params, 1000000*period/rate, &dir);
    if (err < 0) {
        fprintf(stderr, "Unable to set period time %i: %s\n", 1000000*period/rate, snd_strerror(err));
        return err;
    }
    int ps = snd_pcm_hw_params_get_period_size(params, NULL );
    if( ps != period ) {
        fprintf(stderr,  "WARNING: period size does not match: (requested %i, got %i)\n", period, ps );
    }
    /* write the parameters to device */
    err = snd_pcm_hw_params(handle, params);
    if (err < 0) {
        fprintf(stderr, "Unable to set hw params: %s\n", snd_strerror(err));
        return err;
    }
    return 0;
}

static int set_swparams(snd_pcm_t *handle, snd_pcm_sw_params_t *swparams, int period, int nperiods) {
    int err;

    /* get the current swparams */
    err = snd_pcm_sw_params_current(handle, swparams);
    if (err < 0) {
        fprintf(stderr, "Unable to determine current swparams: %s\n", snd_strerror(err));
        return err;
    }
    /* start the transfer when the buffer is full */
    err = snd_pcm_sw_params_set_start_threshold(handle, swparams, period );
    if (err < 0) {
        fprintf(stderr, "Unable to set start threshold mode: %s\n", snd_strerror(err));
        return err;
    }
    snd_pcm_uframes_t boundary;
    err = snd_pcm_sw_params_get_boundary(swparams, &boundary);
    if (err < 0) {
        fprintf(stderr, "Unable to retrieve ringbuffer boundary: %s\n", snd_strerror(err));
        return err;
    }
    err = snd_pcm_sw_params_set_stop_threshold(handle, swparams, boundary);
    if (err < 0) {
        fprintf(stderr, "Unable to set stop threshold mode: %s\n", snd_strerror(err));
        return err;
    }
    /* allow the transfer when at least period_size samples can be processed */
    err = snd_pcm_sw_params_set_avail_min(handle, swparams, 1 );
    if (err < 0) {
        fprintf(stderr, "Unable to set avail min: %s\n", snd_strerror(err));
        return err;
    }
    /* align all transfers to 1 sample */
    err = snd_pcm_sw_params_set_xfer_align(handle, swparams, 1);
    if (err < 0) {
        fprintf(stderr, "Unable to set transfer align: %s\n", snd_strerror(err));
        return err;
    }
    /* write the parameters to the playback device */
    err = snd_pcm_sw_params(handle, swparams);
    if (err < 0) {
        fprintf(stderr, "Unable to set sw params: %s\n", snd_strerror(err));
        return err;
    }
    return 0;
}

// Communicate with ALSA API

static snd_pcm_t *open_audiofd( char *device_name, int capture, int rate, int channels, int period, int nperiods ) {
    int err;
    snd_pcm_t *handle;
    snd_pcm_hw_params_t *hwparams;
    snd_pcm_sw_params_t *swparams;

    snd_pcm_hw_params_alloca(&hwparams);
    snd_pcm_sw_params_alloca(&swparams);

    if ((err = snd_pcm_open(&(handle), device_name, capture ? SND_PCM_STREAM_CAPTURE : SND_PCM_STREAM_PLAYBACK, SND_PCM_NONBLOCK )) < 0) {
        fprintf(stderr, "Capture open error: %s\n", snd_strerror(err));
        return NULL;
    }

    if ((err = set_hwparams(handle, hwparams,SND_PCM_ACCESS_RW_INTERLEAVED, rate, channels, period, nperiods )) < 0) {
        fprintf(stderr, "Setting of hwparams failed: %s\n", snd_strerror(err));
        return NULL;
    }
    if ((err = set_swparams(handle, swparams, period, nperiods)) < 0) {
        fprintf(stderr, "Setting of swparams failed: %s\n", snd_strerror(err));
        return NULL;
    }

    if (capture) {
        snd_pcm_start( handle );
        snd_pcm_wait( handle, 200 );
    } else {
        int num_null_samples = nperiods * period * channels;
        ALSASAMPLE *tmp = (ALSASAMPLE *)alloca( num_null_samples * sizeof( ALSASAMPLE ) ); 
        memset( tmp, 0, num_null_samples * sizeof( ALSASAMPLE ) );
        snd_pcm_writei( handle, tmp, num_null_samples );
    }

    return handle;
}


int process_playback (jack_nframes_t nframes, void *arg)
{
    ALSASAMPLE *outbuf;
    float *floatbuf, *resampbuf;
    int rlen;
    int err;
    snd_pcm_sframes_t delay;

    if (!alsa_playback_handle) return 0;

    snd_pcm_delay( alsa_playback_handle, &delay );

    // Do it the hard way.
    // this is for compensating xruns etc...

    if( delay > (target_delay+max_diff) ) {
        snd_pcm_rewind( alsa_playback_handle, delay - target_delay );
        //snd_pcm_writei( alsa_playback_handle, tmp, target_delay-t_delay );
        if (debug_mode) printf( "delay = %d", (int) delay );
        snd_pcm_delay( alsa_playback_handle, &delay );
        if (debug_mode) printf( "... and delay = %d\n", (int) delay );
        delay = target_delay;
        // XXX: at least set it to that value.
        current_resample_factor = (double) sample_rate / (double) jack_sample_rate;
    }

    if( delay < (target_delay-max_diff) ) {
        ALSASAMPLE *tmp = (ALSASAMPLE *)alloca( (target_delay-delay) * sizeof( ALSASAMPLE ) * num_channels ); 
        memset( tmp, 0, sizeof( ALSASAMPLE ) * num_channels * (target_delay-delay) );
        snd_pcm_writei( alsa_playback_handle, tmp, target_delay-delay );
        if (debug_mode) printf( "delay = %d", (int) delay );
        snd_pcm_delay( alsa_playback_handle, &delay );
        if (debug_mode) printf( "... and delay = %d\n", (int) delay );
        delay = target_delay;
        // XXX: at least set it to that value.
        current_resample_factor = (double) sample_rate / (double) jack_sample_rate;
    }

    /* ok... now we should have target_delay +- max_diff on the alsa side.
     *
     * calculate the number of frames, we want to get.
     */

    double resamp_rate = (double)jack_sample_rate / (double)sample_rate;  // == nframes / alsa_samples.
    double request_samples = nframes / resamp_rate;  //== alsa_samples;

    double offset = delay - target_delay;

    //double frlen = request_samples - offset / catch_factor;
    double frlen = request_samples - offset;

    double compute_factor = frlen / (double) nframes;

    double diff_value =  pow(current_resample_factor - compute_factor, 3) / (double) catch_factor;
    current_resample_factor -= diff_value;
    rlen = lrint(ceil( ((double)nframes) * current_resample_factor ) + 2);

    if( (print_counter--) == 0 ) {
        print_counter = 10;
        if (debug_mode) printf( "res: %f, \tdiff = %f, \toffset = %f \n", (float)current_resample_factor, (float)diff_value, (float) offset );
    }

    /*
     * now this should do it...
     */

    outbuf = (ALSASAMPLE *)alloca( rlen * sizeof( ALSASAMPLE ) * num_channels );

    floatbuf = (float *)alloca( rlen * sizeof( float ) );
    resampbuf = (float *)alloca( nframes * sizeof( float ) );

    /*
     * render jack ports to the outbuf...
     */

    int chn = 0;
    size_t port_index = 0;
    SRC_DATA src;
    
    int to_write = rlen;

    int have_nonzero = 0;
    int i;

    float **bufs = (float **)alloca(num_channels * sizeof(float *));

    if (processing) {

        while (port_index < playback_ports.size()) {

            jack_port_t *port = playback_ports[port_index];

            bufs[chn] = (float *)jack_port_get_buffer(port, nframes);

            if (!have_nonzero) {
                for (i = nframes - 1; i >= 0; --i) {
                    if (fabs(bufs[chn][i]) > noise_floor) {
                        have_nonzero = 1;
                        break;
                    }
                }
            }
            chn++;
            port_index++;
        }
    }

    port_index = 0;
    chn = 0;

    last_playback_frame += nframes;

    if (!processing ||
        (last_playback_frame >
         last_playback_nonzero_frame + jack_sample_rate * 10)) {

        if (have_nonzero) {
            printf("switching on playback\n");
        } else {
            for (chn = 0; chn < num_channels; ++chn) {
                for (i = 0; i < rlen; i++) {
                    outbuf[chn + i * num_channels] = 0;
                }
            }
            goto again;
        }
    }
    if (have_nonzero) {
        last_playback_nonzero_frame = last_playback_frame;
    }

    while (port_index < playback_ports.size()) {

        float *buf = bufs[chn];

        SRC_STATE *src_state = playback_srcs[port_index];

        src.data_in = buf;
        src.input_frames = nframes;

        src.data_out = resampbuf;
        src.output_frames = rlen;
        src.end_of_input = 0;

        //src.src_ratio = (float) frlen / (float) nframes;
        src.src_ratio = current_resample_factor;

        //src_set_ratio( src_state, src.src_ratio );
        src_process( src_state, &src );

        for (i = 0; i < rlen; i++) {
            //!!! should we dither?
            outbuf[chn + i * num_channels] = (ALSASAMPLE)(resampbuf[i] * 32767);
        }

        to_write = src.output_frames_gen;

        port_index++;
        chn++;
    }

    // now write the output...

again:
    err = snd_pcm_writei(alsa_playback_handle, outbuf, to_write);
    if ( err < 0 ) {
        //printf( "err = %d\n", err );
        if (xrun_recovery(alsa_playback_handle, err) < 0) {
            //printf("Write error: %s\n", snd_strerror(err));
            //exit(EXIT_FAILURE);
        }
        goto again;
    }
//  if( err != rlen ) {
//      printf( "write = %d\n", rlen );
//  }

    return 0;      
}

int process_capture (jack_nframes_t nframes, void *arg)
{
    ALSASAMPLE *outbuf;
    float *floatbuf, *resampbuf;
    int rlen;
    int err;
    snd_pcm_sframes_t delay;
    int put_back_samples=0;

    if (!alsa_capture_handle) return 0;

    snd_pcm_delay( alsa_capture_handle, &delay );

    delay = delay;
    // Do it the hard way.
    // this is for compensating xruns etc...

    if( delay > (target_delay+max_diff) ) {
        ALSASAMPLE *tmp = (ALSASAMPLE *)alloca( (delay-target_delay) * sizeof( ALSASAMPLE ) * num_channels ); 
        snd_pcm_readi( alsa_capture_handle, tmp, delay-target_delay );
        if (debug_mode) printf( "delay = %d\n", (int) delay );
        delay = target_delay;
        // XXX: at least set it to that value.
        current_resample_factor = (double) jack_sample_rate / (double) sample_rate;
    }
    if( delay < (target_delay-max_diff) ) {
        snd_pcm_rewind( alsa_capture_handle, target_delay - delay );
        if (debug_mode) printf( "delay = %d\n", (int) delay );
        delay = target_delay;
        // XXX: at least set it to that value.
        current_resample_factor = (double) jack_sample_rate / (double) sample_rate;
    }

    /* ok... now we should have target_delay +- max_diff on the alsa side.
     *
     * calculate the number of frames, we want to get.
     */

    double resamp_rate = (double)jack_sample_rate / (double)sample_rate;  // == nframes / alsa_samples.
    double request_samples = nframes / resamp_rate;  //== alsa_samples;

    double offset = delay - target_delay;

    //double frlen = request_samples - offset / catch_factor;
    double frlen = request_samples + offset;
    double compute_factor = (double) nframes / frlen;

    double diff_value =  pow(current_resample_factor - compute_factor, 3) / (double) catch_factor;
    current_resample_factor -= diff_value;
    rlen = lrint(ceil( ((double)nframes) / current_resample_factor )+2);

    if( (print_counter--) == 0 ) {
        print_counter = 10;
        if (debug_mode) printf( "res: %f, \tdiff = %f, \toffset = %f \n", (float)current_resample_factor, (float)diff_value, (float) offset );
    }

    /*
     * now this should do it...
     */

    outbuf = (ALSASAMPLE *)alloca( rlen * sizeof( ALSASAMPLE ) * num_channels );

    floatbuf = (float *)alloca( rlen * sizeof( float ) );
    resampbuf = (float *)alloca( nframes * sizeof( float ) );

    // get the data...
again:
    err = snd_pcm_readi(alsa_capture_handle, outbuf, rlen);
    if( err < 0 ) {
        if (debug_mode) printf( "err = %d\n", err );
        if (xrun_recovery(alsa_capture_handle, err) < 0) {
            //printf("Write error: %s\n", snd_strerror(err));
            //exit(EXIT_FAILURE);
        }
        goto again;
    }
    if( err != rlen ) {
        if (debug_mode) printf( "read = %d (req = %d)\n", err, rlen );
        rlen = err;
    }

    /*
     * render jack ports to the outbuf...
     */
    int chn = 0;
    size_t port_index = 0;
    int i = 0;

    int have_nonzero = 0;
    
    if (processing) {

        while (port_index < capture_ports.size()) {

            jack_port_t *port = capture_ports[port_index];

            if (jack_port_connected(port) > 0) {
                break;
            }
            
            ++port_index;
            ++i;
        }

        if (port_index < capture_ports.size()) {

            for (i = 0; i < rlen * num_channels; ++i) {
                if (outbuf[i] != 0) {
                    have_nonzero = 1;
                    break;
                }
            }
        }
    }

    port_index = 0;

    last_capture_frame += nframes;

    if (!processing ||
        (last_capture_frame >
         last_capture_nonzero_frame + jack_sample_rate * 10)) {

        if (have_nonzero) {
            printf("switching on capture\n");
        } else {
            while (port_index < capture_ports.size()) {

                jack_port_t *port = capture_ports[port_index];

                float *buf = (float *)jack_port_get_buffer(port, nframes);
                for (i = 0; i < int(nframes); i++) {
                    buf[i] = 0.f;
                }

                port_index++;
            }
            return 0;
        }
    }
    if (have_nonzero) {
        last_capture_nonzero_frame = last_capture_frame;
    }

    port_index = 0;

    while (port_index < capture_ports.size()) {

        jack_port_t *port = capture_ports[port_index];
        float *buf = (float *)jack_port_get_buffer (port, nframes);

        SRC_STATE *src_state = capture_srcs[port_index];
        SRC_DATA src;

        for (i = 0; i < rlen; i++) {
            resampbuf[i] = (float) outbuf[chn + i*num_channels] / 32767;
        }

        src.data_in = resampbuf;
        src.input_frames = rlen;

        src.data_out = buf;
        src.output_frames = nframes;
        src.end_of_input = 0;

        //src.src_ratio = (float) nframes / frlen;
        src.src_ratio = current_resample_factor;

        //src_set_ratio( src_state, src.src_ratio );
        src_process( src_state, &src );

        put_back_samples = rlen-src.input_frames_used;

        if( src.output_frames_gen != int(nframes) )
            if (debug_mode) printf( "did not fill jack_buffer...\n" );

        port_index++;
        chn++;
    }

    //printf( "putback = %d\n", put_back_samples );
    snd_pcm_rewind( alsa_capture_handle, put_back_samples );

    return 0;      
}

int process (jack_nframes_t nframes, void *arg)
{
    if (!playback_only) process_capture(nframes, arg);
    if (!capture_only) process_playback(nframes, arg);
    return 0;
}

/**
 * Allocate the necessary jack ports...
 */

void alloc_ports( int n_capture, int n_playback ) {

    int port_flags = JackPortIsOutput;
    int chn;
    jack_port_t *port;
    char buf[32];

    capture_ports.clear();

    for (chn = 0; chn < n_capture; chn++) {

        snprintf (buf, sizeof(buf) - 1, "capture_%u", chn+1);

        port = jack_port_register (client, buf,
                                   JACK_DEFAULT_AUDIO_TYPE,
                                   port_flags, 0);

        if (!port) {
            fprintf(stderr,  "jack_alsa_bridge: cannot register port for %s", buf);
            break;
        }

        capture_ports.push_back(port);
        capture_srcs.push_back(src_new(SRC_SINC_FASTEST, 1, NULL));
    }

    port_flags = JackPortIsInput;

    playback_ports.clear();

    for (chn = 0; chn < n_playback; chn++) {

        snprintf (buf, sizeof(buf) - 1, "playback_%u", chn+1);

        port = jack_port_register (client, buf,
                                   JACK_DEFAULT_AUDIO_TYPE,
                                   port_flags, 0);

        if (!port) {
            fprintf(stderr,  "jack_alsa_bridge: cannot register port for %s", buf);
            break;
        }

        playback_ports.push_back(port);
        playback_srcs.push_back(src_new(SRC_SINC_FASTEST, 1, NULL));
    }
}

/**
 * This is the shutdown callback for this JACK application.
 * It is called by JACK if the server ever shuts down or
 * decides to disconnect the client.
 */

void jack_shutdown (void *arg)
{
    fprintf(stderr, "jack_shutdown called\n");
    processing = 0;
}

int jack_setup()
{
    if ((client = jack_client_new(jack_name)) == 0) {
        fprintf (stderr, "duplicate client name, or jack server not running?\n");
        return 1;
    }

    /* tell the JACK server to call `jack_shutdown()' if
       it ever shuts down, either entirely, or if it
       just decides to stop calling us.
    */

    jack_on_shutdown (client, jack_shutdown, 0);

    // alloc input ports, which are blasted out to alsa...
    alloc_ports(playback_only ? 0 : num_channels,
                 capture_only ? 0 : num_channels);

    // get jack sample_rate
    
    jack_sample_rate = jack_get_sample_rate( client );

    if( !sample_rate )
        sample_rate = jack_sample_rate;

    current_resample_factor = (double) sample_rate / (double) jack_sample_rate;

    return 0;
}  

int alsa_setup(char *alsa_device, int playback_only, int sample_rate, int num_channels, int period_size, int num_periods)
{
    if (capture_only) {
        alsa_playback_handle = 0;
    } else {
        alsa_playback_handle = open_audiofd
            (alsa_device, 0, sample_rate, num_channels, period_size, num_periods);
        if (alsa_playback_handle < 0) return 1;
    }

    if (playback_only) {
        alsa_capture_handle = 0;
    } else {
        alsa_capture_handle = open_audiofd
            (alsa_device, 1, sample_rate, num_channels, period_size, num_periods);
        if (alsa_capture_handle < 0) return 1;
    }
    
    return 0;
}
 
/**
 * be user friendly.
 * be user friendly.
 * be user friendly.
 */

void printUsage(const char *my_name) {

    fprintf(stderr, "usage: %s [options]\n"
            "\n"
            "  -j <jack name> - reports a different name to jack\n"
            "  -d <alsa_device> \n"
            "  -c <channels> \n"
            "  -p <period_size> \n"
            "  -n <num_period> \n"
            "  -r <sample_rate> \n"
            "  -m <max_diff> \n"
            "  -t <target_delay> \n"
            "  -f <catch_factor> \n"
            "  -P - playback only\n"
            "  -C - capture only\n"
            "  -v - verbose output\n"
            "  -z - reconnect after jack restart\n"
            "\n",
            my_name);
}


/**
 * the main function....
 */


int main (int argc, char *argv[]) {

//    char jack_name[30] = "alsa_aux";
    char alsa_device[30] = "hw:0";

    extern char *optarg;
    extern int optopt;
    int errflg=0;
    int c;

    int period_size_specified = 0;
    int jack_name_specified = 0;

    int persistent = 0;

    while ((c = getopt(argc, argv, ":j:r:c:p:n:d:m:t:f:CPvz")) != -1) {
        switch(c) {
        case 'j':
//            strncpy(jack_name, optarg, 29);
//            jack_name[29] = '\0';
            jack_name = strdup(optarg);
            jack_name_specified = 1;
            break;
        case 'r':
            sample_rate = atoi(optarg);
            break;
        case 'c':
            num_channels = atoi(optarg);
            break;
        case 'p':
            period_size = atoi(optarg);
            period_size_specified = 1;
            break;
        case 'n':
            num_periods = atoi(optarg);
            break;
        case 'd':
            strncpy(alsa_device, optarg, 29);
            alsa_device[29] = '\0';
            break;
        case 't':
            target_delay = atoi(optarg);
            break;
        case 'm':
            max_diff = atoi(optarg);
            break;
        case 'f':
            catch_factor = atoi(optarg);
            break;
        case 'C':
            capture_only = 1;
            break;
        case 'P':
            playback_only = 1;
            break;
        case 'v':
            debug_mode = 1;
            break;
        case 'z':
            persistent = 1;
            break;
        case ':':
            fprintf(stderr,
                    "Option -%c requires an operand\n", optopt);
            errflg++;
            break;
        case '?':
            fprintf(stderr,
                    "Unrecognized option: -%c\n", optopt);
            errflg++;
        }
    }
    if (errflg) {
        printUsage(argv[0]);
        exit(2);
    }

    if (capture_only && playback_only) {
        capture_only = playback_only = 0;
    }

    // Setup target delay and max_diff for the normal user, who does not play with them...

    if( !target_delay ) 
        target_delay = num_periods*period_size / 2;

    if( !max_diff )
        max_diff = period_size / 2;	

    if (!jack_name_specified) {
        if (capture_only) {
            jack_name = strdup("alsa_in");
//            strcpy(jack_name, "alsa_in");
        } else if (playback_only) {
            jack_name = strdup("alsa_out");
//            strcpy(jack_name, "alsa_out");
        }
    }

    do {

        while (jack_setup()) {
            fprintf(stderr, "JACK setup failed");
            if (persistent) {
                fprintf(stderr, ", waiting...\n");
                sleep (5);
            } else {
                fprintf(stderr, "\n");
                return 1;
            }
        }

        if (!period_size_specified) {
            period_size = jack_get_buffer_size(client);
        }

        if (alsa_setup(alsa_device, playback_only, sample_rate, num_channels,
                       period_size, num_periods)) {
            fprintf(stderr, "ALSA setup failed\n");
            return 1;
        }
        
        /* tell the JACK server to call `process()' whenever
           there is work to be done.
        */

        jack_set_process_callback (client, process, 0);

        processing = 1;

        /* tell the JACK server that we are ready to roll */
        
        if (jack_activate (client)) {
            fprintf (stderr, "cannot activate client\n");
            return 1;
        }

        while (processing) sleep(1);

        // We don't want to jack_client_close, because this should
        // only be reached when jack_shutdown callback has been called
        client = 0;

        fprintf(stderr, "unloading\n");

        if (dynamic_jack_unload()) {
            fprintf(stderr, "failed to unload JACK\n");
            return 1;
        }

        int err;
        if (alsa_playback_handle) {
            if ((err = snd_pcm_close(alsa_playback_handle)) < 0) {
                fprintf(stderr, "Playback ALSA close error: %s\n", snd_strerror(err));
                return 1;
            }
            alsa_playback_handle = 0;
        }
        if (alsa_capture_handle) {
            if ((err = snd_pcm_close(alsa_capture_handle)) < 0) {
                fprintf(stderr, "Capture ALSA close error: %s\n", snd_strerror(err));
                return 1;
            }
            alsa_capture_handle = 0;
        }
    } while (persistent);

    exit (0);
}

