// -*- C++ -*-
/* Copyright (c) 2001 Fumitoshi UKAI <ukai@debian.or.jp>

This file is part of groff.

groff is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

groff is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#ifndef ENCODING_H
#define ENCODING_H

#include <config.h>

#ifdef ENABLE_MULTIBYTE
typedef int wchar;	// negative is used for charcode & index
#else
typedef char wchar;
#endif

#include <stdio.h>

#ifdef __cplusplus
class encoding_istream {
public:
  encoding_istream() {};
  virtual ~encoding_istream() {};
  virtual int getbyte() = 0;
  virtual int peekbyte() = 0;
  virtual void ungetbyte(int ch) = 0;
};

class encoding_istream_str: public encoding_istream {
private:
  const unsigned char *s;
  int *i;
  encoding_istream_str() {};
public:
  encoding_istream_str(const unsigned char *s0, int *i0) : s(s0), i(i0) {};
  ~encoding_istream_str() {};
  inline int getbyte() { return s[(*i)++]; };
  inline int peekbyte() { return s[(*i)]; };
  inline void ungetbyte(int ch) { --(*i); };
};

class encoding_istream_fp: public encoding_istream {
private:
  FILE *fp;
public:
  encoding_istream_fp(FILE *fp0) : fp(fp0) {};
  ~encoding_istream_fp() {};
  inline int getbyte() { return fgetc(fp); };
  inline int peekbyte() { int ch = fgetc(fp); ungetc(ch, fp); return ch; };
  inline void ungetbyte(int ch) { ungetc(ch, fp); };
};

class encoding_ostream {
public:
  encoding_ostream() {};
  virtual ~encoding_ostream() {};
  virtual void putbyte(unsigned char ch) = 0;
};

class encoding_ostream_str: public encoding_ostream {
private:
  unsigned char *s;
  int *i;
  int len;
  encoding_ostream_str() {};
public:
  encoding_ostream_str(unsigned char *s0, int *i0, int max) : s(s0), i(i0), len(max) {};
  ~encoding_ostream_str() {};
  inline void putbyte(unsigned char ch) {
    if (*i < len)
      s[(*i)++] = ch;
  }
};

class encoding_ostream_fp: public encoding_ostream {
private:
  FILE *fp;
  const char *format;
public:
  encoding_ostream_fp(FILE *ofp, const char *fmt = "%c") : fp(ofp), format(fmt) {};
  ~encoding_ostream_fp() {};
  inline void putbyte(unsigned char ch) {
    fprintf(fp, format, ch);
  }
};

class encoding_handler {
public:
  encoding_handler() {};
  virtual ~encoding_handler() {};

  // name of this encoding_handler
  virtual const char *name() { return ""; };

  // check if this byte is byte in multibyte character in this encoding?
  virtual int is_wchar_byte(unsigned char c) { return 0; };

  // make new wchar from c0 (beginning of multibytes) and rest from `in'
  virtual wchar make_wchar(unsigned char c0, encoding_istream& in) { 
    return wchar(c0);
  }
  // make new wchar from c0 (beginning of multibytes) and rest from `fp'
  virtual wchar make_wchar(unsigned char c0, FILE *fp) {
    encoding_istream_fp in(fp);
    return make_wchar(c0, in);
  }
  // make new wchar from c0 (beginning of multibtyes) and rest from
  // s[*i], *i will be changed to point the byte of next character.
  virtual wchar make_wchar(unsigned char c0, const unsigned char *s, int *i) {
    encoding_istream_str in(s, i);
    return make_wchar(c0, in);
  }

  // put wchar to outputstream
  // returns number of bytes written
  virtual int put_wchar(wchar wc, encoding_ostream& eos) {
    eos.putbyte((unsigned char)wc);
    return 1;
  }
  // put wchar to `fp' using `fmt'
  // returns number of bytes written
  virtual int put_wchar(wchar wc, FILE *fp, const char *fmt = "%c") {
    encoding_ostream_fp out(fp, fmt);
    return put_wchar(wc, out);
  }
  // put wchar to s[*i] (until maxlen)
  // *i will be changed to point the byte of next character.
  virtual int put_wchar(wchar wc, unsigned char *s, int *i, int maxlen) {
    encoding_ostream_str out(s, i, maxlen);
    return put_wchar(wc, out);
  }

  // maximum number of bytes of multibyte character in this encoding
  virtual int max_wchar_len() { return 1; };

};

encoding_handler* select_input_encoding_handler(const char* encoding_name);
encoding_handler* select_output_encoding_handler(const char* encoding_name);
extern encoding_handler* input_encoding;
extern encoding_handler* output_encoding;
void init_encoding_handler();

// check if wc is wchar?
int is_wchar_code(wchar wc);

// check if wc is wchar & can be represented in single byte?
int is_wchar_singlebyte(wchar wc);

// get singlebyte representation of wchar (if is_wchar_singlebyte(wc))
unsigned char wchar_singlebyte(wchar wc);

// get actual wide character code
int wchar_code(wchar wc);

// make wchar from wide character code
int make_wchar(int w);

#endif

#endif /* ENCODING_H */
