/*
 * IEEE 1394 commander - tool for simple access to IEEE 1394 bus
 *                       using the Linux 1394 subsystem.
 *
 * Copyright (C) 2002-2007, Manfred Weihs <mweihs@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>

#ifdef HAVE_LIBREADLINE
#  if defined(HAVE_READLINE_READLINE_H)
#    include <readline/readline.h>
#  elif defined(HAVE_READLINE_H)
#    include <readline.h>
#  else /* !defined(HAVE_READLINE_H) */
extern char *readline ();
#  endif /* !defined(HAVE_READLINE_H) */
#else /* !defined(HAVE_READLINE_READLINE_H) */
/* no readline */
#endif /* HAVE_LIBREADLINE */

#ifdef HAVE_READLINE_HISTORY
#  if defined(HAVE_READLINE_HISTORY_H)
#    include <readline/history.h>
#  elif defined(HAVE_HISTORY_H)
#    include <history.h>
#  else /* !defined(HAVE_HISTORY_H) */
extern void add_history ();
extern int write_history ();
extern int read_history ();
#  endif /* defined(HAVE_READLINE_HISTORY_H) */
/* no history */
#endif /* HAVE_READLINE_HISTORY */

#include <errno.h>
#include <string.h>
#include <stdlib.h>

#include <libraw1394/raw1394.h>

const char not_compatible[] =
"This libraw1394 does not work with your version of Linux. You need a different\n"
"version that matches your kernel (see kernel help text for the raw1394 option to\n"
"find out which is the correct version).\n";

const char not_loaded[] =
"This probably means that you don't have raw1394 support in the kernel or that\n"
"you haven't loaded the raw1394 module.\n";

raw1394handle_t handle;

#define WRITE 1
#define READ  2
#define LOCK  3
#define BUSRESET 4
#define HELP 5
#define PHY 6
#define INFO 7
#define PRINTVERSION 8

static int bus_reset_handler(struct raw1394_handle *handle, unsigned int gen)
{
	fprintf(stdout,"\nbus reset occurred.\nnew generation number: %d, %d nodes on the bus, local ID: %d\a\n",
	        gen, raw1394_get_nodecount(handle), raw1394_get_local_id(handle) & 0x3f);
        raw1394_update_generation(handle, gen);
#ifdef HAVE_LIBREADLINE
	(void) rl_on_new_line();
	rl_redisplay();
#endif
        return 0;
}

static void reset_bus(int short_br)
{
	if (raw1394_reset_bus_new(handle, short_br?RAW1394_SHORT_RESET:RAW1394_LONG_RESET)) {
		fprintf(stderr,"cannot reset bus!\n");
	} else {
		fprintf(stdout,"successfully reset the bus (%s bus reset)\n",short_br?"short":"long");
	}
}

static void print_help()
{
	fprintf(stdout,"available commands:\n");
	fprintf(stdout,"help                                ... display this help\n");
	fprintf(stdout,"br                                  ... reset the bus\n");
	fprintf(stdout,"br short                            ... reset the bus (short 1394a bus reset)\n");
	fprintf(stdout,"r bus node offset size              ... read from a node\n");
	fprintf(stdout,"w bus node offset size data         ... write from a node\n");
	fprintf(stdout,"l bus node offset extTCode data arg ... lock a node\n");
	fprintf(stdout,"p data                              ... send phy packet\n");
	fprintf(stdout,"i                                   ... print information (nr. of nodes etc.)\n");
	fprintf(stdout,"v                                   ... print version information\n");
	fprintf(stdout,"e                                   ... exit\n\n");
	fprintf(stdout,"For bus and node arguments '.' can be used as local bus or local node.\n\n");
}


static void interpret_response()
{
	int errcode=raw1394_get_errcode(handle);
	if (errcode < 0) {
		switch (errcode) {
			case -1003:
				fprintf(stdout,"Busreset occured\n");
				break;
			case -1100:
				fprintf(stdout,"Send error\n");
				break;
			case -1101:
				fprintf(stdout,"Aborted\n");
				break;
			case -1102:
				fprintf(stdout,"Timeout\n");
				break;
			default:
				fprintf(stdout,"internal raw1394 error\n");
		}
	} else {
		switch(raw1394_get_ack(errcode)) {
			case 0:
				fprintf(stdout,"Ack code: reserved code 0\n");
				break;
			case 1:
				fprintf(stdout,"Ack code: complete\n");
				break;
			case 2:
				fprintf(stdout,"Ack code: pending; ");
				switch (raw1394_get_rcode(errcode)) {
					case 0:
						fprintf(stdout,"Response code: complete\n");
						break;
					case 1:
						fprintf(stdout,"Response code: reserved code 1\n");
						break;
					case 2:
						fprintf(stdout,"Response code: reserved code 2\n");
						break;
					case 3:
						fprintf(stdout,"Response code: reserved code 3\n");
						break;
					case 4:
						fprintf(stdout,"Response code: conflict error\n");
						break;
					case 5:
						fprintf(stdout,"Response code: data error\n");
						break;
					case 6:
						fprintf(stdout,"Response code: type error\n");
						break;
					case 7:
						fprintf(stdout,"Response code: address error\n");
						break;
					case 8:
						fprintf(stdout,"Response code: reserved code 8\n");
						break;
					case 9:
						fprintf(stdout,"Response code: reserved code 9\n");
						break;
					case 10:
						fprintf(stdout,"Response code: reserved code A\n");
						break;
					case 11:
						fprintf(stdout,"Response code: reserved code B\n");
						break;
					case 12:
						fprintf(stdout,"Response code: reserved code C\n");
						break;
					case 13:
						fprintf(stdout,"Response code: reserved code D\n");
						break;
					case 14:
						fprintf(stdout,"Response code: reserved code E\n");
						break;
					case 15:
						fprintf(stdout,"Response code: reserved code F\n");
						break;
					default:
						fprintf(stderr,"error getting response code\n");
				}
				break;
			case 3:
				fprintf(stdout,"Ack code: reserved code 3\n");
				break;
			case 4:
				fprintf(stdout,"Ack code: busy_X\n");
				break;
			case 5:
				fprintf(stdout,"Ack code: busy_A\n");
				break;
			case 6:
				fprintf(stdout,"Ack code: busy_B\n");
				break;
			case 7:
				fprintf(stdout,"Ack code: reserved code 7\n");
				break;
			case 8:
				fprintf(stdout,"Ack code: reserved code 8\n");
				break;
			case 9:
				fprintf(stdout,"Ack code: reserved code 9\n");
				break;
			case 10:
				fprintf(stdout,"Ack code: reserved code A\n");
				break;
			case 11:
				fprintf(stdout,"Ack code: tardy\n");
				break;
			case 12:
				fprintf(stdout,"Ack code: conflict_error\n");
				break;
			case 13:
				fprintf(stdout,"Ack code: data error\n");
				break;
			case 14:
				fprintf(stdout,"Ack code: type error\n");
				break;
			case 15:
				fprintf(stdout,"Ack code: address_error\n");
				break;
			default:
				fprintf(stderr,"error getting ack code\n");
		}
	}
}

static void read_node(unsigned int bus, unsigned int node, unsigned long long int offset, unsigned int bytes)
{
	unsigned char buffer[bytes];
	nodeid_t mynode=(bus << 6) | node;
	unsigned int retval;
	unsigned int i;

	fprintf(stdout,"reading from node %u, bus %u, offset %#.12llX %u bytes\n", node, bus, offset, bytes);
	retval=raw1394_read(handle, mynode, offset, bytes, (quadlet_t*) buffer);
	if (retval)
		fprintf(stdout,"read failed.\n");
	else {
		fprintf(stdout,"read succeeded. Data follows (hex):\n");
		for (i=0; i < bytes; i++) {
			fprintf(stdout,(i+1) % 4?"%.2X ":"%.2X\n",buffer[i]);
		}
		if (i % 4) fprintf(stdout,"\n");
	}
	interpret_response();
}

static void write_node(unsigned int bus, unsigned int node, unsigned long long int offset, unsigned int bytes, unsigned long long int data)
{
	unsigned char buffer[bytes];
	nodeid_t mynode=(bus << 6) | node;
	unsigned int retval;
	unsigned int i;

	for (i=bytes; i; i--) {
		buffer[i-1] = (unsigned char) (data & 0xff);
		data >>= 8;
	}

	fprintf(stdout,"writing to  node %u, bus %u, offset %#.12llX %u bytes:\n", node, bus, offset, bytes);
	for (i=0; i < bytes; i++) {
		fprintf(stdout,(i+1) % 4?"%.2X ":"%.2X\n",buffer[i]);
	}
	if (i % 4) fprintf(stdout,"\n");
	if (data)
		fprintf(stdout,"warning: too much data provided for %u bytes; only use lower significant bytes!\n", bytes);

	retval=raw1394_write(handle, mynode, offset, bytes, (quadlet_t*) buffer);
	if (retval)
		fprintf(stdout,"write failed.\n");
	else {
		fprintf(stdout,"write succeeded.\n");
	}
	interpret_response();
}

static void lock_node(unsigned int bus, unsigned int node, unsigned long long int offset, unsigned int extTCode, unsigned long long int data, unsigned long long int arg)
{
	unsigned char buffer_data[4];
	unsigned char buffer_arg[4];
	unsigned char buffer[4];
	nodeid_t mynode=(bus << 6) | node;
	unsigned int retval;
	unsigned int i;

	/* do byte swapping */
	for (i=4; i; i--) {
		buffer_data[i-1] = (unsigned char) (data & 0xff);
		data >>= 8;
	}

	for (i=4; i; i--) {
		buffer_arg[i-1] = (unsigned char) (arg & 0xff);
		arg >>= 8;
	}

	fprintf(stdout,"locking  node %u, bus %u, offset %#.12llX, extended TCode: %u, data:\n", node, bus, offset, extTCode);
	for (i=0; i < 4; i++) {
		fprintf(stdout,"%.2X ",buffer_data[i]);
	}
	fprintf(stdout,"\narg:\n");
	for (i=0; i < 4; i++) {
		fprintf(stdout,"%.2X ",buffer_arg[i]);
	}
	fprintf(stdout,"\n");

	retval=raw1394_lock(handle, mynode, offset, extTCode, *((quadlet_t*) buffer_data),  *((quadlet_t*) buffer_arg), (quadlet_t*) buffer);
	if (retval)
		fprintf(stdout,"lock failed.\n");
	else {
		fprintf(stdout,"lock succeeded. Data follows (hex):\n");
		for (i=0; i < 4; i++) {
			fprintf(stdout,"%.2X ",buffer[i]);
		}
		fprintf(stdout,"\n");
	}
	interpret_response();
}

static void send_phy_packet(unsigned long int data)
{
	unsigned char buffer[4];
	unsigned int retval;
	unsigned int i;

	for (i=4; i; i--) {
		buffer[i-1] = (unsigned char) (data & 0xff);
		data >>= 8;
	}

	fprintf(stdout,"send phy packet:\n");
	for (i=0; i < 4; i++) {
		fprintf(stdout,"%.2X ",buffer[i]);
	}
	fprintf(stdout,"\n");

	retval=raw1394_phy_packet_write(handle, *((quadlet_t*) buffer));
	if (retval)
		fprintf(stdout,"send phy packet failed.\n");
	else {
		fprintf(stdout,"send phy packet succeeded.\n");
	}
	interpret_response();
}

static void print_info()
{
	fprintf(stdout,"found: %d nodes on bus, local ID is %d, IRM is %d\n",
	        raw1394_get_nodecount(handle),
	        raw1394_get_local_id(handle) & 0x3f,
	        raw1394_get_irm_id(handle) & 0x3f);
	fprintf(stdout,"current generation number (adapter): %d\n", raw1394_get_generation(handle));
}

static void print_version()
{
	fprintf(stdout, "%s %s\n", PACKAGE, VERSION);
}


static void process_command(char *command_str)
{
	char *token;
	int command;
	unsigned int bus;
	unsigned int node;
	unsigned long long int offset;
	unsigned int bytes;
	unsigned long long int data, arg;
	unsigned long int phy_data;
	char *endptr;
      

	token = strtok(command_str, " \n");
	if (!token)
		return;

	if (!strcmp(token,"w"))
		command = WRITE;
	else if (!strcmp(token, "r"))
		command = READ;
	else if (!strcmp(token, "l"))
		command = LOCK;
	else if (!strcmp(token, "p"))
		command = PHY;
	else if (!strcmp(token, "br"))
		command = BUSRESET;
	else if (!strcmp(token, "i"))
		command = INFO;
	else if (!strcmp(token, "v"))
		command = PRINTVERSION;
	else if (!strcmp(token, "help"))
		command = HELP;
	else {
		fprintf(stdout,"illegal command!\n");
		return;
	}

	switch (command) {
		case HELP:
			if (strtok(NULL, " ")) {
				fprintf(stderr,"too many arguments\n");
				break;
			}
			print_help();
			break;
		case BUSRESET:
			if ((token = strtok(NULL, " "))) {
				if (strcmp(token, "short")) {
					fprintf(stderr, "invalid argument\n");
					break;
				}
				if (strtok(NULL, " ")) {
					fprintf(stderr,"too many arguments\n");
					break;
				}
				reset_bus(1);
				break;
			}
			reset_bus(0);
			break;
		case PHY:
			if (!(token = strtok(NULL, " "))) {
				fprintf(stderr,"insufficient arguments for operation!\n");
				break;
			}
			phy_data = strtoul(token, &endptr, 0);
			if (*endptr != '\0') {
				fprintf(stderr,"invalid argument\n");
				break;
			}
			if (strtok(NULL, " ")) {
				fprintf(stderr,"too many arguments\n");
				break;
			}
			send_phy_packet(phy_data);
			break;
		case READ:
		case WRITE:
		case LOCK:
			/* parse bus id */
			if (!(token = strtok(NULL, " "))) {
				fprintf(stderr,"insufficient arguments for operation!\n");
				break;
			}

			if (strcmp(token,".")) {
				bus = strtoul(token, &endptr, 0) & 0x3ff;
				if (*endptr != '\0') {
					fprintf(stderr,"invalid argument\n");
					break;
				}
			} else {
				bus=1023;
			}
			/* parse node id */
			if (!(token = strtok(NULL, " "))) {
				fprintf(stderr,"insufficient arguments for operation!\n");
				break;
			}

			if (strcmp(token,".")) {
				node = strtoul(token, &endptr, 0) &0x3f;
				if (*endptr != '\0') {
					fprintf(stderr,"invalid argument\n");
					break;
				}
			} else {
				node=raw1394_get_local_id(handle) & 0x3f;
			}
			/* parse offset */
			if (!(token = strtok(NULL, " "))) {
				fprintf(stderr,"insufficient arguments for operation!\n");
				break;
			}
  
			offset = strtoull(token, &endptr, 0);
			if (*endptr != '\0') {
				fprintf(stderr,"invalid argument\n");
				break;
			}

			/* parse bytes, for lock, it is the extended TCODE */
			if (!(token = strtok(NULL, " "))) {
				fprintf(stderr,"insufficient arguments for operation!\n");
				break;
			}

			bytes = strtoul(token, &endptr, 0);
			if (*endptr != '\0') {
				fprintf(stderr,"invalid argument\n");
				break;
			}

			if ((command == WRITE) || (command == LOCK)) {
				/* parse data */
				if (!(token = strtok(NULL, " "))) {
					fprintf(stderr,"insufficient arguments for operation!\n");
					break;
				}

				data = strtoull(token, &endptr, 0);
				if (*endptr != '\0') {
					fprintf(stderr,"invalid argument\n");
					break;
				}
			}

			if (command == LOCK) {
				/* parse arg */
				if (!(token = strtok(NULL, " "))) {
					fprintf(stderr,"insufficient arguments for operation!\n");
					break;
				}

				arg = strtoull(token, &endptr, 0);
				if (*endptr != '\0') {
					fprintf(stderr,"invalid argument\n");
					break;
				}
			}

			if (strtok(NULL, " ")) {
				fprintf(stderr,"too many arguments\n");
				break;
			}

			switch (command) {
				case READ:
					read_node(bus, node, offset, bytes);
					break;
				case WRITE:
					write_node(bus, node, offset, bytes, data);
					break;
				case LOCK:
					lock_node(bus, node, offset, bytes, data, arg);
					break;
				default:
					fprintf(stderr,"internal error\n");
			}


			break;
		case INFO:
			if (strtok(NULL, " ")) {
				fprintf(stderr,"too many arguments\n");
				break;
			}
			print_info();
			break;
		case PRINTVERSION:
			if (strtok(NULL, " ")) {
				fprintf(stderr,"too many arguments\n");
				break;
			}
			print_version();
			break;
		default:
		fprintf(stdout,"operation not implemented yet\n");
	}
}

static int done = 0;

#ifdef HAVE_LIBREADLINE
static void rl_handler(char *command) {
	if (!command) {
		fprintf(stdout,"reached EOF -> exit\n");
		rl_callback_handler_remove();
		done=1;
	} else {
#ifdef HAVE_READLINE_HISTORY
		if (*command) {
			add_history(command);
		}
#endif
		if (strcmp(command,"e")) {
			process_command(command);
		} else {
			rl_callback_handler_remove();
			done=1;
		}
		free(command);
	}
}
#endif

int main(int argc, char **argv)
{
	int i, numcards;
	struct raw1394_portinfo pinf[16];
#ifndef HAVE_LIBREADLINE
	char command[100];
#endif
	int AdapterNumber;

	print_version();
	fprintf(stdout,"Copyright (C) 2002-2007 by Manfred Weihs <mweihs@users.sourceforge.net>\n");
	fprintf(stdout,"This software comes with absolutely no warranty.\n\n");
	switch (argc)
	{
	case 1:
		AdapterNumber = 0;
		fprintf(stdout,"No adapter specified!\n");
		break;
	case 2:
		AdapterNumber = atoi(argv[1]);
		break;
	default:
		fprintf(stderr,"syntax error. You may only provide the number of the IEEE 1394 adapter:\n%s [adapterNumber]\n",argv[0]);
		return 1;
	}

	handle = raw1394_new_handle();

	if (!handle) {
		if (!errno) {
			fprintf(stdout,not_compatible);
		} else {
			perror("couldn't get handle");
			fprintf(stdout,not_loaded);
		}
		exit(1);
	}

	fprintf(stdout,"successfully got handle\n");
	fprintf(stdout,"current generation number (driver): %d\n", raw1394_get_generation(handle));

	numcards = raw1394_get_port_info(handle, pinf, 16);
	if (numcards < 0) {
		perror("couldn't get card info");
		exit(1);
	} else {
		fprintf(stdout,"%d card(s) found\n", numcards);
	}

	if (!numcards) {
		exit(0);
	}

	for (i = 0; i < numcards; i++) {
		fprintf(stdout,"  nodes on bus: %2d, card name: %s\n", pinf[i].nodes, pinf[i].name);
	}
        
	fprintf(stdout,"using adapter %d\n", AdapterNumber);
	if (raw1394_set_port(handle, AdapterNumber) < 0) {
		perror("couldn't set port");
		exit(1);
	}

	(void) raw1394_set_bus_reset_handler(handle, bus_reset_handler);
	print_info();

	fprintf(stdout,"\nentering command mode\n");
	fprintf(stdout,"Type 'help' for more information!\n");
#ifdef HAVE_LIBREADLINE
	rl_callback_handler_install("Command: ", rl_handler);
#endif
	while (!done) {
		fd_set rfds;
		FD_ZERO(&rfds);
		FD_SET(raw1394_get_fd(handle), &rfds);
		FD_SET(0, &rfds); /* stdin */
		
#ifndef HAVE_LIBREADLINE
		fprintf(stdout,"Command: ");
		fflush(stdout);
#endif
		if (select(raw1394_get_fd(handle)+1, &rfds, NULL, NULL, NULL) < 0) {
			switch (errno) {
			case EINTR: /* got some signal, try again */
				break;
			default:
				perror("\ninternal error: select on file descriptors did not work");
#ifdef HAVE_LIBREADLINE
				(void) rl_on_new_line();
				rl_redisplay();
#endif
			}
		} else {
			if (FD_ISSET(raw1394_get_fd(handle), &rfds)) {
				raw1394_loop_iterate(handle);
			}
			if (FD_ISSET(0, &rfds)) { /* stdin */
#ifdef HAVE_LIBREADLINE
				rl_callback_read_char();
#else
				if (!fgets(command, sizeof(command), stdin)) {
					fprintf(stdout,"reached EOF -> exit\n");
					done=1;
				} else {
					if (command[strlen(command)-1] == '\n') {
						command[strlen(command)-1] = '\0';
					}
					if (strcmp(command,"e")) {
						process_command(command);
					} else {
						done=1;
					}
				}
#endif
			}
		}
        }
	return 0;
}
