/*

  Kjetil Matheussen 2006.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <stdio.h>
#include <unistd.h>
#include <errno.h>

#include <pthread.h>
#include <pwd.h>

#include <sched.h>
#include <sys/mman.h>

#include <glibtop.h>

#include <glibtop/proclist.h>
#include <glibtop/procstate.h>
#if LIBGTOP_MAJOR_VERSION<2
#  include <glibtop/xmalloc.h>
#endif
#include <glibtop/procuid.h>
#include <glibtop/proctime.h>

#if LIBGTOP_MAJOR_VERSION<2
typedef u_int64_t ui64;
#else
typedef guint64 ui64;
#endif


#define OPTARGS_BEGIN(das_usage) {int lokke;const char *usage=das_usage;for(lokke=1;lokke<argc;lokke++){char *a=argv[lokke];if(!strcmp("--help",a)||!strcmp("-h",a)){printf(usage);return 0
#define OPTARG(name,name2) }else if(!strcmp(name,a)||!strcmp(name2,a)){
#define OPTARG_GETINT() atoi(argv[++lokke])
#define OPTARG_GETFLOAT() atof(argv[++lokke])
#define OPTARG_GETSTRING() argv[++lokke]
#define OPTARG_LAST() }else if(lokke==argc-1){lokke--;
#define OPTARGS_END }else{fprintf(stderr,usage);return(-1);}}}


static int increasetime=1; // Seconds between each time the SCHED_OTHER thread is increasing the counter.
static int checktime=4;    // Seconds between each time the SCHED_FIFO thread checks that the counter is increased.
static int waittime=8;     // Seconds the SCHED_FIFO thread waits before setting the processes back to SCHED_FIFO.

struct das_proclist{
  pid_t pid;
  int policy; //SCHED_OTHER, SCHED_FIFO, SHED_RR
  int priority;
  ui64 start_time; // Creation time of the process.
};

static int verbose=0;
static int counter=0;
static int checkirq=0;


static ui64 get_pid_start_time(int pid){
  glibtop_proc_time buf={0};
  glibtop_get_proc_time(&buf,pid);
  return buf.start_time;
}


static int get_pid_priority(int pid){
  struct sched_param par;
  sched_getparam(pid,&par);
  return par.sched_priority;
}

static int set_pid_priority(int pid,int policy,int priority,char *message,char *name){
  struct sched_param par={0};
  par.sched_priority=priority;
  if((sched_setscheduler(pid,policy,&par)!=0)){
    fprintf(stderr,message,pid,name,strerror(errno));
    return 0;
  }
  return 1;
}


struct das_proclist *get_proclist(int *num_procs){
  int lokke=0;
  glibtop_proclist proclist_def={0};
  unsigned *proclist=glibtop_get_proclist(&proclist_def,GLIBTOP_KERN_PROC_ALL,0); //|GLIBTOP_EXCLUDE_SYSTEM,0);
  struct das_proclist *ret=calloc(sizeof(struct das_proclist),proclist_def.number);

  *num_procs=proclist_def.number;

  for(lokke=0;lokke<proclist_def.number;lokke++){
    ret[lokke].pid=proclist[lokke];
    ret[lokke].policy=sched_getscheduler(proclist[lokke]);
    ret[lokke].priority=get_pid_priority(proclist[lokke]);
    ret[lokke].start_time=get_pid_start_time(proclist[lokke]);
  }

#if LIBGTOP_MAJOR_VERSION<2
  glibtop_free(proclist);
#else
  g_free(proclist);
#endif

  return ret;
}



static int name2pid(char *name){
  int pid=-1;
  int lokke;
  int num_procs=0;
  struct das_proclist *proclist=get_proclist(&num_procs);
    
  for(lokke=0;lokke<num_procs;lokke++){
    glibtop_proc_state state;
    glibtop_get_proc_state(&state,proclist[lokke].pid);
    if(!strcmp(state.cmd,name)){
      pid=proclist[lokke].pid;
      break;
    }
  }
  free(proclist);
  return pid;
}



static int is_a_member(int val,int *vals,int num_vals){
  int lokke;
  for(lokke=0;lokke<num_vals;lokke++)
    if(val==vals[lokke])
      return 1;
  return 0;
}



// Returns a list of users that might be the one owning the proper .Xauthority file.
static int *get_userlist(struct das_proclist *proclist, int num_procs, int *num_users){
  int *ret=calloc(sizeof(int),num_procs);
  int lokke;

  *num_users=0;

  for(lokke=0;lokke<num_procs;lokke++){
    glibtop_proc_uid uid;
    glibtop_get_proc_uid(&uid,proclist[lokke].pid);
    if( ! is_a_member(uid.uid,ret,*num_users)){
      ret[*num_users]=uid.uid;
      (*num_users)++;
    }
  }
  return ret;
}



static int gettimerpid(char *name,int cpu){
  int pid;
  char temp[500];

  if(name==NULL)
    name=&temp[0];

  sprintf(name,"softirq-timer/%d",cpu);

  pid=name2pid(name);

  if(pid==-1){
    sprintf(name,"ksoftirqd/%d",cpu);
    pid=name2pid(name);
  }

  return pid;
}




static int checksoftirq2(int force,int cpu){
  char name[500];
  int pid=gettimerpid(&name[0],cpu);

  if(pid==-1) return 0;


  {
    int policy=sched_getscheduler(pid);
    int priority=get_pid_priority(pid);

    if(priority<sched_get_priority_max(SCHED_FIFO)
       || policy==SCHED_OTHER
       )
      {

	if(force){
	  printf("Forcing %s to SCHED_FIFO priority %d\n",name,sched_get_priority_max(SCHED_FIFO));
	  set_pid_priority(pid,SCHED_FIFO,sched_get_priority_max(SCHED_FIFO),"Could not set %d (\"%s\") to SCHED_FIFO (%s).\n\n",name);
	  return checksoftirq2(0,cpu);
	}	  
	

	if(priority<sched_get_priority_max(SCHED_FIFO))
	  fprintf(stderr,
		  "\n\nWarning. The priority of the \"%s\" process is only %d, and not %d. Unless you are using the High Res Timer,\n"
		  "the watchdog will probably not work. If you are using the High Res Timer, please continue doing that and ignore this message.\n\n",
		  name,
		  priority,
		  sched_get_priority_max(SCHED_FIFO)
		  );
	if(policy==SCHED_OTHER)
	  fprintf(stderr,
		  "\n\nWarning The \"%s\" process is running SCHED_OTHER. Unless you are using the High Res Timer,\n"
		  "the watchdog will probably not work, and the timing on your machine is probably horrible.\n\n",
		  name
		  );

	if(checkirq)
	  printf("\n\nUnless you are using the High Res Timer, you need to add the \"--force\" flag to run das_watchdog reliably.\n\n");
	return -1;
      }
    //printf("name: -%s-\n",state.cmd);
    
    return 1;
  }
}


static int checksoftirq(int force){
  int cpu=0;

  for(;;){
    switch(checksoftirq2(force,cpu)){
    case -1:
      return -1;
    case 1:
      cpu++;
      break;
    case 0:
    default:
      return 0;
    }
  }
  return 0;
}




// The SCHED_OTHER thread.
static void *counter_func(void *arg){

  {
    // Eh, this is not necessary. But, well, just in case.
    set_pid_priority(0,SCHED_OTHER,0,"das_watchdog: Unable to set SCHED_other for %d (\"%s\"). (%s)\n", "the counter_func");
  }

  for(;;){
    counter++;
    if(verbose)
      fprintf(stderr,"counter set to %d\n",counter);
    sleep(increasetime);
  }
  return NULL;
}



int main(int argc,char **argv){
  pid_t mypid=getpid();
  pthread_t counter_thread={0};
  int num_cpus=0;
  int *timerpids;
  int force=0;
  int xmessage_found=1;

  // Get timer pids
  {
    // Find number of timer processes.
    while(gettimerpid(NULL,num_cpus)!=-1)
      num_cpus++;
    timerpids=malloc(sizeof(int)*num_cpus);

    {
      int cpu=0;
      for(cpu=0;cpu<num_cpus;cpu++)
	timerpids[cpu]=gettimerpid(NULL,cpu);
    }
  }


  // Options.
  OPTARGS_BEGIN("Usage: das_watchdog [--force] [--verbose] [--checkirq] [--increasetime n] [--checktime n] [--waittime n]\n"
		"                    [ -f]     [ -v]       [ -c]        [ -it n]           [ -ct n]        [ -wt n]\n");
  OPTARG("--force","-f") force=1;
  OPTARG("--verbose","-v") verbose=1;
  OPTARG("--checkirq","-c") checkirq=1; return(checksoftirq(0));
  OPTARG("--increasetime","-it") increasetime=OPTARG_GETINT();
  OPTARG("--checktime","-ct") checktime=OPTARG_GETINT();
  OPTARG("--waittime","-wt") waittime=OPTARG_GETINT();
  OPTARGS_END;
  

  // Check various.
  {
    if(force && checksoftirq(force)<0)
      return -2;

    checksoftirq(force);
    
    if(getuid()!=0){
      printf("Warning, you are not running as root. das_watchdog should be run as an init-script at startup, and not as a normal user.\n\n");
    }
    
    
    if(access(WHICH_XMESSAGE,X_OK)!=0){
      fprintf(stderr,"Warning, \"xmessage\" is not found or is not an executable. I will try to use the $PATH instead. Hopefully that'll work,\n");
      fprintf(stderr,"but you might not receive messages to the screen in case das_watchdog has to take action.\n");
      xmessage_found=0;
    }
  }


  // Set priority
  {
    if( ! set_pid_priority(0,SCHED_FIFO,sched_get_priority_max(SCHED_FIFO),
			   "das_watchdog: Unable to set SCHED_FIFO realtime priority for %d (\"%s\"). (%s). Exiting.\n",
			   "the watchdog thread"))
      return 0;
    if(mlockall(MCL_CURRENT|MCL_FUTURE)==-1)
      fprintf(stderr,"Could not call mlockalll(MCL_CURRENT|MCL_FUTURE) (%s)\n",strerror(errno));
  }


  // Start child thread.
  {
    pthread_create(&counter_thread,NULL,counter_func,NULL);
  }


  // Main loop. (We are never supposed to exit from this one.)
  for(;;){
    int lastcounter=counter;

    sleep(checktime);
    if(verbose)
      fprintf(stderr,"    counter read to be %d  (lastcounter=%d)\n",counter,lastcounter);
    
    if(lastcounter==counter){ // || lastcounter<5){ // || counter==2){
      int changedsched=0;
      int num_procs=0;
      struct das_proclist *proclist=get_proclist(&num_procs);
      int lokke;

      if(verbose)
	printf("Das_Watchdog: Watchdog kicking!\n");

      for(lokke=0;lokke<num_procs;lokke++){
	if(proclist[lokke].policy!=SCHED_OTHER 
	   && proclist[lokke].pid!=mypid 
	   && (!is_a_member(proclist[lokke].pid,timerpids,num_cpus))
	   )
	  {
	    struct sched_param par={0};
	    par.sched_priority=0;
	    if(verbose)
	      printf("Das_Watchdog: Setting pid %d temporarily to SCHED_OTHER.\n",proclist[lokke].pid);
	    if(set_pid_priority(proclist[lokke].pid,SCHED_OTHER,0,"Could not set pid %d (\"%s\") to SCHED_OTHER (%s).\n\n","no name"))
	      changedsched++;
	  }
      }

      if(changedsched>0){
	{
	  char temp[5000];
	  int num_users;
	  int lokke;
	  int *uids=get_userlist(proclist,num_procs,&num_users);
	  
	  if( ! xmessage_found)
	    sprintf(temp,"xmessage \"WARNING! das_watchdog pauses realtime operations for %d seconds.\" &",waittime);
	  else
	    sprintf(temp,"%s \"WARNING! das_watchdog pauses realtime operations for %d seconds.\" &",WHICH_XMESSAGE,waittime);

	  setenv("DISPLAY",":0.0",1);
	  
	  for(lokke=0;lokke<num_users;lokke++){
	    char xauthpath[5000];
	    struct passwd *pass=getpwuid(uids[lokke]);
	    sprintf(xauthpath,"%s/.Xauthority",pass->pw_dir);
	    if(access(xauthpath,R_OK)==0){
	      setenv("XAUTHORITY",xauthpath,1);
	      if(verbose)
		printf("Trying xauth like -%s-\n",pass->pw_dir);
	      system(temp);
	    }
	  }
	  
	  free(uids);
	}
	
	sleep(waittime);

	for(lokke=0;lokke<num_procs;lokke++){
	  if(proclist[lokke].policy!=SCHED_OTHER 
	     && proclist[lokke].pid!=mypid 
	     && (!is_a_member(proclist[lokke].pid,timerpids,num_cpus))
	     && proclist[lokke].start_time==get_pid_start_time(proclist[lokke].pid) 
	     )
	    {
	      if(get_pid_priority(proclist[lokke].pid)!=0 || sched_getscheduler(proclist[lokke].pid)!=SCHED_OTHER){
		fprintf(stderr,"Das_Watchdog: Someone else changed priority and/or scheduling policy for %d. I'm not going to set it back.\n",proclist[lokke].pid);
	      }else{
		struct sched_param par={0};
		par.sched_priority=proclist[lokke].priority;
		if(verbose)
		  printf("Das_Watchdog: Setting pid %d back to realtime priority.\n",proclist[lokke].pid);
		set_pid_priority(proclist[lokke].pid,proclist[lokke].policy,proclist[lokke].priority,"Could not set pid %d (\"%s\") to SCHED_FIFO/SCHED_RR (%s).\n\n", "no name");
	      }
	    }
	}
      }
      free(proclist);
    }
  }
  
  return 0;
}


