/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __PROCESS_H__
#define __PROCESS_H__

#include <qstringlist.h>

#include <kprocess.h>
#include <kregexp.h>

/*==================================*/
/*	DEFINES
 *==================================*/

enum {
	BLANK_DISK,
	BLANK_FAST,
};

/*========================================================*/

class ArsonConfig;

class ArsonProcessBase : public KProcess
{
	Q_OBJECT

public:
	ArsonProcessBase (const QString &type, RunMode rm,
		Communication comm, int program,
		const ArsonConfig &cfg);
	virtual ~ArsonProcessBase (void);

	virtual bool execute (void);

	void invalidate (const QString &errmsg = QString::null);
	bool successful (int *res = NULL) const;
	QString commandString (void);
	void appendExtra (void);

	ArsonProcessBase &deferArgs (const QStringList &args);
	ArsonProcessBase &deferArg (const QString &arg);
	ArsonProcessBase &resetDefered (void);

	ArsonProcessBase &appendParam (const QString &param)
	{ if (param != QString::null) (*this) << param; return (*this); }
	const QString &type (void) const { return m_strType; }
	const QCString &peekBuffer (void) const { return m_buffer; }

	ArsonProcessBase &appendArgList (const QStringList &sl);

	static QString quoteFile (const QString &filename);

protected:
	struct Output {
		Output(KRegExp *arr, size_t c)
			: re(arr), count(c) {}

		size_t count;
		KRegExp *re;
	};

	KRegExp *findRegexp (Output *arr, const QString &str, int *pindex = NULL);
	void setOutput (Communication comm, KRegExp *arr, size_t count);
	QCString bufferInput (const char *ptr, int len);
	void handleOutput (const char *ptr, int len, bool error);

	bool buffering (void) const { return m_bBuffer; }
	void setBuffering (bool buf) { m_bBuffer = buf; }
	
	virtual void output (const QString &str, bool error) {}
	virtual void onStdout (int index, KRegExp *pre) {}
	virtual void onStderr (int index, KRegExp *pre) {}

	virtual const ArsonConfig &config (void) const = 0;

	Output *m_outOut, *m_outErr;

private:
	const QString m_strType;
	QStringList m_defered;
	RunMode m_runmode;
	uint m_comm;
	int m_program;
	QCString m_buffer;
	bool m_bBuffer;

protected slots:
	virtual void slotGotStdout (KProcess *ptr, char *pbuf, int len);
	virtual void slotGotStderr (KProcess *ptr, char *pbuf, int len);
};

/*========================================================*/

class ArsonProcessMgr;
class ArsonProcessUI;

class ArsonProcess : public ArsonProcessBase
{
	Q_OBJECT

public:
	ArsonProcess (const QString &type, ArsonProcessMgr *pMgr, int program);
	virtual ~ArsonProcess (void);

	virtual bool execute (void);

	ArsonProcessUI *ui (void) const;

protected:
	virtual void output (const QString &str, bool error);
	virtual const ArsonConfig &config (void) const;
	
	ArsonProcessMgr *m_pWriter;

private:
	virtual bool start (RunMode runmode, Communication comm)
	{ return ArsonProcessBase::start(runmode, comm); }

protected slots:
	virtual void slotExited (KProcess *ptr);
};

/*========================================================*/

class ArsonUtilityProcess : public ArsonProcessBase
{
public:
	ArsonUtilityProcess (const ArsonConfig &cfg,
		Communication comm = NoCommunication,
		const QString &name = QString::null,
		int program = -1);

	virtual bool execute (void);
	
protected:
	virtual const ArsonConfig &config (void) const { return m_config; }
	
private:
	const ArsonConfig &m_config;
};

/*========================================================*/
/*	All processes that actually write to the CDR device
 *	should derive from this, so that the process can be
 *	nice'd if possible.
 *========================================================*/

class ArsonWriterProcess : public ArsonProcess
{
public:
	ArsonWriterProcess (ArsonProcessMgr *pWriter,
		int program,
		const QString &type = QString::null);

	bool simulate (void) const { return m_bSimulate; }
	void setSimulate (bool sim) { m_bSimulate = sim; }

	virtual bool execute (void);
	
private:
	virtual void setupSimulate (void) = 0;
	virtual void setupEject (void) = 0;

	virtual int commSetupDoneC (void);

	bool m_bSimulate;
};

/*========================================================*/

class ArsonCdrecordProcess : public ArsonWriterProcess
{
public:
	ArsonCdrecordProcess (ArsonProcessMgr *pWriter);

private:
	enum {
		EVENT_PAUSE,
		EVENT_PAUSE2,
		EVENT_STARTING,
		EVENT_MBWRITTEN,
		EVENT_MBPARTIAL,
		EVENT_TOTAL,
		EVENT_FIXATING,
		EVENT_PROMPT,
		EVENT_BLANK,
		EVENT_WRITETIME,
		_EVENT_MAX
	};

	virtual void onStdout (int type, KRegExp *pre);
	virtual QString driverOpts (void);
	virtual void setupSimulate (void);
	virtual void setupEject (void);

	static KRegExp events[_EVENT_MAX];

	uint m_partialBase;
	int m_trackNo;
};

class ArsonCdrecordDataProcess : public ArsonCdrecordProcess
{
public:
	ArsonCdrecordDataProcess (ArsonProcessMgr *pWriter, const QString &path);
};

class ArsonCdrecordIsosizeProcess : public ArsonCdrecordProcess
{
public:
	ArsonCdrecordIsosizeProcess (ArsonProcessMgr *pWriter, const QString &device);
};

class ArsonCdrecordBlankProcess : public ArsonCdrecordProcess
{
public:
	ArsonCdrecordBlankProcess (ArsonProcessMgr *pMgr, uint how);

	virtual bool execute (void);
};

class ArsonCdrecordFixateProcess : public ArsonCdrecordProcess
{
public:
	ArsonCdrecordFixateProcess (ArsonProcessMgr *pMgr);

	virtual bool execute (void);
};

class ArsonCdrecordAudioProcess : public ArsonCdrecordProcess
{
public:
	ArsonCdrecordAudioProcess (ArsonProcessMgr *pMgr);

	void addTrack (const QString &filename);
};

/*========================================================*/

class ArsonCdrdaoProcess : public ArsonWriterProcess
{
public:
	ArsonCdrdaoProcess (ArsonProcessMgr *pWriter,
		const char *tocfile,
		const char *oper = NULL,
		const QString &type = QString::null);

	static uint calcTime(uint hrs, uint mins, uint secs);
	
private:
	enum {
		EVENT_PAUSE,
		EVENT_CALIBRATE,
		EVENT_TRACKSTART,
		EVENT_MBWRITTEN,
		EVENT_FIXATING,
		EVENT_COPYDATALEN,
		EVENT_COPYSTATUS,
		EVENT_COPYAUDIOLEN,
		EVENT_PROMPT,
		EVENT_RETRY,
		EVENT_RELOAD,
		EVENT_UNLOCK,
		EVENT_BLANK,
		_EVENT_MAX
	};

	virtual void onStderr (int event, KRegExp *pre);
	virtual void setupSimulate (void);
	virtual void setupEject (void);

	void promptContinue (const QString &msg,
		const QString &caption, int how,
		const QString &yes = QString::null,
		const QString &no = QString::null);

	void appendDriverSwitch (void);

	bool m_bInPrompt;

	static KRegExp events[_EVENT_MAX];
};

/*========================================================*/

class ArsonCdrdaoCopyProcess : public ArsonCdrdaoProcess
{
public:
	ArsonCdrdaoCopyProcess (ArsonProcessMgr *pMgr,
		bool deleteImg, bool onTheFly,
		const char *srcdev);
};

/*========================================================*/

class ArsonCdrdaoIsoProcess : public ArsonCdrdaoProcess
{
public:
	ArsonCdrdaoIsoProcess (ArsonProcessMgr *pMgr,
		const char *isofile);
	virtual ~ArsonCdrdaoIsoProcess (void);

protected:
	QString createTocBin (const char *isofile);

	QString m_tempbase;		//	Base filename for TOC/BIN files
};

/*========================================================*/

class ArsonCdrdaoUnlockProcess : public ArsonCdrdaoProcess
{
public:
	ArsonCdrdaoUnlockProcess (ArsonProcessMgr *pMgr);
};

/*========================================================*/

class ArsonCdrdaoBlankProcess : public ArsonCdrdaoProcess
{
public:
	ArsonCdrdaoBlankProcess (ArsonProcessMgr *pMgr, uint how);

	virtual bool execute (void);
};

/*========================================================*/

class ArsonCdrdaoReadCdProcess : public ArsonCdrdaoProcess
{
public:
	ArsonCdrdaoReadCdProcess (ArsonProcessMgr *pMgr, const char *outfile);
	virtual ~ArsonCdrdaoReadCdProcess (void);

private:
	QString m_initDir;
};

/*========================================================*/

class ArsonRipperProcess : public ArsonProcess
{
public:
	ArsonRipperProcess (ArsonProcessMgr *pMgr,
		int program, int trackNo, const char *outfile);

	virtual void setOutputFormat (uint fmt) = 0;
};

class ArsonCdda2WavProcess : public ArsonRipperProcess
{
public:
	ArsonCdda2WavProcess (ArsonProcessMgr *pMgr,
		int trackNo = -1, const char *outfile = NULL);

	virtual void setOutputFormat (uint fmt);

protected:
	virtual void onStderr (int event, KRegExp *pre);

	enum {
		EVENT_PERCENT,
		EVENT_LOAD,
		_EVENT_MAX
	};

	static KRegExp events[_EVENT_MAX];
};

class ArsonCdparanoiaProcess : public ArsonRipperProcess
{
public:
	ArsonCdparanoiaProcess (ArsonProcessMgr *pMgr,
		int trackNo = -1, const char *outfile = NULL);

	virtual void setOutputFormat (uint fmt);
	
protected:
	virtual void onStderr (int event, KRegExp *pre);

	enum {
		EVENT_BEG = 0,
		EVENT_END,
		EVENT_PROGRESS,
		_EVENT_MAX
	};

	ulong m_sectors;
	ulong m_start;

	static KRegExp events[_EVENT_MAX];
};

/*========================================================*/

class ArsonReadCdProcess : public ArsonProcess
{
public:
	ArsonReadCdProcess (ArsonProcessMgr *pMgr, const char *outfile, const char *dev = NULL);

protected:
	virtual void onStderr (int event, KRegExp *pre);
	virtual void output (const QString &str, bool error);

	enum {
		EVENT_LENGTH,
		EVENT_PROGRESS,
		_EVENT_MAX
	};

	int m_max;

	static KRegExp events[_EVENT_MAX];
};

/*========================================================*/

class ArsonEncoderProcess : public ArsonProcess
{
public:
	ArsonEncoderProcess (int program, ArsonProcessMgr *pMgr);

	void setRipper (ArsonProcess *proc) { m_pRipper = proc; }

	virtual bool kill (int signo = SIGTERM);
	virtual void setTag (const char *title,
		int trackNo = -1,
		const char *artist = NULL,
		const char *album = NULL,
		const char *comment = NULL,
		const char *date = NULL,
		const char *genre = NULL) { }

protected:
	ArsonProcess *m_pRipper;	//	The cdda2wav process that rips the track
};

/*========================================================*/

class ArsonBladeProcess : public ArsonEncoderProcess
{
public:
	ArsonBladeProcess (ArsonProcessMgr *pMgr,
		const char *infile, const char *outfile);

protected:
	virtual void onStdout (int event, KRegExp *pre);

	enum {
		EVENT_PERCENT,
		EVENT_DONE,
		_EVENT_MAX
	};

	static KRegExp events[_EVENT_MAX];
};

/*========================================================*/

class ArsonLameProcess : public ArsonEncoderProcess
{
public:
	ArsonLameProcess (ArsonProcessMgr *pMgr,
		const char *infile, const char *outfile);

	virtual void setTag (const char *title,
		int trackNo = -1,
		const char *artist = NULL,
		const char *album = NULL,
		const char *comment = NULL,
		const char *date = NULL,
		const char *genre = NULL);

protected:
	virtual	void onStderr (int event, KRegExp *pre);

	enum {
		EVENT_STATUS,
		_EVENT_MAX
	};

	static KRegExp events[_EVENT_MAX];
};

/*========================================================*/

class ArsonOggencProcess : public ArsonEncoderProcess
{
public:
	ArsonOggencProcess (ArsonProcessMgr *pMgr,
		const char *infile, const char *outfile);

	virtual void setTag (const char *title,
		int trackNo = -1,
		const char *artist = NULL,
		const char *album = NULL,
		const char *comment = NULL,
		const char *date = NULL,
		const char *genre = NULL);

protected:
	virtual	void onStderr (int event, KRegExp *pre);

	enum {
		EVENT_STATUS,
		_EVENT_MAX
	};

	static KRegExp events[_EVENT_MAX];
};

/*========================================================*/
#ifdef FLAC
class ArsonFlacEncoderProcess : public ArsonEncoderProcess
{
public:
	ArsonFlacEncoderProcess (ArsonProcessMgr *pMgr,
		const char *infile, const char *outfile);

private:
	virtual void onStderr (int event, KRegExp *pre);
};
#endif	//	FLAC
/*========================================================*/

enum {
	NORMALIZE_NONE = 0,
	NORMALIZE_BATCH,
	NORMALIZE_MIX,
};

class ArsonNormalizeProcess : public ArsonProcess
{
public:
	ArsonNormalizeProcess (ArsonProcessMgr *pMgr);

	void addTrack (const char *file);
	
protected:
	virtual	void onStderr (int event, KRegExp *pre);

	enum {
		EVENT_COMPUTING,
		EVENT_APPLYING,
		EVENT_STATUS,
		_EVENT_MAX
	};

	enum {
		stateUnknown,
		stateCompute,
		stateApply,
	};
	
	int m_state;
	
	static KRegExp events[_EVENT_MAX];
};

/*========================================================*/
/*	A base class for audio file decoders
 *========================================================*/

class ArsonAudioDecoderProcess : public ArsonProcess
{
public:
	ArsonAudioDecoderProcess (ArsonProcessMgr *pMgr,
		const char *infile, const char *outfile,
		int program);
};

/*========================================================*/

class ArsonMpg123Process : public ArsonAudioDecoderProcess
{
public:
	ArsonMpg123Process (ArsonProcessMgr *pWriter,
		const char *infile, const char *outfile);

protected:
	virtual void onStderr (int event, KRegExp *pre);

	enum {
		EVENT_TIME,
		_EVENT_MAX
	};

	static KRegExp events[_EVENT_MAX];
};

/*========================================================*/
#ifdef OGG
class ArsonOgg123Process : public ArsonAudioDecoderProcess
{
public:
	ArsonOgg123Process (ArsonProcessMgr *pWriter,
		const char *infile, const char *outfile);

protected:
	virtual void onStderr (int event, KRegExp *pre);

	enum {
		EVENT_TIME,
		_EVENT_MAX
	};

	static KRegExp events[_EVENT_MAX];
};
#endif	//	OGG

class ArsonShortenProcess : public ArsonAudioDecoderProcess
{
public:
	ArsonShortenProcess (ArsonProcessMgr *pMgr,
		const char *infile, const char *outfile);

	virtual bool execute (void);
};

#ifdef FLAC
class ArsonFlacDecoderProcess : public ArsonAudioDecoderProcess
{
public:
	ArsonFlacDecoderProcess (ArsonProcessMgr *pMgr,
		const char *infile, const char *outfile);

private:
	virtual void onStderr (int event, KRegExp *pre);
};
#endif	//	FLAC

class ArsonLameDecoderProcess : public ArsonAudioDecoderProcess
{
public:
	ArsonLameDecoderProcess (ArsonProcessMgr *pMgr,
		const char *infile, const char *outfile);

private:
	virtual void onStderr (int event, KRegExp *pre);

	enum {
		EVENT_PROGRESS,
//		EVENT_SUPRESS,
		_EVENT_MAX
	};
	
	static KRegExp events[_EVENT_MAX];
};

/*========================================================*/

class ArsonIsoFlags;

class ArsonMkisoProcess : public ArsonProcess
{
public:
	ArsonMkisoProcess (ArsonProcessMgr *pWriter,
		const ArsonIsoFlags *pFlags,
		const QString &outfile,
		const QStringList &what);

	virtual bool execute (void);

protected:
	virtual void onStderr (int event, KRegExp *pre);

	enum {
		EVENT_PROGRESS,
		_EVENT_MAX
	};

	static KRegExp events[_EVENT_MAX];
};

/*========================================================*/

class QDomElement;

class ArsonVcdxbuildProcess : public ArsonProcess
{
public:
	ArsonVcdxbuildProcess (ArsonProcessMgr *pWriter, const char *basefile);

	virtual bool execute (void);
	
	void addFile (const QString &filename) { m_mpgs.append(filename); }
	uint count (void) const { return m_mpgs.count(); }
	
private:
	virtual void output (const QString &str, bool error);

	void outputProgress (QDomElement &el, ArsonProcessUI *pUI);
	void outputLog (QDomElement &el, ArsonProcessUI *pUI);
	void handleOutputElement (QDomElement &el);

	enum {
		stageUnknown,
		stageScan,
		stageWrite,
		_stage_max
	};

	QStringList m_mpgs;
	QString m_filebase;
	uint m_stage;
};

/*========================================================*/

class ArsonMd5sumProcess : public ArsonProcess
{
public:
	ArsonMd5sumProcess (ArsonProcessMgr *pMgr, const char *infile);
	virtual ~ArsonMd5sumProcess (void);

	const QStringList &errors (void) const { return m_output; }
	
	virtual bool execute (void);
	
private:
	virtual void output (const QString &str, bool error);
	
	QStringList m_output;
	QString m_tempfile;
	QString m_initDir;
};

/*========================================================*/

class ArsonShnlenProcess : public ArsonUtilityProcess
{
public:
	ArsonShnlenProcess (const ArsonConfig &cfg, const char *infile);

	int seconds (void) const { return m_seconds; }
	int bytes (void) const { return m_bytes; }
	
private:
	virtual void onStdout (int event, KRegExp *pre);

	enum {
		EVENT_SHNLEN,
		_EVENT_MAX,
	};

	static KRegExp events[_EVENT_MAX];

	int m_seconds;
	int m_bytes;
};

/*========================================================*/

class ArsonSoxProcess : public ArsonProcess
{
public:
	ArsonSoxProcess (ArsonProcessMgr *pMgr, const char *infile);

	virtual bool execute (void);
	
private:
	virtual void slotExited (KProcess *ptr);

	QString m_outfile;
	QString m_infile;
};

/*========================================================*/
#endif	/*	__PROCESS_H__	*/
