/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Arson configuration code
 *
 *	by Tony Sideris	(06:18AM Aug 08, 2001)
 *================================================*/
#include "arson.h"

#include <qmultilineedit.h>
#include <qcombobox.h>
#include <qdir.h>

#include <kglobalsettings.h>
#include <kmessagebox.h>
#include <kconfig.h>
#include <klocale.h>

#include "_textviewer.h"

#include "process.h"
#include "konfig.h"

/*==================================*/
/*	GLOBALS
 *==================================*/

//	The default size/position of the main frame window
const QRect DEFAULT_GEOMETRY(0,0,640,480);

ArsonConfig g_CONFIG;	//	Global configuration instance

/*	This array must match the PROGRAM_ enum in the
 *	ArsonConfig class. The string specified here is
 *	the string displayed in the 1st column of the
 *	programs tab, and is the string searched for by
 *	the "auto-detect" "feature". Any non-real programs
 *	should be placed at the end, and at least the 1st
 *	non-real program name should have at least one ' '
 *	in it.
 *
 *	For the 'help' member use ' ' to seperate multiple
 *	switches (as in 'cdrdao write -h'), use '|' to specify
 *	a string of failure cases (for version guessing) like
 *	"--longhelp|--help" so that --help is used if --longhelp
 *	fails.
 */
ArsonProgramDef g_programs[ArsonConfig::_PROGRAMS_MAX] = {
	/*
	 *				Program			Help			Version			Version Regexp									URL													Description
	 *				=======			====			=======			==============									===													===========
	 */
	ArsonProgramDef("cdrecord", 	"-help",		"-version",		"Cdrecord ([0-9]+).([0-9]+)",					"http://freshmeat.net/projects/cdrecord/", 			I18N_NOOP("Burns data to CDR.")),
	ArsonProgramDef("cdrdao",		"write -h", 	NULL,			"Cdrdao version ([0-9]+).([0-9]+).([0-9]+)",	"http://cdrdao.sourceforge.net/", 					I18N_NOOP("Burns audio, and data to CDR as well as performs CD copy.")),
	ArsonProgramDef("mkisofs",		"-help",		"-version",		"mkisofs ([0-9]+).([0-9]+)",					"http://freshmeat.net/projects/mkisofs/",			I18N_NOOP("Builds ISO9660 filesystems for burning to CDR.")),
	ArsonProgramDef("mpg123",		"--longhelp",	"--version",	"mpg321 version ([0-9]+).([0-9]+).([0-9]+)",	"http://www.mpg123.org/",							I18N_NOOP("Decodes mp3 files so they can be burned to CDR.")),
#ifdef OGG
	ArsonProgramDef("ogg123",		"--help",		"--version",	"ogg123 .+ ([0-9]+).([0-9]+)",					"http://www.xiph.org/ogg/vorbis/download.html",		I18N_NOOP("Decodes Ogg Vorbis files so they can be burned to CDR.")),
#endif
#ifdef FLAC
	ArsonProgramDef("flac",			"--help",		"--version",	"flac ([0-9]+).([0-9]+).([0-9]+)",				"http://flac.sourceforge.net/",						I18N_NOOP("Decodes, and encodes .flac audio files.")),
#endif
	ArsonProgramDef("cdda2wav",		"-help",		"-help",		"Version ([0-9]+).([0-9]+).+",					"http://www.escape.de/users/colossus/cdda2wav.html",I18N_NOOP("Reads track information from CDs, as well as rips tracks from CD (required for CD ripper).")),
	ArsonProgramDef("cdparanoia",	"-help",		"-V",			"cdparanoia .+ release ([0-9]+).([0-9]+)",		"http://www.xiph.org/paranoia/index.html",			I18N_NOOP("Rips tracks from audio CDs (optional for CD ripper).")),
	ArsonProgramDef("bladeenc",		NULL,			NULL,			"BladeEnc ([0-9]+).([0-9]+).([0-9]+)",			"http://bladeenc.mp3.no",							I18N_NOOP("MP3 encoder.")),
	ArsonProgramDef("lame",			"--longhelp|--help",	NULL,	"LAME version ([0-9]+).([0-9]+)",				"http://www.mp3dev.org/",							I18N_NOOP("Another MP3 encoder.")),
	ArsonProgramDef("oggenc",		"--help",		"--version",	"OggEnc v([0-9]+).([0-9]+)",					"http://www.xiph.org/ogg/vorbis/download.html",		I18N_NOOP("Ogg Vorbis encoder.")),
	ArsonProgramDef("readcd",		"-help",		"-version",		"readcd ([0-9]+).([0-9]+)",						"http://freshmeat.net/projects/cdrecord/",			I18N_NOOP("Reads data CDs into an image file.")),
	ArsonProgramDef("shorten",		"-h",			"-h",			"shorten version ([0-9]+).([0-9]+)",			"http://etree.org/linux.html",						I18N_NOOP("Decodes SHN files so they can be burned to CDR.")),
	ArsonProgramDef("shnlen",		"-h",			NULL,			NULL,											"http://etree.org/linux.html",						I18N_NOOP("Determines track length of SHN files.")),
	ArsonProgramDef("md5sum",		"-help",		NULL,			NULL,											NULL,												I18N_NOOP("Checks the MD5 checksum of files specified in MD5 file.")),
	ArsonProgramDef("sox",			"-help",		"-help",		"sox: Version ([0-9]+).([0-9]+).([0-9]+)",		"http://www.spies.com/Sox/",						I18N_NOOP("Fixes invalid audio tracks (that are not in 44100Khz/16bps/stereo).")),
	ArsonProgramDef("normalize-audio",	"--help",		"--version",	"normalize ([0-9]+).([0-9]+).([0-9]+)",			"http://www.cs.columbia.edu/~cvaill/normalize/",	I18N_NOOP("Evens out the volumes of different audio tracks.")),
	ArsonProgramDef("vcdxbuild",	"--help",		"--version",	"vcdxbuild .+ ([0-9]+).([0-9]+).([0-9]+)",		"http://www.vcdimager.org/",						I18N_NOOP("Builds [S]VCD image suitable for burning to CDR.")),
	ArsonProgramDef("vcdxgen",		"--help",		"--version",	"vcdxgen .+ ([0-9]+).([0-9]+).([0-9]+)",		"http://www.vcdimager.org/",						I18N_NOOP("Generates XML files required by vcdxbuild.")),
	ArsonProgramDef("id3v2",		"--help",		"--version",	"id3v2 ([0-9]+).([0-9]+)",						"http://id3v2.sourceforge.net/",					I18N_NOOP("Adds id3v2 tag information to MP3 files.")),

	ArsonProgramDef(I18N_NOOP("CDR Initialization"), NULL, NULL, NULL, NULL, I18N_NOOP("Run when arson is started to initialize CDR devices (load kernel modules, etc), this won't be necessary for most people.")),
	ArsonProgramDef(I18N_NOOP("CDR Cleanup"), NULL, NULL, NULL, NULL, I18N_NOOP("Run when arson is shutting down to deinitialize CDR devices.")),
};

/*========================================================*/

#define TWO_CHOICES(o,t)	#o "\0" #t "\0"
#define CDRDAO_OR_CDRECORD	TWO_CHOICES(cdrdao, cdrecord)
#define CDRECORD_OR_CDRDAO	TWO_CHOICES(cdrecord, cdrdao)

ArsonProgramPref g_program_prefs[] =
{
	ArsonProgramPref(I18N_NOOP("Audio CD Burner"), CDRDAO_OR_CDRECORD),
	ArsonProgramPref(I18N_NOOP("ISO Image Burner"), CDRECORD_OR_CDRDAO),
	ArsonProgramPref(I18N_NOOP("CD Copy"), TWO_CHOICES(cdrdao, readcd)),
	ArsonProgramPref(I18N_NOOP("MP3 Encoder"), TWO_CHOICES(LAME, bladeenc)),
	ArsonProgramPref(I18N_NOOP("MP3 Decoder"), TWO_CHOICES(mpg123, LAME)),
	ArsonProgramPref(I18N_NOOP("Device Scanner"), CDRECORD_OR_CDRDAO),
	ArsonProgramPref(I18N_NOOP("CD Ripper"), TWO_CHOICES(cdda2wav, cdparanoia)),
	ArsonProgramPref(I18N_NOOP("CDRW Blanking"), CDRECORD_OR_CDRDAO),
};

const char *ArsonProgramPref::groupName (int group)
{
	return g_program_prefs[group].desc;
}

const char *ArsonProgramPref::groupChoices (int group)
{
	return g_program_prefs[group].choices;
}

const char *ArsonProgramPref::groupDefault (int group)
{
	return g_program_prefs[group].choices;
}

/*========================================================*/

#define WRITES(var)		if ((var) != QString::null) pk->writeEntry(#var, var)
#define WRITEI(var) 	pk->writeEntry(#var, var)

#define READS(var)		var = pk->readEntry(#var, var)
#define READI(var)		var = pk->readNumEntry(#var, var)
#define READU(var)		var = pk->readUnsignedNumEntry(#var, var)

/*========================================================*/

bool ArsonConfig::Program::valid (void) const
{
	return arsonIsExecutable(m_program);
}

/*========================================================*/
/*	Base config class implemenatation
 *========================================================*/

bool ArsonConfigBase::load (KConfig *pk)
{
	pk->setGroup(m_name);
	return true;
}

void ArsonConfigBase::save (KConfig *pk, bool finalize)
{
	pk->setGroup(m_name);
}

/*========================================================*/

bool ArsonFeatureConfig::load (KConfig *pk)
{
	ArsonConfigBase::load(pk);
	READI(m_flags);
	return true;
}


void ArsonFeatureConfig::save (KConfig *pk, bool finalize)
{
	ArsonConfigBase::save(pk, finalize);

	WRITEI(m_flags);
}

/*========================================================*/
/*	ArsonConfig class implementation
 *========================================================*/

ArsonConfig::ArsonConfig (void)
	: m_nCdLenMin(74),
	m_nCdLenMB(650),
	m_nSpeed(2),
	m_flags(flagMd5Verify | flagMd5Reset | flagScreenSaver),
	m_startDoc(0),
	m_niceLevel(0),
	m_ripper(this)
{   
   QFile hdFile("/proc/version");
   if (hdFile.exists())
   {
      QString version;
      hdFile.open(IO_ReadOnly);
      hdFile.readLine(version, 1024);
      hdFile.close();

      version = version.section(' ', 2, 2);
      m_kernRel = version.section('.', 0, 0);
      m_kernMaj = version.section('.', 1, 1);
      m_kernMin = version.section('.', 2, 2);
   }
}

/*========================================================*/
/*	Load and save configuration
 *========================================================*/

#define PROGRAM_SEP		","
#define IDE_BURN_KERN   "2545"

bool ArsonConfig::isIDEBurningSupported (void)
{
   QString version = m_kernRel + m_kernMaj + m_kernMin;
   return (version.compare(QString(IDE_BURN_KERN)) > 0);
}

/*========================================================*/

bool ArsonConfig::load (void)
{
	int sb = 0, index;
	QString temp;
	KConfig *pk = kapp->config();

	pk->setGroup("cfg");
	READS(m_strDevice);
	READS(m_strDriver);
	READS(m_strSrcDriver);
	READU(m_flags);
	READI(m_nCdLenMin);
	READI(m_nCdLenMB);
	READI(m_nSpeed);
	READI(m_startDoc);
	READI(m_niceLevel);
	READS(m_tempDirectory);

	m_geometry = pk->readRectEntry(ACFGKEY(m_geometry), &DEFAULT_GEOMETRY);
	
	//	Read in the programs
	pk->setGroup("programs");
	for (index = 0; index < _PROGRAMS_MAX; index++)
	{
		temp = pk->readEntry(ArsonProgramDef::definition(index).programName());

		QStringList sl = QStringList::split(PROGRAM_SEP, temp, TRUE);

		if (!sl.isEmpty())
		{
			Program &prog = m_programs[index];

			prog.m_program = sl[0];
			sl.remove(sl.begin());

			prog.m_params = sl.join(PROGRAM_SEP);
		}
	}

	//	Read in the program preferences
	pk->setGroup("program-prefs");
	for (index = 0; index < _PROGGRP_MAX; ++index)
	{
		m_progPrefs[index] = pk->readEntry(ArsonProgramPref::groupName(index));

		if (m_progPrefs[index].isEmpty())
			m_progPrefs[index] = ArsonProgramPref::groupDefault(index);
	}

	/*
	 *	Autodetect the necessary programs if none exist
	 */
	if (!validDeviceScannersExist())
	{
		autoDetectPrograms();
		
		if (!validDeviceScannersExist())
		{
			throw ArsonStartupError(
				i18n("cdrecord and cdrdao were not found, either of these programs\n") +
				i18n("are used to scan the SCSI bus for CD[RW] devices, so at least one\n") +
				i18n("of them must be installed. If one of them is installed, goto the\n") +
				i18n("Programs tab of the configuration dialog and specify the path.\n") +
				i18n("Otherwise install one or both of these programs and configure your\n") +
				i18n("system to use them properly (see the CD-Writing HOWTO).\n\n") +
				i18n("If you are only using arson as a CD ripper, then this should not\n") +
				i18n("concern you. Check off 'Do not ask again.', and use the\n") +
				i18n("Cooked IOCTL interface (ripper configuration tab)."),
				"noscanners");
		}
	}

	/*	If scanbus fails, initialize the
	 *	CDR and try again.
	 */
	try {
		m_devs.load(pk);
		sb = m_devs.scsiCount() + m_devs.ideCount();
	}
	catch (ArsonStartupError &err)
	{
		/*
		 *	DO NOT REPORT ERROR - CDR init and try again
		 *	if THAT fails the caller should be reporting
		 *	the error.
		 */
		if (cdrInit(true))
		{
			m_private |= privateInitCdr;
			sb = m_devs.scanbus();
		}
	}

	m_ripper.load(pk);
	return true;
}

/*========================================================*/

void ArsonConfig::save (bool finalize)
{
	int index;
	KConfig *pk = kapp->config();

	pk->setGroup("cfg");
	WRITES(m_strDevice);
	WRITES(m_strDriver);
	WRITES(m_strSrcDriver);
	WRITEI(m_flags);
	WRITEI(m_nSpeed);
	WRITEI(m_nCdLenMin);
	WRITEI(m_nCdLenMB);
	WRITEI(m_startDoc);
	WRITEI(m_niceLevel);
	WRITES(m_tempDirectory);
	pk->writeEntry(ACFGKEY(m_geometry), m_geometry);

	//	Write the program configuration
	pk->setGroup("programs");
	for (index = 0; index < _PROGRAMS_MAX; ++index)
	{
		QStringList sl;
		sl << m_programs[index].m_program
		   << m_programs[index].m_params;

		QString qs = sl.join(PROGRAM_SEP);

		pk->writeEntry(ArsonProgramDef::definition(index).programName(), qs);
	}

	//	Write the program preferences
	pk->setGroup("program-prefs");
	for (index = 0; index < _PROGGRP_MAX; ++index)
		pk->writeEntry(ArsonProgramPref::groupName(index), QString(m_progPrefs[index]));

	m_devs.save(pk);
	m_ripper.save(pk, finalize);

	if (finalize && (m_private & privateInitCdr))
		cdrInit(false);
}

/*========================================================*/
/*	Simple iterators
 *========================================================*/

const ArsonConfig::Program *ArsonConfig::program (int index) const
{
	return (index < _PROGRAMS_MAX) ? &(m_programs[index]) : NULL;
}

ArsonConfig::Program *ArsonConfig::program (int index)
{
	return (index < _PROGRAMS_MAX) ? &(m_programs[index]) : NULL;
}

/*========================================================*/
/*	Some users who have CDR-related kernel software
 *	compiled as modules, and load these modules as needed,
 *	have scripts to load the neseccary modules... these
 *	functions execute these scripts if no devices are
 *	available.
 *========================================================*/

bool ArsonConfig::cdrInit (bool init)
{
	ArsonUtilityProcess proc (ACONFIG,
		KProcess::NoCommunication,
		QString::null,
		init ? PROGRAM_CDRINIT : PROGRAM_CDRFIN);

#ifdef ARSON_KDE3
	if (!proc.args().isEmpty())
#else
	if (!proc.args()->isEmpty())
#endif	//	ARSON_KDE_3
	{
		return proc.execute();
	}

	return true;
}

/*========================================================*/
/*	Search the filesystem for neseccary programs, so the
 *	user doesn't need to manually configure the paths...
 *========================================================*/

void ArsonConfig::autoDetectPrograms (const QString &xtra)
{
	for (int index = 0; index < PROGRAM_CDRINIT; index++)
	{
		const QString name = ArsonProgramDef::definition(index).programName();

		if (name.find(' ') != -1)
			continue;

		m_programs[index].m_program = detectProgram(name, xtra);
		ArsonProgramDef::definition(index).resetVersion();
	}
}

QString ArsonConfig::detectProgram (const char *name, const QString &xtra)
{
	QStringList::Iterator it, end;
	QString xpath = xtra;

	if (xpath == QString::null)
		xpath = extraProgramPaths();

	//	Path list built from $PATH and extra paths option
	QStringList paths = QStringList::split(":", xpath);
	paths += QStringList::split(":", getenv("PATH"));

	//	Check each path in the path list
	for (it = paths.begin(), end = paths.end(); it != end; ++it)
	{
		const QString path = QDir(*it).absFilePath(name);

		if (arsonIsExecutable(path))
			return path;
	}

	return QString::null;
}

/*========================================================*/

bool ArsonConfig::validDeviceScannersExist (void) const
{
	return (program(PROGRAM_CDRECORD)->valid() ||
		program(PROGRAM_CDRDAO)->valid());
}

/*========================================================*/

void ArsonConfig::showMissingPrograms (void)
{
	QStringList missing;

	for (int index = 0; index < _PROGRAMS_MAX; ++index)
		if (!program(index)->valid())
			missing.append(
				ArsonProgramDef::definition(index).programName() +
				i18n(" - ") +
				ArsonProgramDef::definition(index).desc());

	if (!missing.isEmpty())
		arsonErrorMsg(
			i18n("The following worker programs were not detected. Whether or not they are required or not depends on what you plan on using arson for (burning CDs requires somethings, while ripping audio requires others). The following was not found:\n\n") +
			missing.join("\n"));
}

/*========================================================*/
/*	Ripper settings implementation
 *========================================================*/

ArsonConfig::RipperCfg::RipperCfg (const ArsonConfig *cfg)
	: ArsonFeatureConfig("ripper"),
	m_outputFormat(1),
	m_cdiFormat("%N %a - %t"),
	m_quality(i18n("CD Quality")),
	m_defaultComment(i18n("Ripped by Arson v%v")),
	m_strip("\"'!@#$%^&*[]|<>?:;\\"),
	m_strOutDir(QDir::home().canonicalPath()),
	m_cfg(cfg)
{
	//	Nothing...
}

/*========================================================*/

bool ArsonConfig::RipperCfg::load (KConfig *pk)
{
	const bool res = ArsonFeatureConfig::load(pk);

	READI(m_outputFormat);
	READS(m_strOutDir);
	READS(m_cdiFormat);
	READS(m_email);
	READS(m_srcdev);
	READS(m_quality);
	READS(m_defaultComment);
 	READI(m_playlistEnabled);
	READS(m_playlistLocation);
	READS(m_playlistFilename);
	READI(m_playlistOverwrite);
     READI(m_PathType);

	m_lookups.load(pk);
	return res;
}

/*========================================================*/

void ArsonConfig::RipperCfg::save (KConfig *pk, bool finalize)
{
	ArsonFeatureConfig::save(pk, finalize);

	WRITEI(m_outputFormat);
	WRITES(m_strOutDir);
	WRITES(m_cdiFormat);
	WRITES(m_email);
	WRITES(m_srcdev);
	WRITES(m_quality);
	WRITES(m_defaultComment);
	WRITEI(m_playlistEnabled);
	WRITES(m_playlistLocation);
	WRITES(m_playlistFilename);
	WRITEI(m_playlistOverwrite);
     WRITEI(m_PathType);

	m_lookups.save(pk);
}

/*========================================================*/

#define ARSON_CDIFORMATS_MRU		"fmtmru"
#define ARSON_MRU_SEP				'\t'
#define ARSON_MRU_MAX				16

void ArsonConfig::RipperCfg::setCdiFormat (const QString &fmt)
{
	QStringList sl = recentCdiFormats();

	if (sl.find(fmt) == sl.end())
	{
		KConfig *pk = kapp->config();

		sl.append(fmt);

		while (sl.count() > ARSON_MRU_MAX)
			sl.remove(sl.begin());
		
		pk->setGroup("ripper");
		pk->writeEntry(ARSON_CDIFORMATS_MRU, sl, ARSON_MRU_SEP);
	}

	m_cdiFormat = fmt;
}

QStringList ArsonConfig::RipperCfg::recentCdiFormats (void) const
{
	KConfig *pk = kapp->config();
	pk->setGroup("ripper");
	return pk->readListEntry(ARSON_CDIFORMATS_MRU, ARSON_MRU_SEP);
}

/*========================================================*/

bool ArsonConfig::RipperCfg::isStrippedChar (char ch) const
{
	return m_strip.contains(ch);
}

/*========================================================*/
/*	Program version class
 *========================================================*/

ArsonVersion::ArsonVersion (int n0, int n1, int n2)
{
	version[0] = n0;
	version[1] = n1;
	version[2] = n2;
}

template<class T> bool version_cmp (const int *pl, const int *pr, int c, T cmp)
{
	for (int index = 0; index < c; ++index)
	{
		if (pl[index] == pr[index])
			continue;

		if (cmp(pl[index], pr[index]))
			return true;

		return false;
	}

	return false;
}

inline bool version_less (int l, int r) { return l < r; }
inline bool version_more (int l, int r) { return l > r; }

bool ArsonVersion::operator< (const ArsonVersion &ver) const
{ return version_cmp(version, ver.version, DEPTH, version_less); }

bool ArsonVersion::operator> (const ArsonVersion &ver) const
{ return version_cmp(version, ver.version, DEPTH, version_more); }

bool ArsonVersion::operator== (const ArsonVersion &ver) const
{
	for (int index = 0; index < DEPTH; ++index)
		if (version[index] != ver.version[index])
			return false;

	return true;
}

bool ArsonVersion::operator!= (const ArsonVersion &ver) const
{ return !((*this) == ver); }

/*========================================================*/

class arsonVersionProcess : public ArsonUtilityProcess
{
public:
	arsonVersionProcess (int program, const char *re)
		: ArsonUtilityProcess(ACONFIG,
			Communication(Stderr|Stdout),
			ArsonProgramDef::definition(program).programName(),
			program),
		regexp(re)
	{ }

	const ArsonVersion &result (void) const { return res; }
	
private:
	virtual void output (const QString &str, bool error)
	{
		if (regexp.match(str))
		{
			const char *g3 = regexp.group(3);
			
			res = ArsonVersion(
				QString(regexp.group(1)).toInt(),
				QString(regexp.group(2)).toInt(),
				g3 ? QString(g3).toInt() : 0
				);
		}
	}

	ArsonVersion res;
	KRegExp regexp;
};

/*========================================================*/
/*	Program definition class
 *========================================================*/

ArsonProgramDef::~ArsonProgramDef (void)
{
	resetVersion();
}

ArsonProgramDef &ArsonProgramDef::definition (int program)
{
	return g_programs[program];
}

/*========================================================*/

const ArsonVersion *ArsonProgramDef::getVersion (void)
{
	if (!pVersion)
	{
		arsonVersionProcess proc (id(), vregexp);

		if (version)
			proc << version;

		proc.execute();

		if (proc.result().valid())
			pVersion = new ArsonVersion(proc.result());
	}

	return pVersion;
}

void ArsonProgramDef::resetVersion (void)
{
	delete pVersion;
	pVersion = NULL;
}

/*========================================================*/

int ArsonProgramDef::id (void) const
{
	for (int index = 0; index < ArsonConfig::_PROGRAMS_MAX; ++index)
		if (programName() == definition(index).programName())
			return index;

	return ArsonConfig::PROGRAM_UNKNOWN;
}

/*========================================================*/
/*	A class for gathering, and displaying
 *	program --help messages.
 *========================================================*/

ArsonProgramHelp::ArsonProgramHelp (int program)
{
	if (program != ArsonConfig::PROGRAM_UNKNOWN)
		setProgram(program);
}

/*========================================================*/
/*	The help process
 *========================================================*/

class arsonProgramHelpProcess : public ArsonUtilityProcess
{
public:
	arsonProgramHelpProcess (int program)
		: ArsonUtilityProcess(ACONFIG,
			Communication(Stderr|Stdout),
			ArsonProgramDef::definition(program).programName(),
			program)
	{
	}

	const QString &help (void) const { return m_help; }

	virtual bool execute (void)
	{
		ArsonUtilityProcess::execute();
		return !m_help.isEmpty();
	}
	
private:
	virtual void output (const QString &str, bool error)
	{
		Trace("(%d) %s\n", error, str.latin1());
		m_help.append(str + "\n");
	}

	QString m_help;
};

/*========================================================*/
/*	The dialog
 *========================================================*/

class arsonTextViewer : public ArsonTextViewerBase
{
public:
	arsonTextViewer (const QString &str, QWidget *parent)
		: ArsonTextViewerBase(parent, NULL, true)
	{
		text->setFont(KGlobalSettings::fixedFont());
		text->setWordWrap(QMultiLineEdit::NoWrap);
		text->setText(str);
	}
};

/*========================================================*/
/*	Do it till it works.
 *========================================================*/

int ArsonProgramHelp::setProgram (int program)
{
	if (program != ArsonConfig::PROGRAM_UNKNOWN &&
		program < ArsonConfig::PROGRAM_CDRINIT)
	{
		const char *switches = ArsonProgramDef::definition(program).helpSwitch();

		m_program = program;
		
		//	For program's that display help with no args
		if (!switches)
		{
			arsonProgramHelpProcess proc (program);

			if (proc.execute())
				m_help = proc.help();
		}
		else	//	All others
		{
			const QStringList sl = QStringList::split(QString("|"), switches);

			for (QStringList::ConstIterator it = sl.begin(), end = sl.end();
				 it != end; ++it)
			{
				arsonProgramHelpProcess proc (program);

				proc.appendArgList(QStringList::split(QString(" "), (*it)));

				if (proc.execute())
				{
					m_help = proc.help();

					if (proc.successful())
						break;
				}
			}
		}
	}
	
	return (m_help.isEmpty() ? -1 : (int) m_help.length());
}

/*========================================================*/

void ArsonProgramHelp::show (QWidget *parent) const
{
	arsonTextViewer dlg (m_help, parent ? parent : kapp->mainWidget());

	dlg.setCaption(i18n("Help for ")
		+ ArsonProgramDef::definition(m_program).programName());

	dlg.exec();
}

/*========================================================*/
