/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	KDE mp3 burner frontend. This file contains
 *	main(), the application class, utility class
 *	implementations, and utility functions.
 *
 *	by Tony Sideris	(05:28AM Jul 31, 2001)
 *================================================*/
#include "arson.h"

#include <stdlib.h>
#include <time.h>

#include <qfileinfo.h>
#include <qstring.h>
#include <qxml.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qheader.h>

#include <kcmdlineargs.h>
#include <kmessagebox.h>
#include <kaboutdata.h>
#include <dcopclient.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kstddirs.h>
#include <kglobal.h>
#include <knotifyclient.h>
#include <kaction.h>

#ifdef ARSON_KDE3
#include <kstdguiitem.h>
#endif

#include "listwnd.h"
#include "mainwnd.h"
#include "konfig.h"
#include "docwidget.h"
#include "logwnd.h"
#include "tempfile.h"
#include "tools.h"
#include "wizard.h"

#ifdef ARSONDBG
arson_instance_counter g_INSTANCE_COUNTER;
#endif	//	ARSONDBG

KCmdLineOptions g_cmdline_options[] = {
	{ "w", 0, 0 },
	{ "write <file>", I18N_NOOP("Burn image file (iso, cue) or directory, then quit"), 0 },
	{ "p", 0, 0 },
	{ "write-prompt", I18N_NOOP("Prompt for an image file, then burn it"), 0 },
	{ "c", 0, 0 },
	{ "copy", I18N_NOOP("CD-to-CD Copy, then quit"), 0 },
	{ "+[arg]", I18N_NOOP(".arson files..."), 0 },
	{ 0, 0, 0 },
};

/*========================================================*/
/*	Main
 *========================================================*/

int main (int argc, char **argv)
{
	KAboutData about(
		PACKAGE,
		I18N_NOOP(DISPLAYNAME),
		VERSION,
		I18N_NOOP(DESCRIPTION),
		KAboutData::License_GPL,
		"(c) 2001,2002, 2003 Tony Sideris",
		I18N_NOOP("A KDE frontend for CD burning, and CD ripping."),
		I18N_NOOP(HOMEPAGE),
		I18N_NOOP(BUGEMAIL)
		);

	about.addAuthor(MYNAME, 0, BUGEMAIL, HOMEPAGE);
	about.addAuthor("Markus Triska",
		I18N_NOOP("Temporary folder option, decoded audio file caching."),
		"triska@gmx.at", "http://triskam.virtualave.net");
	about.addAuthor("Brian Kreulen",
		I18N_NOOP("M3U index file creation, IDE support"),
		"bakreule@users.sourceforge.net", "");

	KCmdLineArgs::init(argc, argv, &about);
	KCmdLineArgs::addCmdLineOptions(g_cmdline_options);
	KApplication::addCmdLineOptions();

	return ArsonApp(argc, argv).main();
}

/*========================================================*/

void ArsonError::report (void) const
{
	const QString &desc = description();

	if (desc != QString::null)
		arsonErrorMsg(desc);
}

void ArsonStartupError::report (void) const
{
	if (arsonWarning(description() + i18n("\nContinue?"), m_cfgKey))
		throw 0xF00;
}

/*========================================================*/
/*	Utility functions
 *========================================================*/

#if defined(ARSONDBG) || defined(ARSONTRACE)
#include <stdarg.h>

void Ts_Trace (const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);

	vfprintf(stderr, fmt, ap);
	va_end(ap);
}
#endif

/*========================================================*/

QString arsonDisplayTime (int secs)
{
	QString temp;

	temp.sprintf("%u:%02d",
		secs / 60,
		secs % 60);

	return temp;
}

/*========================================================*/
/**
 *	Simple find and replace all in string,
 *	but without QRegExp.
 */

QString &arsonReplace (const QString &repl, const QString &with, QString &target)
{
	int index = target.find(repl);

	while (index != -1)
	{
		target.replace(index, repl.length(), with);
		index = target.find(repl, index + with.length());
	}

	return target;
}

/*========================================================*/

#define InBetween(i,n,m)	((i)>=(n)&&(i)<=(m))
#define KILOBYTE		(1024)
#define MEGABYTE		(KILOBYTE*KILOBYTE)
#define GIGABYTE		(MEGABYTE*KILOBYTE)
#define BYTEARG(n)		arg((n), 0, 'f', 2)

QString arsonByteSize (uint nbytes)
{
	if (nbytes < KILOBYTE)
		return i18n("%1").arg(nbytes);

	else if (InBetween(nbytes, KILOBYTE, MEGABYTE))
		return i18n("%1 KB").BYTEARG(double(nbytes) / double(KILOBYTE));

	else if (InBetween(nbytes, MEGABYTE, GIGABYTE))
		return i18n("%1 MB").BYTEARG(double(nbytes) / double(MEGABYTE));

	return i18n("%1 GB").BYTEARG(double(nbytes) / double(GIGABYTE));
}

/*========================================================*/

bool arsonIsExecutable (const char *path)
{
	QFileInfo fi (QFile::encodeName(path));

	if (!fi.isDir() && fi.isExecutable())
		return true;

	return false;
}

/*========================================================*/

QCString arsonDataFile (const char *filename, const char *res, bool local)
{
	QCString result;

	if (!res)
		res = "appdata";

	if (local)
		result = locateLocal(res, filename);
	else
	{
		/*	In debug builds we want the data files in the
		 *	current working dir, these are the devel files
		 *	not the installed copies...
		 */
#ifndef ARSONDBG
		KStandardDirs *dirs = KGlobal::dirs();

		result = dirs->findResource(res, filename);
#endif	//	ARSONDBG
	}

	if (result.isEmpty())
		result = filename;

	return QFile::encodeName(result);
}

/*========================================================*/

QString msgBoxText (const QString &inp)
{
#ifdef ARSON_KDE3
	return inp;	//	KDE3 messageboxes wordwrap text
#else
	const QStringList sl = QStringList::split(" ", inp);
	QString result, current;

	for (int index = 0; index < sl.count();)
	{
		while (current.length() < 80 && index < sl.count())
			current.append(sl[index++] + " ");

		result.append(current + "\n");
		current = QString();
	}

//	result.truncate(result.length() - 1);
	return result;

/*	QString copy (inp);

	for (int index = 0; index < copy.length(); ++index)
	{
		int pos;

		for (pos = copy.find(' ', index);
		pos != -1 && (pos - index) < 80;
		pos = copy.find(' ', pos + 1));

		if (pos != -1)
		{
			copy.insert(pos, "\n");
			index = pos + 1;
		}
		else
			break;
	}

	return copy; */
#endif	//	ARSON_KDE3
}

/*========================================================*/
/**
 *	Display an error message with an OK button only.
 */

void arsonErrorMsg (const QString &msg)
{
	KMessageBox::error(
		ArsonFrame::getFrame(),
		msgBoxText(msg), i18n("Error"));
}

/**
 *	Display a warning message with Continue/Cancel, return
 *	TRUE to Continue. If save_at is not QString::null then
 *	the "Do not show me this again" checkbox will be displayed
 *	and the setting will be stored in the config key specified
 *	by save_at.
 */

bool arsonWarning (const QString &msg, const QString &save_at, const QString &caption)
{
	if (save_at != QString::null)
		return (KMessageBox::warningContinueCancel(ArsonFrame::getFrame(),
					msgBoxText(msg), (caption == QString::null) ? i18n("Warning") : caption,
#ifdef ARSON_KDE3
					KStdGuiItem::guiItem(KStdGuiItem::Continue),
#else
					i18n("Continue"),
#endif	//	ARSON_KDE3
					save_at,
					true) == KMessageBox::Continue);
	else
		return (KMessageBox::warningContinueCancel(ArsonFrame::getFrame(),
					msgBoxText(msg), i18n("Error"),
#ifdef ARSON_KDE3
					KStdGuiItem::guiItem(KStdGuiItem::Continue)
#else
					i18n("Continue")
#endif	//	ARSON_KDE3
					) == KMessageBox::Continue);
}

void arsonNotify (const QString &msg)
{
	KNotifyClient::event(KNotifyClient::notification, msg);
}

/*========================================================*/

void arsonSaveLogWindow (QListBox *pl, const QString &fn)
{
	QString path (fn);

	if (fn == QString::null)
		path = KFileDialog::getSaveFileName();

	if (path != QString::null)
	{
		QFile file (QFile::encodeName(path));

		//	Open the selected file for output
		if (!file.open(IO_WriteOnly | IO_Truncate))
			arsonErrorMsg(
				i18n("Error opening %1 for writing").arg(path));
		else
		{
			const int count = pl->count();

			//	Write each line of the output box to the file
			for (int index = 0; index < count; ++index)
			{
				const QString text = pl->text(index) + "\n";

				file.writeBlock(text, text.length());
			}

			file.close();
		}
	}
}

/*========================================================*/
/*	Open an XML document
 *========================================================*/

bool arsonOpenDomDocument (const KURL &url, QDomDocument &dom)
{
	ArsonNetFile nf (url);

	if (arsonOpenDomDocument(nf.path(), dom))
		return true;

	return false;
}

bool arsonOpenDomDocument (const QString &fn, QDomDocument &dom)
{
	QFile file (QFile::encodeName(fn));

	if (file.open(IO_ReadOnly))
	{
		dom.setContent(&file);
		return true;
	}

	Trace("Failed to open DOM Document: %s\n",
		(const char *) QFile::encodeName(fn));
	return false;
}

/*========================================================*/
/*	Open an arson document (.arson)
 *========================================================*/

class arsonDocHandler : public QXmlDefaultHandler
{
public:
	arsonDocHandler (void)
		: QXmlDefaultHandler(),
		m_pLI(NULL),
		m_pDoc(NULL)
	{}

	~arsonDocHandler (void) { delete m_pLI; }

	ArsonDocWidget *doc (void) { return m_pDoc; }

	virtual bool startElement (const QString &ns, const QString &local,
		const QString &name, const QXmlAttributes &attr)
	{
		if (!m_pDoc)
		{
			ArsonFrame *pFrm = ArsonFrame::getFrame();

			if (pFrm && (m_pDoc = pFrm->findDocument(name)))
			{
				ArsonLvPos pos;

				m_pDoc->newDocument();
				m_pDoc->setSizeCanChange(false);

				m_pDoc->defaultLvPos(pos);
				m_pLI = new ArsonListInserter(pos.parent, pos.after);

				m_pDoc->onStartElement(name, attr, *m_pLI);
				return true;
			}

			return false;
		}
		else
			return m_pDoc->onStartElement(name, attr, *m_pLI);
	}

	virtual bool endElement (const QString &ns,
		const QString &local, const QString &name)
	{
		if (m_pDoc)
			return m_pDoc->onEndElement(name, *m_pLI);

		return false;
	}

	virtual bool endDocument (void)
	{
		if (m_pDoc)
			m_pDoc->setSizeCanChange(true);
		
		return true;
	}

private:
	ArsonListInserter *m_pLI;
	ArsonDocWidget *m_pDoc;
};

void arsonOpenDoc (const KURL &url)
{
	arsonDocHandler adh;
	KRecentFilesAction *pa = (KRecentFilesAction *)
		ArsonFrame::getFrame()->actionCollection()
		->action("file_open_recent");

	if (arsonParseXml(url, &adh))
	{
		if (pa)
			pa->addURL(url);
		
		adh.doc()->setDocumentName(url);
		adh.doc()->setModified(false);
	}
	else
	{
		arsonErrorMsg(
			i18n("Failed to load document from %1")
			.arg(url.url()));

		if (pa)
			pa->removeURL(url);
	}
}

/*========================================================*/
/*	XML parser utilities
 *========================================================*/

bool arsonParseXml (QTextStream *ps, QXmlContentHandler *ph)
{
	QXmlInputSource input (*ps);
	QXmlSimpleReader rdr;

	rdr.setContentHandler(ph);

	if (!rdr.parse(input))
	{
		Trace("XML reader failed! reason:\n%s\n",
			ph->errorString().latin1());

		return false;
	}

	return true;
}

bool arsonParseXml (const KURL &url, QXmlContentHandler *ph)
{
	ArsonNetFile nf (url);
	QFile file (nf.path());

	if (file.open(IO_ReadOnly))
	{
		QTextStream ts (&file);
		return arsonParseXml(&ts, ph);
	}

	return false;
}

/*========================================================*/
/*	Arson application class
 *========================================================*/

ArsonApp::ArsonApp (int argc, char **argv)
	: KApplication()
{
	srand(time(NULL));
	srand(rand());
	
	dcopClient()->registerAs(name(), false);
}

/*========================================================*/
/*	Handle all arson-specific command line parameters
 *========================================================*/

KCmdLineArgs *ArsonApp::handleCmdLine (void)
{
	KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
	QCString imgfile = args->getOption("write");

#define TRACEARG(arg)	if (!(arg).isNull()) Trace("%s: %s\n", #arg, (const char *) arg);

	TRACEARG(imgfile);

	if (!imgfile.isNull() || args->isSet("write-prompt"))
		arsonWriteImg(imgfile);
	else if (args->isSet("copy"))
		arsonCdCopy();
	else
		return args;

	return NULL;
}

/*========================================================*/
/*	Main program.
 *========================================================*/

int ArsonApp::main (void)
{
	int result = 1;
	ArsonFrame *pWnd;
	
	if(isRestored())
		RESTORE(ArsonFrame)
	else
	{
		//	Load the user's configuration
		try {
			ACONFIG.load();
		}
		catch (ArsonStartupError &err)
		{
			try { err.report(); }
			catch (...) {
			}
		}

		/*	Check command line args... will return NULL
		 *	if a single operation was specified on the
		 *	command line, otherwise, operate normally.
		 */
		if (KCmdLineArgs *args = handleCmdLine())
		{
			pWnd = new ArsonFrame;

			pWnd->setGeometry(ACONFIG.geometry());
			setMainWidget(pWnd);
			pWnd->show();

			arsonWarning(
				i18n("<b>arson 0.9.8 Version Notes</b>:") + "<ul>" +
				"<li>" + i18n("arson's program preference (which worker program does what) format changed between 0.9.7, and 0.9.8. Please make sure you check your <b>configuration</b> before doing anything if you have previously installed arson.\n\n") +
				"<li>" + i18n("Doing an on-the-fly burn when the track list contains any <b>Ogg</b> files is currently broken, arson will hang if this is attempted. Hopefully I will find and include a workaround in 0.9.9.") +
				"<li>" + i18n("Adding non-local directories (NFS, ftp, etc.) to a file list is also broken. Download remote directories before adding until I get this fixed.") +
				"</ul>" + i18n("This is all lame i know... so don't bother telling me how much i suck :)"), "version-notes-098", i18n("Version Notes"));

			/*	Open any .arson or .md5 file
			 *	specified on the command line
			 */
			if (args->count() > 0)
			{
				const QCString arg (args->arg(0));

				if (arg.right(3).lower() == "md5")
				{
					ArsonFileListDoc *pd = (ArsonFileListDoc *)
						pWnd->setCurrentDocument(DOCUMENT_AUDIO);

					pd->openMd5File(QString("file:") + arg.data());
				}
				else
					arsonOpenDoc(QString("file:") + arg.data());
			}

			//	Run program, run...
			result = exec();

			ArsonLogWindow::finalize();
			ARSON_INSTANCE_DUMP_ALL();
		}

		//	Save user's config
		ACONFIG.save();
	}

	ArsonTempFile::deleteTempFiles();
	ArsonWizardFactory::freeAll();
	return result;
}

/*========================================================*/
/*	A radio button dialog, so the user can choose an item.
 *========================================================*/

ArsonRadioDlg::ArsonRadioDlg (QWidget *parent)
	: ArsonListviewDlgBase(parent ? parent : kapp->mainWidget(), NULL, true),
	m_group(NULL), m_last(NULL)
{
	setCaption(i18n("Select Disk"));

	teh_list->addColumn("");
	teh_list->header()->hide();

	m_group = new QCheckListItem(teh_list,
		i18n("Disk Sets:"), QCheckListItem::Controller);

	m_group->setPixmap(0, QPixmap());
	m_group->setOpen(true);
}

void ArsonRadioDlg::addItem (const QString &item)
{
	const bool empty = (m_group->firstChild() == NULL);
	QCheckListItem *pi = new QCheckListItem(m_group,
		item, QCheckListItem::RadioButton);

	teh_list->moveItem(pi, m_group, m_last);
	(m_last = pi)->setOn(empty);
}

void ArsonRadioDlg::accept (void)
{
	for (QCheckListItem *pi = (QCheckListItem *) m_group->firstChild();
		 pi; (pi = (QCheckListItem *) pi->nextSibling()))
		if (pi->isOn())
		{
			m_sel = pi->text(0);
			break;
		}

	ArsonListviewDlgBase::accept();
}

/*========================================================*/
