/*
  lily-lexer.cc -- implement Lily_lexer

  source file of the GNU LilyPond music typesetter

  (c) 1997--2005 Han-Wen Nienhuys <hanwen@cs.uu.nl>
*/

#include "lily-lexer.hh"

#include <cctype>
#include <sstream>

#include "scm-hash.hh"
#include "interval.hh"
#include "parser.hh"
#include "keyword.hh"
#include "warn.hh"
#include "source-file.hh"
#include "main.hh"
#include "moment.hh"

static Keyword_ent the_key_tab[]
= {
  {"accepts", ACCEPTS},
  {"addlyrics", ADDLYRICS},
  {"addquote", ADDQUOTE},
  {"alias", ALIAS},
  {"alternative", ALTERNATIVE},
  {"bar", BAR},
  {"book", BOOK},
  {"change", CHANGE},
  {"chordmode", CHORDMODE},
  {"chords", CHORDS},
  {"clef", CLEF},
  {"consists", CONSISTS},
  {"context", CONTEXT},
  {"default", DEFAULT},
  {"defaultchild", DEFAULTCHILD},
  {"denies", DENIES},
  {"description", DESCRIPTION},
  {"drummode", DRUMMODE},
  {"drums", DRUMS},
  {"figuremode", FIGUREMODE},
  {"figures", FIGURES},
  {"grobdescriptions", GROBDESCRIPTIONS},
  {"header", HEADER},
  {"key", KEY},
  {"layout", LAYOUT},
  {"lyricmode", LYRICMODE},
  {"lyrics", LYRICS},
  {"lyricsto", LYRICSTO},
  {"mark", MARK},
  {"markup", MARKUP},
  {"midi", MIDI},
  {"name", NAME},
  {"new", NEWCONTEXT},
  {"notemode", NOTEMODE},
  {"objectid", OBJECTID},
  {"octave", OCTAVE},
  {"once", ONCE},
  {"override", OVERRIDE},
  {"paper", PAPER},
  {"partial", PARTIAL},
  {"relative", RELATIVE},
  {"remove", REMOVE},
  {"repeat", REPEAT},
  {"rest", REST},
  {"revert", REVERT},
  {"score", SCORE},
  {"sequential", SEQUENTIAL},
  {"set", SET},
  {"simultaneous", SIMULTANEOUS},
  {"skip", SKIP},
  {"tag", TAG},
  {"tempo", TEMPO},
  {"time", TIME_T},
  {"times", TIMES},
  {"transpose", TRANSPOSE},
  {"transposition", TRANSPOSITION},
  {"type", TYPE},
  {"unset", UNSET},
  {"with", WITH},
  {0, 0}
};

Lily_lexer::Lily_lexer (Sources *sources)
{
  keytable_ = new Keyword_table (the_key_tab);
  chordmodifier_tab_ = SCM_EOL;
  pitchname_tab_stack_ = SCM_EOL;
  sources_ = sources;
  scopes_ = SCM_EOL;
  error_level_ = 0;
  is_main_input_ = false;
  start_module_ = SCM_EOL;
  smobify_self ();

  add_scope (ly_make_anonymous_module (false));
  push_note_state (scm_c_make_hash_table (0));
  chordmodifier_tab_ = scm_make_vector (scm_from_int (1), SCM_EOL);
}

Lily_lexer::Lily_lexer (Lily_lexer const &src)
  : Includable_lexer ()
{
  keytable_ = (src.keytable_) ? new Keyword_table (*src.keytable_) : 0;
  chordmodifier_tab_ = src.chordmodifier_tab_;
  pitchname_tab_stack_ = src.pitchname_tab_stack_;
  sources_ = src.sources_;
  start_module_ = SCM_EOL;
  
  error_level_ = src.error_level_;
  is_main_input_ = src.is_main_input_;

  scopes_ = SCM_EOL;

  smobify_self ();

  SCM scopes = SCM_EOL;
  SCM *tail = &scopes;
  for (SCM s = src.scopes_; scm_is_pair (s); s = scm_cdr (s))
    {
      SCM newmod = ly_make_anonymous_module (false);
      ly_module_copy (newmod, scm_car (s));
      *tail = scm_cons (newmod, SCM_EOL);
      tail = SCM_CDRLOC (*tail);
    }

  scopes_ = scopes;
  push_note_state (scm_c_make_hash_table (0));
}

Lily_lexer::~Lily_lexer ()
{
  delete keytable_;
}

void
Lily_lexer::add_scope (SCM module)
{
  ly_reexport_module (scm_current_module ());
  if (!scm_is_pair (scopes_))
    start_module_ = scm_current_module ();
  
  for (SCM s = scopes_; scm_is_pair (s); s = scm_cdr (s))
    {
      ly_use_module (module, scm_car (s));
    }
  scopes_ = scm_cons (module, scopes_);

  set_current_scope ();
}


SCM
Lily_lexer::remove_scope ()
{
  SCM sc = scm_car (scopes_);
  scopes_ = scm_cdr (scopes_);
  set_current_scope ();
  return sc;
}

SCM
Lily_lexer::set_current_scope ()
{
  SCM old = scm_current_module ();
  
  if (scm_is_pair (scopes_))
    scm_set_current_module (scm_car (scopes_));
  else
    scm_set_current_module (start_module_);

  return old;
}

int
Lily_lexer::lookup_keyword (String s)
{
  return keytable_->lookup (s.to_str0 ());
}

SCM
Lily_lexer::lookup_identifier_symbol (SCM sym)
{
  for (SCM s = scopes_; scm_is_pair (s); s = scm_cdr (s))
    {
      SCM var = ly_module_lookup (scm_car (s), sym);
      if (var != SCM_BOOL_F)
	return scm_variable_ref (var);
    }

  return SCM_UNDEFINED;
}

SCM
Lily_lexer::lookup_identifier (String name)
{
  return lookup_identifier_symbol (ly_symbol2scm (name.to_str0 ()));
}

void
Lily_lexer::start_main_input ()
{
  // yy_flex_debug = 1;
  new_input (main_input_name_, sources_);

  /* Do not allow \include in --safe-mode */
  allow_includes_b_ = allow_includes_b_ && !be_safe_global;

  scm_module_define (scm_car (scopes_),
		     ly_symbol2scm ("input-file-name"),
		     scm_makfrom0str (main_input_name_.to_str0 ()));
}

void
Lily_lexer::set_identifier (SCM name, SCM s)
{
  SCM sym = name;
  if (scm_is_string (name))
    sym = scm_string_to_symbol (name);

  if (scm_is_symbol (sym))
    {
      if (lookup_keyword (ly_symbol2string (sym)) >= 0)
	{
	  String symstr = ly_symbol2string (sym);
	  warning (_f ("identifier name is a keyword: `%s'", symstr.to_str0 ()));
	}

      SCM mod = scm_car (scopes_);

      scm_module_define (mod, sym, s);
    }
  else
    programming_error ("identifier is not a symbol");
}

void
Lily_lexer::LexerError (char const *s)
{
  if (include_stack_.is_empty ())
    message (_f ("error at EOF: %s", s) + "\n");
  else
    {
      error_level_ |= 1;
      Input spot (*lexloc);
      spot.error (s);
    }
}

char
Lily_lexer::escaped_char (char c) const
{
  switch (c)
    {
    case 'n':
      return '\n';
    case 't':
      return '\t';
    case '\'':
    case '\"':
    case '\\':
      return c;
    }
  return 0;
}

Input
Lily_lexer::here_input () const
{
  return Input(*lexloc);
}

void
Lily_lexer::prepare_for_next_token ()
{
  last_input_ = here_input ();
}

/**
   Since we don't create the buffer state from the bytes directly, we
   don't know about the location of the lexer. Add this as a
   YY_USER_ACTION */
void
Lily_lexer::add_lexed_char (int count)
{
  char const * start = here_str0 ();
  lexloc->set (get_source_file (),
	       start, start + count);
  char_count_stack_.top () += count;
}

#include "ly-smobs.icc"

IMPLEMENT_SMOBS (Lily_lexer);
IMPLEMENT_TYPE_P (Lily_lexer, "ly:lily-lexer?");
IMPLEMENT_DEFAULT_EQUAL_P (Lily_lexer);

SCM
Lily_lexer::mark_smob (SCM s)
{
  Lily_lexer *lexer = (Lily_lexer *) SCM_CELL_WORD_1 (s);

  scm_gc_mark (lexer->chordmodifier_tab_);
  scm_gc_mark (lexer->pitchname_tab_stack_);
  scm_gc_mark (lexer->start_module_);
  return lexer->scopes_;
}

int
Lily_lexer::print_smob (SCM s, SCM port, scm_print_state*)
{
  Lily_lexer *lexer = Lily_lexer::unsmob (s);

  scm_puts ("#<Lily_lexer ", port);
  scm_display (lexer->scopes_, port);
  scm_puts (" >", port);
  return 1;
}
