/*
  dynamic-engraver.cc -- implement Dynamic_engraver

  source file of the GNU LilyPond music typesetter

  (c) 1997--2004 Han-Wen Nienhuys <hanwen@cs.uu.nl>
*/

#include "axis-group-interface.hh"
#include "context.hh"
#include "dimensions.hh"
#include "directional-element-interface.hh"
#include "engraver.hh"
#include "hairpin.hh"
#include "interval.hh"
#include "note-column.hh"
#include "paper-column.hh"
#include "script-interface.hh"
#include "side-position-interface.hh"
#include "staff-symbol-referencer.hh"
#include "warn.hh"

/*
  TODO:

  * direction of text-dynamic-event if not equal to direction of
  line-spanner

  - TODO: this engraver is too complicated. We should split it into
  the handling of the basic grobs and the  linespanner

  - TODO: the line-spanner is not killed after the (de)crescs are
  finished.

*/

/**
   print text & hairpin dynamics.
 */
class Dynamic_engraver : public Engraver
{
  Item *script_;
  Spanner *line_spanner_;
  Spanner *cresc_;

  Spanner *finished_line_spanner_;
  Spanner *finished_cresc_;

  Music *script_ev_;
  Music *current_cresc_ev_;
  
  Drul_array<Music*> accepted_spanreqs_drul_;

  Link_array<Note_column> pending_columns_;
  Link_array<Grob> pending_elements_;
  
  void typeset_all ();

  TRANSLATOR_DECLARATIONS (Dynamic_engraver);
  
protected:
  virtual void finalize ();
  virtual void acknowledge_grob (Grob_info);
  virtual bool try_music (Music *req);
  virtual void stop_translation_timestep ();
  virtual void process_music ();  
};




Dynamic_engraver::Dynamic_engraver ()
{
  script_ = 0;
  finished_cresc_ = 0;
  line_spanner_ = 0;
  finished_line_spanner_ = 0;
  current_cresc_ev_ = 0;
  cresc_ = 0;

  script_ev_ = 0;
  accepted_spanreqs_drul_[START] = 0;
  accepted_spanreqs_drul_[STOP] = 0;
}

bool
Dynamic_engraver::try_music (Music *m)
{
  if (m->is_mus_type ("absolute-dynamic-event"))
    {
      /*
	TODO: probably broken.
      */
      script_ev_ = m;
      return true;
    }
  else if (m->is_mus_type ("decrescendo-event")
	   || m->is_mus_type ("crescendo-event"))
    {
      Direction d = to_dir (m->get_property ("span-direction"));

      accepted_spanreqs_drul_[d] = m;
      if (current_cresc_ev_ && d == START)
	accepted_spanreqs_drul_[STOP] = m;
      return true;
    }
  return false;
}

void
Dynamic_engraver::process_music ()
{
  if (accepted_spanreqs_drul_[START] || accepted_spanreqs_drul_[STOP] || script_ev_)
    {
      if (!line_spanner_)
	{
	  Music * rq = accepted_spanreqs_drul_[START];
	  line_spanner_ = make_spanner ("DynamicLineSpanner", rq ? rq->self_scm (): SCM_EOL );

	  if (script_ev_)
	    rq =  script_ev_;
	}
    }
  
  /*
    During a (de)crescendo, pending event will not be cleared,
    and a line-spanner will always be created, as \< \! are already
    two events.

    Note: line-spanner must always have at least same duration
    as (de)crecsendo, b.o. line-breaking.
  */
  

  /*
    maybe we should leave dynamic texts to the text-engraver and
    simply acknowledge them?
  */
  if (script_ev_)
    {
      script_ = make_item ("DynamicText", script_ev_->self_scm ());
      script_->set_property ("text",
			     script_ev_->get_property ("text"));

      
      if (Direction d = to_dir (script_ev_->get_property ("direction")))
	set_grob_direction (line_spanner_, d);

      Axis_group_interface::add_element (line_spanner_, script_);
    }

  Music *stop_ev = accepted_spanreqs_drul_ [STOP] ?
    accepted_spanreqs_drul_[STOP] : script_ev_;

  if (accepted_spanreqs_drul_[STOP] || script_ev_)
    {
      /*
	finish side position alignment if the (de)cresc ends here, and
	there are no new dynamics.
       */


      if (cresc_)
	{
	  assert (!finished_cresc_ && cresc_);

	  cresc_->set_bound (RIGHT, script_
			       ? script_
			       : unsmob_grob (get_property ("currentMusicalColumn")));
	  add_bound_item (line_spanner_, cresc_->get_bound (RIGHT));
	  

	  finished_cresc_ = cresc_;
	  cresc_ = 0;
	  current_cresc_ev_ = 0;
	}
      else if (accepted_spanreqs_drul_[STOP])
	{
	  accepted_spanreqs_drul_[STOP]->origin ()->warning (_ ("can't find start of (de)crescendo"));
	  stop_ev = 0;
	}
      
    }
  
  if (accepted_spanreqs_drul_[START])
    {
      if (current_cresc_ev_)
	{
	  String msg = _ ("already have a decrescendo");
	  if (current_cresc_ev_->is_mus_type ("decrescendo-event"))
	    msg = _ ("already have a crescendo");

	  accepted_spanreqs_drul_[START]->origin ()->warning (msg);
	  current_cresc_ev_->origin ()->warning (_("Cresc started here"));
	}
      else
	{
	  current_cresc_ev_ = accepted_spanreqs_drul_[START];

	  if (Direction d = to_dir (current_cresc_ev_->get_property ("direction")))
	    set_grob_direction (line_spanner_, d);

	  /*
	    TODO: Use symbols.
	  */

	  String start_type = 
	    ly_symbol2string (current_cresc_ev_->get_property ("name"));

	  /*
	    ugh. Use push/pop?
	  */
	  if (start_type == "DecrescendoEvent")
	    start_type = "decrescendo";
	  else if (start_type == "CrescendoEvent")
	    start_type = "crescendo";
	  
	  SCM s = get_property ((start_type + "Spanner").to_str0 ());
	  if (!scm_is_symbol (s) || s == ly_symbol2scm ("hairpin"))
	    {
	      cresc_  = make_spanner ("Hairpin", accepted_spanreqs_drul_[START]->self_scm ());
	      if (finished_cresc_)
		{
		  Pointer_group_interface::add_grob (finished_cresc_,
						     ly_symbol2scm ("adjacent-hairpins"),
						     cresc_);
		  
		  Pointer_group_interface::add_grob (cresc_,
						     ly_symbol2scm ("adjacent-hairpins"),
						     finished_cresc_);
		}
	      cresc_->set_property ("grow-direction",
				    scm_int2num ((start_type == "crescendo")
						 ? BIGGER : SMALLER));
	      
	    }

	  
	  /*
	    This is a convenient (and legacy) interface to TextSpanners
	    for use in (de)crescendi.
	    Hmm.
	  */
	  else
	    {
	      cresc_  = make_spanner ("DynamicTextSpanner", accepted_spanreqs_drul_[START]->self_scm ());
	      cresc_->set_property ("style", s);
	      context ()->set_property ((start_type
					 + "Spanner").to_str0 (), SCM_EOL);
	      s = get_property ((start_type + "Text").to_str0 ());
	      /*
		FIXME: use get_markup () to check type.
	      */
	      if (scm_is_string (s) || scm_is_pair (s))
		{
		  cresc_->set_property ("edge-text",
					scm_cons (s, scm_makfrom0str ("")));
		  context ()->set_property ((start_type + "Text").to_str0 (),
					    SCM_EOL);
		}
	    }

	  if (script_)
	    {
	      cresc_->set_bound (LEFT, script_);
	      add_bound_item (line_spanner_, cresc_->get_bound (LEFT));
	    }
	  
	  Axis_group_interface::add_element (line_spanner_, cresc_);
	}
    }
}

void
Dynamic_engraver::stop_translation_timestep ()
{
  typeset_all ();
  if (!current_cresc_ev_)
    {
      finished_line_spanner_ = line_spanner_;
      line_spanner_ = 0;
      typeset_all ();
    }

  if (cresc_ && !cresc_->get_bound (LEFT))
    {
      cresc_->set_bound (LEFT, unsmob_grob (get_property ("currentMusicalColumn")));
      add_bound_item (line_spanner_, cresc_->get_bound (LEFT));
    }
  
  script_ev_ = 0;
  accepted_spanreqs_drul_[START] = 0;
  accepted_spanreqs_drul_[STOP] = 0;
}

void
Dynamic_engraver::finalize ()
{
  typeset_all ();
  
  if (line_spanner_
      && !line_spanner_->is_live ())
    line_spanner_ = 0;
  if (line_spanner_)
    {
      finished_line_spanner_ = line_spanner_;
      typeset_all ();
    }

  if (cresc_
      && !cresc_->is_live ())
    cresc_ = 0;
  if (cresc_)
    {
      current_cresc_ev_->origin ()->warning (_ ("unterminated (de)crescendo"));
      cresc_->suicide ();
      cresc_ = 0;
    }
}

void
Dynamic_engraver::typeset_all ()
{
  if (finished_cresc_)
    {
      if (!finished_cresc_->get_bound (RIGHT))
	{
	  finished_cresc_->set_bound (RIGHT, script_
				      ? script_
				      : unsmob_grob (get_property ("currentMusicalColumn")));
	  
	  if (finished_line_spanner_)
	    add_bound_item (finished_line_spanner_,
			    finished_cresc_->get_bound (RIGHT));
	}
      finished_cresc_ = 0;
    }
  
  script_ = 0;
  if (finished_line_spanner_)
    {
      /*
	We used to have
	
	     extend-spanner-over-elements (finished_line_spanner_);

	but this is rather kludgy, since finished_line_spanner_
	typically has a staff-symbol field set , extending it over the
	entire staff.

      */

      Grob * l = finished_line_spanner_->get_bound (LEFT);
      Grob * r = finished_line_spanner_->get_bound (RIGHT);      
      if (!r && l)
	finished_line_spanner_->set_bound (RIGHT, l);
      else if (!l && r)
	finished_line_spanner_->set_bound (LEFT, r);
      else if (!r && !l)
	{
	  /*
	    This is a isolated dynamic apparently, and does not even have
	    any interesting support item.
	   */
	  Grob * cc = unsmob_grob (get_property ("currentMusicalColumn"));
	  Item * ci = dynamic_cast<Item*>(cc);
	  finished_line_spanner_->set_bound (RIGHT, ci);
	  finished_line_spanner_->set_bound (LEFT, ci);	  
	}
	
      finished_line_spanner_ = 0;
    }
}

void
Dynamic_engraver::acknowledge_grob (Grob_info info)
{
  if (!line_spanner_)
    return;

  if (Note_column::has_interface (info.grob_))
    {
      if (line_spanner_
	  /* Don't refill killed spanner */
	  && line_spanner_->is_live ())
	{
	  Side_position_interface::add_support (line_spanner_,info.grob_);
	  add_bound_item (line_spanner_,dynamic_cast<Item*> (info.grob_));
	}

      if (script_ && !script_->get_parent (X_AXIS))
	{
	  SCM head = scm_last_pair (info.grob_->get_property ("note-heads"));
	  if (scm_is_pair (head))
	    script_->set_parent (unsmob_grob (scm_car (head)),  X_AXIS);
	}



      if (cresc_ && !cresc_->get_bound (LEFT))
	{
	  cresc_->set_bound (LEFT, info.grob_);
	  add_bound_item (line_spanner_, cresc_->get_bound (LEFT));
	}
        
    }
  else if (Script_interface::has_interface (info.grob_) && script_)
    {
      SCM p = info.grob_->get_property ("script-priority");

      /*
	UGH.

	DynamicText doesn't really have a script-priority field.
       */
      if (scm_is_number (p)
	  && scm_to_int (p)
	  < scm_to_int (script_->get_property ("script-priority")))
	Side_position_interface::add_support (line_spanner_, info.grob_);
    }
}

ADD_TRANSLATOR (Dynamic_engraver,
/* descr */       
"This engraver creates hairpins, dynamic texts, and their vertical\n"
"alignments.  The symbols are collected onto a DynamicLineSpanner grob\n"
"which takes care of vertical positioning.  "
,
		  
/* creats*/       "DynamicLineSpanner DynamicText Hairpin TextSpanner",
/* accepts */     "absolute-dynamic-event crescendo-event decrescendo-event",
/* acks  */      "note-column-interface script-interface",
/* reads */       "",
/* write */       "");
