;;;; This file is part of LilyPond, the GNU music typesetter.
;;;;
;;;; Copyright (C) 2005--2010 Jan Nieuwenhuizen <janneke@gnu.org>
;;;; Han-Wen Nienhuys <hanwen@xs4all.nl>
;;;;
;;;; LilyPond is free software: you can redistribute it and/or modify
;;;; it under the terms of the GNU General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or
;;;; (at your option) any later version.
;;;;
;;;; LilyPond is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;;; GNU General Public License for more details.
;;;;
;;;; You should have received a copy of the GNU General Public License
;;;; along with LilyPond.  If not, see <http://www.gnu.org/licenses/>.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 
;; backend helpers.

(use-modules (scm ps-to-png)
	     (scm paper-system)
	     (ice-9 optargs))

(define-public (ly:system command . rest)
  (let* ((status 0)
	 (dev-null "/dev/null")
	 (silenced (if (or (ly:get-option 'verbose)
			   (not (access? dev-null W_OK)))
		       command
		       (format #f "~a > ~a 2>&1 " command dev-null))))
    (if (ly:get-option 'verbose)
	(begin
	  (ly:message (_ "Invoking `~a'...") command))
	  (ly:progress "\n"))

    (set! status
	  (if (pair? rest)
	      (system-with-env silenced (car rest))
	      (system silenced)))
	
    (if (> status 0)
	(begin
	  (ly:message (_ "`~a' failed (~a)") command status)
	  (ly:progress "\n")
	  ;; hmmm.  what's the best failure option? 
	  (throw 'ly-file-failed)))))

(define-public (system-with-env cmd env)

  "Execute CMD in fork, with ENV (a list of strings) as the environment"
  (let*
      ;; laziness: should use execle?
      
      ((pid (primitive-fork)))
    (if (= 0 pid)
	;; child
	(begin
	  (environ env)
	  (system cmd))
	
	;; parent
	(cdr (waitpid pid)))))

(define-public (sanitize-command-option str)
  "Kill dubious shell quoting"
  
  (string-append
   "\""
   (regexp-substitute/global #f "[^-_ 0-9,.a-zA-Z'\"\\]" str 'pre 'post)
   "\""))

(define-public (search-executable names)
  (define (helper path lst)
    (if (null? (cdr lst))
	(car lst)
	(if (search-path path (car lst)) (car lst)
	    (helper path (cdr lst)))))

  (let ((path (parse-path (getenv "PATH"))))
    (helper path names)))

(define-public (search-gs)
  
  ;; must be sure that we don't catch stuff from old GUBs.
  (search-executable '("gs")))
  
(define-public (postscript->pdf paper-width paper-height name)
  (let* ((pdf-name (string-append
		    (dir-basename name ".ps" ".eps")
		    ".pdf"))
	 (is-eps (string-match "\\.eps$" name))
	 (paper-size-string (if is-eps
				"-dEPSCrop"
				(ly:format "-dDEVICEWIDTHPOINTS=~$\
 -dDEVICEHEIGHTPOINTS=~$"
					paper-width paper-height)))

	 (cmd (ly:format
		      "~a\
 ~a\
 ~a\
 ~a\
 -dCompatibilityLevel=1.4\
 -dNOPAUSE\
 -dBATCH\
 -r1200\
 -sDEVICE=pdfwrite\
 -sOutputFile=~S\
 -c .setpdfwrite\
 -f ~S\
"
		      (search-gs)
		      (if (ly:get-option 'verbose) "" "-q")
		      (if (or (ly:get-option 'gs-load-fonts)
			      (ly:get-option 'gs-load-lily-fonts))
			  "-dNOSAFER"
			  "-dSAFER")
		      paper-size-string
		      pdf-name
		      name)))
    ;; The wrapper on windows cannot handle `=' signs,
    ;; gs has a workaround with #.
    (if (eq? PLATFORM 'windows)
	(begin
	  (set! cmd (string-regexp-substitute "=" "#" cmd))
	  (set! cmd (string-regexp-substitute "-dSAFER " "" cmd))))

    (ly:message (_ "Converting to `~a'...") pdf-name)
    (ly:progress "\n")
    (ly:system cmd)))

(define-public (postscript->png resolution paper-width paper-height name)
  (let* ((verbose (ly:get-option 'verbose))
	 (rename-page-1 #f))

    ;; Do not try to guess the name of the png file,
    ;; GS produces PNG files like BASE-page%d.png.
    (ly:message (_ "Converting to ~a...") "PNG")
    (make-ps-images name
		    #:resolution resolution
		    #:page-width paper-width
		    #:page-height paper-height
		    #:rename-page-1 rename-page-1
		    #:be-verbose verbose
		    #:anti-alias-factor (ly:get-option 'anti-alias-factor)
		    #:pixmap-format (ly:get-option 'pixmap-format))
    (ly:progress "\n")))

(define-public (postprocess-output paper-book module filename formats)
  (let* ((completed (completize-formats formats))
	 (base (dir-basename filename ".ps" ".eps"))
	 (intermediate (remove (lambda (x) (member x formats)) completed)))
    (for-each (lambda (f)
		((eval (string->symbol (format "convert-to-~a" f))
		       module) paper-book filename)) completed)
    (if (ly:get-option 'delete-intermediate-files)
	(for-each (lambda (f)
		    (if (file-exists? f) (delete-file f)))
		  (map (lambda (x) (string-append base "." x)) intermediate)))))

(define-public (completize-formats formats)
  (define new-fmts '())
  (if (member "png" formats)
      (set! formats (cons "ps" formats)))
  (if (member "pdf" formats)
      (set! formats (cons "ps" formats)))
  (for-each (lambda (x)
	      (if (member x formats) (set! new-fmts (cons x new-fmts))))
	    '("ps" "pdf" "png"))
  (uniq-list (reverse new-fmts)))

(define (header-to-file file-name key value)
  (set! key (symbol->string key))
  (if (not (equal? "-" file-name))
      (set! file-name (string-append file-name "." key)))
  (ly:message (_ "Writing header field `~a' to `~a'...")
	      key
	      (if (equal? "-" file-name) "<stdout>" file-name))
  (if (equal? file-name "-")
      (display value)
      (let ((port (open-file file-name "w")))
	(display value port)
	(close-port port)))

  (ly:progress "\n")
  "")

(define-public (output-scopes scopes fields basename)
  (define (output-scope scope)
    (apply
     string-append
     (module-map
      (lambda (sym var)
	(let ((val (if (variable-bound? var) (variable-ref var) "")))
	  (if (and (memq sym fields) (string? val))
	      (header-to-file basename sym val))
	  ""))
      scope)))
  (apply string-append (map output-scope scopes)))

(define-public (relevant-book-systems book)
  (let ((systems (ly:paper-book-systems book)))
    ;; skip booktitles.
    (if (and (not (ly:get-option 'include-book-title-preview))
	     (pair? systems)
	     (ly:prob-property (car systems) 'is-book-title #f))
	(cdr systems)
	systems)))

(define-public (relevant-dump-systems systems)
  (let ((to-dump-systems '()))
    (for-each
      (lambda (sys)
	(if (or (paper-system-title? sys)
		(not (pair? to-dump-systems))
		(paper-system-title? (car to-dump-systems)))
	    (set! to-dump-systems (cons sys to-dump-systems))))
      systems)
    to-dump-systems))

(define missing-stencil-list '())

(define-public (backend-testing output-module)
  (define (missing-stencil-expression name)
    (begin
      (ly:warning (_ "missing stencil expression `~S'") name)
      ""))

  (map (lambda (x)
	 (if (not (module-defined? output-module x))
	     (begin
	       (module-define! output-module x
			       (lambda* (#:optional y . z)
				 (missing-stencil-expression x)))
	       (set! missing-stencil-list (append (list x)
						  missing-stencil-list)))))
       (ly:all-stencil-commands)))

(define-public (remove-stencil-warnings output-module)
  (for-each
    (lambda (x)
      (module-remove! output-module x))
    missing-stencil-list))
