/*
  stencil.cc -- implement Stencil

  source file of the GNU LilyPond music typesetter

  (c) 1997--2004 Han-Wen Nienhuys <hanwen@cs.uu.nl>
*/

#include <math.h>
#include <libc-extension.hh>	// isinf

#include "font-metric.hh" 
#include "dimensions.hh"
#include "interval.hh"
#include "string.hh"
#include "stencil.hh"
#include "warn.hh"


#include "ly-smobs.icc"


SCM
Stencil::smobbed_copy () const
{
  Stencil * m = new Stencil (*this);

  return m->smobbed_self ();
}

Interval
Stencil::extent (Axis a) const
{
  return dim_[a];
}

Stencil::Stencil (Box b, SCM func)
{
  expr_ = func;
  dim_ = b;
}

Stencil::Stencil ()
{
  expr_ = SCM_EOL;
  set_empty (true);
}

void
Stencil::translate (Offset o)
{
  Axis a = X_AXIS;
  while (a < NO_AXES)
    {
      if (abs (o[a]) > 100 CM
	  || isinf (o[a]) || isnan (o[a]))
	{
	  programming_error ("Improbable offset for translation: setting to zero");
	  o[a] =  0.0;
	}
      incr (a);
    }

  expr_ = scm_list_n (ly_symbol2scm ("translate-stencil"),
		   ly_offset2scm (o),
		   expr_, SCM_UNDEFINED);
  if (!is_empty ())
    dim_.translate (o);
}
  

void
Stencil::translate_axis (Real x,Axis a)
{
  Offset o (0,0);
  o[a] = x;
  translate (o);
}  



void
Stencil::add_stencil (Stencil const &m)
{
  expr_ = scm_list_n (ly_symbol2scm ("combine-stencil"),
		   m.expr_,
		   expr_, SCM_UNDEFINED);
  dim_.unite (m.dim_);
}

void
Stencil::set_empty (bool e)
{
  if (e)
    {
      dim_[X_AXIS].set_empty ();
      dim_[Y_AXIS].set_empty ();
    }
  else
    {
      dim_[X_AXIS] = Interval (0,0);
      dim_[Y_AXIS] = Interval (0,0);
    }
}


void
Stencil::align_to (Axis a, Real x)
{
  if (is_empty ())
    return ;

  Interval i (extent (a));
  translate_axis (-i.linear_combination (x), a);
}

/*
  See scheme Function.
 */
void
Stencil::add_at_edge (Axis a, Direction d, Stencil const &m, Real padding,
		       Real minimum)
{
  Real my_extent= is_empty () ? 0.0 : dim_[a][d];
  Interval i (m.extent (a));
  Real his_extent;
  if (i.is_empty ())
    {
      programming_error ("Stencil::add_at_edge: adding empty stencil.");
      his_extent = 0.0;
    }
  else
    his_extent = i[-d];      

  Real offset = (my_extent -  his_extent)  + d*padding;
  if (minimum > 0  && fabs (offset) <  minimum)
    offset = sign (offset) * minimum; 
  
  Stencil toadd (m);
  toadd.translate_axis (offset, a);
  add_stencil (toadd);
}



/*
  Hmm... maybe this is not such a good idea ; stuff can be empty,
  while expr_ == '()
 */
bool
Stencil::is_empty () const
{
  return expr_ == SCM_EOL;
}

SCM
Stencil::get_expr () const
{
  return expr_;
}



Box
Stencil::extent_box () const
{
  return dim_;
}
IMPLEMENT_SIMPLE_SMOBS (Stencil);


int
Stencil::print_smob (SCM , SCM port, scm_print_state *)
{
  scm_puts ("#<Stencil ", port);
  scm_puts (" >", port);
  
  return 1;
}

  
SCM
Stencil::mark_smob (SCM s)
{
  Stencil  *r = (Stencil *) ly_cdr (s);
  
  return r->expr_;
}

IMPLEMENT_TYPE_P (Stencil, "ly:stencil?");
IMPLEMENT_DEFAULT_EQUAL_P (Stencil);

