/*
  pdf-scheme.cc --  implement Pdf bindings.

  source file of the GNU LilyPond music typesetter

  (c) 2005 Han-Wen Nienhuys <hanwen@xs4all.nl>

*/

#include "pdf.hh"
#include "string.hh"

LY_DEFINE(ly_open_pdf_file, "ly:open-pdf-file",
	  1, 0, 0, (SCM name),
	  "Open @var{name} for writing PDF")
{
  SCM_ASSERT_TYPE(scm_is_string (name), name, SCM_ARG1, __FUNCTION__, "String");
  
  Pdf_file * pdf = new Pdf_file (ly_scm2string (name));
  SCM asscm = pdf->self_scm ();

  scm_gc_unprotect_object (asscm);
  return asscm;
}

LY_DEFINE(ly_make_pdf_object, "ly:make-pdf-object",
	  1, 0, 0, (SCM value),
	  "Make a PDF object for @var{value}.\n"
	  "\n"
	  "This uses the following mapping:\n"
	  "@example\n"
	  "String - String\n"
	  "Dictionary - (dictionary  . ALIST)\n"
	  "Stream - (stream  . STRING)\n"
	  "Null - (null)\n"
	  "Array - vector\n"
	  "Name - symbol\n"
	  "Boolean - Boolean\n"
	  "Number - number\n"
	  "@end example")
{
  Pdf_object * obj = new Pdf_object ();
  obj->set_value (value);
  SCM asscm = obj->self_scm();
  scm_gc_unprotect_object (asscm);
  return asscm;
}


LY_DEFINE(ly_pdf_close, "ly:pdf-close",
	  1, 0, 0, (SCM pdf_file),
	  "Close @var{pdf-file}.")
{
  Pdf_file *file = unsmob_pdf_file (pdf_file);
  SCM_ASSERT_TYPE(file, pdf_file, SCM_ARG1, __FUNCTION__, "PDF File");
  
  file->terminate ();
  return SCM_UNSPECIFIED;
}


LY_DEFINE(ly_pdf_set_root, "ly:pdf-set-root",
	  1, 0, 0, (SCM pdf_file, SCM pdf_obj),
	  "Set the document root of @var{pdf-file}.")
{
  Pdf_file *file = unsmob_pdf_file (pdf_file);
  Pdf_object *obj = unsmob_pdf_object (pdf_obj);
  SCM_ASSERT_TYPE(file, pdf_file, SCM_ARG1, __FUNCTION__, "PDF File");
  SCM_ASSERT_TYPE(obj && obj->is_dict (),
		  pdf_obj, SCM_ARG2, __FUNCTION__, "PDF Dictionary object");

  
  file->set_root_document (obj);
  return SCM_UNSPECIFIED;
}


LY_DEFINE(ly_pdf_write_object, "ly:pdf-write-object",
	  2, 0, 0, (SCM pdf_file, SCM pdf_obj),
	  "Write @var{pdf-obj} to @var{pdf-file}.")
{
  Pdf_file *file = unsmob_pdf_file (pdf_file);
  Pdf_object *obj = unsmob_pdf_object (pdf_obj);
  SCM_ASSERT_TYPE(file, pdf_file, SCM_ARG1, __FUNCTION__, "PDF File");
  SCM_ASSERT_TYPE(obj, pdf_obj, SCM_ARG2, __FUNCTION__, "PDF Object");

  file->write_object (obj);
  return SCM_UNSPECIFIED;
}
