/*
  This file is part of LilyPond, the GNU music typesetter.

  Copyright (C) 1998--2011 Han-Wen Nienhuys <hanwen@xs4all.nl>

  LilyPond is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  LilyPond is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with LilyPond.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "music-sequence.hh"

#include "warn.hh"
#include "program-option.hh"
#include "music.hh"
#include "input.hh"

void
transpose_music_list (SCM lst, Pitch rq)
{
  for (SCM s = lst; scm_is_pair (s); s = scm_cdr (s))
    unsmob_music (scm_car (s))->transpose (rq);
}

Moment
Music_sequence::cumulative_length (SCM l)
{
  Moment cumulative;
  Moment last_len;

  for (SCM s = l; scm_is_pair (s); s = scm_cdr (s))
    {
      Moment l = unsmob_music (scm_car (s))->get_length ();
      if (last_len.grace_part_ && l.main_part_)
        last_len.grace_part_ = Rational (0);
      cumulative += last_len;
      last_len = l;
    }

  last_len.grace_part_ = Rational (0);
  cumulative += last_len;

  return cumulative;
}

Moment
Music_sequence::maximum_length (SCM l)
{
  Moment dur = 0;
  for (SCM s = l; scm_is_pair (s); s = scm_cdr (s))
    {
      Music *m = unsmob_music (scm_car (s));
      if (!m)
        programming_error ("Music sequence should have music elements");
      else
        {
          Moment l = m->get_length ();
          dur = max (dur, l);
        }
    }

  return dur;
}

MAKE_SCHEME_CALLBACK (Music_sequence, maximum_length_callback, 1);
SCM
Music_sequence::maximum_length_callback (SCM m)
{
  Music *me = unsmob_music (m);
  return maximum_length (me->get_property ("elements")).smobbed_copy ();
}

MAKE_SCHEME_CALLBACK (Music_sequence, cumulative_length_callback, 1);
SCM
Music_sequence::cumulative_length_callback (SCM m)
{
  Music *me = unsmob_music (m);
  return cumulative_length (me->get_property ("elements")).smobbed_copy ();
}

MAKE_SCHEME_CALLBACK (Music_sequence, minimum_start_callback, 1);
SCM
Music_sequence::minimum_start_callback (SCM m)
{
  Music *me = unsmob_music (m);
  return minimum_start (me->get_property ("elements")).smobbed_copy ();
}

MAKE_SCHEME_CALLBACK (Music_sequence, first_start_callback, 1);
SCM
Music_sequence::first_start_callback (SCM m)
{
  Music *me = unsmob_music (m);
  return first_start (me->get_property ("elements")).smobbed_copy ();
}

Pitch
music_list_to_relative (SCM l, Pitch p, bool ret_first)
{
  Pitch first = p;
  int count = 0;

  Pitch last = p;
  for (SCM s = l; scm_is_pair (s); s = scm_cdr (s))
    {
      if (Music *m = unsmob_music (scm_car (s)))
        {
          last = m->to_relative_octave (last);
          if (!count++)
            first = last;
        }
    }

  return (ret_first) ? first : last;
}

void
compress_music_list (SCM l, Moment m)
{
  for (SCM s = l; scm_is_pair (s); s = scm_cdr (s))
    unsmob_music (scm_car (s))->compress (m);
}

Moment
Music_sequence::minimum_start (SCM l)
{
  Moment m;

  for (SCM s = l; scm_is_pair (s); s = scm_cdr (s))
    m = min (m, unsmob_music (scm_car (s))->start_mom ());
  return m;
}

Moment
Music_sequence::first_start (SCM l)
{
  Moment m;

  for (SCM s = l; scm_is_pair (s); s = scm_cdr (s))
    {
      Music *mus = unsmob_music (scm_car (s));
      Moment l = mus->get_length ();
      Moment s = mus->start_mom ();
      if (l.to_bool () || s.to_bool ())
        return s;
    }
  return m;
}

MAKE_SCHEME_CALLBACK (Music_sequence, simultaneous_relative_callback, 2);
SCM
Music_sequence::simultaneous_relative_callback (SCM music, SCM pitch)
{
  Music *me = unsmob_music (music);
  Pitch p = *unsmob_pitch (pitch);

  SCM elts = me->get_property ("elements");
  SCM copied = SCM_EOL;
  if (lily_1_8_relative)
    copied = ly_music_deep_copy (elts);

  Pitch retval = music_list_to_relative (elts, p, false);

  if (lily_1_8_relative)
    {

      Pitch retval_1_8 = music_list_to_relative (copied, p, true);
      if (retval_1_8 != retval)
        lily_1_8_compatibility_used = true;

      retval = retval_1_8;
    }

  return retval.smobbed_copy ();
}

MAKE_SCHEME_CALLBACK (Music_sequence, event_chord_relative_callback, 2);
SCM
Music_sequence::event_chord_relative_callback (SCM music, SCM pitch)
{
  Music *me = unsmob_music (music);
  Pitch p = *unsmob_pitch (pitch);
  return music_list_to_relative (me->get_property ("elements"),
                                 p, true).smobbed_copy ();
}

MAKE_SCHEME_CALLBACK (Music_sequence, repeated_chord_relative_callback, 2);
SCM
Music_sequence::repeated_chord_relative_callback (SCM music, SCM pitch)
{
  Music *me = unsmob_music (music);
  Music *repeated_chord = unsmob_music (me->get_property ("element"));
  Music *original_chord = unsmob_music (me->get_property ("original-chord"));

  /* A repeated chord octave is not computed from the previous pitch,
   * (this function `pitch' argument), but from the original chord, so
   * that repeated chords have the same octave have the original chord,
   * even though other simple notes have been entered meanwhile.
   */
  assert (repeated_chord);
  Pitch *p = 0;
  /* Get the original chord first pitch */
  if (original_chord)
    {
      for (SCM s = original_chord->get_property ("elements"); scm_is_pair (s); s = scm_cdr (s))
        {
          if (Music *m = unsmob_music (scm_car (s)))
            {
              p = unsmob_pitch (m->get_property ("pitch"));
              if (p)
                break;
            }
        }
    }
  /* Use the `pitch' argument if no pitch found in original chord. */
  if (! p)
    p = unsmob_pitch (pitch);

  /* Change the first note pitch to -1, to avoid octaviation.  Indeed,
   * the first pitch should be the same as the original chord first
   * pitch. */
  for (SCM s = repeated_chord->get_property ("elements"); scm_is_pair (s); s = scm_cdr (s))
    {
      if (Music *m = unsmob_music (scm_car (s)))
        {
          Pitch *first_pitch = unsmob_pitch (m->get_property ("pitch"));
          if (first_pitch)
            {
              Pitch new_pitch = Pitch (-1,
                                       first_pitch->get_notename (),
                                       first_pitch->get_alteration ());
              m->set_property ("pitch", new_pitch.smobbed_copy ());
              break;
            }
        }
    }
  music_list_to_relative (repeated_chord->get_property ("elements"), *p, true).smobbed_copy ();
  /* Return `pitch' instead of the repeated chord first pitch,
   * because `pitch' is the last explicitly entered pitch */
  return pitch;
}
