/*
  system.cc -- implement System

  source file of the GNU LilyPond music typesetter

  (c) 1996--2003 Han-Wen Nienhuys <hanwen@cs.uu.nl>
*/

#include "input-smob.hh"
#include "axis-group-interface.hh"
#include "warn.hh"
#include "system.hh"
#include "main.hh"
#include "paper-column.hh"
#include "paper-def.hh"
#include "paper-outputter.hh"
#include "paper-score.hh"
#include "string.hh"
#include "warn.hh"
#include "dimensions.hh"
#include "molecule.hh"
#include "all-font-metrics.hh"
#include "spacing-interface.hh"

// todo: use map.
void
fixup_refpoints (SCM s)
{
  for (; gh_pair_p (s); s = ly_cdr (s))
    {
      Grob::fixup_refpoint (ly_car (s));
    }
}


System::System (SCM s)
  : Spanner (s)
{
  rank_ = 0;
}

int
System::element_count () const
{
  return scm_ilength (get_grob_property ("all-elements"));
}

int
System::spanner_count () const
{
  int k =0;
  for (SCM s = get_grob_property ("all-elements");
       gh_pair_p (s); s = ly_cdr (s))
    {
      if (dynamic_cast<Spanner*> (unsmob_grob (gh_car(s))))
	k++;
    }

  return k;
}
  


void
System::typeset_grob (Grob * elem)
{
  elem->pscore_ = pscore_;
  Pointer_group_interface::add_grob (this, ly_symbol2scm ("all-elements"),elem);
  scm_gc_unprotect_object (elem->self_scm ());
}

void
System::output_lines ()
{
  for (SCM s = get_grob_property ("all-elements");
       gh_pair_p (s); s = ly_cdr (s))
    {
      Grob * g = unsmob_grob (ly_car (s));
      if (Spacing_interface::has_interface (g))
	{
	  /*
	    Kill no longer needed grobs. 
	   */
	  Item * it = dynamic_cast<Item*> (g);
	  if (it && Item::breakable_b(it))
	    {
	      it->find_prebroken_piece (LEFT)->suicide();
	      it->find_prebroken_piece (RIGHT)->suicide();
	    }
	  g->suicide ();
	}
      else if (g->live ())
	g->do_break_processing ();
    }

  /*
    fixups must be done in broken line_of_scores, because new elements
    are put over there.  */
  int count = 0;
  for (int i=0; i < broken_intos_.size (); i++)
    {
      Grob *se = broken_intos_[i];
      SCM all = se->get_grob_property ("all-elements");
      for (SCM s = all; gh_pair_p (s); s = ly_cdr (s))
	{
	  fixup_refpoint (ly_car (s));
	}
      count += scm_ilength (all);
    }
  
  /*
    needed for doing items.
   */
  fixup_refpoints (get_grob_property ("all-elements"));

  
  for (SCM s = get_grob_property ("all-elements");
       gh_pair_p (s); s = ly_cdr (s))
    {
      unsmob_grob (ly_car (s))->handle_broken_dependencies ();
    }
  handle_broken_dependencies ();

  if (verbose_global_b)
    progress_indication (_f ("Element count %d.",  count + element_count ()));

  
  for (int i=0; i < broken_intos_.size (); i++)
    {
      System *system = dynamic_cast<System*> (broken_intos_[i]);

      if (verbose_global_b)
	progress_indication ("[");
      system->post_processing (i+1 == broken_intos_.size ());

      if (verbose_global_b)
	{
	  progress_indication (to_string (i));
	  progress_indication ("]");
	}

      if (i < broken_intos_.size () - 1)
	{
	  SCM lastcol =  ly_car (system->get_grob_property ("columns"));
	  Grob*  e = unsmob_grob (lastcol);

	  SCM between = ly_symbol2scm ("between-system-string");
	  SCM inter = e->internal_get_grob_property (between);
	  if (gh_string_p (inter))
	    {
	      pscore_->outputter_
		->output_scheme (scm_list_n (between, 
					     inter, SCM_UNDEFINED));	      
	    }
	}
    }
}




/*
  Find the loose columns in POSNS, and drape them around the columns
  specified in BETWEEN-COLS.  */
void
set_loose_columns (System* which, Column_x_positions const *posns)
{
  for (int i = 0; i < posns->loose_cols_.size (); i++)
    {
      int divide_over = 1;
      Item *loose = dynamic_cast<Item*> (posns->loose_cols_[i]);
      Paper_column* col = dynamic_cast<Paper_column*> (loose);
      
      if (col->system_)
	continue;

      
      Item * left = 0;
      Item * right = 0;
      do
	{
	  SCM between = loose->get_grob_property ("between-cols");
	  if (!gh_pair_p (between))
	    break;


	  Item * l=dynamic_cast<Item*> (unsmob_grob (ly_car (between)));
	  Item * r=dynamic_cast<Item*> (unsmob_grob (ly_cdr (between)));

	  if (!(l && r))
	    break ;
	  
	  if (!left && l)
	    {
	      left = l->get_column ();
	    }

	  divide_over ++;

	  loose = right = r->get_column ();
	}
      while (1);
      
      /*
	We divide the remaining space of the column over the left and
	right side. At the moment, we  
	
      */
      Grob * common = right->common_refpoint (left, X_AXIS);
      
      Real rx =	right->extent(common, X_AXIS)[LEFT];
      Real lx =  left->extent(common, X_AXIS)[RIGHT];
      Real total_dx = rx - lx;
      Interval cval =col->extent (col, X_AXIS);

      /*
	
	We put it in the middle. This is not an ideal solution -- the
	break alignment code inserts a fixed space before the clef
	(about 1 SS), while the space following the clef is
	flexible. In tight situations, the clef will almost be on top
	of the following note. 
	
      */
      Real dx = rx-lx - cval.length ();
      if (total_dx < 2* cval.length ())
	{
	  /*
	    todo: this is discontinuous. I'm too tired to
	    invent a sliding mechanism. Duh.

	    TODO.
	   */
	  dx *= 0.25;
	}
      else
	dx *= 0.5;

      col->system_ = which;
      col->translate_axis (lx + dx - cval[LEFT], X_AXIS); 
    }
}

// const?
void
System::break_into_pieces (Array<Column_x_positions> const &breaking)
{
  for (int i=0; i < breaking.size (); i++)
    {
      System *system = dynamic_cast <System*> (clone ());
      system->rank_ = i;

      Link_array<Grob> c (breaking[i].cols_);
      pscore_->typeset_line (system);
      
      system->set_bound (LEFT,c[0]);
      system->set_bound (RIGHT,c.top ());
      for (int j=0; j < c.size (); j++)
	{
	  c[j]->translate_axis (breaking[i].config_[j],X_AXIS);
	  dynamic_cast<Paper_column*> (c[j])->system_ = system;
	}
      set_loose_columns (system, &breaking[i]);
      broken_intos_.push (system);
    }
}


void
System::output_molecule (SCM expr, Offset o)
{

  while (1)
    {
      if (!gh_pair_p (expr))
	return;
  
      SCM head =ly_car (expr);
      if (unsmob_input (head))
	{
	  Input * ip = unsmob_input (head);
      
	  pscore_->outputter_->output_scheme (scm_list_n (ly_symbol2scm ("define-origin"),
							   scm_makfrom0str (ip->file_string ().to_str0 ()),
							   gh_int2scm (ip->line_number ()),
							   gh_int2scm (ip->column_number ()),
							   SCM_UNDEFINED));
	  expr = ly_cadr (expr);
	}
      else  if (head ==  ly_symbol2scm ("no-origin"))
	{
	  pscore_->outputter_->output_scheme (scm_list_n (head, SCM_UNDEFINED));
	  expr = ly_cadr (expr);
	}
      else if (head == ly_symbol2scm ("translate-molecule"))
	{
	  o += ly_scm2offset (ly_cadr (expr));
	  expr = ly_caddr (expr);
	}
      else if (head == ly_symbol2scm ("combine-molecule"))
	{
	  output_molecule (ly_cadr (expr), o);
	  expr = ly_caddr (expr);
	}
      else
	{
	  pscore_->outputter_->
	    output_scheme (scm_list_n (ly_symbol2scm ("placebox"),
				    gh_double2scm (o[X_AXIS]),
				    gh_double2scm (o[Y_AXIS]),
				    expr,
				    SCM_UNDEFINED));

	  return;
	}
    }
}

void
System::output_scheme (SCM s)
{
  pscore_->outputter_->output_scheme (s);
}

void
System::add_column (Paper_column*p)
{
  Grob *me = this;
  SCM cs = me->get_grob_property ("columns");
  Grob * prev =  gh_pair_p (cs) ? unsmob_grob (ly_car (cs)) : 0;

  p->rank_ = prev ? Paper_column::get_rank (prev) + 1 : 0; 

  me->set_grob_property ("columns",  gh_cons (p->self_scm (), cs));

  Axis_group_interface::add_element (me, p);
}

void
System::pre_processing ()
{
  for (SCM s = get_grob_property ("all-elements"); gh_pair_p (s); s = ly_cdr (s))
    unsmob_grob (ly_car (s))->discretionary_processing ();

  if (verbose_global_b)
    progress_indication (_f ("Grob count %d ",  element_count ()));

  
  for (SCM s = get_grob_property ("all-elements"); gh_pair_p (s); s = ly_cdr (s))
    unsmob_grob (ly_car (s))->handle_prebroken_dependencies ();
  
  fixup_refpoints (get_grob_property ("all-elements"));
  
  for (SCM s = get_grob_property ("all-elements"); gh_pair_p (s); s = ly_cdr (s))
    {
      Grob* sc = unsmob_grob (ly_car (s));
      sc->calculate_dependencies (PRECALCED, PRECALCING, ly_symbol2scm ("before-line-breaking-callback"));
    }
  
  progress_indication ("\n" + _ ("Calculating line breaks...") + " ");
  for (SCM s = get_grob_property ("all-elements"); gh_pair_p (s); s = ly_cdr (s))
    {
      Grob * e = unsmob_grob (ly_car (s));
      SCM proc = e->get_grob_property ("spacing-procedure");
      if (gh_procedure_p (proc))
	gh_call1 (proc, e->self_scm ());
    }
}

void
System::post_processing (bool last_line)
{
  for (SCM s = get_grob_property ("all-elements");
       gh_pair_p (s); s = ly_cdr (s))
    {
      Grob* sc = unsmob_grob (ly_car (s));
      sc->calculate_dependencies (POSTCALCED, POSTCALCING,
				  ly_symbol2scm ("after-line-breaking-callback"));
    }

  Interval i (extent (this, Y_AXIS));
  if (i.is_empty ())
    programming_error ("Huh?  Empty System?");
  else
    translate_axis (- i[MAX], Y_AXIS);

  Real height = i.length ();
  if (height > 50 CM)
    {
      programming_error ("Improbable system height");
      height = 50 CM;
    }

  /*
    generate all molecules  to trigger all font loads.

    (ugh. This is not very memory efficient.)  */
  this->get_molecule();
  for (SCM s = get_grob_property ("all-elements"); gh_pair_p (s); s = ly_cdr (s))
    {
      unsmob_grob (ly_car (s))->get_molecule ();
    }
  /*
    font defs;
   */
  SCM font_names = ly_quote_scm (get_paper ()->font_descriptions ());  
  output_scheme (scm_list_n (ly_symbol2scm ("define-fonts"),
			     font_names,
			     SCM_UNDEFINED));

  /*
    line preamble.
   */
  Interval j (extent (this, X_AXIS));
  Real length = j[RIGHT];
    
  output_scheme (scm_list_n (ly_symbol2scm ("start-system"),
			  gh_double2scm (length),
			  gh_double2scm (height),
			  SCM_UNDEFINED));
  
  /* Output elements in three layers, 0, 1, 2.
     The default layer is 1. */
  {
    Molecule *m = this->get_molecule();
    if (m)
      output_molecule (m->get_expr (), Offset(0,0));
  }
  
  for (int i = 0; i < 3; i++)
    for (SCM s = get_grob_property ("all-elements"); gh_pair_p (s);
	 s = ly_cdr (s))
      {
	Grob *sc = unsmob_grob (ly_car (s));
	Molecule *m = sc->get_molecule ();
	if (!m)
	  continue;
	
	SCM s = sc->get_grob_property ("layer");
	int layer = gh_number_p (s) ? gh_scm2int (s) : 1;
	if (layer != i)
	  continue;
	
	Offset o (sc->relative_coordinate (this, X_AXIS),
		  sc->relative_coordinate (this, Y_AXIS));
	
	SCM e = sc->get_grob_property ("extra-offset");
	if (gh_pair_p (e))
	  {
	    o[X_AXIS] += gh_scm2double (ly_car (e));
	    o[Y_AXIS] += gh_scm2double (ly_cdr (e));      
	  }
	
	output_molecule (m->get_expr (), o);
      }

  
  
  if (last_line)
    {
      output_scheme (scm_list_n (ly_symbol2scm ("stop-last-system"), SCM_UNDEFINED));
    }
  else
    {
      output_scheme (scm_list_n (ly_symbol2scm ("stop-system"), SCM_UNDEFINED));
    }
}


Link_array<Item> 
System::broken_col_range (Item const*l, Item const*r) const
{
  Link_array<Item> ret;

  l = l->get_column ();
  r = r->get_column ();
  SCM s = get_grob_property ("columns");

  while (gh_pair_p (s) && ly_car (s) != r->self_scm ())
    s = ly_cdr (s);
    
  if (gh_pair_p (s))
    s = ly_cdr (s);
  
  while (gh_pair_p (s) && ly_car (s) != l->self_scm ())
    {
      Paper_column*c = dynamic_cast<Paper_column*> (unsmob_grob (ly_car (s)));
      if (Item::breakable_b (c) && !c->system_)
	ret.push (c);

      s = ly_cdr (s);
    }

  ret.reverse ();
  return ret;
}

/**
   Return all columns, but filter out any unused columns , since they might
   disrupt the spacing problem.
 */
Link_array<Grob>
System::columns ()const
{
  Link_array<Grob> acs
    = Pointer_group_interface__extract_grobs (this, (Grob*) 0, "columns");
  bool bfound = false;
  for (int i= acs.size (); i -- ;)
    {
      bool brb = Item::breakable_b (acs[i]);
      bfound = bfound || brb;

      /*
	the last column should be breakable. Weed out any columns that
	seem empty. We need to retain breakable columns, in case
	someone forced a breakpoint.
      */
      if (!bfound || !Paper_column::used_b (acs[i]))
	acs.del (i);
    }
  return acs;
}
  



ADD_INTERFACE (System,"system-interface",
  "Super grob, parent of all: "
"\n\n"
"The columns of a score that form one line.  The toplevel grob.  Any "
"grob has a Line_of_score as both X and Y reference point. The "
"Paper_score contains one grob of this type. Control enters the "
"Grob dependency calculation from this single Line_of_score "
"object.",
  "between-system-string all-elements columns");
