/*
  script-engraver.cc -- implement Script_engraver

  (c)  1997--2002 Han-Wen Nienhuys <hanwen@cs.uu.nl>
*/

#include "script.hh"
#include "side-position-interface.hh"
#include "event.hh"
#include "stem.hh"
#include "rhythmic-head.hh"
#include "engraver.hh"
#include "note-column.hh"


struct Script_tuple
{
  Music *event_;
  Grob * script_;
  SCM description_;
  Script_tuple ()
  {
    event_ = 0;
    script_ = 0;
    description_ = SCM_EOL;
  }
};

class Script_engraver : public Engraver
{
  Array<Script_tuple> scripts_;
public:
  TRANSLATOR_DECLARATIONS(Script_engraver);
protected:
  virtual bool try_music (Music*);
  virtual void initialize ();
  virtual void stop_translation_timestep ();
  virtual void start_translation_timestep ();
  virtual void process_music ();
  virtual void acknowledge_grob (Grob_info);
};

void
Script_engraver::initialize ()
{
  scripts_.clear ();
}

bool
Script_engraver::try_music (Music *r)
{
  if (r->is_mus_type ("articulation-event"))
    {
      Script_tuple t;
      t.event_ =r;
      scripts_.push (t);
      return true;
    }
  return false;
}

void
Script_engraver::process_music ()
{
  for (int i=0; i < scripts_.size (); i++)
    {
      Music* l=scripts_[i].event_;

      SCM alist = get_property ("scriptDefinitions");
      SCM art = scm_assoc (l->get_mus_property ("articulation-type"), alist);

      if (art == SCM_BOOL_F)
	{
	  String a = ly_scm2string (l->get_mus_property ("articulation-type"));
	  l->origin ()->warning (_f ("Don't know how to interpret articulation `%s'", a.to_str0 ()));
			
	  continue;
	}
      
      // todo -> use result of articulation-to-scriptdef directly as basic prop list.
      Grob *p =new Item (get_property ("Script"));
      scripts_[i].script_ = p;
      art = ly_cdr (art);

      scripts_[i].description_ = art;
      
      p->set_grob_property ("script-molecule", ly_car (art));

      art = ly_cdr (art);
      art = ly_cdr (art);
      SCM relative_stem_dir = ly_car (art);
      art = ly_cdr (art);

      SCM force_dir = l->get_mus_property ("direction");
      if (ly_dir_p (force_dir) && !to_dir (force_dir))
	force_dir = ly_car (art);
      
      art = ly_cdr (art);
      int priority = gh_scm2int (ly_car (art));

      SCM s = p->get_grob_property ("script-priority");
      if (gh_number_p (s))
	priority = gh_scm2int (s);

      /*
	Make sure they're in order of user input by adding index i.
	Don't use the direction in this priority. Smaller means closer
	to the head.
      */
      priority += i;

      if (ly_dir_p (force_dir) && to_dir (force_dir))
	p->set_grob_property ("direction", force_dir);
      else if (to_dir (relative_stem_dir))
	p->set_grob_property ("side-relative-direction", relative_stem_dir);


      Side_position_interface::set_axis (p, Y_AXIS);
      p->set_grob_property ("script-priority", gh_int2scm (priority));

      
      announce_grob (p, l->self_scm());
    }
}

void
Script_engraver::acknowledge_grob (Grob_info inf)
{
  if (Stem::has_interface (inf.grob_))
    {
      for (int i=0; i < scripts_.size (); i++)
	{
	  Grob*e = scripts_[i].script_;

	  e->set_grob_property ("direction-source", inf.grob_->self_scm ());
	  e->add_dependency (inf.grob_);
	  Side_position_interface::add_support (e, inf.grob_);
	}
    }
  else if (Rhythmic_head::has_interface (inf.grob_))
    {
      for (int i=0; i < scripts_.size (); i++)
	{
	  Grob *e = scripts_[i].script_;
	  
	  if (Side_position_interface::get_axis (e) == X_AXIS
	      && !e->get_parent (Y_AXIS))
	    {
	      e->set_parent (inf.grob_, Y_AXIS);
	      e->add_dependency (inf.grob_); // ??
	    }
	  Side_position_interface::add_support (e,inf.grob_);
	}
    }
  else if (Note_column::has_interface (inf.grob_))
    {

      /*
	We make note column the parent of the script. That's not
	correct, but due to seconds in a chord, noteheads may be
	swapped around horizontally. We don't know which note head to
	put it on, so we postpone this decision to
	Script_interface::before_line_breaking ().
 
       */
      for (int i=0; i < scripts_.size (); i++)
	{
	  Grob *e = scripts_[i].script_;
	  
	  if (!e->get_parent (X_AXIS) &&
	      Side_position_interface::get_axis (e) == Y_AXIS)
	    {
	      e->set_parent (inf.grob_, X_AXIS);
	    }
	}
    }
}

void
Script_engraver::stop_translation_timestep ()
{
  for (int i=0; i < scripts_.size (); i++) 
    {
      if (!scripts_[i].script_)
	continue;
      
      Grob * sc = scripts_[i].script_;
      if (to_boolean (gh_cadr (scripts_[i].description_)))
	sc->add_offset_callback (Side_position_interface::quantised_position_proc, Y_AXIS);
      else
	Side_position_interface::add_staff_support (sc);
      typeset_grob (sc);
    }
  scripts_.clear ();
}

void
Script_engraver::start_translation_timestep ()
{
  scripts_.clear ();
}



Script_engraver::Script_engraver(){}

ENTER_DESCRIPTION(Script_engraver,
/* descr */       "Handles note scripted articulations.",
/* creats*/       "Script",
/* accepts */     "script-event articulation-event",
/* acks  */      "stem-interface rhythmic-head-interface note-column-interface",
/* reads */       "scriptDefinitions",
/* write */       "");
