/*
  This file is part of Guile glib spawn
  
  Copyright (C) 2010--2011  Jan Nieuwenhuizen <janneke@gnu.org>
  
  Guile-glib-spawn is free software: you can redistribute it and/or modify
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, either version 3 of the
  License, or (at your option) any later version.
  
  Guile-glib-spawn is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Guile-glib-spawn.  If not, see <http://www.gnu.org/licenses/>.

*/


#include <glib.h>
#include <libguile.h>

#if SCM_MAJOR_VERSION == 1 && SCM_MINOR_VERSION > 6 && SCM_MINOR_VERSION < 9
/*
  GUILE V1.7.0 - V1.8.n
*/
#define SCM_VALIDATE_POINTER(x, y)
#define SCM_POINTER_VALUE(x) ((void*)scm_to_ulong (SCM_CAR (x)))
#define scm_from_pointer(pointer, finalizer) scm_cons (scm_from_ulong ((ulong)pointer), scm_from_ulong ((ulong)finalizer))
#endif /* Guile 1.8 */


SCM
close_pid (SCM pid)
#define FUNC_NAME __FUNCTION__
{

  SCM_VALIDATE_POINTER (1, pid);

  g_spawn_close_pid ((GPid)SCM_POINTER_VALUE (pid));
  return SCM_BOOL_T;
}
#undef FUNC_NAME

void
init_close_pid (void *unused)
{

  (void) unused;
  scm_c_define_gsubr ("me:close-pid", 1, 0, 0, close_pid);
  scm_c_export ("me:close-pid", NULL);
}

static int
run_command_async (char *argv[], GPid *child_pid)
{

  GError *error = 0;
  int exit_status = 0;
  int flags = G_SPAWN_SEARCH_PATH;
  if (!g_spawn_async (0, argv, 0, (GSpawnFlags) flags,
		      0, 0,
		      child_pid,
		      &error))
    {
      fprintf (stderr, "failed: %s: %s\n", argv[0], error->message);
      g_error_free (error);
      exit_status = -1;
    }

  return exit_status;
}

static char *
scm2str0 (SCM str)
{

  return scm_to_locale_string (str);
}

SCM
spawn_async (SCM command, SCM args)
{

  int argc = scm_is_pair (args) ? scm_ilength (args) : 0;
  char **argv = malloc (sizeof (char *) * (argc + 2));

  int n = 0;
  argv[n++] = scm2str0 (command);
  for (SCM s = args; scm_is_pair (s); s = scm_cdr (s))
    argv[n++] = scm2str0 (scm_car (s));
  argv[n] = 0;

  GPid child_pid;
  int exit_status = run_command_async (argv, &child_pid);
  
  for (int i = 0; i < n; i++)
    free (argv[i]);

  return scm_from_pointer (child_pid, g_spawn_close_pid);
}

void
init_spawn_async (void *unused)
{

  (void) unused;
  scm_c_define_gsubr ("me:spawn-async", 1, 0, 1, spawn_async);
  scm_c_export ("me:spawn-async", NULL);
}

void
scm_init_glib_spawn_module ()
{

  scm_c_define_module ("glib spawn", init_close_pid, NULL);
  scm_c_define_module ("glib spawn", init_spawn_async, NULL);
}
