#include <stdio.h>
#include <stdlib.h>
#include <basics.h>
#include <ustring.h>
#include <utype.h>
#include "splinefont.h"
#include <intl.h>

#define ISOL	0
#define INIT	1
#define MEDI	2
#define FINA	3

struct letter_frequencies {
    char *utf8_letter;
    float frequency[4];
    float *afters;
};

/* This is over many languages, used when we have no specific info */
static float word_lengths[] = {
    0.000000, 0.064717, 0.202506, 0.187441, 0.133263, 
    0.111502, 0.094215, 0.071569, 0.051841, 0.035586, 
    0.021651, 0.012286, 0.006966, 0.003683, 0.001541, 
    0.000605, 0.000421, 0.000116, 0.000051, 0.000017, 
    0.000007, 0.000002, 0.000007, 0.000002, 0.000001, 
    0.000001, 0.000000, 0.000000, 0.000000, 0.000000
};

/* These data are bigram frequences for various scripts and languages. */
/* Obviously there is no "perfect" bigram table because every text and writer */
/*  will be slightly different. These should be representative, however. */
/* They were derived by analyzing texts downloaded from Project Gutenberg */
/*  In many cases I have no idea what these texts are... */

/* Welsh -- Gwaith Mynyddog.  Cyfrol II  by  Mynyddog */
/* Swedish -- Ada  by  Hasse Zetterström */
/* Spanish -- La Niña de Luzmela  by  Concha Espina */
/* Russian -- Duhovnye ody (Spiritual Odes)  by  Gavrila Romanovich Derzhavin */
/* Polish -- Ironia Pozorow  by  Maciej hr. Lubienski */
/* Italian -- Le rive della Bormida nel 1794  by  Giuseppe Cesare Abba */
/* Hungarian -- Az arany szalamandra  by  Ferenc Donászy */
/* Hebrew -- Hunger Book One  by  Knut Hamsun */
/* Greek -- Autumn  by  Konstantinos Chatzopoulos */
/* German -- Faust  Eine Tragödie  by  Johann Wolfgang von Goethe */
/* French -- 20000 Lieues sous les mers  by  Jules Verne */
/* English -- Life And Adventures Of Martin Chuzzlewit  by  Charles Dickens */
/* Dutch -- Columbus De ontdekker van Amerika by J.S.C. Abbott */
/* Czech -- R.U.R. by Karel Čapek */

/* The text sample for hiragana comes from: */
/*   http://www-rohan.sdsu.edu/~gawron/crypto/lectures/hiragana.html */
/* It represents many news articles transcribed so the Kanji are in hiragana */

/* The Hindi text sample was composed of ~10 articles from http://www.bhaskar.com/ */
/*  (I did the composition with no real idea what I was doing). This is a */
/*  small sample, too small to get bigram data */
/* (Other indic language newspapers may be found at http://www.indiagrid.com/cgi-bin/news.cgi?channel=malayalam&mchannel=lang ) */

/* The declaration of human rights (from the UN) is at: http://unicode.org/udhr/index_by_name.html */
/* a short text in many languages. I used for Lithuanian, Turkish, Hindi, Sanskrit */

/* Sometimes I consider the text too short for valid bigram data */

/******************************************************************************/
static float CSY_afters_0[] = { 0.000747, 0.056510, 0.019417, 0.063978, 0.000498, 0.000996, 0.000996, 0.022405, 0, 0.030122, 0.119492, 0.218073, 0.048793, 0.104556, 0.000996, 0.019417, 0.047050, 0.040329, 0.064725, 0.007468, 0.049042, 0, 0, 0, 0.012447, 0.000498, 0, 0, 0, 0, 0, 0.012945, 0.001743, 0, 0.004481, 0.003236, 0.015932, 0.003236, 0, 0.019666, 0.010207, 0};
static float CSY_afters_1[] = { 0.032907, 0, 0, 0.000609, 0.054845, 0, 0, 0, 0.029250, 0.005484, 0.003047, 0.023157, 0, 0.009141, 0.260817, 0, 0.124924, 0.000609, 0.002438, 0.146862, 0, 0, 0, 0.209628, 0.000609, 0.006703, 0.000609, 0.016453, 0, 0, 0.018891, 0, 0, 0.031079, 0, 0.009750, 0.000609, 0, 0.010360, 0, 0.001219, 0};
static float CSY_afters_2[] = { 0.011455, 0, 0, 0, 0.328751, 0, 0, 0, 0.152348, 0, 0.049255, 0.002291, 0, 0.003436, 0.342497, 0.002291, 0, 0, 0.027491, 0.008018, 0.004582, 0, 0, 0.003436, 0, 0, 0, 0.063001, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001145, 0, 0, 0};
static float CSY_afters_3[] = { 0.053445, 0.002254, 0.005473, 0.000644, 0.103026, 0, 0, 0.001932, 0.060528, 0.001288, 0.003220, 0.021571, 0.004507, 0.071797, 0.250805, 0.005473, 0.065035, 0.016420, 0.002576, 0.032196, 0.028332, 0, 0, 0.070509, 0, 0.037669, 0.014488, 0.043786, 0, 0.000322, 0.008049, 0.000644, 0, 0.084675, 0, 0.002576, 0.000966, 0, 0.003220, 0.000966, 0.001610, 0};
static float CSY_afters_4[] = { 0.000931, 0.027178, 0.010797, 0.063291, 0.000186, 0.002234, 0.001303, 0.009494, 0.021035, 0.048771, 0.025503, 0.152829, 0.111876, 0.210536, 0.002234, 0.014706, 0.045979, 0.040395, 0.042628, 0.002234, 0.031087, 0, 0.001117, 0, 0.020663, 0.000186, 0, 0, 0, 0, 0, 0.026806, 0.008191, 0, 0.001303, 0.009121, 0.021966, 0.000186, 0, 0.014892, 0.030343, 0};
static float CSY_afters_5[] = { 0.786585, 0, 0, 0, 0, 0, 0, 0, 0.030488, 0, 0, 0.006098, 0, 0, 0.054878, 0, 0.024390, 0, 0.012195, 0.030488, 0, 0, 0, 0.042683, 0, 0.012195, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float CSY_afters_6[] = { 0.645695, 0, 0, 0, 0.043046, 0, 0, 0, 0.033113, 0, 0, 0.225166, 0, 0, 0.003311, 0, 0.019868, 0, 0, 0.016556, 0, 0, 0, 0.003311, 0, 0.003311, 0.003311, 0, 0.003311, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float CSY_afters_7[] = { 0.160253, 0, 0, 0.002298, 0.354968, 0, 0, 0, 0.002298, 0, 0.001723, 0.112005, 0.011488, 0.040781, 0.186674, 0, 0.048248, 0, 0.000574, 0.023550, 0.004595, 0, 0, 0.018380, 0, 0.013785, 0.005169, 0, 0, 0, 0.006893, 0, 0, 0, 0, 0.005169, 0, 0, 0.001149, 0, 0, 0};
static float CSY_afters_8[] = { 0.005765, 0.003843, 0.045734, 0.088778, 0.055342, 0, 0.004612, 0.009224, 0.004612, 0.014988, 0.047656, 0.077248, 0.052652, 0.214450, 0.005380, 0.003843, 0.002690, 0.118755, 0.076095, 0.020369, 0.047271, 0, 0, 0, 0.008455, 0.002690, 0, 0, 0.003075, 0, 0, 0.024596, 0.001922, 0, 0.001922, 0, 0.015373, 0.000769, 0, 0.004996, 0.036895, 0};
static float CSY_afters_9[] = { 0.111740, 0, 0.000471, 0.062235, 0.351249, 0, 0, 0.000471, 0.080622, 0.000471, 0.001886, 0.004715, 0.014616, 0.010372, 0.001414, 0, 0.000471, 0.173032, 0.019802, 0.008015, 0.001886, 0, 0, 0, 0.000943, 0.059406, 0, 0.067421, 0, 0, 0, 0.000471, 0.008958, 0, 0, 0, 0.016502, 0.000471, 0.001414, 0.000471, 0.000471, 0};
static float CSY_afters_10[] = { 0.122960, 0.000544, 0.001632, 0.085419, 0.028836, 0, 0, 0.002720, 0.000544, 0, 0, 0.060392, 0.000544, 0.063656, 0.208923, 0.001088, 0.063112, 0.003264, 0.042437, 0.091948, 0.015778, 0, 0, 0.060392, 0, 0.052775, 0.040805, 0, 0, 0, 0.029380, 0, 0, 0, 0, 0.013602, 0, 0, 0.004353, 0.004897, 0, 0};
static float CSY_afters_11[] = { 0.123012, 0.000746, 0.000497, 0, 0.336978, 0, 0, 0.000497, 0.129473, 0, 0.005716, 0.083748, 0.001740, 0.015159, 0.100646, 0.000497, 0, 0.001243, 0.006710, 0.034046, 0.001988, 0, 0, 0.015656, 0.000746, 0.037525, 0.012674, 0.024354, 0.002734, 0, 0.007455, 0.002734, 0, 0, 0.000497, 0, 0.001243, 0, 0.000994, 0.001243, 0, 0.049453};
static float CSY_afters_12[] = { 0.106618, 0.002859, 0.000408, 0.000408, 0.170752, 0.000408, 0, 0.001634, 0.237745, 0, 0.000408, 0.031454, 0, 0.051879, 0.075980, 0.002859, 0.011029, 0.006127, 0.000408, 0.084150, 0, 0, 0, 0.053922, 0.000408, 0.047386, 0.011438, 0.037582, 0.000408, 0, 0.003268, 0, 0, 0.029412, 0, 0.005310, 0, 0, 0.022467, 0.003268, 0, 0};
static float CSY_afters_13[] = { 0.223088, 0.000200, 0.006191, 0.003595, 0.233074, 0, 0.000799, 0.000599, 0.077292, 0, 0.005392, 0, 0, 0.001997, 0.119633, 0, 0, 0.002397, 0.006591, 0.024366, 0, 0, 0, 0.021969, 0.000599, 0.068105, 0.028959, 0.080687, 0, 0, 0.015378, 0.000599, 0, 0.072499, 0, 0, 0.000799, 0, 0.003795, 0.001398, 0, 0};
static float CSY_afters_14[] = { 0.000418, 0.094174, 0.009605, 0.068281, 0, 0.000209, 0.002088, 0.056588, 0.000418, 0.022343, 0.038004, 0.032157, 0.113802, 0.036333, 0.000209, 0.022343, 0.032783, 0.054709, 0.096053, 0.091460, 0.097098, 0, 0, 0, 0.031739, 0, 0, 0, 0, 0, 0, 0.031739, 0.004385, 0, 0.003341, 0.010023, 0.011276, 0.000418, 0, 0.031531, 0.006473, 0};
static float CSY_afters_15[] = { 0.133767, 0, 0.001393, 0, 0.021830, 0, 0, 0, 0.029726, 0.000464, 0, 0.031584, 0, 0.004180, 0.232234, 0, 0.275430, 0.013470, 0.009289, 0.023688, 0, 0, 0, 0.006038, 0, 0.038086, 0.001393, 0.025081, 0, 0, 0.000929, 0, 0, 0.013934, 0, 0.116117, 0.007431, 0, 0.013934, 0, 0, 0};
static float CSY_afters_16[] = { 0.121833, 0.004098, 0.005216, 0.012668, 0.026826, 0, 0.002981, 0.004471, 0.049180, 0.000373, 0.002981, 0.001490, 0.005589, 0.032787, 0.373323, 0.000745, 0.027943, 0.005216, 0.016021, 0.042846, 0.009687, 0, 0, 0.110283, 0.005961, 0.085693, 0.017139, 0, 0, 0, 0.016021, 0.001118, 0, 0, 0, 0, 0.002235, 0, 0.006334, 0.006706, 0.002235, 0};
static float CSY_afters_17[] = { 0.017673, 0.001767, 0, 0, 0.192931, 0, 0, 0.000884, 0.047423, 0, 0.036819, 0.080707, 0.068925, 0.030633, 0.034462, 0.022680, 0.002651, 0.012960, 0.314875, 0.031811, 0.036230, 0.000295, 0, 0.009426, 0, 0.008837, 0, 0.037408, 0, 0, 0, 0.000589, 0, 0, 0.000295, 0, 0, 0.003535, 0.003240, 0, 0.002946, 0};
static float CSY_afters_18[] = { 0.142492, 0.000766, 0.000255, 0.000511, 0.225485, 0, 0, 0, 0.084780, 0, 0.012257, 0.006639, 0.000511, 0.019918, 0.207099, 0, 0.037794, 0.001277, 0.000255, 0.027324, 0.027835, 0, 0, 0.039326, 0, 0.021195, 0.006895, 0.032176, 0.000255, 0, 0.005107, 0, 0, 0.058223, 0, 0.021961, 0.002298, 0, 0.017365, 0, 0, 0};
static float CSY_afters_19[] = { 0.000644, 0.013531, 0.021907, 0.115335, 0, 0.005155, 0, 0.021907, 0, 0.068299, 0.063789, 0.046392, 0.081186, 0.011598, 0, 0.033505, 0.012242, 0.233247, 0.047036, 0, 0.033505, 0, 0, 0, 0.022552, 0, 0, 0, 0, 0, 0, 0.021263, 0.005799, 0, 0, 0.005155, 0.034794, 0, 0, 0.074098, 0.027062, 0};
static float CSY_afters_20[] = { 0.073821, 0.001080, 0.000720, 0.004321, 0.093266, 0, 0, 0, 0.066979, 0.000360, 0.012243, 0.024487, 0, 0.030248, 0.091106, 0.007202, 0.022326, 0.018725, 0.002161, 0.007922, 0, 0, 0, 0.082823, 0.014044, 0.119914, 0.028448, 0.070220, 0, 0, 0.043212, 0.005762, 0, 0.095067, 0.000360, 0.011163, 0.053295, 0, 0.007922, 0.009723, 0.001080, 0};
static float CSY_afters_21[] = { 0, 0, 0, 0, 0.500000, 0, 0, 0.500000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float CSY_afters_22[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0.333333, 0, 0, 0, 0, 0, 0, 0.333333, 0, 0, 0.166667, 0, 0, 0, 0, 0.166667, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float CSY_afters_23[] = { 0, 0.045576, 0.012064, 0.010724, 0, 0, 0, 0.034853, 0, 0.014745, 0.026810, 0.152815, 0.009383, 0.061662, 0.083110, 0.016086, 0.028150, 0.168901, 0.044236, 0, 0.006702, 0, 0, 0, 0.013405, 0, 0, 0, 0, 0, 0, 0.025469, 0.001340, 0, 0.002681, 0.010724, 0.041555, 0.029491, 0, 0.045576, 0.113941, 0};
static float CSY_afters_24[] = { 0.301105, 0.032228, 0.002762, 0.045120, 0.086556, 0, 0, 0.007366, 0.030387, 0.002762, 0.044199, 0.032228, 0.033149, 0.080110, 0.023020, 0.028545, 0.023941, 0.005525, 0.018416, 0.065378, 0.026703, 0, 0, 0.011971, 0, 0.044199, 0, 0.035912, 0, 0, 0, 0.000921, 0, 0, 0, 0.005525, 0.000921, 0, 0.010129, 0.000921, 0, 0};
static float CSY_afters_25[] = { 0, 0.014545, 0.037818, 0.080000, 0, 0, 0, 0.020364, 0, 0.010182, 0.010182, 0.066909, 0.126545, 0.142545, 0, 0.005818, 0.050909, 0.117818, 0.109091, 0, 0.069091, 0, 0, 0, 0.044364, 0.002909, 0, 0, 0, 0, 0, 0.010909, 0, 0, 0, 0.023273, 0.030545, 0, 0, 0.013091, 0.013091, 0};
static float CSY_afters_26[] = { 0, 0.012658, 0, 0.006329, 0, 0.018987, 0, 0.455696, 0, 0, 0.018987, 0.075949, 0.183544, 0.069620, 0, 0.037975, 0.006329, 0.025316, 0.050633, 0, 0.012658, 0, 0, 0, 0.025316, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float CSY_afters_27[] = { 0, 0.005923, 0.076999, 0.016782, 0, 0, 0, 0.021718, 0, 0.005923, 0.063179, 0.030602, 0.260612, 0.019743, 0, 0.004936, 0.038500, 0.034551, 0.202369, 0, 0.044423, 0, 0, 0, 0.018756, 0, 0, 0, 0, 0, 0, 0.007897, 0.000987, 0, 0.002962, 0.006910, 0.080948, 0, 0, 0.030602, 0.024679, 0};
static float CSY_afters_28[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.882353, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.117647, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float CSY_afters_29[] = { 0, 0, 0.022222, 0.022222, 0, 0, 0, 0, 0, 0.022222, 0.044444, 0.022222, 0.022222, 0.044444, 0, 0.022222, 0.022222, 0.244444, 0.044444, 0, 0, 0, 0, 0, 0.177778, 0, 0, 0, 0, 0, 0, 0.066667, 0, 0, 0, 0.111111, 0, 0, 0, 0.111111, 0, 0};
static float CSY_afters_30[] = { 0, 0.056604, 0, 0.023585, 0, 0, 0, 0.023585, 0, 0.004717, 0.009434, 0.009434, 0.165094, 0.004717, 0, 0.004717, 0.202830, 0.009434, 0.141509, 0, 0.066038, 0, 0, 0, 0.023585, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.009434, 0.009434, 0, 0, 0.004717, 0.231132, 0};
static float CSY_afters_31[] = { 0.026490, 0, 0, 0, 0.246689, 0, 0, 0, 0.086093, 0, 0.105960, 0.091060, 0.001656, 0.225166, 0, 0.004967, 0, 0, 0.076159, 0.001656, 0, 0, 0, 0, 0, 0.008278, 0, 0.124172, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001656, 0, 0, 0};
static float CSY_afters_32[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.066667, 0, 0, 0, 0, 0, 0.866667, 0, 0, 0, 0, 0, 0, 0.066667, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float CSY_afters_33[] = { 0, 0, 0.071661, 0.031488, 0, 0, 0, 0.010858, 0, 0.094463, 0.103149, 0.274701, 0.019544, 0.034745, 0, 0, 0.003257, 0.016287, 0.222584, 0, 0.021716, 0, 0, 0, 0.011944, 0, 0, 0, 0, 0, 0, 0.014115, 0, 0, 0.001086, 0.016287, 0.006515, 0.001086, 0, 0.029316, 0.015201, 0};
static float CSY_afters_34[] = { 0.173913, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.043478, 0, 0, 0, 0.043478, 0, 0, 0.043478, 0.565217, 0.086957, 0, 0, 0, 0, 0, 0.043478, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float CSY_afters_35[] = { 0.015172, 0.001379, 0.001379, 0, 0.468966, 0, 0, 0, 0.208276, 0, 0.012414, 0, 0.009655, 0.001379, 0.002759, 0, 0, 0.006897, 0.005517, 0.001379, 0.001379, 0, 0, 0, 0, 0.038621, 0, 0.223448, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001379, 0, 0, 0};
static float CSY_afters_36[] = { 0.017945, 0, 0.001631, 0, 0.324633, 0, 0, 0, 0.143556, 0, 0.050571, 0.058728, 0.001631, 0.029364, 0.001631, 0.021207, 0, 0.003263, 0.132137, 0.006525, 0.001631, 0, 0, 0, 0, 0.003263, 0.004894, 0.171289, 0, 0, 0, 0, 0, 0, 0, 0, 0.006525, 0.019576, 0, 0, 0, 0};
static float CSY_afters_37[] = { 0.500000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.388889, 0.111111, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float CSY_afters_38[] = { 0, 0.047368, 0, 0.021053, 0, 0, 0, 0.078947, 0, 0.163158, 0.010526, 0.121053, 0.078947, 0, 0, 0, 0, 0.100000, 0, 0, 0.115789, 0, 0, 0, 0.042105, 0, 0, 0, 0, 0, 0, 0.021053, 0, 0, 0.005263, 0.010526, 0, 0, 0, 0.184211, 0, 0};
static float CSY_afters_39[] = { 0.014045, 0.002809, 0.001404, 0.077247, 0.485955, 0, 0, 0, 0.148876, 0, 0.014045, 0.009831, 0, 0.028090, 0.005618, 0, 0.001404, 0.002809, 0.022472, 0.009831, 0.001404, 0, 0, 0, 0, 0.050562, 0, 0.120787, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.002809, 0, 0, 0};
static float CSY_afters_40[] = { 0.046000, 0, 0.160000, 0, 0.014000, 0, 0, 0.002000, 0.002000, 0, 0, 0.090000, 0.004000, 0.216000, 0.126000, 0, 0.032000, 0.002000, 0.106000, 0.054000, 0.018000, 0, 0, 0.030000, 0, 0.072000, 0.010000, 0, 0, 0, 0.012000, 0, 0, 0, 0, 0, 0, 0, 0.004000, 0, 0, 0};
static float CSY_afters_41[] = { 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static struct letter_frequencies CSY_counts[] = {
    { "a", 0.432464, 0.031587, 0.071800, 0.090461, CSY_afters_0 },
    { "b", 0.000000, 0.036345, 0.020861, 0.000631, CSY_afters_1 },
    { "c", 0.000000, 0.018688, 0.011331, 0.008886, CSY_afters_2 },
    { "d", 0.001185, 0.074753, 0.037329, 0.009975, CSY_afters_3 },
    { "e", 0.002370, 0.001720, 0.110662, 0.167622, CSY_afters_4 },
    { "f", 0.000000, 0.008140, 0.000456, 0.000401, CSY_afters_5 },
    { "g", 0.000000, 0.014733, 0.000932, 0.000000, CSY_afters_6 },
    { "h", 0.000000, 0.064205, 0.012864, 0.004185, CSY_afters_7 },
    { "i", 0.029621, 0.001605, 0.053322, 0.063632, CSY_afters_8 },
    { "j", 0.003555, 0.081805, 0.014377, 0.006421, CSY_afters_9 },
    { "k", 0.088863, 0.029236, 0.027510, 0.022357, CSY_afters_10 },
    { "l", 0.000000, 0.020179, 0.076067, 0.028892, CSY_afters_11 },
    { "m", 0.000000, 0.043453, 0.035009, 0.047638, CSY_afters_12 },
    { "n", 0.002370, 0.097455, 0.068506, 0.040071, CSY_afters_13 },
    { "o", 0.046209, 0.024765, 0.090257, 0.110812, CSY_afters_14 },
    { "p", 0.000000, 0.092983, 0.011000, 0.000344, CSY_afters_15 },
    { "r", 0.005924, 0.034912, 0.042985, 0.017026, CSY_afters_16 },
    { "s", 0.100711, 0.081805, 0.040768, 0.019262, CSY_afters_17 },
    { "t", 0.000000, 0.076588, 0.053446, 0.051536, CSY_afters_18 },
    { "u", 0.054502, 0.018287, 0.025542, 0.049759, CSY_afters_19 },
    { "v", 0.113744, 0.067186, 0.033248, 0.002408, CSY_afters_20 },
    { "w", 0.000000, 0.000057, 0.000021, 0.000000, CSY_afters_21 },
    { "x", 0.000000, 0.000000, 0.000124, 0.000000, CSY_afters_22 },
    { "y", 0.000000, 0.000000, 0.015454, 0.059046, CSY_afters_23 },
    { "z", 0.088863, 0.032045, 0.010917, 0.001949, CSY_afters_24 },
    { "á", 0.000000, 0.000000, 0.028484, 0.031014, CSY_afters_25 },
    { "é", 0.000000, 0.000000, 0.003273, 0.020637, CSY_afters_26 },
    { "í", 0.000000, 0.000000, 0.020985, 0.052224, CSY_afters_27 },
    { "ó", 0.028436, 0.000057, 0.000331, 0.000459, CSY_afters_28 },
    { "ú", 0.001185, 0.002522, 0.000021, 0.000000, CSY_afters_29 },
    { "ý", 0.000000, 0.000000, 0.004392, 0.012382, CSY_afters_30 },
    { "č", 0.000000, 0.009516, 0.009073, 0.006191, CSY_afters_31 },
    { "ď", 0.000000, 0.000115, 0.000580, 0.004529, CSY_afters_32 },
    { "ě", 0.000000, 0.000000, 0.019079, 0.020064, CSY_afters_33 },
    { "ň", 0.000000, 0.000000, 0.000476, 0.001949, CSY_afters_34 },
    { "ř", 0.000000, 0.006191, 0.012781, 0.000917, CSY_afters_35 },
    { "š", 0.000000, 0.003268, 0.011518, 0.009688, CSY_afters_36 },
    { "ť", 0.000000, 0.000115, 0.000331, 0.002866, CSY_afters_37 },
    { "ů", 0.000000, 0.000000, 0.003936, 0.005389, CSY_afters_38 },
    { "ž", 0.000000, 0.016969, 0.008618, 0.014446, CSY_afters_39 },
    { "ch", 0.000000, 0.008714, 0.007209, 0.013930, CSY_afters_40 },
    { "qu", 0.000000, 0.000000, 0.004122, 0.000000, CSY_afters_41 },
    NULL
};	/* CSY */
static float CSY_word_lens[] = {
    0.000000, 0.046150, 0.176728, 0.113462, 0.149934, 0.169237, 0.174869, 0.078412, 0.047572, 0.020615, 0.015037, 0.004812, 0.002187, 0.000273, 0.000492, 0.000164, 0.000055, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float CSY_all_consonants[] = {
    0.000000, 0.018427, 0.000711, 0.000219, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 
};

static float CSY_consonant_run[] = {
    0.000000, 0.796697, 0.187858, 0.013973, 0.001472, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float CSY_vowel_run[] = {
    0.000000, 0.906900, 0.074713, 0.010665, 0.007401, 0.000321, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float NLD_afters_0[] = { 0.192205, 0.006329, 0.002389, 0.047234, 0.000084, 0.017058, 0.029380, 0.000126, 0.002640, 0.001215, 0.021961, 0.074769, 0.028583, 0.279925, 0.001551, 0.011819, 0.124057, 0.046773, 0.070830, 0.002221, 0.008969, 0, 0.000042, 0.000461, 0.002808, 0, 0, 0, 0, 0.002179, 0, 0, 0, 0.000042, 0.024350, 0};
static float NLD_afters_1[] = { 0.074178, 0.023055, 0, 0.000200, 0.402366, 0, 0, 0, 0.029471, 0, 0, 0.058941, 0, 0, 0.122895, 0, 0.076183, 0.000200, 0.002606, 0.137330, 0, 0, 0, 0, 0.000200, 0, 0, 0, 0, 0, 0, 0, 0.072374, 0, 0, 0};
static float NLD_afters_2[] = { 0.179439, 0, 0.001869, 0, 0.055140, 0, 0, 0, 0.055140, 0, 0.000935, 0, 0, 0, 0.652336, 0, 0.003738, 0, 0.011215, 0.039252, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.000935, 0, 0, 0};
static float NLD_afters_3[] = { 0.105211, 0.001681, 0, 0.023593, 0.616312, 0, 0.001275, 0.004985, 0.121964, 0.000638, 0.000812, 0.000406, 0.006840, 0.000058, 0.051359, 0.000464, 0.024578, 0.016289, 0.003246, 0.011188, 0.001043, 0.004116, 0, 0, 0.002782, 0, 0, 0, 0, 0, 0, 0, 0.001159, 0, 0, 0};
static float NLD_afters_4[] = { 0.000858, 0.007466, 0.000359, 0.029104, 0.099347, 0.005609, 0.019622, 0.006048, 0.033675, 0.000240, 0.019662, 0.074016, 0.020121, 0.370741, 0.000220, 0.009003, 0.151546, 0.024373, 0.070783, 0.008204, 0.019542, 0.009561, 0.000299, 0.000040, 0.014871, 0, 0.000080, 0.000040, 0.000619, 0, 0, 0, 0, 0.000020, 0.003932, 0};
static float NLD_afters_5[] = { 0.014790, 0.005752, 0, 0.260477, 0.159408, 0.066557, 0.060805, 0.011504, 0.020542, 0.000822, 0.012325, 0.023007, 0.006574, 0.001643, 0.043550, 0.002465, 0.042728, 0.110929, 0.123254, 0, 0.004108, 0.018077, 0, 0, 0.004108, 0, 0.001643, 0, 0, 0, 0, 0, 0.004930, 0, 0, 0};
static float NLD_afters_6[] = { 0.051614, 0.001680, 0, 0.040211, 0.655624, 0, 0.009963, 0.012003, 0.036610, 0.000120, 0.003721, 0.005882, 0.002521, 0.000240, 0.049214, 0.000600, 0.066619, 0.030128, 0.009723, 0.014284, 0.001801, 0.000720, 0, 0, 0.003961, 0, 0, 0, 0, 0, 0, 0, 0.002761, 0, 0, 0};
static float NLD_afters_7[] = { 0.160385, 0, 0, 0, 0.477472, 0, 0, 0, 0.037404, 0, 0, 0.000142, 0, 0.000142, 0.108813, 0, 0, 0, 0, 0.065174, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.150468, 0, 0, 0};
static float NLD_afters_8[] = { 0.013756, 0.000941, 0.002824, 0.061613, 0.231972, 0.003041, 0.104402, 0, 0.000217, 0, 0.026064, 0.054083, 0.006588, 0.274327, 0.005720, 0.009485, 0.009702, 0.064292, 0.068781, 0.000362, 0.005502, 0, 0.000507, 0, 0.006661, 0, 0, 0, 0.004561, 0, 0, 0, 0, 0, 0.043151, 0.001448};
static float NLD_afters_9[] = { 0.359098, 0, 0, 0, 0.452496, 0, 0, 0, 0.008052, 0, 0, 0, 0, 0, 0.086957, 0, 0.001610, 0, 0, 0.091787, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float NLD_afters_10[] = { 0.075148, 0.007438, 0, 0.003815, 0.324814, 0, 0.001335, 0.013542, 0.032424, 0.002670, 0.059126, 0.051497, 0.000191, 0.005722, 0.150486, 0.001717, 0.062178, 0.023651, 0.100896, 0.027274, 0.000763, 0.050734, 0, 0, 0.003433, 0, 0, 0, 0, 0, 0, 0, 0.001144, 0, 0, 0};
static float NLD_afters_11[] = { 0.164783, 0.000847, 0, 0.092279, 0.192561, 0.015254, 0.016102, 0.003861, 0.085782, 0, 0.036158, 0.079473, 0.004520, 0.000471, 0.069397, 0.004143, 0.001224, 0.044727, 0.009134, 0.080885, 0.007062, 0.001224, 0, 0, 0.000753, 0, 0, 0, 0, 0, 0, 0, 0.088983, 0.000188, 0.000094, 0.000094};
static float NLD_afters_12[] = { 0.195680, 0.115135, 0, 0.041572, 0.394284, 0.000650, 0.003085, 0.005196, 0.072426, 0, 0.000650, 0.001137, 0.014453, 0, 0.084443, 0.007470, 0.001949, 0.020624, 0.003735, 0.005196, 0.000487, 0.000162, 0, 0.000162, 0.001137, 0, 0, 0, 0, 0, 0, 0, 0.030367, 0, 0, 0};
static float NLD_afters_13[] = { 0.081066, 0.018566, 0.002409, 0.264598, 0.091908, 0, 0.158305, 0.008716, 0.086947, 0.016298, 0.024589, 0.011480, 0.004748, 0.028203, 0.046273, 0.000354, 0.002126, 0.063492, 0.052863, 0.010558, 0.008007, 0.005385, 0, 0, 0.011834, 0, 0.000071, 0, 0, 0, 0, 0, 0.001063, 0, 0.000142, 0};
static float NLD_afters_14[] = { 0.001926, 0.002231, 0.001622, 0.015513, 0.102712, 0.029354, 0.025551, 0.000101, 0.004259, 0.001977, 0.014246, 0.063169, 0.055665, 0.145196, 0.203701, 0.065653, 0.123549, 0.014753, 0.042028, 0.038530, 0.024842, 0.001166, 0, 0.000456, 0.004005, 0, 0, 0, 0, 0, 0, 0, 0, 0.000811, 0.016984, 0};
static float NLD_afters_15[] = { 0.195382, 0.007815, 0.002131, 0.006394, 0.295204, 0, 0.033037, 0, 0.039076, 0.002842, 0.002487, 0.086679, 0.005684, 0.001776, 0.071048, 0.067140, 0.084192, 0.028774, 0.025577, 0.018828, 0.002131, 0.003197, 0, 0.000355, 0.004973, 0, 0, 0, 0, 0, 0, 0, 0.015275, 0, 0, 0};
static float NLD_afters_16[] = { 0.074457, 0.015290, 0.000591, 0.145664, 0.192495, 0.003693, 0.026518, 0.021864, 0.077338, 0.000222, 0.024228, 0.031762, 0.020461, 0.017728, 0.103043, 0.008716, 0.008051, 0.065593, 0.048826, 0.036490, 0.017654, 0.015216, 0, 0.000074, 0.014847, 0, 0, 0, 0, 0, 0, 0, 0.028143, 0, 0.001034, 0};
static float NLD_afters_17[] = { 0.033227, 0.004190, 0.003034, 0.009679, 0.031927, 0, 0.004190, 0.004190, 0.008090, 0.003034, 0.002889, 0.042473, 0.010835, 0.007079, 0.017480, 0.088269, 0.001011, 0.026437, 0.418665, 0.000867, 0.006212, 0.002311, 0, 0, 0.002167, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.271742, 0};
static float NLD_afters_18[] = { 0.053227, 0.007477, 0, 0.010724, 0.484750, 0, 0.011806, 0.007871, 0.055982, 0.003640, 0.002460, 0.001476, 0.002853, 0.000295, 0.119146, 0.001377, 0.066608, 0.050177, 0.039256, 0.028630, 0.011905, 0.020366, 0, 0.000394, 0.005313, 0, 0.000098, 0, 0, 0, 0, 0, 0.014168, 0, 0, 0};
static float NLD_afters_19[] = { 0.014997, 0.008332, 0.000926, 0.067580, 0.002777, 0.000370, 0.039993, 0, 0.189965, 0, 0.040178, 0.036475, 0.119052, 0.094797, 0, 0.000741, 0.068321, 0.168858, 0.013146, 0.017774, 0.004629, 0.078319, 0.000555, 0.000185, 0.001666, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.030365, 0};
static float NLD_afters_20[] = { 0.314184, 0, 0, 0, 0.365793, 0, 0, 0, 0.041001, 0, 0, 0.024672, 0, 0, 0.175685, 0, 0.062813, 0, 0, 0.005483, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.003218, 0.007151, 0, 0, 0};
static float NLD_afters_21[] = { 0.423702, 0.000350, 0, 0.012410, 0.325468, 0, 0, 0.000175, 0.073064, 0, 0.001748, 0.000175, 0.000175, 0.000175, 0.105401, 0, 0.006817, 0.003321, 0.000350, 0.001923, 0.000175, 0.000175, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.044398, 0, 0, 0};
static float NLD_afters_22[] = { 0.263158, 0, 0.052632, 0, 0, 0, 0, 0.052632, 0.526316, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.105263, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float NLD_afters_23[] = { 0.142857, 0, 0, 0, 0.035714, 0, 0, 0, 0, 0, 0, 0.285714, 0, 0, 0.178571, 0, 0.285714, 0.071429, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float NLD_afters_24[] = { 0.083839, 0, 0, 0, 0.286479, 0, 0, 0, 0.132893, 0, 0, 0, 0, 0, 0.195505, 0, 0, 0, 0, 0.028898, 0, 0.020514, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.000714, 0.251159, 0, 0, 0};
static float NLD_afters_26[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.520000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.480000, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float NLD_afters_27[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float NLD_afters_28[] = { 0, 0, 0, 0, 0.084746, 0, 0, 0, 0.016949, 0, 0, 0.016949, 0, 0.627119, 0, 0, 0.254237, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float NLD_afters_29[] = { 0, 0.365385, 0.192308, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.019231, 0.423077, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float NLD_afters_30[] = { 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float NLD_afters_31[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.017241, 0, 0.465517, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.517241, 0, 0, 0, 0};
static float NLD_afters_32[] = { 0.015075, 0.000670, 0, 0.084422, 0.008710, 0.028476, 0.029816, 0.006030, 0, 0, 0.334003, 0.053601, 0.001675, 0.346399, 0, 0.004020, 0, 0.018760, 0.006030, 0, 0.027471, 0.002010, 0, 0, 0.032831, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float NLD_afters_33[] = { 0.150000, 0, 0, 0, 0.050000, 0, 0, 0, 0.250000, 0, 0, 0, 0, 0, 0.550000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float NLD_afters_34[] = { 0.064189, 0.000307, 0, 0.000307, 0.248771, 0, 0, 0.003686, 0.071867, 0.002457, 0, 0.000614, 0, 0, 0.063268, 0, 0.060197, 0, 0.441953, 0.019963, 0.000307, 0, 0, 0, 0.000921, 0, 0, 0, 0, 0, 0, 0, 0.021192, 0, 0, 0};
static float NLD_afters_35[] = { 0, 0, 0, 0, 0.695652, 0, 0, 0, 0.304348, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static struct letter_frequencies NLD_counts[] = {
    { "a", 0.024390, 0.042671, 0.114334, 0.009916, NLD_afters_0 },
    { "b", 0.000000, 0.043781, 0.011649, 0.000308, NLD_afters_1 },
    { "c", 0.000000, 0.012923, 0.001258, 0.000015, NLD_afters_2 },
    { "d", 0.004878, 0.132331, 0.046995, 0.061331, NLD_afters_3 },
    { "e", 0.053659, 0.081610, 0.242864, 0.167754, NLD_afters_4 },
    { "f", 0.009756, 0.003177, 0.005480, 0.010672, NLD_afters_5 },
    { "g", 0.000000, 0.054515, 0.025997, 0.036240, NLD_afters_6 },
    { "h", 0.043902, 0.090153, 0.006570, 0.000000, NLD_afters_7 },
    { "i", 0.000000, 0.029609, 0.064460, 0.002776, NLD_afters_8 },
    { "j", 0.019512, 0.003208, 0.002239, 0.000015, NLD_afters_9 },
    { "k", 0.000000, 0.028761, 0.018310, 0.022854, NLD_afters_10 },
    { "l", 0.014634, 0.019724, 0.050633, 0.021050, NLD_afters_11 },
    { "m", 0.048780, 0.048624, 0.016288, 0.019616, NLD_afters_12 },
    { "n", 0.034146, 0.029532, 0.066113, 0.340504, NLD_afters_13 },
    { "o", 0.029268, 0.054854, 0.087638, 0.007418, NLD_afters_14 },
    { "p", 0.000000, 0.010271, 0.011649, 0.016624, NLD_afters_15 },
    { "r", 0.000000, 0.010795, 0.069588, 0.066219, NLD_afters_16 },
    { "s", 0.073171, 0.035485, 0.025048, 0.059172, NLD_afters_17 },
    { "t", 0.521951, 0.046850, 0.038626, 0.110695, NLD_afters_18 },
    { "u", 0.092683, 0.007202, 0.026745, 0.004010, NLD_afters_19 },
    { "v", 0.014634, 0.090678, 0.013605, 0.000000, NLD_afters_20 },
    { "w", 0.009756, 0.059897, 0.009957, 0.001588, NLD_afters_21 },
    { "x", 0.000000, 0.000077, 0.000076, 0.000185, NLD_afters_22 },
    { "y", 0.000000, 0.000062, 0.000130, 0.000093, NLD_afters_23 },
    { "z", 0.000000, 0.061377, 0.008814, 0.000586, NLD_afters_24 },
    { "à", 0.004878, 0.000000, 0.000000, 0.000000, NULL },
    { "é", 0.000000, 0.000740, 0.000282, 0.000062, NLD_afters_26 },
    { "ê", 0.000000, 0.000000, 0.000005, 0.000015, NLD_afters_27 },
    { "ë", 0.000000, 0.000000, 0.000320, 0.000540, NLD_afters_28 },
    { "ï", 0.000000, 0.000000, 0.000282, 0.000000, NLD_afters_29 },
    { "ò", 0.000000, 0.000154, 0.000000, 0.000000, NLD_afters_30 },
    { "ó", 0.000000, 0.000000, 0.000314, 0.000046, NLD_afters_31 },
    { "ij", 0.000000, 0.000355, 0.016055, 0.031197, NLD_afters_32 },
    { "ph", 0.000000, 0.000000, 0.000108, 0.000000, NLD_afters_33 },
    { "ch", 0.000000, 0.000555, 0.017454, 0.008497, NLD_afters_34 },
    { "qu", 0.000000, 0.000031, 0.000114, 0.000000, NLD_afters_35 },
    NULL
};	/* NLD */
static float NLD_word_lens[] = {
    0.000000, 0.003151, 0.190822, 0.254404, 0.120538, 0.082414, 0.105626, 0.069500, 0.063151, 0.037202, 0.027825, 0.020277, 0.011760, 0.005503, 0.004274, 0.001691, 0.001030, 0.000492, 0.000215, 0.000046, 0.000031, 0.000000, 0.000031, 0.000015, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float NLD_all_consonants[] = {
    0.000000, 0.002506, 0.000523, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 
};

static float NLD_consonant_run[] = {
    0.000000, 0.754949, 0.212540, 0.028519, 0.003811, 0.000181, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float NLD_vowel_run[] = {
    0.000000, 0.782897, 0.211836, 0.004677, 0.000589, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float ENG_afters_0[] = { 0.000375, 0.020106, 0.025732, 0.064442, 0.000151, 0.008723, 0.019045, 0.001265, 0.053992, 0.000278, 0.017509, 0.070841, 0.030251, 0.214759, 0.000218, 0.017673, 0, 0.103368, 0.110119, 0.137760, 0.009733, 0.032648, 0.009332, 0.000361, 0.036624, 0.001225, 0, 0, 0.003909, 0.001817, 0.006323, 0.000134, 0.000003, 0.000020, 0.000184, 0.001011, 0.000050, 0.000020};
static float ENG_afters_1[] = { 0.061138, 0.005943, 0, 0.000763, 0.365821, 0, 0.000016, 0.000065, 0.037576, 0.010717, 0.000049, 0.107726, 0.000877, 0.000081, 0.100760, 0, 0, 0.057955, 0.018171, 0.010084, 0.144783, 0.000244, 0.000146, 0, 0.077084, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_2[] = { 0.134120, 0.000014, 0.024019, 0.000122, 0.212892, 0, 0, 0, 0.046291, 0, 0.088980, 0.042838, 0, 0, 0.251994, 0, 0, 0.061766, 0.000948, 0.082996, 0.038655, 0, 0.000014, 0, 0.008895, 0, 0, 0, 0.000081, 0, 0, 0, 0.002789, 0, 0.002586, 0, 0, 0};
static float ENG_afters_3[] = { 0.073894, 0.001846, 0.002393, 0.023617, 0.276124, 0.003678, 0.019422, 0.000591, 0.158467, 0.001123, 0.001315, 0.033543, 0.006159, 0.019807, 0.170504, 0.000118, 0, 0.048283, 0.060926, 0.000015, 0.026837, 0.006824, 0.001167, 0, 0.060882, 0, 0, 0, 0.000118, 0.001979, 0.000222, 0, 0.000030, 0, 0, 0, 0.000118, 0};
static float ENG_afters_4[] = { 0.081118, 0.001789, 0.032190, 0.117980, 0.048386, 0.013859, 0.007020, 0.002988, 0.017625, 0.000742, 0.001740, 0.060331, 0.029044, 0.125325, 0.004176, 0.014427, 0, 0.219474, 0.099849, 0.040050, 0.000522, 0.025784, 0.010106, 0.013301, 0.021828, 0.000190, 0, 0, 0.000844, 0.000827, 0.005450, 0.000289, 0.002137, 0.000532, 0.000059, 0.000016, 0, 0};
static float ENG_afters_5[] = { 0.107103, 0.000059, 0, 0, 0.131549, 0.104348, 0, 0, 0.117421, 0, 0, 0.032066, 0, 0.000078, 0.267807, 0.000274, 0, 0.113786, 0.005511, 0.057040, 0.059326, 0, 0.000039, 0, 0.003322, 0, 0, 0, 0, 0.000020, 0.000254, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_6[] = { 0.128514, 0.000152, 0, 0.000152, 0.281242, 0.000076, 0.023504, 0, 0.090826, 0, 0, 0.053569, 0.003039, 0, 0.156755, 0.000152, 0, 0.128514, 0.052074, 0.005547, 0.067145, 0, 0.000203, 0, 0.003267, 0.000025, 0, 0, 0, 0, 0.005243, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_7[] = { 0.276938, 0, 0, 0, 0.353611, 0, 0, 0, 0.253368, 0, 0, 0.000012, 0.000012, 0.010607, 0.088974, 0.000012, 0, 0, 0, 0, 0.015910, 0, 0, 0, 0.000557, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_8[] = { 0.013119, 0.005361, 0.027012, 0.052247, 0.035839, 0.030972, 0.005120, 0.000061, 0.000437, 0.000126, 0.007908, 0.045653, 0.054671, 0.286247, 0.040120, 0.006035, 0, 0.044999, 0.126992, 0.105824, 0.000751, 0.019698, 0.000027, 0.001574, 0, 0.001061, 0, 0, 0.017132, 0.007237, 0.039109, 0.000008, 0.000149, 0, 0, 0.002298, 0, 0.022214};
static float ENG_afters_9[] = { 0.103109, 0, 0, 0, 0.196859, 0, 0, 0, 0.019194, 0, 0, 0, 0, 0, 0.455584, 0, 0, 0, 0, 0, 0.225095, 0, 0, 0, 0.000159, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_10[] = { 0.008575, 0.000251, 0.000552, 0.000150, 0.540217, 0.010330, 0.000953, 0.000602, 0.251078, 0, 0.000050, 0.017751, 0.001655, 0, 0.002407, 0.000100, 0, 0.005566, 0.148280, 0.000050, 0.001304, 0, 0.003259, 0, 0.006218, 0, 0, 0, 0, 0.000351, 0, 0, 0, 0.000301, 0, 0, 0, 0};
static float ENG_afters_11[] = { 0.088569, 0.000856, 0.001394, 0.103368, 0.194917, 0.032533, 0.000435, 0.000007, 0.125307, 0.000007, 0.011663, 0.168810, 0.005075, 0.002021, 0.094286, 0.002789, 0, 0.002102, 0.016739, 0.012556, 0.014894, 0.005407, 0.007945, 0.000074, 0.105536, 0, 0, 0, 0.000059, 0.000052, 0.001992, 0.000583, 0, 0, 0, 0, 0.000022, 0};
static float ENG_afters_12[] = { 0.184981, 0.018526, 0.000011, 0.000053, 0.281120, 0.004102, 0, 0.000011, 0.093988, 0.000011, 0.000011, 0.001297, 0.019506, 0.003743, 0.100725, 0.044094, 0, 0.093640, 0.035132, 0.000011, 0.040035, 0.000032, 0.000011, 0, 0.076812, 0, 0, 0.000011, 0.000021, 0.000032, 0.000264, 0.001824, 0, 0, 0, 0, 0, 0};
static float ENG_afters_13[] = { 0.026773, 0.000316, 0.041426, 0.248847, 0.103545, 0.007043, 0.179730, 0.001084, 0.036640, 0.001625, 0.016544, 0.010630, 0.000923, 0.013755, 0.094345, 0.000682, 0, 0.000547, 0.042414, 0.126511, 0.005047, 0.003983, 0.000863, 0.000968, 0.024385, 0.000030, 0, 0.000005, 0.005322, 0.000502, 0.001605, 0, 0.001956, 0.000211, 0.000231, 0.000025, 0, 0.001485};
static float ENG_afters_14[] = { 0.007013, 0.008710, 0.012520, 0.020446, 0.003829, 0.098964, 0.004181, 0.006982, 0.011454, 0.000176, 0.018249, 0.032198, 0.068264, 0.156198, 0.050046, 0.018532, 0, 0.115099, 0.031621, 0.040931, 0.190640, 0.015600, 0.062340, 0.000680, 0.004288, 0.000443, 0, 0, 0.000034, 0.000069, 0.019361, 0.000428, 0.000325, 0.000065, 0, 0.000306, 0.000008, 0};
static float ENG_afters_15[] = { 0.120488, 0.000401, 0.000017, 0.000017, 0.227401, 0, 0, 0, 0.067575, 0.000017, 0.001856, 0.092368, 0.000368, 0.000167, 0.165694, 0.075466, 0, 0.134264, 0.028220, 0.028103, 0.039321, 0, 0.000518, 0, 0.016484, 0, 0, 0, 0, 0.000518, 0.000552, 0, 0, 0, 0, 0, 0.000184, 0};
static float ENG_afters_17[] = { 0.068720, 0.002320, 0.008583, 0.046882, 0.309364, 0.004916, 0.011815, 0.004249, 0.089283, 0.000116, 0.013401, 0.019762, 0.015005, 0.034877, 0.102605, 0.005865, 0, 0.024078, 0.083498, 0.056647, 0.019621, 0.009446, 0.003245, 0, 0.054198, 0, 0, 0, 0.003869, 0.000496, 0.006404, 0.000165, 0.000178, 0.000122, 0.000263, 0, 0, 0.000006};
static float ENG_afters_18[] = { 0.107139, 0.010878, 0.016428, 0.000924, 0.203870, 0.002570, 0.001058, 0, 0.099830, 0.000015, 0.012666, 0.011146, 0.012219, 0.019088, 0.099726, 0.036605, 0.000007, 0.000238, 0.085115, 0.198491, 0.060841, 0.000007, 0.008702, 0, 0.004984, 0, 0, 0, 0.001490, 0, 0.001512, 0.000387, 0.004038, 0, 0.000022, 0, 0, 0};
static float ENG_afters_19[] = { 0.080273, 0.000190, 0.000471, 0.000021, 0.194691, 0.002474, 0.000063, 0, 0.139495, 0.000014, 0, 0.046564, 0.003093, 0.002811, 0.277508, 0.000246, 0, 0.052433, 0.049657, 0.058991, 0.039500, 0, 0.016046, 0, 0.028578, 0.000401, 0, 0, 0.006269, 0.000148, 0.000056, 0, 0, 0, 0.000007, 0, 0, 0};
static float ENG_afters_20[] = { 0.014455, 0.018690, 0.015003, 0.015717, 0.018416, 0.007061, 0.005653, 0, 0.008694, 0, 0.000694, 0.133322, 0.027394, 0.112529, 0.001007, 0.071962, 0, 0.166124, 0.119141, 0.154104, 0.000010, 0.001311, 0, 0.000411, 0.000479, 0.003325, 0, 0, 0.039805, 0.003658, 0.007022, 0.000137, 0, 0, 0, 0.000088, 0, 0.053790};
static float ENG_afters_21[] = { 0.047399, 0, 0, 0, 0.784816, 0, 0, 0.004835, 0.113323, 0, 0, 0.001711, 0, 0.000462, 0.040989, 0, 0, 0.000326, 0.000027, 0.000027, 0.001005, 0, 0, 0, 0.005025, 0, 0.000054, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_22[] = { 0.292660, 0.000279, 0.000124, 0.003577, 0.203081, 0.001394, 0.000046, 0, 0.251533, 0, 0.001146, 0.010019, 0.000263, 0.064912, 0.152385, 0, 0, 0, 0.015965, 0.000062, 0.000170, 0, 0, 0, 0.001812, 0.000015, 0, 0, 0, 0.000031, 0.000526, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_23[] = { 0.107576, 0, 0.157143, 0, 0.077489, 0.001299, 0, 0.016234, 0.102381, 0, 0, 0.006926, 0, 0, 0.004762, 0.304545, 0, 0, 0.001732, 0.163203, 0.010823, 0.008442, 0, 0.019481, 0.003463, 0, 0, 0, 0.008442, 0, 0.001082, 0, 0.004978, 0, 0, 0, 0, 0};
static float ENG_afters_24[] = { 0.008773, 0.017312, 0.018217, 0.003985, 0.137997, 0.002143, 0.000502, 0.001574, 0.039614, 0, 0, 0.004889, 0.012591, 0.000871, 0.564746, 0.001172, 0, 0.003248, 0.136323, 0.000268, 0, 0.000603, 0.000703, 0, 0, 0, 0, 0.000033, 0.000067, 0.004956, 0.035931, 0.000033, 0, 0.003449, 0, 0, 0, 0};
static float ENG_afters_25[] = { 0.017544, 0, 0, 0, 0.397243, 0, 0.001253, 0.012531, 0.082707, 0, 0, 0.221178, 0, 0, 0.013158, 0, 0, 0.000627, 0.000627, 0, 0.001880, 0, 0, 0, 0.022556, 0.228697, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_27[] = { 0, 0, 0.500000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.500000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_28[] = { 0.419007, 0.000237, 0, 0.001066, 0.204171, 0.005332, 0, 0, 0.176561, 0, 0, 0.002607, 0.008769, 0.000474, 0.068610, 0, 0, 0.017301, 0.009243, 0.000355, 0.078208, 0, 0.003199, 0, 0.004858, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_29[] = { 0.138016, 0, 0, 0, 0.483690, 0.000800, 0, 0, 0.080782, 0, 0, 0.001601, 0.012274, 0.002335, 0.248149, 0, 0, 0.011207, 0, 0, 0.018411, 0, 0.000133, 0, 0.002602, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_30[] = { 0.183892, 0.000010, 0.000010, 0.001082, 0.633902, 0.000978, 0.000010, 0.000042, 0.108647, 0, 0.000010, 0.000666, 0.000104, 0.000031, 0.045047, 0.000062, 0, 0.017170, 0.002456, 0, 0.003049, 0, 0.000177, 0, 0.002633, 0, 0, 0, 0, 0, 0, 0, 0.000021, 0, 0, 0, 0, 0};
static float ENG_afters_31[] = { 0.204315, 0, 0, 0, 0.290609, 0, 0, 0, 0.237310, 0, 0, 0.003807, 0, 0, 0.024112, 0, 0, 0.044416, 0.019036, 0, 0.091371, 0, 0, 0, 0.083756, 0, 0, 0, 0, 0, 0.001269, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_32[] = { 0.165177, 0, 0, 0, 0.234466, 0, 0, 0, 0.588064, 0, 0, 0, 0, 0, 0.011176, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001118, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_33[] = { 0.203270, 0, 0, 0, 0.279842, 0, 0, 0, 0.274937, 0, 0, 0, 0, 0, 0.191824, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.050127, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_34[] = { 0.005520, 0, 0, 0, 0.191295, 0, 0, 0, 0.018259, 0, 0, 0, 0, 0, 0.783227, 0, 0, 0, 0, 0, 0.001699, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_35[] = { 0.204751, 0.001131, 0, 0, 0.352941, 0, 0, 0, 0.292986, 0, 0, 0.002262, 0.001131, 0, 0.063348, 0, 0, 0, 0.053167, 0, 0.028281, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_36[] = { 0.071429, 0, 0, 0, 0.110627, 0, 0, 0, 0.489547, 0, 0, 0, 0, 0, 0.317073, 0, 0, 0, 0, 0, 0.006098, 0, 0, 0, 0.005226, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ENG_afters_37[] = { 0.003538, 0.016547, 0, 0, 0.023508, 0.001712, 0.000114, 0, 0.015177, 0, 0, 0.015406, 0.000114, 0.001484, 0.044505, 0, 0, 0, 0.005249, 0.871277, 0, 0, 0.000456, 0, 0.000342, 0, 0, 0, 0, 0.000114, 0.000456, 0, 0, 0, 0, 0, 0, 0};
static struct letter_frequencies ENG_counts[] = {
    { "a", 0.501060, 0.095293, 0.108824, 0.003057, ENG_afters_0 },
    { "b", 0.000530, 0.048801, 0.008651, 0.000391, ENG_afters_1 },
    { "c", 0.001767, 0.032520, 0.022667, 0.000752, ENG_afters_2 },
    { "d", 0.000984, 0.032480, 0.019517, 0.121421, ENG_afters_3 },
    { "e", 0.000101, 0.018004, 0.148337, 0.189578, ENG_afters_4 },
    { "f", 0.000025, 0.032287, 0.011096, 0.036847, ENG_afters_5 },
    { "g", 0.000278, 0.020747, 0.010528, 0.033397, ENG_afters_6 },
    { "h", 0.000202, 0.090505, 0.001605, 0.001423, ENG_afters_7 },
    { "i", 0.488818, 0.058449, 0.106860, 0.000183, ENG_afters_8 },
    { "j", 0.000151, 0.005263, 0.000759, 0.000000, ENG_afters_9 },
    { "k", 0.000025, 0.002343, 0.009162, 0.011442, ENG_afters_10 },
    { "l", 0.001161, 0.028047, 0.056553, 0.025469, ENG_afters_11 },
    { "m", 0.000833, 0.057327, 0.021750, 0.020870, ENG_afters_12 },
    { "n", 0.000177, 0.025605, 0.090556, 0.078689, ENG_afters_13 },
    { "o", 0.000883, 0.056856, 0.107880, 0.052202, ENG_afters_14 },
    { "p", 0.000252, 0.027689, 0.017718, 0.006153, ENG_afters_15 },
    { "q", 0.000000, 0.000000, 0.000000, 0.000001, NULL },
    { "r", 0.000454, 0.019834, 0.074745, 0.070490, ENG_afters_17 },
    { "s", 0.001060, 0.062450, 0.039607, 0.113369, ENG_afters_18 },
    { "t", 0.000379, 0.052947, 0.048247, 0.114371, ENG_afters_19 },
    { "u", 0.000328, 0.012323, 0.046832, 0.012972, ENG_afters_20 },
    { "v", 0.000076, 0.006862, 0.015716, 0.000062, ENG_afters_21 },
    { "w", 0.000202, 0.052472, 0.008460, 0.012264, ENG_afters_22 },
    { "x", 0.000076, 0.000127, 0.002319, 0.000632, ENG_afters_23 },
    { "y", 0.000025, 0.020911, 0.005497, 0.066219, ENG_afters_24 },
    { "z", 0.000000, 0.000050, 0.000798, 0.000040, ENG_afters_25 },
    { "æ", 0.000000, 0.000000, 0.000000, 0.000002, NULL },
    { "é", 0.000000, 0.000001, 0.000001, 0.000002, ENG_afters_27 },
    { "ch", 0.000000, 0.005988, 0.001516, 0.010771, ENG_afters_28 },
    { "sh", 0.000000, 0.013821, 0.001189, 0.001422, ENG_afters_29 },
    { "th", 0.000000, 0.091952, 0.006035, 0.012078, ENG_afters_30 },
    { "ph", 0.000000, 0.000303, 0.000263, 0.000085, ENG_afters_31 },
    { "qu", 0.000000, 0.002729, 0.001015, 0.000000, ENG_afters_32 },
    { "wh", 0.000000, 0.018938, 0.000183, 0.000000, ENG_afters_33 },
    { "kn", 0.000000, 0.004744, 0.000184, 0.000000, ENG_afters_34 },
    { "gn", 0.000000, 0.000021, 0.000445, 0.000148, ENG_afters_35 },
    { "wr", 0.000000, 0.001208, 0.000020, 0.000000, ENG_afters_36 },
    { "gh", 0.000000, 0.000102, 0.004464, 0.003198, ENG_afters_37 },
    NULL
};	/* ENG */
static float ENG_word_lens[] = {
    0.000000, 0.041384, 0.244968, 0.241823, 0.169986, 0.087025, 0.073533, 0.055951, 0.033833, 0.025196, 0.013542, 0.006853, 0.003469, 0.001813, 0.000468, 0.000104, 0.000032, 0.000017, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000001
};

static float ENG_all_consonants[] = {
    0.000000, 0.000364, 0.006883, 0.002565, 0.000004, 0.000000, 0.000000, 0.000000, 0.000000, 0.000001, 0.000000, 0.000000, 
};

static float ENG_consonant_run[] = {
    0.000000, 0.746291, 0.226731, 0.025320, 0.001647, 0.000010, 0.000001, 0.000000, 0.000000, 0.000001, 0.000000, 0.000000
};

static float ENG_vowel_run[] = {
    0.000000, 0.869523, 0.114786, 0.015672, 0.000019, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float FRA_afters_0[] = { 0.000164, 0.023933, 0.023516, 0.012673, 0.000063, 0.004102, 0.029588, 0.004569, 0.239895, 0.001376, 0.000404, 0.046376, 0.024097, 0.175885, 0.000909, 0.036303, 0, 0.103620, 0.056664, 0.045253, 0.095706, 0.049860, 0.000025, 0.001338, 0.005491, 0.000732, 0, 0, 0, 0.001325, 0, 0.000164, 0, 0.000050, 0.005655, 0.000480, 0.000013, 0, 0, 0.000682, 0.005112, 0, 0.003976};
static float FRA_afters_1[] = { 0.142976, 0.001384, 0, 0.001568, 0.095840, 0, 0, 0, 0.127848, 0.004428, 0, 0.223872, 0.000553, 0.000092, 0.146758, 0, 0, 0.166590, 0.031916, 0.003136, 0.020570, 0, 0, 0, 0.001753, 0, 0, 0.007195, 0, 0, 0.000738, 0.014574, 0.004428, 0, 0.002029, 0, 0.000092, 0, 0.000646, 0, 0, 0.001015, 0};
static float FRA_afters_2[] = { 0.114933, 0, 0.017857, 0, 0.305350, 0, 0, 0, 0.069508, 0, 0.001316, 0.030876, 0, 0.000285, 0.269707, 0, 0, 0.053892, 0.006510, 0.032406, 0.044287, 0, 0, 0, 0.001743, 0, 0, 0.000747, 0, 0, 0.003628, 0.033224, 0.000107, 0, 0, 0, 0.008004, 0, 0, 0, 0.000036, 0.004482, 0.001103};
static float FRA_afters_3[] = { 0.117003, 0, 0, 0.000101, 0.487486, 0.000025, 0.000051, 0.005148, 0.113833, 0.000228, 0, 0.000127, 0.003474, 0.000076, 0.058096, 0.000076, 0, 0.036034, 0.013491, 0.000076, 0.102548, 0.000304, 0.000304, 0, 0.000659, 0, 0, 0.000558, 0, 0, 0.002206, 0.053480, 0.001420, 0, 0.001496, 0, 0.000051, 0, 0.000888, 0, 0.000025, 0.000710, 0.000025};
static float FRA_afters_4[] = { 0.020638, 0.000944, 0.018629, 0.009497, 0.000152, 0.006078, 0.004809, 0.001694, 0.020070, 0.000477, 0.000051, 0.041905, 0.055126, 0.208815, 0.000954, 0.009751, 0, 0.116441, 0.257924, 0.098137, 0.087767, 0.014175, 0.000365, 0.008482, 0.000446, 0.012825, 0, 0.000172, 0, 0.000507, 0, 0, 0, 0, 0, 0, 0.000010, 0, 0.001938, 0.000041, 0.000345, 0, 0.000832};
static float FRA_afters_5[] = { 0.212984, 0, 0, 0, 0.107366, 0.092468, 0, 0, 0.143154, 0, 0, 0.060008, 0, 0, 0.168456, 0, 0, 0.100125, 0.009405, 0.000250, 0.073908, 0, 0, 0, 0, 0, 0, 0.002414, 0, 0, 0.002913, 0.017312, 0.001248, 0, 0.000832, 0, 0, 0, 0.007158, 0, 0, 0, 0};
static float FRA_afters_6[] = { 0.129606, 0, 0, 0.000091, 0.286209, 0, 0.001186, 0.000730, 0.057005, 0, 0, 0.041499, 0.002007, 0.100967, 0.053265, 0, 0, 0.133528, 0.007023, 0.025812, 0.092576, 0, 0, 0, 0.001824, 0.000182, 0, 0.000912, 0, 0, 0.010945, 0.051806, 0.002371, 0, 0.000456, 0, 0, 0, 0, 0, 0, 0, 0};
static float FRA_afters_7[] = { 0.247934, 0, 0, 0, 0.196281, 0, 0, 0, 0.057851, 0, 0.000258, 0.002066, 0.037190, 0.002583, 0.248192, 0, 0, 0.002583, 0.001550, 0.000517, 0.092717, 0, 0, 0, 0.014205, 0, 0, 0.005940, 0, 0, 0.012913, 0.065857, 0.000517, 0, 0, 0, 0.010847, 0, 0, 0, 0, 0, 0};
static float FRA_afters_8[] = { 0.010961, 0.009348, 0.017125, 0.016515, 0.120201, 0.008752, 0.018642, 0.000027, 0.001003, 0.000163, 0.000285, 0.135226, 0.021786, 0.123235, 0.038342, 0.004932, 0, 0.069774, 0.150942, 0.189690, 0.000488, 0.018941, 0, 0.008535, 0.000014, 0.002005, 0, 0.000081, 0, 0, 0.012668, 0.003970, 0, 0, 0, 0, 0, 0, 0, 0.000095, 0.001707, 0.000014, 0.014537};
static float FRA_afters_9[] = { 0.146773, 0, 0, 0, 0.582306, 0.000145, 0, 0.000435, 0.002611, 0, 0, 0.000290, 0.000290, 0, 0.154605, 0.000435, 0, 0, 0.000145, 0, 0.071066, 0.001595, 0, 0, 0.003191, 0, 0.019579, 0, 0, 0, 0, 0.016389, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.000145, 0};
static float FRA_afters_10[] = { 0.201681, 0, 0, 0.046218, 0.075630, 0, 0, 0.008403, 0.294118, 0, 0, 0.109244, 0.004202, 0.008403, 0.159664, 0, 0, 0.033613, 0.037815, 0, 0.016807, 0, 0, 0, 0.004202, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float FRA_afters_11[] = { 0.221407, 0.000777, 0.001873, 0.001184, 0.390462, 0.001855, 0.001767, 0.008675, 0.058325, 0, 0.000053, 0.096631, 0.001979, 0.001095, 0.065728, 0.001060, 0, 0.000018, 0.015920, 0.003410, 0.072230, 0.001237, 0.000053, 0.000035, 0.001237, 0.000035, 0.010548, 0.001113, 0, 0, 0.004647, 0.023270, 0.000159, 0, 0.001219, 0, 0.000018, 0, 0.000194, 0.000141, 0.000018, 0.000989, 0.010637};
static float FRA_afters_12[] = { 0.192565, 0.039152, 0, 0.000029, 0.304502, 0, 0.000087, 0.000232, 0.087743, 0, 0.000261, 0.000058, 0.082428, 0.001133, 0.144583, 0.063114, 0, 0, 0.000755, 0.001220, 0.017398, 0, 0.000029, 0, 0.002875, 0, 0, 0.001830, 0, 0, 0.017165, 0.027505, 0.014087, 0, 0.000116, 0, 0.000145, 0, 0.000319, 0.000465, 0.000029, 0.000174, 0};
static float FRA_afters_13[] = { 0.065788, 0.000030, 0.043564, 0.087773, 0.168487, 0.010880, 0.028818, 0.000689, 0.032310, 0.000525, 0.000090, 0.000869, 0.000210, 0.032385, 0.060963, 0.000045, 0.002638, 0.001663, 0.126347, 0.271067, 0.013937, 0.007358, 0.000045, 0.000060, 0.001858, 0.001798, 0, 0.000405, 0, 0.004915, 0.001394, 0.018328, 0.001139, 0, 0.000075, 0, 0.000075, 0, 0.000030, 0, 0.006489, 0.000944, 0.006009};
static float FRA_afters_14[] = { 0.000457, 0.011311, 0.010482, 0.009008, 0, 0.009432, 0.003641, 0.001626, 0.099177, 0.000728, 0.000254, 0.029159, 0.076114, 0.285644, 0.001033, 0.010431, 0.000119, 0.092810, 0.028719, 0.028075, 0.264969, 0.001981, 0.000288, 0.000508, 0.011244, 0.000068, 0, 0, 0, 0, 0.000491, 0.000339, 0.000169, 0.000474, 0.000322, 0.000423, 0, 0.006689, 0.001422, 0.001372, 0.006468, 0, 0.004555};
static float FRA_afters_15[] = { 0.244081, 0, 0, 0, 0.126414, 0, 0, 0, 0.067948, 0, 0, 0.093897, 0, 0.000032, 0.165241, 0.038988, 0, 0.144674, 0.017524, 0.014480, 0.037033, 0.000032, 0, 0, 0.000641, 0, 0, 0.002659, 0, 0.000320, 0.010828, 0.025180, 0.007432, 0, 0.000064, 0, 0.002018, 0, 0.000481, 0, 0, 0.000032, 0};
static float FRA_afters_16[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float FRA_afters_17[] = { 0.131622, 0.005341, 0.011641, 0.031880, 0.301277, 0.005721, 0.010913, 0.000562, 0.096022, 0.000364, 0.000298, 0.005672, 0.023547, 0.014154, 0.089027, 0.007458, 0, 0.032426, 0.059743, 0.047986, 0.018139, 0.010483, 0.000529, 0, 0.000513, 0.000231, 0, 0.001538, 0, 0.003820, 0.014684, 0.052996, 0.007011, 0, 0.000149, 0, 0.001058, 0, 0.000744, 0.000215, 0.007772, 0.000099, 0.004365};
static float FRA_afters_18[] = { 0.134308, 0.000114, 0.014397, 0.000091, 0.225950, 0.001002, 0.000114, 0.001230, 0.114855, 0.000046, 0.000228, 0.000433, 0.001686, 0.000114, 0.121848, 0.022985, 0.000023, 0.000205, 0.124559, 0.094216, 0.077678, 0.000137, 0, 0, 0.001913, 0, 0, 0.000501, 0, 0, 0.002073, 0.035081, 0.000159, 0, 0.000046, 0, 0, 0, 0.001185, 0.002916, 0.000478, 0.000182, 0.019249};
static float FRA_afters_19[] = { 0.148129, 0.000023, 0.000548, 0.000023, 0.253008, 0.000023, 0, 0.005412, 0.135889, 0.000023, 0.000046, 0.001576, 0.001530, 0.000046, 0.080652, 0.000023, 0, 0.163314, 0.047085, 0.060421, 0.034640, 0, 0, 0, 0.000822, 0.001165, 0, 0.001187, 0, 0, 0.003585, 0.047747, 0.005983, 0, 0.000137, 0, 0.006828, 0, 0, 0, 0.000137, 0, 0};
static float FRA_afters_20[] = { 0.006827, 0.007855, 0.011385, 0.014856, 0.030974, 0.007913, 0.008883, 0.000659, 0.076202, 0.005314, 0.000058, 0.038790, 0.014546, 0.136094, 0.000330, 0.020074, 0, 0.223991, 0.169977, 0.102056, 0, 0.042106, 0, 0.062781, 0.001552, 0.001358, 0, 0.000407, 0, 0, 0.000892, 0.004267, 0, 0.000252, 0.000485, 0.000078, 0, 0, 0, 0.000272, 0.007196, 0, 0.001571};
static float FRA_afters_21[] = { 0.214337, 0, 0, 0, 0.297481, 0, 0, 0, 0.178744, 0, 0, 0.000890, 0, 0.000111, 0.202770, 0, 0, 0.051332, 0, 0, 0.013904, 0, 0, 0, 0, 0, 0, 0.000723, 0, 0, 0.001835, 0.030421, 0.005172, 0, 0.001502, 0, 0.000334, 0, 0, 0, 0, 0.000445, 0};
static float FRA_afters_22[] = { 0.584507, 0, 0.007042, 0, 0.084507, 0, 0, 0.007042, 0.274648, 0, 0.007042, 0, 0, 0.021127, 0, 0, 0, 0, 0.007042, 0.007042, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float FRA_afters_23[] = { 0.132108, 0, 0.116360, 0, 0.073491, 0, 0, 0.004374, 0.222222, 0, 0, 0, 0, 0, 0.008749, 0.179353, 0, 0, 0.000875, 0.148731, 0.003500, 0.013998, 0, 0.011374, 0.020122, 0, 0, 0, 0, 0, 0.003500, 0.032371, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.028871};
static float FRA_afters_24[] = { 0.245851, 0.001037, 0.007261, 0.013485, 0.314834, 0, 0.015041, 0, 0.002075, 0, 0, 0.022303, 0.016079, 0.007261, 0.071577, 0.042531, 0, 0.036307, 0.115664, 0.030083, 0, 0.002075, 0, 0.003112, 0, 0.000519, 0, 0.002075, 0, 0, 0.002075, 0.044606, 0, 0, 0, 0, 0, 0, 0, 0.003631, 0.000519, 0, 0};
static float FRA_afters_25[] = { 0.120370, 0.005556, 0, 0, 0.414815, 0, 0, 0, 0.062963, 0, 0.003704, 0.007407, 0, 0, 0.305556, 0, 0, 0, 0, 0, 0.018519, 0, 0, 0, 0, 0.005556, 0, 0.001852, 0, 0, 0.001852, 0.051852, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float FRA_afters_27[] = { 0.001312, 0.044619, 0.069554, 0, 0, 0, 0.040682, 0.001312, 0.011811, 0, 0, 0.074803, 0.149606, 0.028871, 0, 0.003937, 0, 0, 0.001312, 0.443570, 0, 0.001312, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.125984, 0, 0.001312};
static float FRA_afters_28[] = { 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float FRA_afters_29[] = { 0.542453, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.262972, 0, 0, 0, 0, 0, 0.168632, 0, 0, 0, 0, 0, 0.016509, 0.001179, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.008255, 0, 0, 0, 0};
static float FRA_afters_30[] = { 0, 0.009011, 0.035795, 0.011014, 0, 0, 0.022528, 0, 0, 0, 0, 0.008761, 0.037797, 0.038798, 0, 0, 0, 0.462829, 0.210763, 0.117397, 0, 0.030288, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.000501, 0.008260, 0, 0.006258};
static float FRA_afters_31[] = { 0.020225, 0.011973, 0.082884, 0.021589, 0.135989, 0.015944, 0.035734, 0.001427, 0.002047, 0.011105, 0.000062, 0.042745, 0.033067, 0.031640, 0.004591, 0.088095, 0, 0.089460, 0.114337, 0.190707, 0.003660, 0.029220, 0, 0, 0, 0.000310, 0, 0.000682, 0, 0, 0, 0.000310, 0, 0, 0, 0.000186, 0, 0, 0, 0.001365, 0.018736, 0, 0.011911};
static float FRA_afters_32[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.028966, 0.237241, 0.018391, 0, 0, 0, 0, 0, 0.597701, 0, 0.023908, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.091494, 0, 0.002299};
static float FRA_afters_33[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.702703, 0, 0, 0, 0, 0, 0, 0.297297, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float FRA_afters_34[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.223862, 0.097171, 0.227552, 0, 0, 0, 0, 0, 0.407134, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.044280, 0, 0};
static float FRA_afters_35[] = { 0.016129, 0, 0.016129, 0.225806, 0.112903, 0.048387, 0, 0, 0, 0, 0, 0, 0, 0.016129, 0.048387, 0, 0, 0.048387, 0.177419, 0, 0, 0.032258, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.258065};
static float FRA_afters_36[] = { 0, 0, 0.001383, 0.015214, 0, 0, 0, 0, 0, 0, 0, 0.179806, 0.015214, 0.015214, 0, 0.006916, 0, 0, 0, 0.763485, 0, 0.002766, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float FRA_afters_37[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float FRA_afters_38[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.073786, 0.089320, 0.003883, 0, 0, 0, 0.116505, 0, 0.704854, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.011650, 0, 0};
static float FRA_afters_39[] = { 0.146947, 0, 0, 0, 0.053435, 0, 0, 0, 0.124046, 0, 0, 0.007634, 0, 0, 0.190840, 0, 0, 0.045802, 0, 0, 0, 0, 0, 0, 0.175573, 0, 0, 0, 0, 0, 0.169847, 0.085878, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float FRA_afters_40[] = { 0.346650, 0, 0, 0, 0.418152, 0, 0, 0, 0.070887, 0, 0.000205, 0.000820, 0, 0.003483, 0.069453, 0, 0, 0.005737, 0.001024, 0.001229, 0.005327, 0, 0, 0, 0.002049, 0, 0, 0.010244, 0, 0, 0.005941, 0.054087, 0.002868, 0, 0, 0, 0.000615, 0, 0, 0, 0, 0.001229, 0};
static float FRA_afters_41[] = { 0, 0, 0, 0.002667, 0, 0, 0, 0, 0.250667, 0, 0.002667, 0.018667, 0, 0.005333, 0, 0.002667, 0, 0.002667, 0.008000, 0.002667, 0.704000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float FRA_afters_42[] = { 0.109115, 0, 0, 0, 0.530398, 0, 0, 0.000079, 0.284670, 0.000079, 0, 0, 0, 0, 0.029087, 0, 0, 0, 0, 0.000159, 0.015815, 0.000079, 0, 0, 0.000954, 0, 0.015656, 0.000397, 0, 0, 0.000079, 0.012795, 0.000397, 0, 0.000159, 0, 0, 0, 0, 0, 0, 0.000079, 0};
static struct letter_frequencies FRA_counts[] = {
    { "a", 0.134638, 0.047332, 0.105793, 0.045096, FRA_afters_0 },
    { "b", 0.001161, 0.021899, 0.008889, 0.000128, FRA_afters_1 },
    { "c", 0.000497, 0.072640, 0.017306, 0.006741, FRA_afters_2 },
    { "d", 0.000995, 0.126092, 0.015511, 0.011980, FRA_afters_3 },
    { "e", 0.000000, 0.059491, 0.131391, 0.291674, FRA_afters_4 },
    { "f", 0.000166, 0.030547, 0.007578, 0.001058, FRA_afters_5 },
    { "g", 0.000332, 0.012412, 0.012517, 0.001002, FRA_afters_6 },
    { "h", 0.000166, 0.010429, 0.002230, 0.001461, FRA_afters_7 },
    { "i", 0.002653, 0.023283, 0.106108, 0.034384, FRA_afters_8 },
    { "j", 0.000497, 0.026182, 0.001214, 0.000000, FRA_afters_9 },
    { "k", 0.000000, 0.000557, 0.000168, 0.000257, FRA_afters_10 },
    { "l", 0.015089, 0.114258, 0.046432, 0.026932, FRA_afters_11 },
    { "m", 0.028022, 0.061127, 0.031283, 0.001538, FRA_afters_12 },
    { "n", 0.000332, 0.041327, 0.088581, 0.066974, FRA_afters_13 },
    { "o", 0.000995, 0.014070, 0.086549, 0.002968, FRA_afters_14 },
    { "p", 0.000166, 0.082140, 0.018680, 0.001936, FRA_afters_15 },
    { "q", 0.000000, 0.000000, 0.000002, 0.000784, FRA_afters_16 },
    { "r", 0.000332, 0.031395, 0.082476, 0.061230, FRA_afters_17 },
    { "s", 0.000166, 0.077047, 0.040209, 0.217822, FRA_afters_18 },
    { "t", 0.018902, 0.033206, 0.055929, 0.147756, FRA_afters_19 },
    { "u", 0.000000, 0.021021, 0.072398, 0.031198, FRA_afters_20 },
    { "v", 0.001492, 0.030298, 0.016926, 0.000090, FRA_afters_21 },
    { "w", 0.000000, 0.000257, 0.000127, 0.000103, FRA_afters_22 },
    { "x", 0.000829, 0.000231, 0.001690, 0.016168, FRA_afters_23 },
    { "y", 0.047090, 0.001195, 0.002559, 0.001576, FRA_afters_24 },
    { "z", 0.000000, 0.000493, 0.000660, 0.005641, FRA_afters_25 },
    { "à", 0.745150, 0.000000, 0.000000, 0.004039, NULL },
    { "â", 0.000000, 0.000201, 0.001110, 0.000000, FRA_afters_27 },
    { "æ", 0.000000, 0.000004, 0.000000, 0.000000, FRA_afters_28 },
    { "ç", 0.000000, 0.000531, 0.001124, 0.000000, FRA_afters_29 },
    { "è", 0.000000, 0.000004, 0.006198, 0.000000, FRA_afters_30 },
    { "é", 0.000000, 0.012900, 0.020340, 0.017415, FRA_afters_31 },
    { "ê", 0.000000, 0.001747, 0.002742, 0.000000, FRA_afters_32 },
    { "ë", 0.000000, 0.000000, 0.000057, 0.000034, FRA_afters_33 },
    { "î", 0.000000, 0.000424, 0.001108, 0.000000, FRA_afters_34 },
    { "ï", 0.000000, 0.000000, 0.000096, 0.000034, FRA_afters_35 },
    { "ô", 0.000332, 0.000030, 0.001111, 0.000009, FRA_afters_36 },
    { "ù", 0.000000, 0.000000, 0.000002, 0.001688, FRA_afters_37 },
    { "û", 0.000000, 0.000000, 0.000799, 0.000133, FRA_afters_38 },
    { "ph", 0.000000, 0.000732, 0.000548, 0.000013, FRA_afters_39 },
    { "ch", 0.000000, 0.008502, 0.004494, 0.000124, FRA_afters_40 },
    { "oe", 0.000000, 0.000244, 0.000493, 0.000017, FRA_afters_41 },
    { "qu", 0.000000, 0.035750, 0.006574, 0.000000, FRA_afters_42 },
    NULL
};	/* FRA */
static float FRA_word_lens[] = {
    0.000000, 0.025181, 0.256787, 0.132516, 0.142587, 0.103407, 0.099858, 0.078911, 0.062987, 0.043597, 0.027377, 0.012730, 0.007432, 0.003728, 0.001829, 0.000656, 0.000359, 0.000042, 0.000017, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float FRA_all_consonants[] = {
    0.000000, 0.001741, 0.000071, 0.000008, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 
};

static float FRA_consonant_run[] = {
    0.000000, 0.759497, 0.220746, 0.019328, 0.000428, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float FRA_vowel_run[] = {
    0.000000, 0.806392, 0.172361, 0.020171, 0.001008, 0.000069, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float DEU_afters_0[] = { 0.003104, 0.046810, 0.000654, 0.008823, 0.001144, 0.015358, 0.044849, 0.025161, 0.000245, 0, 0.000817, 0.099910, 0.032840, 0.173760, 0.000082, 0.006127, 0, 0.103586, 0.102688, 0.052038, 0.180541, 0.003676, 0, 0.000163, 0.001797, 0.002369, 0.031778, 0, 0, 0, 0.003676, 0.053999, 0.003921, 0, 0.000082};
static float DEU_afters_1[] = { 0.051220, 0.000732, 0, 0, 0.532195, 0.000244, 0.006585, 0.001951, 0.067317, 0, 0.000488, 0.069024, 0, 0.001707, 0.018780, 0, 0, 0.082927, 0.033902, 0.043171, 0.022195, 0.000488, 0.001220, 0, 0.000976, 0.002195, 0, 0.006829, 0.006098, 0.006829, 0, 0.004878, 0, 0.034878, 0.003171};
static float DEU_afters_2[] = { 0.096774, 0, 0.021505, 0, 0.064516, 0, 0, 0, 0.129032, 0, 0, 0.032258, 0, 0, 0.118280, 0, 0, 0.043011, 0, 0.301075, 0.193548, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float DEU_afters_3[] = { 0.158529, 0.000543, 0, 0.000109, 0.446289, 0.000760, 0.000760, 0.000109, 0.220920, 0.000109, 0.000217, 0.009440, 0.000326, 0.003581, 0.032878, 0.000217, 0.000109, 0.030490, 0.002387, 0.004557, 0.060764, 0.000434, 0.001302, 0, 0, 0, 0, 0.002170, 0, 0.002604, 0, 0.004015, 0, 0.013997, 0.002387};
static float DEU_afters_4[] = { 0.000608, 0.027397, 0.000465, 0.017776, 0.003648, 0.009550, 0.026539, 0.036339, 0, 0.000036, 0.002933, 0.063200, 0.035230, 0.306806, 0.000429, 0.011982, 0.000429, 0.283308, 0.095819, 0.027469, 0, 0.000393, 0.010265, 0.001788, 0.012626, 0.001466, 0.005043, 0.000036, 0.000429, 0.000143, 0.004220, 0.007940, 0.005580, 0.000107, 0};
static float DEU_afters_5[] = { 0.170068, 0.001984, 0, 0, 0.169501, 0.062358, 0.011621, 0.001417, 0.037132, 0.000283, 0.000283, 0.069728, 0.003968, 0.017290, 0.038265, 0, 0, 0.131519, 0.013039, 0.106859, 0.016440, 0, 0.000850, 0, 0, 0.008503, 0, 0.018707, 0.002834, 0.100340, 0, 0.001417, 0, 0.006236, 0.009354};
static float DEU_afters_6[] = { 0.076174, 0.000160, 0, 0.001278, 0.521239, 0.001437, 0.002076, 0.000160, 0.019962, 0, 0.003673, 0.058448, 0.000319, 0.012456, 0.055893, 0, 0, 0.095497, 0.028106, 0.070265, 0.024752, 0, 0.000160, 0, 0, 0.000319, 0, 0.005749, 0.004312, 0.002395, 0, 0, 0, 0.015011, 0.000160};
static float DEU_afters_7[] = { 0.140203, 0.000156, 0, 0, 0.207182, 0, 0.000156, 0.000156, 0.132865, 0.000156, 0.000156, 0.048556, 0.034348, 0.089774, 0.029820, 0.000156, 0, 0.158626, 0.006401, 0.030289, 0.014520, 0.000156, 0.000312, 0, 0.000625, 0.000468, 0, 0.030133, 0.029820, 0.007026, 0, 0, 0, 0.030601, 0.007338};
static float DEU_afters_8[] = { 0.000474, 0.001152, 0.000881, 0.001626, 0.284650, 0.002981, 0.045048, 0.052635, 0.000271, 0, 0.001355, 0.020255, 0.041864, 0.120309, 0.002100, 0.001761, 0.000135, 0.057851, 0.076616, 0.046538, 0.000542, 0.000271, 0, 0.000135, 0, 0.000339, 0.005555, 0.000135, 0, 0, 0.010026, 0.220160, 0.004335, 0, 0};
static float DEU_afters_9[] = { 0.333333, 0, 0, 0, 0.466667, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.004762, 0, 0, 0, 0, 0, 0.171429, 0, 0, 0, 0, 0, 0, 0.009524, 0, 0.014286, 0, 0, 0, 0, 0};
static float DEU_afters_10[] = { 0.132648, 0.003426, 0, 0, 0.154185, 0, 0, 0.001468, 0.032795, 0, 0, 0.108174, 0, 0.023495, 0.139011, 0, 0, 0.086148, 0.013216, 0.049927, 0.053842, 0, 0, 0, 0, 0.001468, 0, 0.014195, 0.057758, 0.033774, 0, 0.001468, 0, 0.092022, 0.000979};
static float DEU_afters_11[] = { 0.089490, 0.019299, 0, 0.026798, 0.188937, 0.006392, 0.007130, 0.000738, 0.162508, 0, 0.005040, 0.148617, 0.002459, 0.010940, 0.026798, 0.001229, 0, 0.000615, 0.052735, 0.090965, 0.027289, 0.000246, 0.000615, 0, 0.000246, 0.005040, 0, 0.020529, 0.002704, 0.017701, 0, 0.013522, 0, 0.067732, 0.003688};
static float DEU_afters_12[] = { 0.159431, 0.000912, 0, 0.004925, 0.247720, 0.002189, 0.002919, 0.002007, 0.216162, 0, 0.001459, 0.005472, 0.124225, 0.003283, 0.013499, 0.013864, 0, 0.000730, 0.013681, 0.019701, 0.059467, 0, 0.001459, 0, 0, 0.001094, 0, 0.015140, 0.017147, 0.016235, 0, 0.000730, 0, 0.056549, 0};
static float DEU_afters_13[] = { 0.041994, 0.010152, 0.000846, 0.242501, 0.155361, 0.007307, 0.099292, 0.004845, 0.084756, 0.000154, 0.020920, 0.004538, 0.007076, 0.080834, 0.017536, 0.000461, 0.000077, 0.002769, 0.056684, 0.055684, 0.047762, 0.001769, 0.003846, 0, 0, 0.025688, 0, 0.004768, 0.001231, 0.005153, 0.000077, 0.004384, 0.000923, 0.005999, 0.004615};
static float DEU_afters_14[] = { 0.000206, 0.026365, 0.002472, 0.021833, 0.001442, 0.026982, 0.021421, 0.072709, 0.000206, 0, 0.010505, 0.111226, 0.050875, 0.154274, 0.001236, 0.082595, 0, 0.212152, 0.033986, 0.026571, 0.001236, 0.000618, 0.000618, 0.000206, 0, 0.000206, 0.025747, 0, 0, 0, 0.014006, 0.094130, 0.006179, 0, 0};
static float DEU_afters_15[] = { 0.065556, 0, 0, 0, 0.092778, 0.151667, 0, 0.338889, 0.057222, 0, 0, 0.039444, 0, 0, 0.014444, 0.041667, 0, 0.107778, 0.001667, 0.015556, 0.031667, 0, 0, 0, 0.000556, 0.001667, 0, 0.008889, 0.001667, 0.010556, 0, 0.001111, 0, 0.017222, 0};
static float DEU_afters_16[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float DEU_afters_17[] = { 0.083172, 0.020309, 0, 0.047389, 0.183946, 0.018859, 0.038395, 0.017602, 0.064023, 0.000774, 0.020986, 0.025435, 0.019052, 0.043520, 0.035590, 0.003385, 0.000774, 0.030174, 0.067602, 0.076692, 0.040039, 0.003191, 0.012379, 0, 0.000290, 0.025048, 0, 0.017311, 0.004642, 0.032205, 0, 0.018859, 0.000097, 0.035880, 0.012379};
static float DEU_afters_18[] = { 0.043307, 0.001708, 0.000513, 0.000598, 0.134279, 0.001281, 0.005211, 0.002135, 0.139745, 0, 0.001452, 0.001708, 0.001708, 0.000769, 0.069446, 0.029042, 0.000085, 0.002477, 0.042539, 0.289058, 0.006577, 0.000769, 0.001538, 0, 0.000085, 0.001879, 0, 0.004186, 0.000683, 0.002135, 0, 0.181772, 0.000256, 0.032459, 0.000598};
static float DEU_afters_19[] = { 0.051222, 0.002356, 0, 0.000620, 0.469180, 0.003845, 0.003845, 0.047997, 0.056059, 0, 0.000868, 0.012402, 0.005085, 0.002108, 0.054818, 0.000372, 0, 0.068957, 0.030014, 0.100583, 0.020588, 0.001240, 0.008930, 0, 0.000124, 0, 0, 0.009178, 0.005581, 0.023192, 0, 0.006821, 0.000124, 0.007937, 0.005953};
static float DEU_afters_20[] = { 0.001883, 0.014817, 0.000502, 0.007408, 0.023732, 0.090156, 0.025741, 0.016323, 0.002888, 0.000126, 0.002762, 0.011678, 0.075967, 0.359618, 0.000251, 0.004520, 0, 0.100954, 0.117906, 0.047840, 0.000126, 0.000502, 0.001381, 0, 0.000628, 0.001256, 0.028880, 0.000879, 0, 0.000126, 0.005399, 0.052737, 0.002888, 0.000126, 0};
static float DEU_afters_21[] = { 0.076659, 0, 0, 0, 0.376430, 0, 0, 0, 0.103547, 0, 0, 0, 0, 0, 0.430778, 0, 0, 0.001144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.002860, 0.008581, 0, 0, 0, 0, 0, 0};
static float DEU_afters_22[] = { 0.228232, 0, 0, 0, 0.230700, 0, 0, 0, 0.253366, 0, 0, 0, 0, 0, 0.107496, 0, 0, 0, 0, 0, 0.026706, 0, 0, 0, 0, 0, 0, 0.040395, 0.010548, 0.023564, 0, 0, 0, 0.078995, 0};
static float DEU_afters_23[] = { 0, 0, 0, 0, 0.826923, 0, 0, 0, 0.057692, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.096154, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.019231, 0, 0, 0};
static float DEU_afters_24[] = { 0, 0, 0, 0.177033, 0.229665, 0.009569, 0.009569, 0.033493, 0.014354, 0, 0, 0.047847, 0.095694, 0.224880, 0, 0.004785, 0, 0.019139, 0.057416, 0.066986, 0, 0, 0, 0, 0, 0.004785, 0.004785, 0, 0, 0, 0, 0, 0, 0, 0};
static float DEU_afters_25[] = { 0.024138, 0.000431, 0, 0.000431, 0.110345, 0, 0.000431, 0.000862, 0.116810, 0, 0, 0.012500, 0.000431, 0.000431, 0.015517, 0.000431, 0, 0, 0.000431, 0.020690, 0.543103, 0, 0.055603, 0, 0, 0.000862, 0, 0.004741, 0.001724, 0.003448, 0, 0, 0, 0.078017, 0.008621};
static float DEU_afters_26[] = { 0.001880, 0.015038, 0, 0, 0.428571, 0.001880, 0, 0.007519, 0.030075, 0, 0, 0.026316, 0.013158, 0.001880, 0, 0, 0, 0.005639, 0.005639, 0.434211, 0, 0.009398, 0.001880, 0, 0, 0, 0, 0, 0, 0, 0, 0.016917, 0, 0, 0};
static float DEU_afters_27[] = { 0, 0.008397, 0, 0.034351, 0, 0.049618, 0.036641, 0.100763, 0, 0, 0.000763, 0.064122, 0.023664, 0.198473, 0, 0.001527, 0, 0.134351, 0.028244, 0.104580, 0.086260, 0, 0, 0, 0, 0, 0.030534, 0, 0, 0, 0.002290, 0.079389, 0.016031, 0, 0};
static float DEU_afters_28[] = { 0, 0.022305, 0, 0.014870, 0, 0.063197, 0.064436, 0.086741, 0, 0, 0, 0.052045, 0.012392, 0.232962, 0, 0.024783, 0, 0.234201, 0.052045, 0.039653, 0, 0, 0.002478, 0, 0, 0, 0.024783, 0, 0, 0, 0.008674, 0.055762, 0.008674, 0, 0};
static float DEU_afters_29[] = { 0, 0.163391, 0, 0.004492, 0, 0.008984, 0.034250, 0.151039, 0, 0, 0, 0.037058, 0.010107, 0.064009, 0, 0.003930, 0, 0.245929, 0.048849, 0.022459, 0, 0, 0, 0, 0.000561, 0, 0.038742, 0, 0, 0, 0.133071, 0.019090, 0.014037, 0, 0};
static float DEU_afters_30[] = { 0.002128, 0.002128, 0, 0.002128, 0.410638, 0, 0.023404, 0.006383, 0.008511, 0, 0.010638, 0.068085, 0, 0.012766, 0, 0, 0, 0.002128, 0.082979, 0.314894, 0.014894, 0, 0.012766, 0, 0, 0.017021, 0, 0, 0, 0, 0, 0.008511, 0, 0, 0};
static float DEU_afters_31[] = { 0.037972, 0.003547, 0, 0.002086, 0.217400, 0.001460, 0.002921, 0.004799, 0.021698, 0, 0.006259, 0.063009, 0.022950, 0.014605, 0.049238, 0, 0.000209, 0.029001, 0.021072, 0.355936, 0.008554, 0.000417, 0.068642, 0, 0, 0.005007, 0, 0.013979, 0.019612, 0.008971, 0, 0.001043, 0.000209, 0.016691, 0.002712};
static float DEU_afters_32[] = { 0, 0, 0, 0.051205, 0.319277, 0, 0, 0, 0.033133, 0, 0, 0.054217, 0, 0, 0.006024, 0, 0, 0, 0.003012, 0.469880, 0.015060, 0, 0.012048, 0, 0, 0, 0, 0, 0, 0.024096, 0, 0.006024, 0, 0.003012, 0.003012};
static float DEU_afters_33[] = { 0, 0.032644, 0, 0.025834, 0.003992, 0.015500, 0.025599, 0.007750, 0, 0, 0.000235, 0.037341, 0.012447, 0.565054, 0, 0.000939, 0, 0.000705, 0.064349, 0.115782, 0, 0.000705, 0.000470, 0, 0, 0.001174, 0.018084, 0, 0, 0, 0, 0.071160, 0.000235, 0, 0};
static float DEU_afters_34[] = { 0.001715, 0, 0, 0.063465, 0.164666, 0.087479, 0.068611, 0, 0.005146, 0, 0, 0.017153, 0.001715, 0.080617, 0, 0, 0, 0.063465, 0.012007, 0.140652, 0.005146, 0, 0, 0, 0, 0.010292, 0, 0, 0, 0, 0, 0.276158, 0.001715, 0, 0};
static struct letter_frequencies DEU_counts[] = {
    { "a", 0.011990, 0.060185, 0.069880, 0.006894, DEU_afters_0 },
    { "b", 0.004796, 0.035694, 0.017542, 0.006734, DEU_afters_1 },
    { "c", 0.002398, 0.000521, 0.000507, 0.000040, DEU_afters_2 },
    { "d", 0.014388, 0.134460, 0.018964, 0.049583, DEU_afters_3 },
    { "e", 0.014388, 0.043731, 0.194988, 0.136123, DEU_afters_4 },
    { "f", 0.007194, 0.033911, 0.013888, 0.013648, DEU_afters_5 },
    { "g", 0.047962, 0.056898, 0.025893, 0.022507, DEU_afters_6 },
    { "h", 0.004796, 0.045254, 0.031370, 0.005672, DEU_afters_7 },
    { "i", 0.009592, 0.061147, 0.088587, 0.000100, DEU_afters_8 },
    { "j", 0.002398, 0.008117, 0.000113, 0.000000, DEU_afters_9 },
    { "k", 0.002398, 0.026074, 0.005613, 0.001423, DEU_afters_10 },
    { "l", 0.007194, 0.024010, 0.052474, 0.020443, DEU_afters_11 },
    { "m", 0.009592, 0.061067, 0.018419, 0.039342, DEU_afters_12 },
    { "n", 0.033573, 0.043090, 0.082089, 0.216811, DEU_afters_13 },
    { "o", 0.122302, 0.005331, 0.034713, 0.009840, DEU_afters_14 },
    { "p", 0.002398, 0.007115, 0.010931, 0.000220, DEU_afters_15 },
    { "q", 0.000000, 0.000802, 0.000197, 0.000000, DEU_afters_16 },
    { "r", 0.014388, 0.012446, 0.073519, 0.143599, DEU_afters_17 },
    { "s", 0.604317, 0.109408, 0.047262, 0.073373, DEU_afters_18 },
    { "t", 0.064748, 0.018819, 0.053889, 0.130632, DEU_afters_19 },
    { "u", 0.002398, 0.045114, 0.043215, 0.023369, DEU_afters_20 },
    { "v", 0.002398, 0.031906, 0.001180, 0.000100, DEU_afters_21 },
    { "w", 0.002398, 0.067821, 0.008109, 0.000120, DEU_afters_22 },
    { "x", 0.000000, 0.000020, 0.000386, 0.000080, DEU_afters_23 },
    { "y", 0.000000, 0.000000, 0.001581, 0.003768, DEU_afters_24 },
    { "z", 0.002398, 0.033309, 0.004977, 0.003728, DEU_afters_25 },
    { "ß", 0.000000, 0.000000, 0.004024, 0.012867, DEU_afters_26 },
    { "ä", 0.000000, 0.000621, 0.009675, 0.000000, DEU_afters_27 },
    { "ö", 0.000000, 0.000561, 0.005893, 0.000000, DEU_afters_28 },
    { "ü", 0.004796, 0.004169, 0.011899, 0.000020, DEU_afters_29 },
    { "ck", 0.000000, 0.000000, 0.003555, 0.004008, DEU_afters_30 },
    { "ch", 0.000000, 0.001082, 0.035848, 0.071248, DEU_afters_31 },
    { "tz", 0.000000, 0.000000, 0.002511, 0.001243, DEU_afters_32 },
    { "ei", 0.000000, 0.023810, 0.023223, 0.002104, DEU_afters_33 },
    { "eu", 0.000000, 0.003507, 0.003086, 0.000361, DEU_afters_34 },
    NULL
};	/* DEU */
static float DEU_word_lens[] = {
    0.000000, 0.008288, 0.118916, 0.312186, 0.150021, 0.131318, 0.107050, 0.053048, 0.038241, 0.029376, 0.018305, 0.012780, 0.007175, 0.004134, 0.007374, 0.000656, 0.000358, 0.000437, 0.000258, 0.000040, 0.000020, 0.000020, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float DEU_all_consonants[] = {
    0.000000, 0.006877, 0.000139, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 
};

static float DEU_consonant_run[] = {
    0.000000, 0.686703, 0.263751, 0.043695, 0.005309, 0.000533, 0.000009, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float DEU_vowel_run[] = {
    0.000000, 0.905405, 0.092181, 0.002413, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float ELL_afters_0[] = { 0, 0, 0, 0, 0, 0.976190, 0, 0, 0, 0, 0, 0, 0, 0, 0.023810, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ELL_afters_1[] = { 0, 0, 0, 0, 0, 0, 0.006857, 0.023587, 0.035930, 0.006034, 0.026056, 0.000549, 0.086945, 0.007954, 0.043335, 0.198300, 0.050741, 0.120132, 0.018651, 0, 0.023039, 0.056500, 0.030170, 0.047175, 0.149205, 0.000823, 0.029622, 0.034558, 0.002468, 0.001371, 0, 0, 0, 0};
static float ELL_afters_2[] = { 0, 0, 0, 0, 0, 0.001224, 0.038556, 0.010608, 0.016524, 0.001224, 0.008160, 0, 0.003264, 0, 0.031824, 0.050592, 0.058344, 0.142187, 0.044472, 0.001020, 0.102611, 0.116891, 0.023460, 0.084047, 0.044064, 0, 0.207670, 0.008772, 0.004284, 0.000204, 0, 0, 0, 0};
static float ELL_afters_3[] = { 0, 0, 0, 0, 0, 0, 0, 0.089897, 0.005137, 0, 0, 0, 0.183219, 0, 0.127568, 0.029110, 0.045377, 0.010274, 0.029966, 0, 0.015411, 0.130993, 0.012842, 0.110445, 0.198630, 0, 0, 0.011130, 0, 0, 0, 0, 0, 0};
static float ELL_afters_4[] = { 0, 0, 0, 0, 0, 0.231830, 0.000197, 0.085877, 0.055545, 0.009060, 0.010242, 0, 0.002364, 0, 0.121922, 0.026591, 0.020091, 0.068741, 0.005712, 0.007485, 0.099271, 0.004530, 0.010242, 0.051014, 0.114438, 0, 0.005712, 0.063226, 0.000985, 0.004924, 0, 0, 0, 0};
static float ELL_afters_5[] = { 0.011865, 0.000141, 0.004238, 0.000636, 0.009252, 0, 0.002966, 0.046684, 0.007275, 0.001766, 0.029522, 0.000424, 0.016032, 0.181298, 0.020482, 0.032771, 0.035101, 0.260188, 0.024578, 0.000212, 0.058620, 0.071050, 0.031923, 0.042941, 0.058761, 0.016809, 0.020411, 0.008546, 0.002825, 0, 0.000353, 0.000494, 0.001836, 0};
static float ELL_afters_6[] = { 0, 0.076484, 0.013699, 0.050228, 0.020548, 0.149543, 0, 0.099315, 0, 0.039954, 0, 0.092466, 0, 0.025114, 0, 0.155251, 0, 0, 0, 0.085616, 0, 0.154110, 0, 0, 0, 0.007991, 0, 0, 0, 0.003425, 0, 0.017123, 0, 0.009132};
static float ELL_afters_7[] = { 0, 0.065311, 0.026926, 0.019479, 0.005443, 0.063019, 0, 0.028072, 0.000573, 0.088227, 0, 0.006302, 0, 0.233171, 0.128330, 0.061300, 0.057863, 0.011172, 0, 0.055285, 0, 0.018046, 0, 0, 0, 0.017474, 0, 0.011744, 0, 0.008021, 0, 0.023203, 0.067030, 0.004010};
static float ELL_afters_8[] = { 0, 0.017611, 0.021464, 0.002752, 0.027518, 0.045680, 0, 0, 0, 0.468905, 0, 0.015960, 0, 0.182719, 0, 0, 0, 0, 0, 0.031370, 0, 0.040176, 0, 0, 0, 0.086957, 0, 0, 0, 0.017061, 0, 0.011558, 0.010457, 0.019813};
static float ELL_afters_9[] = { 0, 0, 0, 0, 0.221281, 0.000862, 0.003016, 0.019960, 0.006749, 0.000287, 0.002728, 0, 0.000718, 0.206491, 0.026852, 0.033458, 0.031447, 0.092188, 0.012924, 0.000144, 0.015939, 0.093194, 0.054566, 0.018524, 0.016657, 0.092763, 0.016370, 0.008329, 0.009190, 0.000287, 0, 0, 0.011344, 0.003733};
static float ELL_afters_10[] = { 0, 0.041575, 0.008753, 0.012035, 0.056893, 0.163020, 0, 0, 0, 0.518600, 0, 0.013129, 0, 0.056893, 0, 0, 0, 0, 0, 0.090810, 0, 0, 0, 0, 0, 0.004376, 0, 0, 0, 0.017505, 0, 0.010941, 0, 0.005470};
static float ELL_afters_11[] = { 0, 0, 0, 0, 0, 0, 0, 0.010201, 0.003967, 0, 0, 0, 0.013885, 0, 0.167186, 0.022669, 0.033721, 0.241145, 0.007651, 0, 0, 0.017569, 0.228393, 0.211958, 0.036554, 0, 0.000283, 0.004817, 0, 0, 0, 0, 0, 0};
static float ELL_afters_12[] = { 0, 0.053051, 0.031113, 0.015556, 0.201835, 0.063422, 0, 0, 0, 0.185082, 0, 0.138014, 0, 0.050658, 0, 0.002792, 0.001994, 0, 0, 0.029118, 0, 0.009174, 0, 0, 0, 0.075389, 0, 0, 0, 0.003590, 0, 0.021141, 0.067411, 0.050658};
static float ELL_afters_13[] = { 0, 0.113876, 0.004095, 0, 0, 0.221295, 0.002678, 0.061585, 0.011970, 0.003938, 0.029926, 0.001733, 0.026303, 0.001418, 0.038116, 0.014333, 0.014333, 0.107418, 0.021893, 0.047724, 0.002835, 0.014333, 0.036856, 0.111829, 0.064420, 0, 0.009765, 0.009450, 0.000630, 0.016853, 0, 0.003780, 0, 0.006615};
static float ELL_afters_14[] = { 0, 0.059785, 0.005373, 0.006287, 0.019433, 0.413123, 0, 0, 0, 0.091449, 0, 0.001257, 0, 0.038294, 0.011545, 0.016004, 0, 0.001029, 0, 0.125972, 0, 0.036465, 0, 0.000229, 0.001715, 0.072588, 0, 0, 0, 0.051097, 0, 0.028121, 0.015089, 0.005144};
static float ELL_afters_15[] = { 0, 0.076578, 0.031928, 0.023946, 0.027189, 0.195311, 0, 0, 0, 0.114742, 0, 0.081317, 0, 0.146919, 0.003991, 0.082564, 0.000998, 0.000748, 0, 0.117486, 0.000748, 0, 0, 0, 0.003492, 0.006984, 0.000499, 0, 0, 0.030681, 0, 0.031429, 0.012971, 0.009479};
static float ELL_afters_16[] = { 0, 0.086285, 0.098519, 0.031230, 0.015454, 0.272859, 0.000322, 0, 0, 0.158403, 0, 0.027366, 0, 0.085802, 0.000161, 0, 0.013683, 0.003542, 0, 0.065518, 0.073406, 0, 0, 0, 0, 0.002093, 0.000322, 0, 0.004507, 0.013522, 0, 0.041372, 0.002415, 0.003220};
static float ELL_afters_17[] = { 0, 0.023505, 0.009271, 0.016976, 0.006529, 0.282711, 0.000131, 0, 0.000261, 0.115174, 0, 0.059415, 0.069209, 0.041656, 0, 0, 0, 0.007704, 0, 0.208279, 0, 0, 0, 0.000131, 0.094280, 0.000914, 0, 0, 0, 0.017367, 0, 0.028598, 0.003134, 0.014756};
static float ELL_afters_18[] = { 0, 0.006494, 0.059740, 0.003247, 0.005844, 0.348701, 0, 0, 0, 0.461688, 0, 0.007792, 0, 0.018182, 0, 0, 0, 0, 0, 0.005844, 0, 0, 0, 0, 0, 0.007792, 0, 0, 0, 0.064935, 0, 0.000649, 0.008442, 0.000649};
static float ELL_afters_19[] = { 0, 0, 0.003014, 0.003836, 0.072877, 0.000548, 0.005616, 0.007397, 0.002466, 0.000411, 0, 0.000822, 0.001096, 0.062329, 0.015753, 0.025753, 0.027260, 0.118904, 0.000274, 0.000274, 0.005753, 0.040000, 0.146027, 0.023151, 0.012466, 0.312740, 0.001370, 0.012055, 0.000685, 0, 0, 0, 0.097123, 0};
static float ELL_afters_20[] = { 0, 0.092561, 0.028818, 0.029601, 0.026938, 0.118089, 0, 0, 0, 0.182302, 0, 0.020673, 0, 0.026625, 0, 0.036022, 0, 0.010963, 0, 0.165074, 0.012529, 0.108379, 0, 0, 0.000157, 0.002036, 0, 0, 0, 0.048551, 0, 0.080971, 0.001096, 0.008614};
static float ELL_afters_21[] = { 0, 0.059022, 0.023609, 0.012788, 0.181563, 0.136594, 0.000703, 0.012788, 0.002248, 0.070545, 0.000141, 0.013912, 0.032884, 0.151630, 0.001265, 0.000141, 0.006745, 0.011804, 0, 0.093170, 0.005059, 0.002530, 0, 0, 0.020517, 0.013772, 0.009415, 0.021079, 0, 0.022063, 0, 0.060006, 0.007589, 0.026419};
static float ELL_afters_23[] = { 0, 0.006670, 0.002808, 0.009244, 0.008776, 0.122396, 0.003861, 0.000234, 0.000117, 0.246197, 0, 0.015914, 0.006436, 0.039434, 0.038966, 0, 0.027966, 0, 0, 0.021297, 0.014159, 0, 0, 0.015797, 0.353499, 0.013106, 0.005032, 0.004915, 0, 0.024339, 0, 0.002925, 0.009127, 0.006787};
static float ELL_afters_24[] = { 0, 0.042071, 0.071215, 0.013293, 0.039004, 0.206778, 0, 0, 0, 0.044993, 0.002922, 0.195822, 0, 0.069754, 0, 0, 0.000073, 0, 0, 0.226718, 0, 0.039296, 0, 0.006208, 0.000073, 0.004017, 0, 0, 0, 0.011833, 0, 0.014827, 0.003287, 0.007815};
static float ELL_afters_25[] = { 0, 0.002193, 0, 0.000627, 0, 0.159148, 0.007832, 0.032268, 0.011905, 0.011278, 0.000627, 0, 0.008459, 0, 0.014098, 0.037594, 0.047619, 0.072368, 0, 0.033835, 0.025689, 0.255952, 0.089599, 0.046679, 0.077694, 0, 0.014098, 0.024123, 0.024123, 0.000313, 0, 0.000940, 0, 0.000940};
static float ELL_afters_26[] = { 0, 0.042674, 0.018882, 0.016616, 0.029079, 0.354985, 0, 0, 0, 0.058535, 0, 0.023414, 0.007175, 0.040785, 0, 0.008308, 0, 0.064577, 0, 0.069864, 0, 0.011707, 0, 0, 0.052870, 0.034366, 0, 0, 0, 0.071752, 0, 0.020015, 0.035121, 0.039275};
static float ELL_afters_27[] = { 0, 0.039430, 0.090361, 0.011501, 0.018072, 0.168127, 0, 0, 0, 0.163198, 0, 0.068456, 0.003834, 0.050383, 0, 0.029573, 0.000548, 0.043264, 0, 0.058598, 0, 0.047097, 0, 0, 0.128149, 0.008762, 0, 0.003834, 0, 0.036145, 0, 0.010405, 0.003834, 0.016429};
static float ELL_afters_28[] = { 0, 0.021654, 0.005906, 0.001969, 0.003937, 0.066929, 0, 0, 0, 0.362205, 0, 0.135827, 0, 0.340551, 0, 0, 0, 0, 0, 0.007874, 0, 0, 0, 0, 0, 0.017717, 0, 0, 0, 0.019685, 0, 0.003937, 0.001969, 0.009843};
static float ELL_afters_29[] = { 0, 0, 0, 0, 0.012295, 0, 0.000683, 0.003415, 0.001366, 0, 0, 0.005464, 0.026639, 0.004098, 0.001366, 0.001366, 0.077869, 0.157787, 0.002049, 0, 0.074454, 0.084699, 0.344262, 0.123634, 0.049180, 0, 0.004781, 0.024590, 0, 0, 0, 0, 0, 0};
static float ELL_afters_30[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0.200000, 0, 0, 0, 0, 0, 0, 0, 0.200000, 0.200000, 0, 0, 0, 0, 0, 0, 0.400000, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ELL_afters_31[] = { 0, 0, 0, 0, 0, 0, 0.007065, 0.018841, 0.025907, 0.000942, 0.001413, 0.005181, 0.000471, 0.001413, 0.032030, 0.057466, 0.095148, 0.158267, 0.002355, 0, 0.048987, 0.091380, 0.104569, 0.096090, 0.196420, 0, 0.023552, 0.020725, 0.011776, 0, 0, 0, 0, 0};
static float ELL_afters_32[] = { 0, 0, 0, 0, 0, 0.011897, 0.005636, 0.029430, 0.021916, 0.006888, 0.001252, 0, 0.011271, 0.002505, 0.041954, 0.050094, 0.035066, 0.051972, 0, 0.004383, 0.017533, 0.351284, 0.046337, 0.206011, 0.066374, 0, 0.010019, 0.020038, 0.005009, 0.003131, 0, 0, 0, 0};
static float ELL_afters_33[] = { 0, 0, 0, 0, 0, 0.001056, 0, 0.001056, 0.001056, 0, 0, 0, 0.053854, 0.001056, 0, 0.006336, 0.055966, 0.229145, 0.001056, 0.003168, 0.053854, 0.176346, 0.168955, 0.050686, 0.192186, 0, 0.001056, 0.003168, 0, 0, 0, 0, 0, 0};
static struct letter_frequencies ELL_counts[] = {
    { "ΐ", 0.000000, 0.000000, 0.001601, 0.000000, ELL_afters_0 },
    { "ά", 0.000000, 0.014387, 0.029789, 0.041502, ELL_afters_1 },
    { "έ", 0.000000, 0.057881, 0.026777, 0.001328, ELL_afters_2 },
    { "ή", 0.005941, 0.013447, 0.006499, 0.016905, ELL_afters_3 },
    { "ί", 0.000330, 0.002130, 0.047647, 0.013447, ELL_afters_4 },
    { "α", 0.004290, 0.055336, 0.115867, 0.208920, ELL_afters_5 },
    { "β", 0.000000, 0.012478, 0.004050, 0.000055, ELL_afters_6 },
    { "γ", 0.000000, 0.029245, 0.023194, 0.000000, ELL_afters_7 },
    { "δ", 0.000000, 0.025067, 0.008681, 0.000000, ELL_afters_8 },
    { "ε", 0.004950, 0.056055, 0.047056, 0.172454, ELL_afters_9 },
    { "ζ", 0.000000, 0.001466, 0.008205, 0.000028, ELL_afters_10 },
    { "η", 0.517822, 0.000553, 0.033438, 0.036051, ELL_afters_11 },
    { "θ", 0.001320, 0.014138, 0.019021, 0.000000, ELL_afters_12 },
    { "ι", 0.000660, 0.001107, 0.060121, 0.118086, ELL_afters_13 },
    { "κ", 0.107591, 0.142434, 0.034306, 0.000000, ELL_afters_14 },
    { "λ", 0.000000, 0.009684, 0.034868, 0.000055, ELL_afters_15 },
    { "μ", 0.008581, 0.091249, 0.027769, 0.000000, ELL_afters_16 },
    { "ν", 0.016832, 0.031901, 0.061988, 0.092770, ELL_afters_17 },
    { "ξ", 0.000000, 0.015909, 0.009196, 0.000000, ELL_afters_18 },
    { "ο", 0.293069, 0.008826, 0.066524, 0.078936, ELL_afters_19 },
    { "π", 0.000000, 0.088869, 0.030237, 0.000194, ELL_afters_20 },
    { "ρ", 0.000000, 0.006170, 0.065686, 0.000083, ELL_afters_21 },
    { "ς", 0.000000, 0.000000, 0.000000, 0.123841, NULL },
    { "σ", 0.019142, 0.112802, 0.042587, 0.000028, ELL_afters_23 },
    { "τ", 0.013201, 0.144620, 0.080656, 0.000194, ELL_afters_24 },
    { "υ", 0.000000, 0.001854, 0.029779, 0.042968, ELL_afters_25 },
    { "φ", 0.000000, 0.021692, 0.017763, 0.000000, ELL_afters_26 },
    { "χ", 0.000330, 0.015494, 0.012064, 0.000360, ELL_afters_27 },
    { "ψ", 0.000000, 0.006142, 0.002725, 0.000000, ELL_afters_28 },
    { "ω", 0.003300, 0.002324, 0.013150, 0.023324, ELL_afters_29 },
    { "ϊ", 0.000000, 0.000000, 0.000048, 0.000000, ELL_afters_30 },
    { "ό", 0.002640, 0.013308, 0.015647, 0.018454, ELL_afters_31 },
    { "ύ", 0.000000, 0.001356, 0.014751, 0.005783, ELL_afters_32 },
    { "ώ", 0.000000, 0.002075, 0.008310, 0.004233, ELL_afters_33 },
    NULL
};	/* ELL */
static float ELL_word_lens[] = {
    0.000000, 0.077349, 0.120951, 0.213106, 0.131162, 0.099890, 0.128354, 0.103081, 0.067061, 0.028361, 0.014193, 0.009267, 0.003931, 0.002119, 0.000817, 0.000332, 0.000026, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float ELL_all_consonants[] = {
    0.000000, 0.052970, 0.010798, 0.037194, 0.005974, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 
};

static float ELL_consonant_run[] = {
    0.000000, 0.791628, 0.150499, 0.047080, 0.010342, 0.000425, 0.000027, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float ELL_vowel_run[] = {
    0.000000, 0.791348, 0.192927, 0.015444, 0.000267, 0.000014, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float IWR_afters_0[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.363636, 0, 0, 0, 0.090909, 0.090909, 0, 0, 0, 0.090909, 0.181818, 0, 0, 0.090909, 0, 0, 0, 0, 0.090909, 0};
static float IWR_afters_1[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.012987, 0.051948, 0.038961, 0.012987, 0, 0.064935, 0, 0.025974, 0, 0.233766, 0, 0.064935, 0.129870, 0, 0.051948, 0, 0.012987, 0.025974, 0, 0, 0.012987, 0, 0.012987, 0.064935, 0.025974, 0.064935, 0.090909};
static float IWR_afters_2[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.045455, 0.060606, 0, 0.030303, 0.030303, 0, 0, 0.060606, 0, 0.318182, 0, 0.015152, 0.045455, 0.015152, 0.030303, 0, 0.090909, 0, 0.090909, 0, 0.030303, 0, 0.015152, 0.030303, 0.090909, 0, 0};
static float IWR_afters_3[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.020000, 0.060000, 0.020000, 0.040000, 0.150000, 0.010000, 0, 0.010000, 0.050000, 0, 0, 0.020000, 0.080000, 0, 0.060000, 0.110000, 0.150000, 0.020000, 0, 0, 0.020000, 0, 0, 0.010000, 0.170000, 0, 0};
static float IWR_afters_4[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.023077, 0.046154, 0.011538, 0.030769, 0.057692, 0.030769, 0.030769, 0.042308, 0.011538, 0.169231, 0.003846, 0.023077, 0.084615, 0, 0.042308, 0.019231, 0.134615, 0.023077, 0.007692, 0.019231, 0.007692, 0, 0.007692, 0.023077, 0.096154, 0.034615, 0.019231};
static float IWR_afters_5[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.094118, 0.023529, 0.035294, 0.058824, 0.070588, 0.035294, 0.023529, 0.035294, 0, 0.035294, 0.011765, 0.023529, 0.082353, 0.011765, 0.070588, 0.035294, 0.105882, 0.011765, 0.047059, 0, 0.047059, 0, 0, 0.035294, 0.035294, 0.035294, 0.035294};
static float IWR_afters_6[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.021978, 0.032967, 0.010989, 0.054945, 0, 0.021978, 0, 0.032967, 0.021978, 0.010989, 0, 0.043956, 0.153846, 0.010989, 0.164835, 0.054945, 0.153846, 0.010989, 0.032967, 0, 0.010989, 0, 0, 0.021978, 0.076923, 0.043956, 0.010989};
static float IWR_afters_7[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.090909, 0, 0, 0, 0.045455, 0, 0.136364, 0.181818, 0, 0.090909, 0, 0.090909, 0, 0, 0, 0.045455, 0, 0, 0, 0.181818, 0.136364, 0};
static float IWR_afters_8[] = { 0.037879, 0.075758, 0.030303, 0.037879, 0.196970, 0.075758, 0.007576, 0.007576, 0, 0, 0, 0.022727, 0.045455, 0.015152, 0, 0.007576, 0.037879, 0.007576, 0.007576, 0.015152, 0.030303, 0, 0, 0.098485, 0, 0.022727, 0.007576, 0.022727, 0.007576, 0.022727, 0, 0.022727, 0, 0, 0.015152, 0.113636, 0.007576, 0};
static float IWR_afters_9[] = { 0, 0.225000, 0.150000, 0.075000, 0.325000, 0.100000, 0, 0.050000, 0.025000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.025000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.025000, 0, 0};
static float IWR_afters_10[] = { 0, 0.062500, 0, 0, 0, 0.062500, 0, 0, 0, 0, 0, 0, 0.062500, 0.062500, 0, 0, 0.125000, 0, 0, 0, 0.125000, 0, 0.125000, 0, 0.125000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.187500, 0.062500, 0};
static float IWR_afters_11[] = { 0, 0.002260, 0.004931, 0.001438, 0.009657, 0.002055, 0.005958, 0.001849, 0, 0, 0, 0.000411, 0.020957, 0.005958, 0.026094, 0.027943, 0.115677, 0.007808, 0.068215, 0.008013, 0.107253, 0.020547, 0.011712, 0.092870, 0.027327, 0.042737, 0.000616, 0.055476, 0.004315, 0.003698, 0.010273, 0.020957, 0.000205, 0.014177, 0.007191, 0.023423, 0.079926, 0.168071};
static float IWR_afters_12[] = { 0.000269, 0, 0.001075, 0.001075, 0.000538, 0.000269, 0.000269, 0.000269, 0.007527, 0, 0, 0.041935, 0.020161, 0.011559, 0.029839, 0.050000, 0.092204, 0.009946, 0.017204, 0.013978, 0.156183, 0.003226, 0.036290, 0.066667, 0.000269, 0.042204, 0.012097, 0.017742, 0.006989, 0.063441, 0, 0.012097, 0.000269, 0.012366, 0.048925, 0.134677, 0.037366, 0.051075};
static float IWR_afters_13[] = { 0, 0, 0, 0.004614, 0.006920, 0, 0.003460, 0, 0.006920, 0, 0, 0.009227, 0.058824, 0.003460, 0.070358, 0.018454, 0.123414, 0.023068, 0.009227, 0, 0.089965, 0, 0, 0.138408, 0.092272, 0.047290, 0.014994, 0.025375, 0.005767, 0.076125, 0, 0.001153, 0, 0, 0, 0.061130, 0.102653, 0.006920};
static float IWR_afters_14[] = { 0, 0.001347, 0.000673, 0.008081, 0.000673, 0.000673, 0, 0.000673, 0, 0, 0, 0.026263, 0.126599, 0.000673, 0.010101, 0.077441, 0.165657, 0, 0.011448, 0.003367, 0.171044, 0.002694, 0.001347, 0.037037, 0.019529, 0.051852, 0.002694, 0.017508, 0.010101, 0.074747, 0.004040, 0.016835, 0, 0, 0.012795, 0.080808, 0.012795, 0.050505};
static float IWR_afters_15[] = { 0, 0.000472, 0.000472, 0.001415, 0.006368, 0.001179, 0, 0.000236, 0.000236, 0, 0, 0.047877, 0.043632, 0.025708, 0.028774, 0.011321, 0.094104, 0.035142, 0.038915, 0.003538, 0.112500, 0, 0.028066, 0.040094, 0.013915, 0.078774, 0.016274, 0.055189, 0.023821, 0.044575, 0, 0.040330, 0, 0.016745, 0.020047, 0.053302, 0.071934, 0.045047};
static float IWR_afters_16[] = { 0, 0.000303, 0.000455, 0.000303, 0, 0.001516, 0.007884, 0.000303, 0.011674, 0, 0, 0.117192, 0.060036, 0.008945, 0.054730, 0.026380, 0.001364, 0.003942, 0.030473, 0.013190, 0.033505, 0.009400, 0.023196, 0.078381, 0.025773, 0.035628, 0.040782, 0.044421, 0.013796, 0.028805, 0.006822, 0.021680, 0.003184, 0.016677, 0.015464, 0.067314, 0.029260, 0.167223};
static float IWR_afters_17[] = { 0, 0, 0, 0.006969, 0.002323, 0.004646, 0, 0, 0, 0, 0, 0.142857, 0.036005, 0.006969, 0.004646, 0.420441, 0.072009, 0, 0, 0, 0.032520, 0.003484, 0.031359, 0.018583, 0, 0.053426, 0.001161, 0.016260, 0, 0.020906, 0, 0.001161, 0, 0, 0.044135, 0.073171, 0, 0.006969};
static float IWR_afters_18[] = { 0, 0.002161, 0.000540, 0.002701, 0.002701, 0.001621, 0, 0, 0, 0, 0, 0.001621, 0.023771, 0.001080, 0.085359, 0.042139, 0.150729, 0.028633, 0.000540, 0.005943, 0.088061, 0.003241, 0.010265, 0.099406, 0.010805, 0.034036, 0.015667, 0.021070, 0.011345, 0, 0.003782, 0.029714, 0.002161, 0.019989, 0.019449, 0.122636, 0.068071, 0.090762};
static float IWR_afters_19[] = { 0, 0.007722, 0, 0.003861, 0.013514, 0.003861, 0, 0, 0, 0, 0, 0.007722, 0.042471, 0, 0, 0.113900, 0.243243, 0, 0.025097, 0.015444, 0.142857, 0, 0, 0.027027, 0.001931, 0.007722, 0.017375, 0.042471, 0.001931, 0.021236, 0.003861, 0.048263, 0, 0, 0, 0.129344, 0.009653, 0.069498};
static float IWR_afters_20[] = { 0, 0.000205, 0.000410, 0.000205, 0.002255, 0.000615, 0, 0, 0.000820, 0, 0, 0.023990, 0.017429, 0.004921, 0.045315, 0.056592, 0.113594, 0.022350, 0.015993, 0.015173, 0.021325, 0.017429, 0.025220, 0.043264, 0.179414, 0.013738, 0.040394, 0.046340, 0.010867, 0.029731, 0.005741, 0.010047, 0.001435, 0.011072, 0.016814, 0.064179, 0.047365, 0.095756};
static float IWR_afters_22[] = { 0, 0, 0, 0.000407, 0.002849, 0, 0, 0.000407, 0.005291, 0, 0, 0.031746, 0.065934, 0.000814, 0.010582, 0.052910, 0.055759, 0.019129, 0.024827, 0, 0.144078, 0.034595, 0.007733, 0.199023, 0.001628, 0.044363, 0.049654, 0.024420, 0.014652, 0.016687, 0.014652, 0.027676, 0, 0.000814, 0, 0.049654, 0.010989, 0.088726};
static float IWR_afters_23[] = { 0.000442, 0.000664, 0.000442, 0.002434, 0.003319, 0.001106, 0, 0, 0.000442, 0, 0, 0.123673, 0.051991, 0.007965, 0.014381, 0.117478, 0.093805, 0.006858, 0.038717, 0.013274, 0.201106, 0.025442, 0.027655, 0.026106, 0.019912, 0.047788, 0.000442, 0.007522, 0.006416, 0.029425, 0.003097, 0.019690, 0.000221, 0.006858, 0.012832, 0.010619, 0.024779, 0.053097};
static float IWR_afters_25[] = { 0.000521, 0.001822, 0.000781, 0.001041, 0.006767, 0.001822, 0, 0, 0, 0, 0, 0.047892, 0.030713, 0.011713, 0.046851, 0.083811, 0.100208, 0.008850, 0.036439, 0.013274, 0.090057, 0.002343, 0.023165, 0.033837, 0.002082, 0.024466, 0.019521, 0.033576, 0.020562, 0.076783, 0, 0.016918, 0.003644, 0.039823, 0.028110, 0.085893, 0.060906, 0.045809};
static float IWR_afters_27[] = { 0, 0.004296, 0, 0.001074, 0.003580, 0.001432, 0, 0, 0.000716, 0, 0, 0.021482, 0.008235, 0.022198, 0.025779, 0.103473, 0.134622, 0.001074, 0.019334, 0.008235, 0.374150, 0.008951, 0.023272, 0.002864, 0.014322, 0.008951, 0.007161, 0.041532, 0.026853, 0.019692, 0, 0.037594, 0, 0.003938, 0.009667, 0.009667, 0.036520, 0.019334};
static float IWR_afters_28[] = { 0, 0.001395, 0.002789, 0.006974, 0.004184, 0.002789, 0, 0.002789, 0, 0, 0, 0.005579, 0.078103, 0.027894, 0.019526, 0.022315, 0.160391, 0, 0.022315, 0.034868, 0.112971, 0.005579, 0.023710, 0.086471, 0.001395, 0.016736, 0.020921, 0.001395, 0.001395, 0.019526, 0.034868, 0.160391, 0, 0, 0.030683, 0.040446, 0, 0.051604};
static float IWR_afters_29[] = { 0, 0.000964, 0.000643, 0.000643, 0.003213, 0.001285, 0.000643, 0.000321, 0, 0, 0, 0.000321, 0.038560, 0.003856, 0.049807, 0.037596, 0.143316, 0.012853, 0, 0.013817, 0.098008, 0, 0.001285, 0.223972, 0.048201, 0.046915, 0.018638, 0.041131, 0.004820, 0.000643, 0.001928, 0.004820, 0.003856, 0.039846, 0.006427, 0.038560, 0.034383, 0.078728};
static float IWR_afters_31[] = { 0, 0.000589, 0.000589, 0.000589, 0.005297, 0, 0, 0, 0.010594, 0, 0, 0.012360, 0, 0.012949, 0.001177, 0.049441, 0.080636, 0.005886, 0.020600, 0.022366, 0.103590, 0.007652, 0.005297, 0.046498, 0.000589, 0, 0.024720, 0.115951, 0.030606, 0.097116, 0.003531, 0.004709, 0.012360, 0.016480, 0.023543, 0.087110, 0.083579, 0.113596};
static float IWR_afters_33[] = { 0, 0, 0.000959, 0, 0.002876, 0.001918, 0, 0, 0.000959, 0, 0, 0.129434, 0.061361, 0.001918, 0.030681, 0.064238, 0.098754, 0, 0.040268, 0.006711, 0.107383, 0, 0, 0.127517, 0.003835, 0.078619, 0.000959, 0.005753, 0, 0.048897, 0.004794, 0.028763, 0.002876, 0.006711, 0.000959, 0.093960, 0, 0.048897};
static float IWR_afters_34[] = { 0.000838, 0, 0.000838, 0, 0.008382, 0.000838, 0.000838, 0, 0, 0, 0, 0.017603, 0.036882, 0, 0.021794, 0.056161, 0.192791, 0, 0.032691, 0.050293, 0.064543, 0, 0, 0.041911, 0.009220, 0.019279, 0.022632, 0.017603, 0.001676, 0.019279, 0.000838, 0.030176, 0.010059, 0.031014, 0.002515, 0.218776, 0.062867, 0.027661};
static float IWR_afters_35[] = { 0, 0.001389, 0.002431, 0.005556, 0.005903, 0.002778, 0, 0.000347, 0, 0, 0, 0.096181, 0.047569, 0.049653, 0.018403, 0.060069, 0.182639, 0.002778, 0.039583, 0.010764, 0.168403, 0.025347, 0.029167, 0.003125, 0.006944, 0.011806, 0.005556, 0.011458, 0.011111, 0.045139, 0.003125, 0.009028, 0.004514, 0.012500, 0.025347, 0.008681, 0.019792, 0.072917};
static float IWR_afters_36[] = { 0, 0, 0, 0, 0.000688, 0, 0.000688, 0, 0, 0.013750, 0.005500, 0.047439, 0.082503, 0.005156, 0.004125, 0.040564, 0.115504, 0.000344, 0.007906, 0.005156, 0.093159, 0.008250, 0.041595, 0.103128, 0.025438, 0.073909, 0.004469, 0.049845, 0, 0.048470, 0.002406, 0.008250, 0, 0.000344, 0.019938, 0.124441, 0.008250, 0.058783};
static float IWR_afters_37[] = { 0, 0, 0, 0, 0.000700, 0, 0, 0, 0.002100, 0, 0, 0.029751, 0.027301, 0.009450, 0.002800, 0.067203, 0.107805, 0.000350, 0.037802, 0.000350, 0.474974, 0.006300, 0.023451, 0.007000, 0.009100, 0.022751, 0.010501, 0.026601, 0.000350, 0.019951, 0.002800, 0.022051, 0, 0.001050, 0.021701, 0.048302, 0.005950, 0.011551};
static struct letter_frequencies IWR_counts[] = {
    { "ְ", 0.000000, 0.000000, 0.000271, 0.000000, IWR_afters_0 },
    { "ִ", 0.000000, 0.000000, 0.001894, 0.000000, IWR_afters_1 },
    { "ֵ", 0.000000, 0.000000, 0.001624, 0.000000, IWR_afters_2 },
    { "ֶ", 0.000000, 0.000000, 0.002460, 0.000000, IWR_afters_3 },
    { "ַ", 0.000000, 0.000000, 0.006396, 0.000050, IWR_afters_4 },
    { "ָ", 0.000000, 0.000000, 0.002091, 0.000099, IWR_afters_5 },
    { "ֹ", 0.000000, 0.000000, 0.002239, 0.000000, IWR_afters_6 },
    { "ֻ", 0.000000, 0.000000, 0.000541, 0.000000, IWR_afters_7 },
    { "ּ", 0.000000, 0.000000, 0.003247, 0.001340, IWR_afters_8 },
    { "ׁ", 0.000000, 0.000000, 0.000984, 0.000000, IWR_afters_9 },
    { "ׂ", 0.000000, 0.000000, 0.000394, 0.000000, IWR_afters_10 },
    { "א", 0.083333, 0.136046, 0.052277, 0.044207, IWR_afters_11 },
    { "ב", 0.125000, 0.085487, 0.049128, 0.024659, IWR_afters_12 },
    { "ג", 0.041667, 0.012205, 0.015277, 0.002778, IWR_afters_13 },
    { "ד", 0.041667, 0.012850, 0.030161, 0.030960, IWR_afters_14 },
    { "ה", 0.250000, 0.167948, 0.021034, 0.135946, IWR_afters_15 },
    { "ו", 0.062500, 0.064847, 0.130114, 0.047135, IWR_afters_16 },
    { "ז", 0.041667, 0.015480, 0.013506, 0.003423, IWR_afters_17 },
    { "ח", 0.041667, 0.017663, 0.036778, 0.010915, IWR_afters_18 },
    { "ט", 0.062500, 0.004118, 0.010701, 0.011015, IWR_afters_19 },
    { "י", 0.062500, 0.027338, 0.106423, 0.187943, IWR_afters_20 },
    { "ך", 0.000000, 0.000000, 0.000000, 0.031754, NULL },
    { "כ", 0.083333, 0.059340, 0.031022, 0.000000, IWR_afters_22 },
    { "ל", 0.041667, 0.099677, 0.061773, 0.082014, IWR_afters_23 },
    { "ם", 0.000000, 0.000000, 0.000000, 0.089407, NULL },
    { "מ", 0.020833, 0.076507, 0.056582, 0.000149, IWR_afters_25 },
    { "ן", 0.000000, 0.000000, 0.000000, 0.053833, NULL },
    { "נ", 0.000000, 0.027289, 0.055180, 0.000050, IWR_afters_27 },
    { "ס", 0.000000, 0.007194, 0.014072, 0.004912, IWR_afters_28 },
    { "ע", 0.000000, 0.063508, 0.045069, 0.013743, IWR_afters_29 },
    { "ף", 0.000000, 0.000000, 0.000000, 0.012900, NULL },
    { "פ", 0.020833, 0.020888, 0.031440, 0.000000, IWR_afters_31 },
    { "ץ", 0.000000, 0.000000, 0.000000, 0.005458, NULL },
    { "צ", 0.000000, 0.010866, 0.020271, 0.000000, IWR_afters_33 },
    { "ק", 0.000000, 0.016224, 0.021304, 0.010221, IWR_afters_34 },
    { "ר", 0.000000, 0.020590, 0.060641, 0.072141, IWR_afters_35 },
    { "ש", 0.020833, 0.042917, 0.050284, 0.018655, IWR_afters_36 },
    { "ת", 0.000000, 0.011015, 0.064823, 0.104292, IWR_afters_37 },
    NULL
};	/* IWR */
static float IWR_word_lens[] = {
    0.000000, 0.002376, 0.175716, 0.201010, 0.268326, 0.212840, 0.089591, 0.034896, 0.009553, 0.003316, 0.001287, 0.000247, 0.000198, 0.000297, 0.000099, 0.000148, 0.000049, 0.000000, 0.000000, 0.000000, 0.000049, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static struct letter_frequencies HIN_counts[] = {
    { "ं", 0.000000, 0.000000, 0.035891, 0.079458 },
    { "ः", 0.000000, 0.000000, 0.000093, 0.000507 },
    { "अ", 0.000000, 0.040806, 0.000093, 0.000000 },
    { "आ", 0.002273, 0.014320, 0.001398, 0.000634 },
    { "इ", 0.000000, 0.015968, 0.001585, 0.000000 },
    { "ई", 0.001136, 0.000634, 0.001585, 0.012926 },
    { "उ", 0.000000, 0.018502, 0.000000, 0.000000 },
    { "ऊ", 0.001136, 0.000127, 0.000093, 0.000000 },
    { "ए", 0.001136, 0.011659, 0.006059, 0.018756 },
    { "ऐ", 0.000000, 0.002788, 0.000000, 0.000000 },
    { "ऑ", 0.000000, 0.002281, 0.000000, 0.000000 },
    { "ओ", 0.000000, 0.000634, 0.002890, 0.000127 },
    { "औ", 0.000000, 0.019136, 0.000000, 0.000000 },
    { "क", 0.052273, 0.171968, 0.040645, 0.049423 },
    { "ख", 0.001136, 0.007350, 0.004009, 0.003929 },
    { "ग", 0.009091, 0.016855, 0.016873, 0.007223 },
    { "घ", 0.002273, 0.003929, 0.000466, 0.000253 },
    { "च", 0.002273, 0.017488, 0.005593, 0.011152 },
    { "छ", 0.000000, 0.007477, 0.001119, 0.001521 },
    { "ज", 0.006818, 0.033836, 0.013517, 0.006210 },
    { "झ", 0.000000, 0.001141, 0.000466, 0.000253 },
    { "ञ", 0.000000, 0.000887, 0.000000, 0.000000 },
    { "ट", 0.117045, 0.006463, 0.008390, 0.006210 },
    { "ठ", 0.002273, 0.001647, 0.000653, 0.000760 },
    { "ड", 0.009091, 0.004182, 0.002144, 0.009251 },
    { "ढ", 0.000000, 0.000634, 0.000093, 0.002408 },
    { "ण", 0.007955, 0.000507, 0.003263, 0.005576 },
    { "त", 0.075000, 0.032822, 0.032162, 0.039159 },
    { "थ", 0.004545, 0.014954, 0.002144, 0.002535 },
    { "द", 0.032955, 0.022557, 0.014636, 0.020023 },
    { "ध", 0.012500, 0.003675, 0.013704, 0.005196 },
    { "न", 0.045455, 0.038271, 0.062086, 0.047396 },
    { "प", 0.198864, 0.040553, 0.017526, 0.010772 },
    { "फ", 0.007955, 0.003168, 0.002517, 0.004816 },
    { "ब", 0.002273, 0.029401, 0.005687, 0.006463 },
    { "भ", 0.000000, 0.019896, 0.007271, 0.000253 },
    { "म", 0.014773, 0.067545, 0.021255, 0.014574 },
    { "य", 0.064773, 0.036751, 0.032162, 0.008364 },
    { "र", 0.042045, 0.069573, 0.046332, 0.111519 },
    { "ल", 0.007955, 0.020403, 0.027594, 0.015461 },
    { "व", 0.080682, 0.041820, 0.021348, 0.007857 },
    { "श", 0.017045, 0.011532, 0.007644, 0.009631 },
    { "ष", 0.025000, 0.007857, 0.002517, 0.008111 },
    { "स", 0.115909, 0.072868, 0.028619, 0.032695 },
    { "ह", 0.000000, 0.060068, 0.027780, 0.012419 },
    { "ा", 0.013636, 0.001267, 0.161462, 0.114181 },
    { "ि", 0.000000, 0.001394, 0.104037, 0.028640 },
    { "ी", 0.014773, 0.000000, 0.025357, 0.088962 },
    { "ु", 0.000000, 0.000380, 0.035145, 0.000507 },
    { "ू", 0.000000, 0.000000, 0.014822, 0.001774 },
    { "ृ", 0.000000, 0.000000, 0.002517, 0.000000 },
    { "ॆ", 0.000000, 0.000000, 0.000093, 0.000000 },
    { "े", 0.007955, 0.000253, 0.065349, 0.131416 },
    { "ै", 0.000000, 0.000000, 0.013144, 0.021037 },
    { "ॉ", 0.000000, 0.000000, 0.000839, 0.001141 },
    { "ो", 0.000000, 0.000253, 0.050247, 0.037258 },
    { "ौ", 0.000000, 0.000000, 0.005593, 0.000127 },
    { "क़", 0.000000, 0.001014, 0.000000, 0.001014 },
    { "ज़", 0.000000, 0.000507, 0.001119, 0.000000 },
    { "ड़", 0.000000, 0.000000, 0.000093, 0.000000 },
    { "ढ़", 0.000000, 0.000000, 0.000186, 0.000000 },
    { "फ़", 0.000000, 0.000000, 0.000093, 0.000127 },
    NULL
};	/* HIN */
static float HIN_word_lens[] = {
    0.000000, 0.100331, 0.338160, 0.212405, 0.166116, 0.102383, 0.050963, 0.014822, 0.011629, 0.002508, 0.000456, 0.000114, 0.000000, 0.000114, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float HUN_afters_0[] = { 0.000533, 0.015604, 0.005868, 0.041211, 0.001734, 0.001200, 0.036410, 0.003468, 0.018538, 0.028274, 0.106428, 0.135103, 0.039877, 0.124833, 0, 0.024540, 0, 0.073220, 0.027074, 0.112030, 0.001200, 0.010536, 0, 0.000267, 0.000533, 0.112830, 0.000133, 0.000533, 0.000133, 0, 0, 0.000667, 0, 0.000133, 0, 0.000133, 0.000934, 0, 0.040144, 0.000667, 0.010403, 0.017338, 0.007202, 0.000267};
static float HUN_afters_1[] = { 0.216724, 0.117763, 0, 0.000495, 0.309253, 0.000495, 0, 0, 0.033152, 0.002474, 0.000495, 0.015339, 0, 0.001979, 0.103909, 0, 0, 0.011875, 0.001484, 0.000990, 0.011381, 0.000495, 0, 0, 0, 0, 0.051954, 0.013855, 0, 0.015834, 0.026719, 0.013360, 0.006927, 0.003958, 0.032657, 0.004453, 0, 0, 0, 0, 0, 0.001979, 0, 0};
static float HUN_afters_2[] = { 0.121951, 0.009756, 0.039024, 0, 0.095122, 0, 0, 0.131707, 0.053659, 0, 0.019512, 0.012195, 0.002439, 0.004878, 0.075610, 0, 0, 0.009756, 0, 0.051220, 0.058537, 0.004878, 0, 0, 0.131707, 0.002439, 0.100000, 0.041463, 0, 0.002439, 0, 0.002439, 0.002439, 0, 0, 0, 0.026829, 0, 0, 0, 0, 0, 0, 0};
static float HUN_afters_3[] = { 0.056442, 0.005060, 0, 0.021798, 0.222655, 0.000389, 0.000389, 0.009731, 0.075516, 0.038536, 0.005060, 0.000779, 0.002725, 0.026469, 0.113663, 0, 0, 0.024912, 0.002336, 0.071234, 0.036979, 0.044375, 0, 0, 0, 0, 0.039704, 0.056831, 0, 0.007007, 0.035812, 0.025302, 0.001168, 0.016349, 0.052939, 0.000779, 0, 0, 0, 0, 0, 0.005060, 0, 0};
static float HUN_afters_4[] = { 0.000871, 0.008223, 0.001645, 0.046338, 0.001548, 0.001741, 0.077005, 0.009577, 0.009287, 0.014511, 0.060462, 0.160201, 0.089291, 0.126632, 0.002128, 0.005804, 0.000097, 0.079327, 0.048950, 0.112122, 0.000677, 0.013834, 0, 0.000774, 0.000193, 0.044694, 0.004063, 0.000484, 0, 0.000387, 0, 0, 0, 0.000580, 0, 0, 0.001548, 0.000097, 0.040147, 0.015285, 0.002225, 0.018477, 0.000484, 0.000290};
static float HUN_afters_5[] = { 0.044865, 0, 0, 0, 0.422732, 0.001994, 0, 0, 0.067797, 0.009970, 0, 0.003988, 0, 0, 0.192423, 0, 0, 0.002991, 0, 0, 0.016949, 0, 0, 0, 0, 0, 0.019940, 0.096710, 0, 0, 0, 0.049850, 0.002991, 0.029910, 0.033898, 0.002991, 0, 0, 0, 0, 0, 0, 0, 0};
static float HUN_afters_6[] = { 0.125939, 0.023420, 0.000442, 0.004861, 0.156871, 0.007954, 0.029165, 0.014141, 0.053911, 0.023420, 0.008396, 0.012815, 0.008838, 0.029607, 0.124171, 0.002209, 0, 0.023862, 0.010605, 0.051259, 0.008838, 0.019001, 0, 0, 0, 0.004419, 0.072912, 0.083517, 0, 0.016350, 0.016792, 0.015466, 0.003977, 0.006186, 0.015908, 0.000442, 0.002209, 0, 0.003977, 0.003535, 0.005303, 0.008838, 0, 0.000442};
static float HUN_afters_7[] = { 0.327447, 0, 0, 0, 0.139253, 0, 0, 0.000505, 0.078204, 0, 0, 0.000505, 0, 0.000505, 0.230575, 0, 0, 0.003532, 0.000505, 0.004036, 0.011604, 0.000505, 0, 0, 0.020686, 0, 0.105954, 0.022200, 0, 0.026741, 0.004541, 0.006054, 0.005045, 0.001009, 0.007064, 0.003027, 0, 0, 0, 0.000505, 0, 0, 0, 0};
static float HUN_afters_8[] = { 0.042590, 0.018993, 0.008633, 0.060719, 0.014964, 0.005755, 0.076547, 0.007770, 0.002590, 0.004029, 0.075396, 0.044317, 0.018417, 0.151079, 0.003453, 0.002302, 0, 0.037986, 0.149640, 0.103597, 0.014388, 0.021295, 0, 0.002014, 0, 0.029065, 0.035396, 0.009209, 0, 0, 0.006331, 0, 0.003165, 0.000288, 0, 0, 0.003741, 0, 0.009496, 0.009496, 0.002590, 0.022446, 0.000288, 0.002014};
static float HUN_afters_9[] = { 0.142857, 0.002857, 0, 0.026429, 0.148571, 0.010714, 0.002143, 0.001429, 0, 0.028571, 0.005000, 0.010000, 0, 0.006429, 0.150000, 0, 0, 0.010000, 0.000714, 0.108571, 0.047143, 0, 0, 0, 0, 0.004286, 0.140000, 0.044286, 0, 0, 0.032857, 0.043571, 0.009286, 0.011429, 0.005000, 0.007857, 0, 0, 0, 0, 0, 0, 0, 0};
static float HUN_afters_10[] = { 0.117403, 0.019271, 0, 0, 0.209902, 0.002075, 0, 0.007412, 0.114142, 0.004151, 0.033798, 0.029647, 0.000593, 0.015417, 0.117403, 0, 0, 0.020753, 0.003854, 0.012155, 0.018381, 0.000593, 0, 0, 0, 0.001482, 0.045953, 0.086273, 0, 0.013638, 0.003854, 0.077972, 0.002075, 0.033501, 0.005929, 0, 0, 0, 0, 0.000296, 0, 0.002075, 0, 0};
static float HUN_afters_11[] = { 0.110329, 0.004575, 0.008446, 0.015837, 0.182298, 0.005631, 0.009678, 0.014605, 0.033257, 0.016189, 0.026043, 0.079887, 0.041175, 0.022171, 0.038712, 0.001760, 0, 0.005103, 0.011790, 0.110329, 0.010734, 0.026570, 0, 0, 0, 0.000880, 0.096252, 0.050677, 0, 0.004575, 0.022171, 0.008798, 0.000176, 0.001760, 0.023051, 0.000352, 0.007039, 0, 0, 0.000880, 0.001408, 0.006687, 0.000176, 0};
static float HUN_afters_12[] = { 0.120439, 0.035391, 0.000274, 0.000549, 0.278738, 0.000823, 0.000549, 0.001646, 0.215912, 0.000549, 0.000274, 0.014540, 0.013992, 0.003292, 0.089438, 0.007407, 0, 0.006036, 0.002195, 0.005487, 0.030178, 0.000823, 0, 0, 0.000274, 0.000549, 0.067215, 0.063649, 0, 0.008505, 0.003018, 0.007133, 0.008505, 0.000823, 0.000274, 0.007682, 0.001372, 0, 0, 0, 0, 0.002195, 0, 0.000274};
static float HUN_afters_13[] = { 0.138312, 0.012410, 0.005190, 0.109657, 0.174865, 0.003384, 0.054829, 0.004738, 0.067013, 0.004061, 0.057536, 0.004287, 0.000451, 0.034522, 0.030009, 0, 0.000226, 0.002482, 0.011733, 0.133574, 0.011958, 0.003610, 0, 0, 0, 0.004061, 0.029332, 0.041065, 0, 0.001579, 0.002482, 0.006543, 0, 0.009251, 0.008800, 0.002933, 0.011282, 0, 0.000677, 0, 0.014892, 0.002256, 0, 0};
static float HUN_afters_14[] = { 0, 0.018754, 0.003959, 0.041884, 0, 0, 0.043967, 0.011044, 0, 0.001250, 0.082100, 0.096895, 0.080642, 0.104605, 0, 0.005626, 0, 0.121692, 0.098979, 0.119817, 0.000417, 0.008127, 0, 0, 0.000208, 0.060638, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.002917, 0, 0.053136, 0.021880, 0.013961, 0.007293, 0, 0.000208};
static float HUN_afters_15[] = { 0.130129, 0, 0.001172, 0.003517, 0.230950, 0, 0, 0.018757, 0.094959, 0.022274, 0.007034, 0.018757, 0.004689, 0.003517, 0.105510, 0.051583, 0, 0.057444, 0.003517, 0.051583, 0.017585, 0.004689, 0, 0, 0.001172, 0.008206, 0.090270, 0.019930, 0, 0.010551, 0.008206, 0.003517, 0, 0.010551, 0.004689, 0.002345, 0.011723, 0, 0, 0, 0, 0.001172, 0, 0};
static float HUN_afters_16[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float HUN_afters_17[] = { 0.167012, 0.007602, 0.018199, 0.038701, 0.167934, 0.004377, 0.015434, 0.006680, 0.045612, 0.012900, 0.018429, 0.006450, 0.025801, 0.028104, 0.061276, 0.001152, 0.000230, 0.028334, 0.011057, 0.102281, 0.012440, 0.011057, 0, 0, 0.000230, 0.007832, 0.070721, 0.017738, 0, 0.006450, 0.020272, 0.022575, 0.003225, 0.015434, 0.019120, 0.005298, 0.003225, 0, 0.000691, 0, 0.006220, 0.006450, 0.002995, 0.000461};
static float HUN_afters_18[] = { 0.078378, 0.005405, 0.000772, 0.007722, 0.149807, 0.001158, 0.000386, 0.006564, 0.045174, 0.000386, 0.015830, 0.001931, 0.020849, 0.016988, 0.055212, 0.006564, 0, 0.011197, 0.057143, 0.136293, 0.018919, 0.001544, 0.000386, 0, 0.006950, 0, 0.077220, 0.132432, 0, 0.011969, 0.016216, 0.003089, 0.005405, 0.006564, 0.014672, 0.000772, 0, 0, 0.001158, 0, 0, 0.084942, 0, 0};
static float HUN_afters_19[] = { 0.142630, 0.006220, 0.001591, 0, 0.179228, 0.001302, 0.000434, 0.027774, 0.032403, 0.006799, 0.013453, 0.021554, 0.000434, 0.009547, 0.082887, 0.000145, 0, 0.009692, 0.011572, 0.169680, 0.042095, 0.018516, 0, 0, 0, 0.000289, 0.069868, 0.047736, 0, 0.000579, 0.028931, 0.020975, 0.001447, 0.014610, 0.019962, 0.010704, 0.000145, 0, 0, 0, 0, 0.006365, 0.000434, 0};
static float HUN_afters_20[] = { 0.002861, 0.002861, 0.003577, 0.215308, 0.004292, 0, 0.042918, 0.007153, 0.005722, 0.012160, 0.035050, 0.170959, 0.032904, 0.083691, 0, 0.005722, 0, 0.055079, 0.119456, 0.155937, 0.001431, 0.002861, 0, 0, 0, 0.003577, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001431, 0, 0.025751, 0, 0.004292, 0.002861, 0.002146, 0};
static float HUN_afters_21[] = { 0.297079, 0.002247, 0, 0, 0.253933, 0, 0, 0.000449, 0.082697, 0.000449, 0, 0, 0, 0.004045, 0.069663, 0, 0, 0.000449, 0, 0.004494, 0, 0.004045, 0, 0, 0, 0, 0.133483, 0.099775, 0, 0.003596, 0.003596, 0.021124, 0.000449, 0.003596, 0.009888, 0.001348, 0, 0, 0, 0, 0, 0.003596, 0, 0};
static float HUN_afters_22[] = { 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float HUN_afters_23[] = { 0.230769, 0, 0, 0, 0, 0, 0, 0, 0.230769, 0, 0, 0, 0, 0, 0.076923, 0.307692, 0, 0, 0, 0.076923, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.076923, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float HUN_afters_24[] = { 0.008929, 0, 0, 0.026786, 0.026786, 0, 0, 0, 0, 0, 0, 0, 0.410714, 0.491071, 0, 0, 0, 0.008929, 0.026786, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float HUN_afters_25[] = { 0.073548, 0.020396, 0, 0.050062, 0.122373, 0.001236, 0.012979, 0.010507, 0.036465, 0, 0.007417, 0.003708, 0.009271, 0.033375, 0.106304, 0.000618, 0, 0.001236, 0, 0.144623, 0.042645, 0.021632, 0, 0, 0, 0.067985, 0.068603, 0.053152, 0, 0.011125, 0.030902, 0.032138, 0.007417, 0.015451, 0.014215, 0, 0, 0, 0, 0, 0.000618, 0, 0, 0};
static float HUN_afters_26[] = { 0.000267, 0.045091, 0.001334, 0.008271, 0, 0.000800, 0.044023, 0.007471, 0.006937, 0.016809, 0.025614, 0.139541, 0.039755, 0.119264, 0.000267, 0.009338, 0, 0.165155, 0.104055, 0.142209, 0.000267, 0.019744, 0, 0, 0, 0.027481, 0, 0.000534, 0, 0, 0, 0, 0, 0, 0, 0, 0.001868, 0, 0.007471, 0.006670, 0.041889, 0.010406, 0.006137, 0.001334};
static float HUN_afters_27[] = { 0, 0.033227, 0, 0.011557, 0, 0, 0.115862, 0.004912, 0.000867, 0.015025, 0.041606, 0.090436, 0.014736, 0.082057, 0, 0.038428, 0, 0.120196, 0.207454, 0.078301, 0, 0.016758, 0, 0, 0, 0.019648, 0, 0.000289, 0, 0, 0, 0, 0, 0.000289, 0, 0, 0, 0, 0.006068, 0.022537, 0.031494, 0.047385, 0, 0.000867};
static float HUN_afters_28[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float HUN_afters_29[] = { 0, 0.027833, 0, 0.003976, 0, 0, 0.053678, 0, 0, 0.003976, 0, 0.019881, 0.015905, 0.057654, 0, 0, 0, 0.168986, 0.013917, 0.413519, 0, 0.147117, 0, 0, 0, 0.023857, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.041750, 0, 0.003976, 0.003976, 0, 0};
static float HUN_afters_30[] = { 0.009036, 0.076807, 0.009036, 0.037651, 0.006024, 0.006024, 0.003012, 0.022590, 0.018072, 0.025602, 0.051205, 0.331325, 0.003012, 0.031627, 0, 0.001506, 0, 0.082831, 0.137048, 0.060241, 0, 0.030120, 0, 0, 0, 0.022590, 0, 0, 0, 0, 0, 0, 0, 0.001506, 0, 0, 0.007530, 0.007530, 0.004518, 0.003012, 0.001506, 0.003012, 0, 0.006024};
static float HUN_afters_31[] = { 0, 0.035000, 0.006667, 0.021667, 0, 0, 0.034167, 0.002500, 0, 0.002500, 0.031667, 0.088333, 0.031667, 0.090000, 0, 0.008333, 0, 0.231667, 0.093333, 0.123333, 0, 0.056667, 0, 0, 0, 0.100000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001667, 0, 0, 0, 0.033333, 0.007500, 0, 0};
static float HUN_afters_32[] = { 0.012295, 0, 0, 0.004098, 0, 0, 0.020492, 0, 0, 0.110656, 0.008197, 0.176230, 0, 0.004098, 0, 0, 0, 0.151639, 0.036885, 0.110656, 0, 0.020492, 0, 0, 0, 0.057377, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.020492, 0, 0.217213, 0.008197, 0.012295, 0.028689, 0, 0};
static float HUN_afters_33[] = { 0, 0.006144, 0, 0.032258, 0, 0, 0.021505, 0.015361, 0.001536, 0.003072, 0.101382, 0.436252, 0.009217, 0.153610, 0, 0, 0, 0.053763, 0.029186, 0.035330, 0, 0.027650, 0, 0, 0, 0.049155, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.010753, 0, 0.001536, 0.010753, 0.001536, 0};
static float HUN_afters_34[] = { 0.012719, 0.034976, 0, 0.041335, 0.004769, 0.009539, 0.006359, 0.020668, 0.003180, 0.011129, 0.058824, 0.289348, 0.006359, 0.162162, 0, 0, 0, 0.041335, 0.069952, 0.100159, 0.001590, 0.017488, 0, 0, 0, 0.052464, 0.003180, 0.001590, 0, 0.001590, 0, 0, 0.003180, 0.001590, 0, 0, 0.001590, 0.003180, 0, 0, 0.001590, 0.036566, 0, 0.001590};
static float HUN_afters_35[] = { 0, 0.020833, 0, 0, 0.027778, 0, 0, 0.041667, 0, 0.020833, 0.041667, 0.048611, 0, 0.097222, 0, 0, 0, 0.076389, 0.055556, 0.006944, 0, 0.138889, 0, 0, 0, 0.375000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.013889, 0, 0, 0, 0, 0.013889, 0, 0.020833};
static float HUN_afters_36[] = { 0.407843, 0, 0, 0, 0.143137, 0, 0, 0, 0.060784, 0, 0.033333, 0.001961, 0.005882, 0, 0.076471, 0, 0, 0, 0, 0, 0.064706, 0.003922, 0, 0, 0, 0, 0.066667, 0.041176, 0, 0.003922, 0.013725, 0.013725, 0.011765, 0.025490, 0.023529, 0.001961, 0, 0, 0, 0, 0, 0, 0, 0};
static float HUN_afters_37[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.125000, 0.250000, 0, 0, 0, 0, 0, 0, 0.375000, 0, 0, 0, 0, 0, 0, 0, 0, 0.250000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float HUN_afters_38[] = { 0.121835, 0.007911, 0, 0, 0.265823, 0.006329, 0, 0.007911, 0.055380, 0.006329, 0.007911, 0, 0.045886, 0.015823, 0.145570, 0, 0, 0.006329, 0.001582, 0.009494, 0.020570, 0.012658, 0, 0, 0, 0.009494, 0.023734, 0.012658, 0, 0.006329, 0.020570, 0.014241, 0.015823, 0.036392, 0.026899, 0.017405, 0, 0, 0, 0, 0, 0.079114, 0, 0};
static float HUN_afters_39[] = { 0.113772, 0.044910, 0, 0.002994, 0.350299, 0.002994, 0.041916, 0.008982, 0.035928, 0, 0.008982, 0, 0.002994, 0.065868, 0.047904, 0.002994, 0, 0.017964, 0.011976, 0.095808, 0.014970, 0.002994, 0, 0, 0, 0.005988, 0.026946, 0.053892, 0, 0, 0.008982, 0, 0, 0.014970, 0, 0.005988, 0, 0, 0, 0, 0.002994, 0.005988, 0, 0};
static float HUN_afters_40[] = { 0.122689, 0.021849, 0, 0.010084, 0.121008, 0.001681, 0, 0.015126, 0.102521, 0.001681, 0.023529, 0.008403, 0.005042, 0.011765, 0.141176, 0, 0, 0.008403, 0.003361, 0.031933, 0.058824, 0.045378, 0, 0, 0, 0.015126, 0.075630, 0.030252, 0, 0.021849, 0.003361, 0.028571, 0.016807, 0.003361, 0, 0.015126, 0.003361, 0, 0, 0, 0.001681, 0.050420, 0, 0};
static float HUN_afters_41[] = { 0.152803, 0.001848, 0, 0.001232, 0.264325, 0.001232, 0, 0.001848, 0.048059, 0, 0.017868, 0.011091, 0.001848, 0.005545, 0.091805, 0, 0, 0.007394, 0.004929, 0.109057, 0.004929, 0.000616, 0, 0, 0.000616, 0, 0.064695, 0.052372, 0, 0.044978, 0.041282, 0.025262, 0.013555, 0.021565, 0.004929, 0.002465, 0, 0, 0.001232, 0, 0.000616, 0, 0, 0};
static float HUN_afters_42[] = { 0.144231, 0, 0, 0, 0.019231, 0, 0, 0, 0, 0.163462, 0, 0.009615, 0.009615, 0, 0.028846, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.548077, 0, 0, 0, 0, 0, 0.019231, 0.009615, 0, 0.048077, 0, 0, 0, 0, 0, 0, 0, 0};
static float HUN_afters_43[] = { 0.170732, 0, 0, 0, 0, 0, 0.146341, 0, 0.121951, 0, 0, 0.073171, 0, 0, 0.097561, 0, 0, 0, 0, 0, 0.097561, 0, 0, 0, 0, 0.048780, 0.073171, 0.073171, 0, 0.073171, 0, 0.024390, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static struct letter_frequencies HUN_counts[] = {
    { "a", 0.836748, 0.084498, 0.082949, 0.087073, HUN_afters_0 },
    { "b", 0.000000, 0.032824, 0.019804, 0.006083, HUN_afters_1 },
    { "c", 0.000000, 0.005589, 0.004290, 0.000438, HUN_afters_2 },
    { "d", 0.000000, 0.017535, 0.031322, 0.021426, HUN_afters_3 },
    { "e", 0.084600, 0.069702, 0.126248, 0.068387, HUN_afters_4 },
    { "f", 0.000000, 0.044496, 0.002660, 0.000219, HUN_afters_5 },
    { "g", 0.000000, 0.005644, 0.030082, 0.032166, HUN_afters_6 },
    { "h", 0.000000, 0.064497, 0.011211, 0.002356, HUN_afters_7 },
    { "i", 0.000000, 0.032111, 0.040235, 0.042797, HUN_afters_8 },
    { "j", 0.000000, 0.022960, 0.013662, 0.004000, HUN_afters_9 },
    { "k", 0.000000, 0.079785, 0.026698, 0.082196, HUN_afters_10 },
    { "l", 0.000000, 0.038194, 0.069440, 0.064277, HUN_afters_11 },
    { "m", 0.000000, 0.110527, 0.022673, 0.051510, HUN_afters_12 },
    { "n", 0.000000, 0.041208, 0.051251, 0.093759, HUN_afters_13 },
    { "o", 0.000000, 0.013261, 0.063465, 0.000932, HUN_afters_14 },
    { "p", 0.000000, 0.018357, 0.007214, 0.003452, HUN_afters_15 },
    { "q", 0.000000, 0.000164, 0.000042, 0.000000, HUN_afters_16 },
    { "r", 0.000661, 0.022193, 0.054817, 0.021316, HUN_afters_17 },
    { "s", 0.040317, 0.021700, 0.030556, 0.084772, HUN_afters_18 },
    { "t", 0.001322, 0.063182, 0.080219, 0.162255, HUN_afters_19 },
    { "u", 0.000000, 0.010521, 0.016796, 0.000110, HUN_afters_20 },
    { "v", 0.000000, 0.051784, 0.017827, 0.000548, HUN_afters_21 },
    { "w", 0.000000, 0.000000, 0.000014, 0.000000, HUN_afters_22 },
    { "x", 0.000000, 0.000000, 0.000181, 0.000219, HUN_afters_23 },
    { "y", 0.000000, 0.000000, 0.001560, 0.000658, HUN_afters_24 },
    { "z", 0.000000, 0.003343, 0.021684, 0.044824, HUN_afters_25 },
    { "á", 0.000000, 0.010192, 0.049608, 0.014521, HUN_afters_26 },
    { "é", 0.000661, 0.051236, 0.035180, 0.017700, HUN_afters_27 },
    { "ë", 0.000000, 0.000000, 0.000014, 0.000000, HUN_afters_28 },
    { "í", 0.000000, 0.002959, 0.006253, 0.000000, HUN_afters_29 },
    { "ó", 0.000000, 0.002575, 0.008593, 0.016110, HUN_afters_30 },
    { "ö", 0.000000, 0.008110, 0.014651, 0.000000, HUN_afters_31 },
    { "ú", 0.000000, 0.005973, 0.001880, 0.003891, HUN_afters_32 },
    { "ü", 0.000000, 0.003178, 0.008259, 0.000219, HUN_afters_33 },
    { "ő", 0.035691, 0.002411, 0.008147, 0.013371, HUN_afters_34 },
    { "ű", 0.000000, 0.000110, 0.001978, 0.003836, HUN_afters_35 },
    { "cs", 0.000000, 0.017700, 0.002604, 0.001370, HUN_afters_36 },
    { "dz", 0.000000, 0.000000, 0.000111, 0.000000, HUN_afters_37 },
    { "gy", 0.000000, 0.004055, 0.007771, 0.034796, HUN_afters_38 },
    { "ly", 0.000000, 0.000219, 0.004596, 0.005096, HUN_afters_39 },
    { "ny", 0.000000, 0.004329, 0.007186, 0.005370, HUN_afters_40 },
    { "sz", 0.000000, 0.032440, 0.014359, 0.007946, HUN_afters_41 },
    { "ty", 0.000000, 0.000000, 0.001448, 0.000000, HUN_afters_42 },
    { "zs", 0.000000, 0.000438, 0.000460, 0.000000, HUN_afters_43 },
    NULL
};	/* HUN */
static float HUN_word_lens[] = {
    0.000000, 0.076561, 0.123621, 0.099990, 0.083696, 0.140117, 0.124734, 0.100142, 0.081267, 0.054043, 0.045896, 0.030159, 0.017863, 0.010879, 0.004757, 0.003390, 0.001720, 0.000658, 0.000506, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float HUN_all_consonants[] = {
    0.000000, 0.003239, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 
};

static float HUN_consonant_run[] = {
    0.000000, 0.741932, 0.243628, 0.012389, 0.000178, 0.001874, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float HUN_vowel_run[] = {
    0.000000, 0.978892, 0.020588, 0.000498, 0.000023, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float ITA_afters_0[] = { 0, 0.007372, 0.042435, 0.036477, 0.002143, 0.005679, 0.024453, 0.001050, 0.021625, 0.000171, 0, 0.145028, 0.039584, 0.213738, 0.000214, 0.026940, 0, 0.160373, 0.058551, 0.112237, 0.003065, 0.089606, 0, 0, 0.008744, 0, 0, 0, 0, 0, 0, 0, 0, 0.000364, 0.000150};
static float ITA_afters_1[] = { 0.168732, 0.175861, 0, 0, 0.248005, 0, 0, 0, 0.184179, 0, 0, 0.007299, 0, 0, 0.087931, 0, 0, 0.079952, 0, 0, 0.047021, 0, 0, 0, 0, 0, 0, 0.000170, 0.000340, 0.000340, 0, 0.000170, 0, 0, 0};
static float ITA_afters_2[] = { 0.183689, 0, 0.080355, 0, 0.118014, 0, 0, 0, 0.140513, 0, 0, 0.021492, 0, 0, 0.342480, 0, 0, 0.018422, 0, 0.000144, 0.044423, 0, 0, 0, 0, 0, 0.001055, 0.002111, 0, 0.002111, 0.000048, 0, 0, 0.040633, 0.004509};
static float ITA_afters_3[] = { 0.185684, 0, 0, 0.014204, 0.229068, 0, 0, 0, 0.315278, 0, 0, 0, 0, 0, 0.181406, 0, 0, 0.026056, 0.000086, 0, 0.038506, 0, 0, 0, 0, 0.000599, 0.000513, 0.003252, 0, 0.005348, 0, 0, 0, 0, 0};
static float ITA_afters_4[] = { 0.003068, 0.018612, 0.025387, 0.047528, 0.000128, 0.001508, 0.042312, 0.000741, 0.027151, 0, 0, 0.151762, 0.028634, 0.149870, 0.001074, 0.005190, 0, 0.233727, 0.118781, 0.058010, 0.000435, 0.074066, 0, 0.000026, 0.011403, 0, 0, 0, 0, 0, 0, 0, 0, 0.000153, 0.000435};
static float ITA_afters_5[] = { 0.284641, 0, 0, 0, 0.123399, 0.071271, 0, 0, 0.156089, 0, 0, 0.005154, 0, 0, 0.149904, 0, 0, 0.115152, 0, 0, 0.092917, 0, 0, 0, 0, 0, 0.001473, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_6[] = { 0.047207, 0, 0, 0, 0.068709, 0, 0.062727, 0.023038, 0.217767, 0, 0, 0.255113, 0.000081, 0.139924, 0.055533, 0, 0, 0.062242, 0, 0, 0.064991, 0, 0, 0, 0, 0, 0, 0.000485, 0, 0.002183, 0, 0, 0, 0, 0};
static float ITA_afters_7[] = { 0.323964, 0, 0, 0, 0.235207, 0, 0, 0, 0.207101, 0, 0, 0, 0.001479, 0, 0.227811, 0, 0, 0, 0, 0, 0.002959, 0, 0, 0, 0, 0, 0.001479, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_8[] = { 0.147927, 0.004585, 0.047035, 0.020300, 0.067887, 0.004281, 0.048858, 0, 0.001105, 0, 0, 0.081973, 0.043638, 0.162123, 0.095368, 0.006380, 0, 0.040766, 0.064987, 0.057144, 0.035573, 0.026542, 0, 0.000110, 0.006960, 0.004612, 0.001353, 0, 0, 0.004916, 0, 0.022233, 0, 0.003121, 0.000221};
static float ITA_afters_9[] = { 0.090909, 0, 0, 0, 0.181818, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.363636, 0, 0, 0, 0, 0, 0.363636, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_10[] = { 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_11[] = { 0.263894, 0.001899, 0.011395, 0.005083, 0.172345, 0.001061, 0.001676, 0.001424, 0.153382, 0, 0, 0.198877, 0.003770, 0.000028, 0.083198, 0.002234, 0, 0, 0.001927, 0.044574, 0.030581, 0.004161, 0, 0, 0.002486, 0.007233, 0.000084, 0.003044, 0, 0.001368, 0, 0, 0, 0.004273, 0};
static float ITA_afters_12[] = { 0.330873, 0.024712, 0, 0, 0.219296, 0, 0, 0.000880, 0.130196, 0, 0, 0, 0.027420, 0, 0.161611, 0.077116, 0, 0, 0, 0, 0.014624, 0, 0, 0, 0, 0.000068, 0.001490, 0.000068, 0, 0.011645, 0, 0, 0, 0, 0};
static float ITA_afters_13[] = { 0.117817, 0, 0.045069, 0.104372, 0.109867, 0.007102, 0.021892, 0.000175, 0.078740, 0, 0, 0, 0, 0.042351, 0.245893, 0, 0.000029, 0, 0.021892, 0.132519, 0.018881, 0.006284, 0, 0, 0.024084, 0, 0.005407, 0.000380, 0, 0.004472, 0, 0, 0, 0.009470, 0.003303};
static float ITA_afters_14[] = { 0.000698, 0.003953, 0.041393, 0.010232, 0.000299, 0.004385, 0.036111, 0.004651, 0.025181, 0, 0, 0.140256, 0.070627, 0.227460, 0.000066, 0.024384, 0, 0.202810, 0.113680, 0.036941, 0, 0.050395, 0.000066, 0, 0.003089, 0, 0.000066, 0.000066, 0, 0, 0, 0, 0, 0.002857, 0.000332};
static float ITA_afters_15[] = { 0.206642, 0, 0, 0, 0.230938, 0, 0, 0, 0.159270, 0, 0, 0.009157, 0, 0, 0.195470, 0.042366, 0, 0.124168, 0.000061, 0, 0.029669, 0, 0, 0, 0, 0, 0.000122, 0.001404, 0, 0.000733, 0, 0, 0, 0, 0};
static float ITA_afters_16[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_17[] = { 0.200149, 0.005299, 0.013548, 0.017643, 0.249356, 0.000229, 0.016612, 0, 0.143438, 0, 0.000029, 0.025892, 0.013003, 0.028384, 0.122558, 0.001661, 0, 0.027553, 0.040958, 0.047918, 0.010483, 0.010225, 0, 0, 0.001948, 0.005700, 0, 0.000372, 0, 0.007017, 0, 0, 0, 0.010025, 0};
static float ITA_afters_18[] = { 0.119067, 0.003548, 0.060547, 0.000686, 0.177707, 0.002474, 0.002951, 0.000179, 0.160386, 0, 0, 0.000089, 0.002564, 0.000149, 0.090180, 0.039143, 0, 0.000030, 0.118173, 0.127594, 0.064602, 0.001968, 0, 0, 0, 0.000030, 0.004442, 0.010762, 0, 0.002504, 0, 0.002862, 0, 0.005426, 0.001938};
static float ITA_afters_19[] = { 0.203562, 0, 0, 0, 0.167143, 0, 0, 0.000491, 0.127407, 0, 0, 0, 0, 0, 0.220390, 0, 0, 0.098050, 0, 0.120774, 0.044741, 0, 0, 0, 0, 0.008353, 0.001996, 0.002027, 0, 0.004483, 0, 0.000553, 0.000031, 0, 0};
static float ITA_afters_20[] = { 0.066379, 0.011901, 0.015727, 0.028691, 0.031666, 0.008643, 0.017215, 0, 0.064891, 0.000142, 0, 0.101941, 0.018065, 0.255384, 0.125531, 0.014593, 0, 0.072613, 0.041301, 0.115118, 0, 0.002196, 0, 0, 0.003330, 0, 0, 0.000921, 0, 0.003542, 0, 0, 0, 0.000213, 0};
static float ITA_afters_21[] = { 0.383732, 0, 0, 0, 0.301691, 0, 0, 0.000574, 0.154386, 0, 0, 0, 0, 0, 0.098947, 0, 0, 0.020096, 0, 0, 0.013461, 0.017671, 0, 0, 0, 0, 0.002041, 0.000128, 0, 0.007145, 0.000128, 0, 0, 0, 0};
static float ITA_afters_23[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0.517241, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.275862, 0, 0.206897, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_24[] = { 0.289507, 0, 0, 0, 0.043616, 0, 0, 0, 0.291403, 0, 0, 0, 0, 0, 0.097661, 0, 0, 0, 0, 0, 0.009798, 0, 0, 0, 0.260430, 0, 0, 0, 0, 0.007585, 0, 0, 0, 0, 0};
static float ITA_afters_25[] = { 0, 0, 0, 0, 0, 0, 0.500000, 0, 0.250000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.250000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_26[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0.250000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.250000, 0.500000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_27[] = { 0.090909, 0, 0, 0, 0.181818, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.727273, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_28[] = { 0.500000, 0, 0, 0, 0.500000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_29[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.500000, 0, 0, 0, 0, 0, 0, 0.500000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_30[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.666667, 0, 0, 0, 0, 0, 0.333333, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_32[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_33[] = { 0.000299, 0, 0, 0, 0.642783, 0, 0, 0, 0.301687, 0, 0, 0, 0, 0, 0.000149, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.054934, 0.000149, 0, 0, 0, 0, 0, 0, 0};
static float ITA_afters_34[] = { 0.349547, 0, 0, 0, 0.588105, 0, 0, 0, 0.061021, 0, 0, 0, 0, 0, 0.000442, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.000884, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static struct letter_frequencies ITA_counts[] = {
    { "a", 0.350389, 0.067801, 0.110597, 0.230215, ITA_afters_0 },
    { "b", 0.000505, 0.021537, 0.009475, 0.000000, ITA_afters_1 },
    { "c", 0.013220, 0.071198, 0.035259, 0.000000, ITA_afters_2 },
    { "d", 0.013927, 0.118102, 0.026298, 0.004809, ITA_afters_3 },
    { "e", 0.453830, 0.020798, 0.105175, 0.219559, ITA_afters_4 },
    { "f", 0.000000, 0.043207, 0.004565, 0.000000, ITA_afters_5 },
    { "g", 0.002826, 0.034976, 0.023443, 0.000008, ITA_afters_6 },
    { "h", 0.000000, 0.002268, 0.001158, 0.001669, ITA_afters_7 },
    { "i", 0.100010, 0.049985, 0.086729, 0.180936, ITA_afters_8 },
    { "j", 0.000000, 0.000008, 0.000029, 0.000000, ITA_afters_9 },
    { "k", 0.000000, 0.000000, 0.000003, 0.000000, ITA_afters_10 },
    { "l", 0.000202, 0.067868, 0.079392, 0.046181, ITA_afters_11 },
    { "m", 0.000404, 0.056223, 0.022986, 0.000075, ITA_afters_12 },
    { "n", 0.000404, 0.036970, 0.085505, 0.054587, ITA_afters_13 },
    { "o", 0.036936, 0.010706, 0.082776, 0.195746, ITA_afters_14 },
    { "p", 0.000101, 0.086930, 0.016993, 0.000000, ITA_afters_15 },
    { "q", 0.000000, 0.000000, 0.000003, 0.000000, ITA_afters_16 },
    { "r", 0.000908, 0.021562, 0.092845, 0.020233, ITA_afters_17 },
    { "s", 0.000303, 0.113019, 0.057276, 0.000133, ITA_afters_18 },
    { "t", 0.000000, 0.035192, 0.081382, 0.000050, ITA_afters_19 },
    { "u", 0.000000, 0.025483, 0.031739, 0.005141, ITA_afters_20 },
    { "v", 0.000303, 0.041712, 0.030605, 0.000050, ITA_afters_21 },
    { "w", 0.000000, 0.000000, 0.000000, 0.000017, NULL },
    { "x", 0.000202, 0.000199, 0.000014, 0.000050, ITA_afters_23 },
    { "z", 0.000000, 0.000939, 0.008765, 0.000000, ITA_afters_24 },
    { "à", 0.000000, 0.000000, 0.000011, 0.007583, ITA_afters_25 },
    { "è", 0.025532, 0.000000, 0.000011, 0.007625, ITA_afters_26 },
    { "ì", 0.000000, 0.000000, 0.000063, 0.005889, ITA_afters_27 },
    { "í", 0.000000, 0.000000, 0.000006, 0.000000, ITA_afters_28 },
    { "ò", 0.000000, 0.000000, 0.000006, 0.011803, ITA_afters_29 },
    { "ô", 0.000000, 0.000000, 0.000009, 0.000000, ITA_afters_30 },
    { "ù", 0.000000, 0.000000, 0.000000, 0.007641, NULL },
    { "ü", 0.000000, 0.000000, 0.000003, 0.000000, ITA_afters_32 },
    { "ch", 0.000000, 0.038357, 0.005978, 0.000000, ITA_afters_33 },
    { "qu", 0.000000, 0.034960, 0.000902, 0.000000, ITA_afters_34 },
    NULL
};	/* ITA */
static float ITA_word_lens[] = {
    0.000000, 0.076045, 0.214735, 0.119696, 0.090449, 0.171682, 0.096397, 0.086628, 0.060159, 0.037343, 0.024335, 0.011719, 0.006815, 0.002425, 0.001036, 0.000422, 0.000069, 0.000008, 0.000038, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float ITA_all_consonants[] = {
    0.000000, 0.002533, 0.000031, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 
};

static float ITA_consonant_run[] = {
    0.000000, 0.713476, 0.273440, 0.013079, 0.000004, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float ITA_vowel_run[] = {
    0.000000, 0.920855, 0.076637, 0.002265, 0.000243, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float JAP_afters_0[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float JAP_afters_1[] = { 0, 0, 0, 0.110857, 0, 0.005714, 0, 0.001143, 0, 0.002286, 0.046857, 0.005714, 0.069714, 0, 0.011429, 0, 0.008000, 0.102857, 0, 0, 0.026286, 0, 0.016000, 0.009143, 0.002286, 0.019429, 0, 0, 0, 0, 0.064000, 0, 0.001143, 0, 0.045714, 0.024000, 0, 0.010286, 0, 0.022857, 0, 0.001143, 0, 0, 0, 0, 0, 0.001143, 0, 0, 0.001143, 0, 0.012571, 0.002286, 0, 0, 0, 0, 0, 0, 0, 0.008000, 0, 0.003429, 0.005714, 0, 0, 0.002286, 0, 0.001143, 0, 0, 0.034286, 0.038857, 0.152000, 0.003429, 0.001143, 0.010286, 0, 0.115429};
static float JAP_afters_2[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.375000, 0, 0, 0, 0, 0, 0.625000};
static float JAP_afters_3[] = { 0, 0.003314, 0, 0.022464, 0, 0.014730, 0, 0.007365, 0, 0.003867, 0.033143, 0.010495, 0.040140, 0.012889, 0.001473, 0.000552, 0.031854, 0.004051, 0.042902, 0.004051, 0.023384, 0.012152, 0.069048, 0.024857, 0.004235, 0.001473, 0.021359, 0.002578, 0.002210, 0.003867, 0.026883, 0.008286, 0.198306, 0, 0.022832, 0.008838, 0.000368, 0.034248, 0.002762, 0.024121, 0.009575, 0.006260, 0.006813, 0.000368, 0.009575, 0.002210, 0.004603, 0.002578, 0, 0.010495, 0.002025, 0, 0.018229, 0.005524, 0, 0.002762, 0.000184, 0.000368, 0.007918, 0.000921, 0, 0.002210, 0.004235, 0.002946, 0.003683, 0.001289, 0, 0.004419, 0, 0.000552, 0, 0.006444, 0.006629, 0.019886, 0.060026, 0.011784, 0.000921, 0.003867, 0, 0.055607};
static float JAP_afters_4[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float JAP_afters_5[] = { 0, 0.006747, 0.000409, 0.031691, 0, 0.000204, 0.006952, 0.013903, 0, 0.010018, 0.073196, 0.005111, 0.056021, 0.030260, 0.001227, 0.000613, 0.028420, 0.005725, 0.023513, 0.032509, 0.041914, 0.001022, 0.111634, 0.032100, 0.002453, 0, 0.061951, 0.008383, 0.010223, 0.001636, 0.010018, 0.013290, 0.030669, 0, 0.003680, 0.001840, 0.000204, 0.015948, 0.002453, 0.008996, 0.041709, 0.013699, 0.028420, 0.000204, 0.000409, 0.002045, 0.012063, 0.015743, 0, 0.014516, 0.000409, 0, 0.004294, 0.005520, 0, 0.001227, 0.003885, 0, 0.011654, 0.001636, 0, 0.003680, 0.004498, 0.005520, 0.010427, 0.004294, 0, 0.003680, 0, 0.000409, 0, 0.011450, 0.010018, 0.067062, 0.002658, 0.003476, 0.010836, 0.011041, 0, 0.008587};
static float JAP_afters_6[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.264706, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.323529, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.029412, 0.058824, 0, 0, 0, 0, 0, 0.323529};
static float JAP_afters_7[] = { 0, 0, 0, 0.231111, 0, 0, 0, 0, 0, 0.004444, 0, 0.001481, 0.071111, 0, 0, 0, 0, 0, 0.001481, 0, 0.001481, 0.001481, 0.005926, 0.001481, 0.004444, 0, 0, 0, 0, 0, 0.068148, 0.001481, 0, 0, 0.001481, 0, 0, 0.031111, 0, 0, 0, 0, 0, 0, 0, 0, 0.001481, 0.007407, 0, 0, 0.044444, 0, 0.002963, 0.001481, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.002963, 0, 0, 0, 0, 0, 0, 0, 0.002963, 0.010370, 0.016296, 0.060741, 0.004444, 0, 0, 0, 0.417778};
static float JAP_afters_8[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0};
static float JAP_afters_9[] = { 0, 0, 0, 0.029851, 0, 0.130597, 0, 0.004664, 0, 0.138993, 0.031716, 0.002799, 0.046642, 0, 0.210821, 0.002799, 0.000933, 0, 0.027052, 0.005597, 0.021455, 0.005597, 0.068097, 0, 0.007463, 0.000933, 0.000933, 0.001866, 0.011194, 0.000933, 0.000933, 0, 0.012127, 0, 0.001866, 0, 0, 0.017724, 0, 0.006530, 0.015858, 0.007463, 0.006530, 0, 0, 0.004664, 0.027052, 0, 0, 0, 0.001866, 0, 0, 0, 0, 0, 0, 0, 0, 0.000933, 0, 0.003731, 0, 0, 0, 0.019590, 0, 0.002799, 0, 0, 0, 0.009328, 0.000933, 0.060634, 0, 0, 0.004664, 0.033582, 0, 0.010261};
static float JAP_afters_10[] = { 0, 0.001557, 0, 0.281445, 0, 0.002802, 0, 0.010274, 0, 0, 0.008095, 0.007161, 0.000311, 0.003736, 0.096824, 0, 0.010585, 0.009963, 0.005915, 0.004981, 0.001868, 0.000623, 0.017123, 0.012453, 0.000623, 0.005293, 0.002802, 0.004359, 0.001245, 0.000623, 0.030822, 0.002802, 0.002802, 0.000311, 0.025218, 0.019925, 0.000311, 0.002179, 0.000623, 0.001557, 0.001245, 0.006227, 0.010274, 0, 0.011519, 0.019303, 0.002179, 0.001557, 0, 0.000934, 0, 0, 0.001245, 0.031133, 0, 0, 0.002179, 0, 0.000623, 0, 0, 0.004047, 0.010897, 0.001245, 0.013076, 0.002802, 0, 0.000623, 0, 0, 0, 0, 0.082192, 0.005915, 0.005915, 0.001245, 0, 0.018369, 0, 0.188045};
static float JAP_afters_11[] = { 0, 0, 0, 0.291188, 0, 0, 0, 0.063218, 0, 0, 0.005747, 0, 0.005747, 0, 0.183908, 0, 0.007663, 0, 0, 0, 0.005747, 0, 0.040230, 0, 0.003831, 0, 0.001916, 0, 0, 0, 0.051724, 0, 0.005747, 0, 0.036398, 0, 0, 0, 0, 0, 0, 0, 0.001916, 0, 0.013410, 0.001916, 0, 0, 0, 0, 0.001916, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.007663, 0, 0.001916, 0, 0.005747, 0, 0, 0.005747, 0, 0, 0, 0, 0.049808, 0.019157, 0.021073, 0.015326, 0, 0.136015, 0, 0.015326};
static float JAP_afters_12[] = { 0, 0.002380, 0, 0.006800, 0, 0.000680, 0, 0, 0, 0.003740, 0.018361, 0.002040, 0.007140, 0.012921, 0.006120, 0.002040, 0.000340, 0.001020, 0.007480, 0.006120, 0.004080, 0.000680, 0.014281, 0.005780, 0.001360, 0.002720, 0.009181, 0, 0.005440, 0, 0.029242, 0.004080, 0.004080, 0, 0.002040, 0.003400, 0.000680, 0.007820, 0.002040, 0.001700, 0.005100, 0.022781, 0.020741, 0.000340, 0.002040, 0.002040, 0.001700, 0, 0, 0.001020, 0.002720, 0, 0.000340, 0.000680, 0, 0, 0, 0, 0.003400, 0.003740, 0, 0.003060, 0, 0.000340, 0.008160, 0.002380, 0.007820, 0.001020, 0.313839, 0.004760, 0.282897, 0.004760, 0.013261, 0.006460, 0.007820, 0.005100, 0.003060, 0.002380, 0, 0.090445};
static float JAP_afters_13[] = { 0, 0.001727, 0, 0.056995, 0, 0.003454, 0, 0.001727, 0, 0, 0.050086, 0, 0.001727, 0, 0, 0, 0.001727, 0, 0, 0, 0, 0, 0.082902, 0.024180, 0.001727, 0, 0.010363, 0, 0, 0, 0.001727, 0.001727, 0.046632, 0, 0.008636, 0.001727, 0, 0.001727, 0.001727, 0, 0, 0.005181, 0.001727, 0, 0, 0.005181, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001727, 0.008636, 0, 0.008636, 0.001727, 0, 0, 0, 0.478411, 0, 0.003454, 0.018998, 0.001727, 0, 0.008636, 0.015544, 0, 0.139896};
static float JAP_afters_14[] = { 0, 0.002913, 0, 0.032775, 0, 0.025492, 0, 0.010197, 0, 0.018208, 0.003642, 0.000728, 0.033503, 0.024035, 0.001457, 0.000728, 0, 0.019665, 0, 0.015295, 0.059723, 0.002185, 0.057538, 0.027677, 0.005827, 0.001457, 0.024763, 0.025492, 0.005098, 0.000728, 0.004370, 0.030590, 0.019665, 0, 0.001457, 0.001457, 0.000728, 0.055353, 0.001457, 0.014567, 0.002185, 0.037145, 0.061908, 0, 0.015295, 0.004370, 0.020393, 0.002913, 0, 0.014567, 0, 0, 0.001457, 0.006555, 0, 0, 0.015295, 0, 0.011653, 0.008740, 0, 0.003642, 0.045885, 0.011653, 0.012382, 0.014567, 0, 0.000728, 0, 0, 0, 0.008740, 0.039330, 0.042243, 0.014567, 0.030590, 0.021850, 0.016023, 0, 0.006555};
static float JAP_afters_15[] = { 0, 0.033613, 0, 0, 0, 0.008403, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.067227, 0, 0.142857, 0, 0.050420, 0, 0, 0, 0, 0, 0, 0.008403, 0.016807, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.008403, 0.042017, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.033613, 0.092437, 0.100840, 0.016807, 0.042017, 0.008403, 0, 0.327731};
static float JAP_afters_16[] = { 0, 0.000756, 0, 0.352986, 0, 0, 0, 0, 0, 0, 0, 0, 0.006803, 0.000756, 0, 0, 0, 0, 0, 0, 0, 0.000756, 0.002268, 0.000756, 0, 0, 0, 0, 0, 0, 0.026455, 0.001512, 0, 0, 0.062736, 0.087680, 0, 0.029478, 0.002268, 0.004535, 0, 0.001512, 0, 0, 0.006047, 0, 0.003023, 0.003023, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001512, 0.004535, 0, 0.001512, 0, 0, 0, 0, 0, 0, 0, 0, 0.018141, 0.010582, 0, 0.001512, 0, 0.368859};
static float JAP_afters_17[] = { 0, 0, 0, 0.008333, 0, 0, 0, 0, 0, 0, 0.002083, 0, 0.070833, 0, 0, 0, 0, 0, 0.006250, 0, 0, 0, 0.025000, 0.002083, 0, 0, 0, 0, 0, 0, 0.035417, 0.020833, 0, 0, 0.006250, 0.141667, 0, 0.018750, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.020833, 0, 0.018750, 0, 0, 0, 0, 0.622917};
static float JAP_afters_18[] = { 0, 0, 0, 0.000935, 0, 0.324918, 0, 0.005610, 0, 0, 0, 0.008883, 0.005143, 0.001403, 0.219729, 0.000468, 0, 0, 0.004208, 0, 0.001403, 0, 0.009350, 0.009350, 0.002338, 0, 0.000468, 0, 0.000468, 0, 0.003740, 0.001403, 0.000468, 0, 0.020103, 0, 0, 0.000935, 0, 0.108462, 0.001870, 0.011688, 0, 0, 0, 0.034596, 0.006545, 0.004208, 0, 0, 0, 0, 0.000468, 0.001403, 0, 0, 0.000935, 0.000468, 0, 0, 0, 0.009350, 0.005143, 0.007480, 0.070594, 0.003273, 0, 0.000935, 0, 0, 0, 0.006545, 0, 0.003740, 0.001403, 0.021505, 0.016830, 0.000468, 0, 0.060776};
static float JAP_afters_19[] = { 0, 0, 0, 0.028986, 0, 0.215781, 0, 0, 0, 0.024155, 0.004831, 0, 0.064412, 0, 0.064412, 0, 0, 0, 0.001610, 0.114332, 0.045089, 0, 0.022544, 0, 0, 0, 0, 0.140097, 0, 0, 0, 0, 0.008052, 0, 0, 0, 0, 0.057971, 0, 0.009662, 0, 0.019324, 0.022544, 0, 0, 0, 0.027375, 0, 0, 0.001610, 0.014493, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.006441, 0.012882, 0.003221, 0, 0, 0, 0.004831, 0, 0, 0, 0.008052, 0, 0.004831, 0, 0, 0.072464, 0, 0, 0};
static float JAP_afters_20[] = { 0, 0, 0, 0.238695, 0, 0.001717, 0, 0.008014, 0, 0.001145, 0.016600, 0.013165, 0.029193, 0.004579, 0.076703, 0.001145, 0.006297, 0.013738, 0, 0, 0.005152, 0, 0.005152, 0, 0.000572, 0, 0, 0, 0, 0, 0, 0.006869, 0.001717, 0, 0.004007, 0.018317, 0, 0, 0, 0.001717, 0, 0.000572, 0, 0, 0, 0, 0.002290, 0.000572, 0, 0.006869, 0.001145, 0, 0, 0.000572, 0, 0, 0.002290, 0, 0, 0, 0, 0.002290, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001717, 0.017745, 0, 0, 0.000572, 0, 0.002290, 0, 0.506583};
static float JAP_afters_21[] = { 0, 0, 0, 0.814394, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.003788, 0, 0, 0, 0, 0.003788, 0, 0.030303, 0, 0.037879, 0, 0, 0, 0, 0, 0, 0, 0.003788, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.011364, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.003788, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.003788, 0, 0.011364, 0, 0, 0.022727, 0, 0.053030};
static float JAP_afters_22[] = { 0, 0.012964, 0, 0.010729, 0, 0.000224, 0, 0.002459, 0, 0.000447, 0.011623, 0.001118, 0.040232, 0.000224, 0.006035, 0.000224, 0.002459, 0.001341, 0.001118, 0, 0.007376, 0.000671, 0.004247, 0.025481, 0.010729, 0.001118, 0.004470, 0.000447, 0.000224, 0, 0.117792, 0.002012, 0.004023, 0, 0.001565, 0.011846, 0, 0.102816, 0, 0.002459, 0.003576, 0.004023, 0.000447, 0, 0.000224, 0.000224, 0.006705, 0.001788, 0, 0.002459, 0, 0, 0.000224, 0.001565, 0, 0, 0, 0, 0.002682, 0.000894, 0, 0.003353, 0.002235, 0.001118, 0.015199, 0.003576, 0.109522, 0, 0.162271, 0, 0.179705, 0.002682, 0.003353, 0.002682, 0.000671, 0.002459, 0.001341, 0.000447, 0, 0.096111};
static float JAP_afters_23[] = { 0, 0.008880, 0, 0.001366, 0, 0, 0, 0.004781, 0, 0.000683, 0.036885, 0.000683, 0.012295, 0.026639, 0.001366, 0, 0.019126, 0, 0.005464, 0, 0, 0, 0.010246, 0.012978, 0, 0, 0, 0.002732, 0, 0.002049, 0.029372, 0.000683, 0.006831, 0, 0.045765, 0.043033, 0, 0.012295, 0, 0.000683, 0.012978, 0, 0.010929, 0, 0, 0.002049, 0.000683, 0, 0, 0, 0, 0, 0, 0.003415, 0.000683, 0.003415, 0, 0, 0, 0, 0, 0.012295, 0.013661, 0.019126, 0.019126, 0.004098, 0.007514, 0, 0.161202, 0.008197, 0.353142, 0, 0, 0.004098, 0.008197, 0.001366, 0.001366, 0, 0, 0.067623};
static float JAP_afters_24[] = { 0, 0.001122, 0, 0.093154, 0, 0.114478, 0, 0.003367, 0, 0, 0.003367, 0.006734, 0.026936, 0.016835, 0.037037, 0.005612, 0, 0.001122, 0.010101, 0, 0, 0.001122, 0, 0.024691, 0.032548, 0, 0, 0, 0, 0, 0.004489, 0, 0, 0, 0, 0, 0, 0.007856, 0.013468, 0.016835, 0, 0.006734, 0.041526, 0, 0.005612, 0, 0, 0, 0, 0.006734, 0.003367, 0, 0, 0, 0, 0, 0, 0.002245, 0, 0, 0, 0.002245, 0.010101, 0.011223, 0.016835, 0.005612, 0, 0, 0, 0, 0, 0, 0.003367, 0.001122, 0.430976, 0.013468, 0.005612, 0, 0, 0.012346};
static float JAP_afters_25[] = { 0, 0, 0, 0, 0, 0, 0, 0.077778, 0, 0.011111, 0.233333, 0, 0.044444, 0, 0, 0, 0.133333, 0, 0.022222, 0, 0, 0, 0.055556, 0, 0, 0, 0, 0, 0, 0, 0, 0.033333, 0, 0, 0, 0.033333, 0, 0, 0, 0, 0, 0, 0, 0.044444, 0, 0, 0, 0, 0, 0.011111, 0, 0, 0, 0, 0, 0.033333, 0.011111, 0, 0, 0, 0, 0.077778, 0.022222, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.022222, 0.022222, 0.011111, 0.100000, 0, 0, 0, 0};
static float JAP_afters_26[] = { 0, 0, 0, 0.492950, 0, 0, 0, 0, 0, 0, 0.010716, 0, 0.050761, 0, 0, 0.000564, 0, 0, 0, 0, 0, 0, 0, 0, 0.000564, 0, 0, 0, 0, 0, 0.006768, 0, 0, 0, 0.019741, 0.059222, 0, 0.002820, 0, 0, 0, 0, 0, 0, 0, 0, 0.000564, 0.000564, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.002820, 0, 0, 0.000564, 0, 0, 0, 0, 0, 0, 0.002256, 0, 0.001128, 0.024253, 0, 0.001692, 0, 0, 0.322053};
static float JAP_afters_27[] = { 0, 0, 0, 0.045630, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.002320, 0.009281, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001547, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.002320, 0, 0.730085, 0, 0, 0.208817};
static float JAP_afters_28[] = { 0, 0, 0, 0.006221, 0, 0.566096, 0, 0, 0, 0, 0, 0.001555, 0, 0, 0.135303, 0, 0, 0, 0.015552, 0, 0, 0.006221, 0.057543, 0.001555, 0, 0, 0, 0.001555, 0.001555, 0, 0, 0.003110, 0.023328, 0, 0.004666, 0.010886, 0, 0, 0, 0, 0, 0.006221, 0, 0, 0.001555, 0.068429, 0, 0.001555, 0, 0, 0, 0, 0.003110, 0, 0, 0, 0, 0, 0, 0, 0, 0.001555, 0, 0, 0, 0, 0, 0.001555, 0, 0, 0, 0, 0.001555, 0.001555, 0, 0.037325, 0.007776, 0.001555, 0, 0.031104};
static float JAP_afters_29[] = { 0, 0, 0, 0.014151, 0, 0.603774, 0, 0, 0, 0, 0, 0, 0.004717, 0, 0.278302, 0, 0.009434, 0, 0.004717, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.004717, 0.004717, 0.004717, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.028302, 0, 0, 0, 0.042453};
static float JAP_afters_30[] = { 0, 0, 0, 0.412738, 0, 0.000827, 0, 0.038048, 0, 0, 0.031431, 0.014061, 0.002481, 0, 0.054591, 0, 0.006617, 0.000827, 0.000827, 0.000827, 0.001654, 0.000827, 0.010753, 0.001654, 0.005790, 0.000827, 0.000827, 0, 0, 0, 0.013234, 0.011580, 0.049628, 0, 0.023160, 0.006617, 0.002481, 0.013234, 0, 0.005790, 0, 0.010753, 0.004963, 0, 0.001654, 0.004136, 0.001654, 0.000827, 0, 0.000827, 0.005790, 0, 0.003309, 0, 0, 0, 0, 0, 0.000827, 0, 0, 0.014888, 0.000827, 0.000827, 0.106700, 0.005790, 0, 0, 0, 0, 0, 0.001654, 0.023160, 0.022333, 0.015715, 0, 0.001654, 0.001654, 0, 0.058726};
static float JAP_afters_31[] = { 0, 0, 0, 0.439181, 0, 0, 0, 0.002561, 0, 0.002561, 0.034571, 0, 0.010243, 0, 0, 0, 0.024328, 0.001280, 0, 0, 0.007682, 0, 0.046095, 0, 0.002561, 0, 0.002561, 0, 0, 0, 0.001280, 0, 0.006402, 0, 0.093470, 0.010243, 0, 0.002561, 0, 0.005122, 0, 0, 0, 0, 0.003841, 0, 0.001280, 0, 0, 0, 0.001280, 0, 0, 0.001280, 0, 0, 0, 0.003841, 0, 0, 0, 0.002561, 0, 0.001280, 0.014085, 0, 0, 0, 0, 0, 0, 0, 0, 0.002561, 0, 0.002561, 0.003841, 0.001280, 0, 0.267606};
static float JAP_afters_32[] = { 0, 0.003400, 0, 0.048567, 0, 0, 0.010685, 0, 0, 0.020884, 0.021370, 0.002914, 0.047110, 0.006314, 0.003885, 0, 0.002914, 0, 0.009713, 0.033511, 0.020884, 0, 0.002914, 0.007771, 0.000486, 0, 0, 0.075279, 0, 0, 0.002428, 0.006799, 0.007285, 0.000971, 0, 0.001457, 0.000486, 0.048567, 0.001457, 0, 0, 0.013113, 0.038854, 0.000486, 0, 0.000486, 0.018456, 0.003885, 0, 0, 0.000486, 0, 0, 0.010199, 0, 0, 0.000971, 0, 0.020884, 0, 0, 0.006314, 0, 0, 0.000971, 0, 0.016027, 0.002428, 0.107819, 0, 0.323944, 0.015542, 0.001457, 0.001457, 0, 0.000971, 0.021370, 0, 0, 0.005828};
static float JAP_afters_33[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.250000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.500000, 0, 0, 0, 0, 0, 0, 0, 0, 0.250000, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float JAP_afters_34[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.076169, 0, 0.026395, 0, 0.003017, 0, 0.015837, 0, 0.037707, 0, 0.011312, 0, 0.076923, 0, 0.003017, 0, 0.043741, 0, 0.012066, 0, 0.263198, 0, 0.033937, 0, 0, 0.002262, 0, 0.164404, 0, 0.021870, 0.001508, 0, 0, 0, 0, 0, 0, 0, 0.027903, 0, 0, 0.060332, 0, 0, 0.003771, 0, 0, 0.003017, 0, 0, 0.111614, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float JAP_afters_35[] = { 0, 0, 0.008666, 0.128250, 0, 0.099653, 0.007799, 0.004333, 0, 0.025130, 0.027730, 0.002600, 0.220971, 0.028596, 0.010399, 0.004333, 0.019064, 0.026863, 0, 0, 0.000867, 0.000867, 0.016464, 0.008666, 0, 0, 0, 0.006066, 0, 0, 0.009532, 0.007799, 0.003466, 0, 0.000867, 0.006932, 0.061525, 0.004333, 0.001733, 0.013865, 0.014731, 0.019064, 0.004333, 0, 0.000867, 0.005199, 0, 0.003466, 0, 0.003466, 0.028596, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.008666, 0.006066, 0.000867, 0.013865, 0.012998, 0, 0.004333, 0, 0, 0, 0.074523, 0.000867, 0.025997, 0, 0.006932, 0.006066, 0.000867, 0, 0.000867};
static float JAP_afters_36[] = { 0, 0, 0, 0.126126, 0, 0, 0, 0, 0, 0, 0, 0, 0.306306, 0, 0.126126, 0, 0.342342, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.018018, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.009009, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.054054, 0.018018, 0, 0, 0, 0, 0, 0, 0};
static float JAP_afters_37[] = { 0, 0.000834, 0, 0.262719, 0, 0, 0, 0, 0, 0, 0, 0.001668, 0.158465, 0, 0, 0.000834, 0, 0, 0, 0, 0, 0, 0, 0.001668, 0, 0, 0, 0, 0, 0, 0.003336, 0, 0, 0, 0.010842, 0.035863, 0, 0, 0, 0, 0, 0, 0, 0, 0.000834, 0, 0, 0, 0, 0, 0.000834, 0, 0, 0, 0, 0.005004, 0, 0, 0, 0, 0, 0, 0, 0.001668, 0, 0.006672, 0, 0, 0, 0, 0, 0, 0.005004, 0, 0.004170, 0, 0, 0, 0, 0.499583};
static float JAP_afters_38[] = { 0, 0.226316, 0, 0, 0, 0, 0, 0, 0, 0.005263, 0, 0, 0.184211, 0, 0, 0, 0, 0, 0, 0, 0.005263, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.026316, 0, 0, 0, 0, 0, 0.005263, 0.036842, 0, 0, 0, 0, 0.052632, 0, 0, 0, 0.010526, 0, 0, 0, 0.021053, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.126316, 0, 0, 0, 0, 0, 0, 0, 0.005263, 0.015789, 0, 0, 0, 0, 0.278947};
static float JAP_afters_39[] = { 0, 0, 0, 0.002807, 0, 0.530526, 0, 0.004211, 0, 0.032281, 0.002807, 0, 0.014035, 0.002105, 0.056842, 0, 0.005614, 0, 0.021754, 0, 0.000702, 0, 0.036491, 0.001404, 0, 0.002105, 0.000702, 0, 0, 0, 0, 0.001404, 0.011228, 0, 0.018246, 0.001404, 0.003509, 0, 0, 0.001404, 0.014035, 0.007018, 0.001404, 0, 0, 0.001404, 0, 0.001404, 0, 0, 0.000702, 0, 0, 0, 0, 0, 0, 0, 0, 0.004912, 0, 0.002105, 0.002105, 0.000702, 0.061053, 0.035088, 0, 0, 0, 0, 0, 0.000702, 0.003509, 0.058947, 0.009825, 0.001404, 0.000702, 0.002105, 0, 0.039298};
static float JAP_afters_40[] = { 0, 0.001149, 0, 0.041379, 0.002299, 0.651724, 0, 0, 0, 0.006897, 0, 0, 0.014943, 0, 0.028736, 0.001149, 0.001149, 0, 0.003448, 0, 0.003448, 0, 0.018391, 0, 0.003448, 0, 0, 0, 0.001149, 0, 0, 0.002299, 0.002299, 0, 0.004598, 0, 0, 0, 0, 0, 0, 0, 0.002299, 0, 0, 0.004598, 0, 0.001149, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001149, 0, 0.017241, 0, 0, 0.005747, 0.004598, 0, 0, 0, 0, 0, 0, 0, 0.018391, 0.102299, 0, 0.001149, 0, 0, 0.052874};
static float JAP_afters_41[] = { 0, 0, 0, 0.207823, 0, 0.007589, 0, 0.002335, 0, 0.021600, 0.050204, 0.025102, 0.001751, 0, 0.022183, 0, 0.006421, 0.000584, 0, 0.014594, 0.009924, 0.000584, 0.011675, 0.003503, 0.001168, 0.003503, 0.001168, 0.023351, 0, 0, 0, 0.007005, 0.001168, 0, 0.070636, 0, 0, 0.015762, 0, 0.011675, 0.141856, 0.154116, 0.014594, 0, 0, 0, 0.004670, 0, 0, 0.000584, 0.000584, 0, 0, 0.021600, 0, 0, 0.001751, 0, 0, 0, 0, 0.001168, 0.012843, 0.002919, 0.001168, 0.001751, 0, 0.002919, 0, 0, 0, 0.004086, 0.016929, 0.011675, 0.040280, 0.002919, 0.002919, 0.004086, 0, 0.033275};
static float JAP_afters_42[] = { 0, 0.023871, 0, 0.010379, 0, 0, 0, 0, 0, 0.006746, 0, 0, 0.185781, 0.001038, 0.002076, 0, 0, 0.001038, 0, 0.023352, 0.006227, 0, 0.008822, 0.000519, 0.001038, 0, 0.000519, 0.036845, 0, 0, 0, 0, 0.353918, 0, 0.065387, 0, 0, 0.022314, 0, 0, 0, 0.007784, 0.008303, 0, 0, 0, 0.022314, 0, 0, 0, 0.002076, 0, 0, 0.002076, 0, 0, 0, 0, 0.006746, 0, 0, 0.002595, 0, 0, 0, 0.000519, 0, 0, 0.069538, 0, 0.000519, 0.007265, 0.001038, 0, 0, 0, 0.012974, 0.006746, 0, 0.099637};
static float JAP_afters_43[] = { 0, 0, 0, 0.555556, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.222222, 0, 0, 0, 0, 0, 0.111111, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.111111, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float JAP_afters_44[] = { 0, 0.001887, 0, 0.005660, 0, 0, 0, 0, 0, 0.001887, 0.007547, 0, 0.001887, 0, 0.003774, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001887, 0, 0, 0, 0.013208, 0.003774, 0.001887, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001887, 0.011321, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.005660, 0, 0, 0, 0, 0, 0, 0.013208, 0.003774, 0, 0, 0, 0, 0, 0.920755};
static float JAP_afters_45[] = { 0, 0, 0, 0, 0, 0.274788, 0, 0, 0, 0, 0.002833, 0.002833, 0.022663, 0, 0, 0, 0, 0, 0.042493, 0.053824, 0.016997, 0, 0, 0.008499, 0, 0, 0, 0, 0, 0.033994, 0.011331, 0, 0.019830, 0, 0.002833, 0, 0, 0, 0.036827, 0, 0, 0.002833, 0.002833, 0, 0, 0, 0, 0.014164, 0, 0, 0.118980, 0, 0, 0, 0, 0, 0.130312, 0, 0.002833, 0.016997, 0, 0.014164, 0.014164, 0, 0, 0.022663, 0, 0, 0, 0, 0, 0.016997, 0, 0.014164, 0.090652, 0, 0, 0, 0, 0.008499};
static float JAP_afters_46[] = { 0, 0, 0, 0.056478, 0, 0, 0, 0, 0, 0, 0.011628, 0, 0, 0, 0.009967, 0, 0.019934, 0.006645, 0.002492, 0, 0.000831, 0, 0.011628, 0.036545, 0, 0.003322, 0, 0, 0, 0, 0.008306, 0, 0.248339, 0, 0.111296, 0.058970, 0, 0, 0, 0, 0.003322, 0.054817, 0.006645, 0, 0, 0, 0.002492, 0.040698, 0, 0, 0, 0, 0.000831, 0, 0, 0.001661, 0, 0, 0, 0, 0, 0.006645, 0, 0, 0, 0.000831, 0, 0.011628, 0, 0, 0, 0, 0.019934, 0.000831, 0.004983, 0.013289, 0, 0, 0, 0.245017};
static float JAP_afters_47[] = { 0, 0.065657, 0, 0.328283, 0, 0, 0, 0, 0, 0, 0.015152, 0, 0.005051, 0, 0.136364, 0.005051, 0, 0, 0.005051, 0, 0, 0, 0.055556, 0, 0.020202, 0, 0, 0, 0.005051, 0, 0.005051, 0, 0, 0, 0, 0.035354, 0, 0, 0, 0, 0, 0.025253, 0.050505, 0, 0, 0, 0, 0, 0, 0.005051, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.005051, 0.015152, 0, 0, 0.005051, 0, 0, 0, 0, 0.035354, 0, 0, 0.025253, 0, 0.005051, 0, 0.146465};
static float JAP_afters_48[] = { 0, 0, 0, 0.097561, 0, 0, 0, 0, 0, 0, 0, 0, 0.024390, 0, 0.073171, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.402439, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.109756, 0.012195, 0, 0, 0, 0, 0.280488};
static float JAP_afters_49[] = { 0, 0.001548, 0, 0.001548, 0, 0, 0, 0, 0, 0, 0.027864, 0.029412, 0.108359, 0, 0.023220, 0.001548, 0.029412, 0, 0.015480, 0.001548, 0.003096, 0, 0.032508, 0.018576, 0.001548, 0, 0.001548, 0, 0, 0, 0.001548, 0.006192, 0.003096, 0, 0.003096, 0.054180, 0.013932, 0.003096, 0, 0.071207, 0.001548, 0.010836, 0.003096, 0, 0, 0.003096, 0.041796, 0.003096, 0, 0, 0.035604, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.007740, 0, 0, 0, 0.255418, 0.001548, 0.051084, 0.013932, 0.003096, 0.021672, 0.041796, 0, 0, 0.051084};
static float JAP_afters_50[] = { 0, 0.162393, 0, 0.017094, 0, 0, 0, 0.008547, 0, 0, 0.051282, 0, 0.017094, 0, 0.025641, 0, 0, 0, 0, 0, 0, 0, 0.213675, 0, 0.008547, 0, 0, 0, 0, 0.017094, 0.042735, 0.034188, 0, 0, 0.017094, 0.034188, 0, 0.017094, 0, 0.042735, 0, 0.119658, 0.008547, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.136752, 0, 0.008547, 0, 0, 0, 0, 0, 0, 0.017094};
static float JAP_afters_51[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.934066, 0, 0, 0, 0, 0, 0, 0, 0, 0.065934};
static float JAP_afters_52[] = { 0.009009, 0.009009, 0.009009, 0, 0, 0.009009, 0.002252, 0.040541, 0.004505, 0, 0.020270, 0.002252, 0.027027, 0, 0.281532, 0, 0.020270, 0, 0.004505, 0.002252, 0.018018, 0, 0.027027, 0.033784, 0.002252, 0, 0.006757, 0.020270, 0.024775, 0, 0.022523, 0.024775, 0.004505, 0, 0.004505, 0.004505, 0, 0, 0, 0.011261, 0.038288, 0, 0, 0, 0, 0, 0, 0, 0, 0.002252, 0.002252, 0, 0.002252, 0, 0, 0.002252, 0, 0, 0.009009, 0, 0, 0.029279, 0.029279, 0, 0.015766, 0, 0, 0.006757, 0, 0.006757, 0, 0, 0.045045, 0.029279, 0, 0.006757, 0, 0, 0, 0.128378};
static float JAP_afters_53[] = { 0, 0.010959, 0, 0.002740, 0, 0, 0, 0, 0, 0, 0.060274, 0, 0.024658, 0, 0.008219, 0, 0, 0, 0, 0, 0.101370, 0, 0.104110, 0, 0, 0, 0, 0, 0.008219, 0, 0.016438, 0.008219, 0.021918, 0, 0.060274, 0.027397, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.013699, 0, 0, 0, 0.005479, 0, 0, 0, 0.019178, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.027397, 0, 0, 0, 0, 0, 0, 0.010959, 0.095890, 0.005479, 0.005479, 0, 0, 0, 0.361644};
static float JAP_afters_54[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.400000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.200000, 0, 0, 0, 0, 0, 0.200000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.200000};
static float JAP_afters_55[] = { 0, 0, 0, 0.453237, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.007194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.014388, 0, 0, 0, 0.028777, 0, 0, 0.028777, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.079137, 0, 0.086331, 0, 0, 0, 0, 0.302158};
static float JAP_afters_56[] = { 0, 0, 0, 0.444934, 0, 0, 0, 0, 0, 0.013216, 0, 0, 0.052863, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.105727, 0, 0, 0, 0, 0.145374, 0, 0.013216, 0, 0.022026, 0, 0, 0.004405, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.035242, 0, 0, 0, 0, 0.162996};
static float JAP_afters_57[] = { 0, 0, 0, 0.222222, 0, 0, 0, 0, 0, 0, 0, 0, 0.407407, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.111111, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.074074, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.037037, 0.037037, 0, 0, 0, 0, 0.111111};
static float JAP_afters_58[] = { 0, 0, 0, 0, 0, 0.559937, 0, 0, 0, 0, 0.053628, 0, 0, 0, 0.015773, 0, 0.031546, 0, 0.001577, 0.001577, 0, 0, 0.034700, 0.012618, 0, 0, 0.003155, 0.001577, 0.001577, 0.011041, 0, 0, 0.001577, 0, 0.028391, 0, 0, 0.003155, 0, 0.020505, 0.017350, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.014196, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.006309, 0, 0.001577, 0, 0.009464, 0, 0, 0, 0, 0, 0.168770};
static float JAP_afters_59[] = { 0, 0, 0, 0, 0, 0.577381, 0, 0.005952, 0, 0, 0, 0, 0, 0, 0.011905, 0, 0, 0, 0, 0, 0, 0, 0.041667, 0.011905, 0.291667, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.011905, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.011905, 0.011905, 0.005952, 0, 0, 0, 0.017857};
static float JAP_afters_60[] = { 0, 0, 0, 0.005000, 0, 0.390000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.015000, 0, 0, 0.005000, 0, 0, 0.585000};
static float JAP_afters_61[] = { 0, 0, 0, 0.040843, 0, 0.001318, 0, 0.104084, 0, 0, 0, 0.003953, 0.003953, 0, 0.003953, 0.013175, 0, 0, 0.002635, 0.001318, 0.025033, 0.001318, 0.013175, 0, 0.009223, 0.001318, 0.001318, 0, 0, 0, 0.050066, 0.002635, 0.031621, 0, 0.050066, 0.093544, 0, 0, 0.092227, 0.030303, 0.007905, 0.005270, 0.007905, 0, 0.009223, 0.001318, 0.002635, 0.002635, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.011858, 0, 0, 0.005270, 0.013175, 0, 0.002635, 0, 0, 0, 0, 0.003953, 0.031621, 0.089592, 0.011858, 0.003953, 0.056653, 0, 0.155468};
static float JAP_afters_62[] = { 0, 0.027613, 0, 0.021696, 0, 0.013807, 0, 0.011834, 0, 0.017751, 0.041420, 0.001972, 0.007890, 0.001972, 0, 0.003945, 0, 0, 0.057199, 0, 0, 0, 0.001972, 0.001972, 0.019724, 0.025641, 0.017751, 0, 0, 0, 0.007890, 0.005917, 0.011834, 0, 0.007890, 0.071006, 0, 0.041420, 0.003945, 0.096647, 0, 0.088757, 0.003945, 0, 0, 0.007890, 0.013807, 0, 0, 0, 0, 0, 0, 0.007890, 0, 0, 0, 0, 0.005917, 0, 0, 0.007890, 0, 0, 0.007890, 0.011834, 0.001972, 0.007890, 0, 0, 0.025641, 0.007890, 0.001972, 0, 0.017751, 0.005917, 0, 0, 0, 0.254438};
static float JAP_afters_63[] = { 0, 0, 0, 0.006536, 0, 0, 0, 0, 0, 0, 0.058824, 0, 0.084967, 0.032680, 0, 0, 0.091503, 0, 0.098039, 0, 0.006536, 0.013072, 0.196078, 0, 0.045752, 0.045752, 0, 0, 0, 0, 0.006536, 0.006536, 0.019608, 0, 0, 0, 0.013072, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.248366, 0.026144, 0, 0, 0, 0, 0, 0};
static float JAP_afters_64[] = { 0, 0.001890, 0, 0.389414, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001890, 0, 0.047259, 0, 0, 0, 0, 0.011342, 0.024575, 0.077505, 0, 0.018904, 0.001890, 0, 0, 0, 0, 0.111531, 0.011342, 0, 0, 0, 0.005671, 0, 0.102079, 0, 0, 0.005671, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001890, 0, 0, 0.081285, 0, 0, 0, 0, 0.105860};
static float JAP_afters_65[] = { 0, 0, 0, 0.001522, 0, 0.044140, 0, 0, 0, 0, 0, 0, 0.001522, 0, 0.057839, 0, 0, 0, 0.004566, 0, 0.001522, 0, 0.013699, 0, 0.016743, 0, 0, 0, 0, 0, 0.006088, 0, 0.041096, 0, 0.039574, 0.024353, 0, 0.003044, 0, 0.214612, 0.019787, 0.022831, 0.030441, 0, 0, 0.161339, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001522, 0.004566, 0, 0, 0.007610, 0, 0.003044, 0, 0, 0, 0.007610, 0.001522, 0.056317, 0.007610, 0.007610, 0.004566, 0.006088, 0, 0.187215};
static float JAP_afters_66[] = { 0, 0, 0, 0.048780, 0, 0, 0, 0, 0, 0.006098, 0.189024, 0, 0, 0, 0.353659, 0, 0, 0, 0.012195, 0, 0.036585, 0.006098, 0, 0, 0.012195, 0, 0.018293, 0, 0, 0, 0, 0, 0.103659, 0, 0.079268, 0, 0, 0, 0, 0.006098, 0, 0.024390, 0, 0, 0, 0.012195, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.012195, 0.012195, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.006098, 0, 0.036585, 0, 0.024390};
static float JAP_afters_67[] = { 0, 0, 0, 0.009202, 0, 0, 0, 0.006135, 0, 0.003067, 0.009202, 0.003067, 0.012270, 0, 0.371166, 0, 0.012270, 0, 0.018405, 0, 0.006135, 0, 0.012270, 0, 0.205521, 0, 0.003067, 0, 0, 0, 0, 0, 0.003067, 0, 0.003067, 0, 0, 0, 0, 0.055215, 0, 0, 0, 0, 0, 0, 0.003067, 0.003067, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.202454, 0.012270, 0.006135, 0.009202, 0, 0, 0.015337, 0, 0, 0, 0, 0, 0.006135, 0.003067, 0, 0, 0, 0, 0.006135};
static float JAP_afters_68[] = { 0, 0, 0, 0.001408, 0, 0.772877, 0, 0, 0, 0, 0.000469, 0, 0.001877, 0.007039, 0.015016, 0, 0.004223, 0, 0, 0, 0.001408, 0.000939, 0.038480, 0.000939, 0, 0, 0.000939, 0.000469, 0, 0, 0, 0.000469, 0.011262, 0, 0.023463, 0.041764, 0, 0.002816, 0, 0.010324, 0.004223, 0, 0.003754, 0, 0.000469, 0.004693, 0.000939, 0, 0, 0, 0, 0, 0.001408, 0, 0, 0, 0, 0, 0.000469, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.007039, 0, 0.001877, 0, 0, 0, 0.000469, 0, 0.038480};
static float JAP_afters_69[] = { 0, 0, 0, 0, 0, 0.698413, 0, 0, 0, 0, 0, 0, 0.031746, 0, 0.026455, 0, 0, 0, 0, 0, 0.005291, 0, 0.068783, 0, 0, 0.015873, 0, 0, 0.005291, 0, 0.005291, 0.005291, 0, 0, 0, 0, 0, 0, 0.005291, 0, 0, 0, 0.047619, 0, 0, 0, 0, 0, 0, 0, 0.005291, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.005291, 0, 0.015873, 0, 0, 0, 0, 0, 0, 0, 0.015873, 0.037037, 0.005291, 0, 0, 0, 0};
static float JAP_afters_70[] = { 0, 0, 0, 0, 0, 0.875278, 0, 0, 0, 0.013363, 0.001392, 0.001392, 0.003341, 0.000278, 0.063474, 0, 0, 0, 0.002506, 0, 0, 0, 0.000278, 0.001114, 0, 0, 0.004733, 0, 0, 0.000557, 0, 0.000278, 0.000835, 0, 0, 0, 0, 0.001114, 0, 0.007238, 0, 0, 0.000835, 0, 0, 0, 0, 0.000278, 0, 0.004176, 0, 0, 0, 0.003062, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.008352, 0, 0, 0, 0, 0.000557, 0, 0.000278, 0, 0.004454, 0, 0, 0, 0, 0, 0.000835};
static float JAP_afters_71[] = { 0, 0, 0, 0.020917, 0, 0.288013, 0, 0, 0, 0, 0.001609, 0, 0.003218, 0.001609, 0.038616, 0, 0, 0, 0.013677, 0, 0.048270, 0, 0.008045, 0.000805, 0, 0, 0, 0, 0.045857, 0, 0.000805, 0.002414, 0.003218, 0, 0.008850, 0.001609, 0, 0.022526, 0, 0.026549, 0, 0, 0.000805, 0, 0.000805, 0, 0, 0.001609, 0, 0, 0.010459, 0, 0, 0, 0, 0, 0, 0, 0.002414, 0.005632, 0, 0.006436, 0.006436, 0, 0.004827, 0, 0, 0.003218, 0, 0, 0, 0, 0.000805, 0.036203, 0.088496, 0, 0.002414, 0.004023, 0, 0.288817};
static float JAP_afters_72[] = { 0, 0, 0, 0.288189, 0, 0.017323, 0, 0.001575, 0, 0, 0.089764, 0.004724, 0.007874, 0.001575, 0.075591, 0, 0, 0, 0, 0, 0, 0, 0.053543, 0.001575, 0.009449, 0.004724, 0.001575, 0, 0.003150, 0, 0.020472, 0.004724, 0, 0, 0.026772, 0.001575, 0.004724, 0, 0, 0, 0, 0.003150, 0.048819, 0, 0, 0, 0, 0.003150, 0, 0, 0.012598, 0, 0, 0, 0, 0, 0.031496, 0, 0, 0, 0, 0.006299, 0.012598, 0.009449, 0, 0.001575, 0, 0.031496, 0, 0, 0, 0, 0, 0.003150, 0.004724, 0.108661, 0, 0.007874, 0, 0.096063};
static float JAP_afters_73[] = { 0, 0.081232, 0, 0.005602, 0, 0.000934, 0, 0.014006, 0, 0.001867, 0.040149, 0, 0.005602, 0.001867, 0.007470, 0.000934, 0.001867, 0, 0.015873, 0.001867, 0, 0, 0.011204, 0.025210, 0.000934, 0, 0, 0, 0.001867, 0.002801, 0.001867, 0.001867, 0, 0, 0.005602, 0.176471, 0, 0.003735, 0.004669, 0.006536, 0, 0.002801, 0.006536, 0, 0.022409, 0.003735, 0.002801, 0.000934, 0, 0.036415, 0, 0.003735, 0.000934, 0.000934, 0, 0, 0, 0, 0.000934, 0, 0, 0.020542, 0, 0, 0, 0.005602, 0.001867, 0.003735, 0.033613, 0.019608, 0.373483, 0.007470, 0, 0.005602, 0, 0, 0, 0.001867, 0, 0.022409};
static float JAP_afters_74[] = { 0, 0, 0, 0.020725, 0, 0.155440, 0, 0, 0, 0, 0.005181, 0.015544, 0, 0.005181, 0.233161, 0.020725, 0, 0, 0.031088, 0.005181, 0.010363, 0, 0.119171, 0, 0.015544, 0, 0, 0, 0, 0, 0.010363, 0.036269, 0, 0, 0, 0.046632, 0, 0, 0.005181, 0, 0, 0, 0.005181, 0, 0, 0.005181, 0, 0, 0, 0, 0.088083, 0, 0, 0, 0, 0, 0.010363, 0, 0, 0, 0, 0.031088, 0, 0, 0.010363, 0, 0, 0.082902, 0, 0, 0, 0, 0, 0.005181, 0, 0, 0, 0.010363, 0, 0.015544};
static float JAP_afters_75[] = { 0, 0, 0, 0.091190, 0, 0, 0, 0, 0, 0, 0.001546, 0, 0.004637, 0, 0, 0, 0, 0, 0.001546, 0, 0.010819, 0, 0, 0, 0.001546, 0, 0, 0, 0, 0.009274, 0.157651, 0.004637, 0, 0, 0.001546, 0.021638, 0, 0.125193, 0, 0, 0.001546, 0.003091, 0, 0, 0, 0, 0.001546, 0.064915, 0, 0, 0, 0, 0.018547, 0.001546, 0, 0, 0, 0, 0.001546, 0, 0, 0, 0, 0, 0, 0.004637, 0, 0, 0, 0.001546, 0, 0.001546, 0, 0, 0.166924, 0.004637, 0, 0.012365, 0, 0.284389};
static float JAP_afters_76[] = { 0, 0.002558, 0, 0.029838, 0, 0.044331, 0, 0.000853, 0, 0.052003, 0, 0.003410, 0.005115, 0, 0.307758, 0.000853, 0, 0.003410, 0, 0.013640, 0.016198, 0.001705, 0.042626, 0.005115, 0.002558, 0.005968, 0.001705, 0.180733, 0.001705, 0, 0, 0, 0.001705, 0, 0.002558, 0, 0, 0.078431, 0, 0, 0, 0.002558, 0.009378, 0, 0, 0, 0.013640, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.054561, 0, 0, 0.000853, 0.000853, 0, 0, 0, 0, 0, 0.015345, 0, 0, 0, 0, 0.016198, 0, 0, 0.081841};
static float JAP_afters_77[] = { 0, 0, 0, 0.005025, 0, 0.002513, 0, 0.017588, 0, 0.002513, 0.072864, 0.002513, 0.007538, 0, 0.030151, 0.002513, 0.017588, 0.012563, 0.060302, 0, 0.002513, 0, 0.052764, 0, 0.002513, 0.015075, 0.067839, 0, 0, 0, 0.062814, 0.007538, 0.002513, 0, 0.050251, 0, 0, 0.002513, 0, 0.032663, 0, 0, 0.005025, 0, 0.005025, 0.002513, 0, 0, 0, 0, 0.002513, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.030151, 0, 0.005025, 0.015075, 0.002513, 0, 0, 0, 0, 0, 0, 0.032663, 0.175879, 0.050251, 0.103015, 0, 0.002513, 0, 0.035176};
static float JAP_afters_79[] = { 0, 0.004064, 0, 0.026310, 0, 0, 0, 0.006845, 0, 0.002995, 0.045775, 0.011979, 0.057754, 0.012193, 0.000428, 0.000642, 0.026952, 0.010481, 0.037433, 0.040642, 0.019037, 0.008128, 0.046203, 0.027594, 0.009412, 0, 0.023316, 0.061390, 0.011765, 0.004706, 0.017540, 0.033369, 0.024385, 0.000214, 0, 0.000428, 0.000642, 0.028877, 0.006417, 0.089840, 0.041283, 0.036150, 0.021176, 0, 0.014973, 0.001711, 0.012834, 0.004920, 0.008984, 0.000214, 0.004278, 0.001283, 0.001070, 0.017112, 0.001283, 0, 0.000214, 0.000428, 0.000642, 0.003850, 0.009198, 0.005775, 0.006417, 0.001070, 0.004706, 0.002567, 0, 0.004492, 0, 0.011123, 0, 0.018396, 0.008128, 0.023743, 0, 0.007701, 0.021390, 0.005134, 0, 0};
static struct letter_frequencies JAP_counts[] = {
    { "ぁ", 0.000000, 0.000000, 0.000085, 0.000000, JAP_afters_0 },
    { "あ", 0.000078, 0.019588, 0.005883, 0.004324, JAP_afters_1 },
    { "ぃ", 0.000000, 0.000000, 0.000340, 0.000000, JAP_afters_2 },
    { "い", 0.004665, 0.057224, 0.078248, 0.101182, JAP_afters_3 },
    { "ぅ", 0.000000, 0.000000, 0.000042, 0.000000, JAP_afters_4 },
    { "う", 0.000078, 0.011104, 0.096684, 0.152183, JAP_afters_5 },
    { "ぇ", 0.000000, 0.000000, 0.000722, 0.001048, JAP_afters_6 },
    { "え", 0.000466, 0.011989, 0.006563, 0.005437, JAP_afters_7 },
    { "ぉ", 0.000000, 0.000000, 0.000021, 0.000033, JAP_afters_8 },
    { "お", 0.000311, 0.017557, 0.011385, 0.002064, JAP_afters_9 },
    { "か", 0.009174, 0.063284, 0.027187, 0.016149, JAP_afters_10 },
    { "が", 0.096556, 0.006093, 0.007137, 0.000295, JAP_afters_11 },
    { "き", 0.002488, 0.043238, 0.034430, 0.035605, JAP_afters_12 },
    { "ぎ", 0.000155, 0.008353, 0.006882, 0.005929, JAP_afters_13 },
    { "く", 0.006686, 0.006780, 0.024766, 0.060140, JAP_afters_14 },
    { "ぐ", 0.000000, 0.001474, 0.001572, 0.000295, JAP_afters_15 },
    { "け", 0.000000, 0.020734, 0.014656, 0.003407, JAP_afters_16 },
    { "げ", 0.000155, 0.007599, 0.005268, 0.002850, JAP_afters_17 },
    { "こ", 0.001788, 0.043467, 0.017247, 0.003210, JAP_afters_18 },
    { "ご", 0.006530, 0.008418, 0.007731, 0.010809, JAP_afters_19 },
    { "さ", 0.014305, 0.033575, 0.015335, 0.006748, JAP_afters_20 },
    { "ざ", 0.000000, 0.003570, 0.003292, 0.000033, JAP_afters_21 },
    { "し", 0.019047, 0.086704, 0.038805, 0.018376, JAP_afters_22 },
    { "じ", 0.005209, 0.027351, 0.013360, 0.008287, JAP_afters_23 },
    { "す", 0.000466, 0.019424, 0.006330, 0.003734, JAP_afters_24 },
    { "ず", 0.002099, 0.000131, 0.001827, 0.000884, JAP_afters_25 },
    { "せ", 0.000855, 0.034164, 0.015505, 0.001015, JAP_afters_26 },
    { "ぜ", 0.000000, 0.010646, 0.020560, 0.000491, JAP_afters_27 },
    { "そ", 0.000155, 0.013725, 0.004758, 0.001048, JAP_afters_28 },
    { "ぞ", 0.000078, 0.003996, 0.001912, 0.000000, JAP_afters_29 },
    { "た", 0.000311, 0.020898, 0.012128, 0.045465, JAP_afters_30 },
    { "だ", 0.004976, 0.014052, 0.007476, 0.004422, JAP_afters_31 },
    { "ち", 0.000466, 0.019883, 0.030840, 0.043369, JAP_afters_32 },
    { "ぢ", 0.000000, 0.000000, 0.000085, 0.000000, JAP_afters_33 },
    { "っ", 0.000000, 0.000033, 0.028143, 0.000033, JAP_afters_34 },
    { "つ", 0.001166, 0.022831, 0.009707, 0.028366, JAP_afters_35 },
    { "づ", 0.000155, 0.000131, 0.002273, 0.000000, JAP_afters_36 },
    { "て", 0.000777, 0.010744, 0.018500, 0.032264, JAP_afters_37 },
    { "で", 0.070901, 0.005208, 0.000658, 0.004881, JAP_afters_38 },
    { "と", 0.089870, 0.022732, 0.015526, 0.022307, JAP_afters_39 },
    { "ど", 0.001866, 0.014544, 0.009048, 0.014675, JAP_afters_40 },
    { "な", 0.000700, 0.032166, 0.015526, 0.009237, JAP_afters_41 },
    { "に", 0.120501, 0.050346, 0.008284, 0.013659, JAP_afters_42 },
    { "ぬ", 0.000000, 0.000066, 0.000149, 0.000066, JAP_afters_43 },
    { "ね", 0.000078, 0.011465, 0.003823, 0.002358, JAP_afters_44 },
    { "の", 0.224909, 0.006879, 0.003037, 0.008091, JAP_afters_45 },
    { "は", 0.118402, 0.025681, 0.008921, 0.000688, JAP_afters_46 },
    { "ば", 0.000078, 0.002489, 0.002591, 0.005077, JAP_afters_47 },
    { "ぱ", 0.019669, 0.000360, 0.001508, 0.000459, JAP_afters_48 },
    { "ひ", 0.004120, 0.014773, 0.004142, 0.001867, JAP_afters_49 },
    { "び", 0.000000, 0.000622, 0.002082, 0.004913, JAP_afters_50 },
    { "ぴ", 0.000000, 0.000066, 0.001890, 0.000033, JAP_afters_51 },
    { "ふ", 0.000544, 0.012021, 0.001635, 0.003079, JAP_afters_52 },
    { "ぶ", 0.001011, 0.005798, 0.003993, 0.005437, JAP_afters_53 },
    { "ぷ", 0.000000, 0.000033, 0.000085, 0.000262, JAP_afters_54 },
    { "へ", 0.007386, 0.003308, 0.000807, 0.000000, JAP_afters_55 },
    { "べ", 0.000000, 0.004782, 0.001720, 0.001572, JAP_afters_56 },
    { "ぺ", 0.000000, 0.000426, 0.000297, 0.000000, JAP_afters_57 },
    { "ほ", 0.000000, 0.015330, 0.003526, 0.001572, JAP_afters_58 },
    { "ぼ", 0.000000, 0.003538, 0.001274, 0.000983, JAP_afters_59 },
    { "ぽ", 0.000622, 0.000557, 0.003887, 0.000393, JAP_afters_60 },
    { "ま", 0.000078, 0.013561, 0.007328, 0.004062, JAP_afters_61 },
    { "み", 0.002876, 0.010613, 0.003887, 0.004455, JAP_afters_62 },
    { "む", 0.000000, 0.003112, 0.001232, 0.003669, JAP_afters_63 },
    { "め", 0.001166, 0.003865, 0.008730, 0.011596, JAP_afters_64 },
    { "も", 0.019902, 0.013692, 0.005076, 0.002260, JAP_afters_65 },
    { "ゃ", 0.000000, 0.000000, 0.003483, 0.013168, JAP_afters_66 },
    { "や", 0.008785, 0.005962, 0.003059, 0.000950, JAP_afters_67 },
    { "ゅ", 0.000000, 0.000000, 0.045262, 0.004782, JAP_afters_68 },
    { "ゆ", 0.000078, 0.003439, 0.001784, 0.001015, JAP_afters_69 },
    { "ょ", 0.000000, 0.000000, 0.076294, 0.006093, JAP_afters_70 },
    { "よ", 0.004976, 0.026008, 0.009537, 0.000426, JAP_afters_71 },
    { "ら", 0.005286, 0.005437, 0.009962, 0.012611, JAP_afters_72 },
    { "り", 0.000000, 0.008877, 0.016992, 0.016738, JAP_afters_73 },
    { "る", 0.003265, 0.000131, 0.004014, 0.050280, JAP_afters_74 },
    { "れ", 0.002721, 0.010711, 0.006797, 0.006256, JAP_afters_75 },
    { "ろ", 0.000233, 0.008516, 0.019392, 0.016050, JAP_afters_76 },
    { "わ", 0.000233, 0.004127, 0.005777, 0.005306, JAP_afters_77 },
    { "を", 0.110472, 0.000000, 0.000000, 0.000000, NULL },
    { "ん", 0.000078, 0.000000, 0.099297, 0.139179, JAP_afters_79 },
    NULL
};	/* JAP */
static float JAP_word_lens[] = {
    0.000000, 0.296451, 0.210486, 0.162134, 0.200000, 0.079442, 0.025812, 0.011546, 0.004287, 0.002835, 0.002374, 0.000968, 0.001083, 0.000599, 0.000461, 0.000254, 0.000300, 0.000161, 0.000161, 0.000138, 0.000000, 0.000069, 0.000161, 0.000046, 0.000023, 0.000023, 0.000046, 0.000069, 0.000023, 0.000046
};

static struct letter_frequencies JAP_kata_counts[] = {
    { "ァ", 0.000000, 0.000000, 0.000085, 0.000000, JAP_afters_0 },
    { "ア", 0.000078, 0.019588, 0.005883, 0.004324, JAP_afters_1 },
    { "ィ", 0.000000, 0.000000, 0.000340, 0.000000, JAP_afters_2 },
    { "イ", 0.004665, 0.057224, 0.078248, 0.101182, JAP_afters_3 },
    { "ゥ", 0.000000, 0.000000, 0.000042, 0.000000, JAP_afters_4 },
    { "ウ", 0.000078, 0.011104, 0.096684, 0.152183, JAP_afters_5 },
    { "ェ", 0.000000, 0.000000, 0.000722, 0.001048, JAP_afters_6 },
    { "エ", 0.000466, 0.011989, 0.006563, 0.005437, JAP_afters_7 },
    { "ォ", 0.000000, 0.000000, 0.000021, 0.000033, JAP_afters_8 },
    { "オ", 0.000311, 0.017557, 0.011385, 0.002064, JAP_afters_9 },
    { "カ", 0.009174, 0.063284, 0.027187, 0.016149, JAP_afters_10 },
    { "ガ", 0.096556, 0.006093, 0.007137, 0.000295, JAP_afters_11 },
    { "キ", 0.002488, 0.043238, 0.034430, 0.035605, JAP_afters_12 },
    { "ギ", 0.000155, 0.008353, 0.006882, 0.005929, JAP_afters_13 },
    { "ク", 0.006686, 0.006780, 0.024766, 0.060140, JAP_afters_14 },
    { "グ", 0.000000, 0.001474, 0.001572, 0.000295, JAP_afters_15 },
    { "ケ", 0.000000, 0.020734, 0.014656, 0.003407, JAP_afters_16 },
    { "ゲ", 0.000155, 0.007599, 0.005268, 0.002850, JAP_afters_17 },
    { "コ", 0.001788, 0.043467, 0.017247, 0.003210, JAP_afters_18 },
    { "ゴ", 0.006530, 0.008418, 0.007731, 0.010809, JAP_afters_19 },
    { "サ", 0.014305, 0.033575, 0.015335, 0.006748, JAP_afters_20 },
    { "ザ", 0.000000, 0.003570, 0.003292, 0.000033, JAP_afters_21 },
    { "シ", 0.019047, 0.086704, 0.038805, 0.018376, JAP_afters_22 },
    { "ジ", 0.005209, 0.027351, 0.013360, 0.008287, JAP_afters_23 },
    { "ス", 0.000466, 0.019424, 0.006330, 0.003734, JAP_afters_24 },
    { "ズ", 0.002099, 0.000131, 0.001827, 0.000884, JAP_afters_25 },
    { "セ", 0.000855, 0.034164, 0.015505, 0.001015, JAP_afters_26 },
    { "ゼ", 0.000000, 0.010646, 0.020560, 0.000491, JAP_afters_27 },
    { "ソ", 0.000155, 0.013725, 0.004758, 0.001048, JAP_afters_28 },
    { "ゾ", 0.000078, 0.003996, 0.001912, 0.000000, JAP_afters_29 },
    { "タ", 0.000311, 0.020898, 0.012128, 0.045465, JAP_afters_30 },
    { "ダ", 0.004976, 0.014052, 0.007476, 0.004422, JAP_afters_31 },
    { "チ", 0.000466, 0.019883, 0.030840, 0.043369, JAP_afters_32 },
    { "ヂ", 0.000000, 0.000000, 0.000085, 0.000000, JAP_afters_33 },
    { "ッ", 0.000000, 0.000033, 0.028143, 0.000033, JAP_afters_34 },
    { "ツ", 0.001166, 0.022831, 0.009707, 0.028366, JAP_afters_35 },
    { "ヅ", 0.000155, 0.000131, 0.002273, 0.000000, JAP_afters_36 },
    { "テ", 0.000777, 0.010744, 0.018500, 0.032264, JAP_afters_37 },
    { "デ", 0.070901, 0.005208, 0.000658, 0.004881, JAP_afters_38 },
    { "ト", 0.089870, 0.022732, 0.015526, 0.022307, JAP_afters_39 },
    { "ド", 0.001866, 0.014544, 0.009048, 0.014675, JAP_afters_40 },
    { "ナ", 0.000700, 0.032166, 0.015526, 0.009237, JAP_afters_41 },
    { "ニ", 0.120501, 0.050346, 0.008284, 0.013659, JAP_afters_42 },
    { "ヌ", 0.000000, 0.000066, 0.000149, 0.000066, JAP_afters_43 },
    { "ネ", 0.000078, 0.011465, 0.003823, 0.002358, JAP_afters_44 },
    { "ノ", 0.224909, 0.006879, 0.003037, 0.008091, JAP_afters_45 },
    { "ハ", 0.118402, 0.025681, 0.008921, 0.000688, JAP_afters_46 },
    { "バ", 0.000078, 0.002489, 0.002591, 0.005077, JAP_afters_47 },
    { "パ", 0.019669, 0.000360, 0.001508, 0.000459, JAP_afters_48 },
    { "ヒ", 0.004120, 0.014773, 0.004142, 0.001867, JAP_afters_49 },
    { "ビ", 0.000000, 0.000622, 0.002082, 0.004913, JAP_afters_50 },
    { "ピ", 0.000000, 0.000066, 0.001890, 0.000033, JAP_afters_51 },
    { "フ", 0.000544, 0.012021, 0.001635, 0.003079, JAP_afters_52 },
    { "ブ", 0.001011, 0.005798, 0.003993, 0.005437, JAP_afters_53 },
    { "プ", 0.000000, 0.000033, 0.000085, 0.000262, JAP_afters_54 },
    { "ヘ", 0.007386, 0.003308, 0.000807, 0.000000, JAP_afters_55 },
    { "ベ", 0.000000, 0.004782, 0.001720, 0.001572, JAP_afters_56 },
    { "ペ", 0.000000, 0.000426, 0.000297, 0.000000, JAP_afters_57 },
    { "ホ", 0.000000, 0.015330, 0.003526, 0.001572, JAP_afters_58 },
    { "ボ", 0.000000, 0.003538, 0.001274, 0.000983, JAP_afters_59 },
    { "ポ", 0.000622, 0.000557, 0.003887, 0.000393, JAP_afters_60 },
    { "マ", 0.000078, 0.013561, 0.007328, 0.004062, JAP_afters_61 },
    { "ミ", 0.002876, 0.010613, 0.003887, 0.004455, JAP_afters_62 },
    { "ム", 0.000000, 0.003112, 0.001232, 0.003669, JAP_afters_63 },
    { "メ", 0.001166, 0.003865, 0.008730, 0.011596, JAP_afters_64 },
    { "モ", 0.019902, 0.013692, 0.005076, 0.002260, JAP_afters_65 },
    { "ャ", 0.000000, 0.000000, 0.003483, 0.013168, JAP_afters_66 },
    { "ヤ", 0.008785, 0.005962, 0.003059, 0.000950, JAP_afters_67 },
    { "ュ", 0.000000, 0.000000, 0.045262, 0.004782, JAP_afters_68 },
    { "ユ", 0.000078, 0.003439, 0.001784, 0.001015, JAP_afters_69 },
    { "ョ", 0.000000, 0.000000, 0.076294, 0.006093, JAP_afters_70 },
    { "ヨ", 0.004976, 0.026008, 0.009537, 0.000426, JAP_afters_71 },
    { "ラ", 0.005286, 0.005437, 0.009962, 0.012611, JAP_afters_72 },
    { "リ", 0.000000, 0.008877, 0.016992, 0.016738, JAP_afters_73 },
    { "ル", 0.003265, 0.000131, 0.004014, 0.050280, JAP_afters_74 },
    { "レ", 0.002721, 0.010711, 0.006797, 0.006256, JAP_afters_75 },
    { "ロ", 0.000233, 0.008516, 0.019392, 0.016050, JAP_afters_76 },
    { "ワ", 0.000233, 0.004127, 0.005777, 0.005306, JAP_afters_77 },
    { "ヲ", 0.110472, 0.000000, 0.000000, 0.000000, NULL },
    { "ン", 0.000078, 0.000000, 0.099297, 0.139179, JAP_afters_79 },
    NULL
};	/* JAP */
static struct letter_frequencies LTH_counts[] = {
    { "a", 0.000000, 0.058488, 0.127579, 0.048502 },
    { "b", 0.000000, 0.048502, 0.011108, 0.000000 },
    { "c", 0.000000, 0.000000, 0.006665, 0.000000 },
    { "d", 0.000000, 0.024964, 0.014757, 0.012839 },
    { "e", 0.000000, 0.004993, 0.070771, 0.028531 },
    { "f", 0.000000, 0.000713, 0.000476, 0.000000 },
    { "g", 0.000000, 0.030670, 0.025389, 0.002853 },
    { "h", 0.000000, 0.000000, 0.000159, 0.000000 },
    { "i", 0.025641, 0.085592, 0.169470, 0.158345 },
    { "j", 0.000000, 0.042796, 0.015233, 0.000713 },
    { "k", 0.000000, 0.089158, 0.039987, 0.010699 },
    { "l", 0.000000, 0.042796, 0.026658, 0.007846 },
    { "m", 0.000000, 0.017118, 0.051254, 0.008559 },
    { "n", 0.000000, 0.064194, 0.055855, 0.001427 },
    { "o", 0.025641, 0.009986, 0.051888, 0.079173 },
    { "p", 0.000000, 0.076320, 0.013646, 0.011412 },
    { "r", 0.000000, 0.015692, 0.056490, 0.084879 },
    { "s", 0.000000, 0.094151, 0.057125, 0.298859 },
    { "t", 0.000000, 0.147646, 0.063948, 0.010699 },
    { "u", 0.000000, 0.002853, 0.050619, 0.026391 },
    { "v", 0.000000, 0.064907, 0.025547, 0.000000 },
    { "y", 0.000000, 0.006419, 0.020311, 0.000000 },
    { "z", 0.000000, 0.000000, 0.000952, 0.000000 },
    { "é", 0.000000, 0.000000, 0.000159, 0.000000 },
    { "ą", 0.000000, 0.000000, 0.001904, 0.057775 },
    { "č", 0.000000, 0.000000, 0.002222, 0.001427 },
    { "ė", 0.000000, 0.000000, 0.016027, 0.017118 },
    { "ę", 0.000000, 0.000000, 0.000317, 0.044936 },
    { "į", 0.948718, 0.015692, 0.000952, 0.009986 },
    { "š", 0.000000, 0.018545, 0.007934, 0.003566 },
    { "ū", 0.000000, 0.000000, 0.008251, 0.000000 },
    { "ų", 0.000000, 0.000000, 0.000952, 0.071327 },
    { "ž", 0.000000, 0.037803, 0.005395, 0.002140 },
    NULL
};	/* LTH */
static float LTH_word_lens[] = {
    0.000000, 0.027065, 0.114504, 0.072172, 0.117280, 0.095073, 0.124913, 0.091603, 0.081888, 0.077030, 0.101319, 0.047883, 0.015961, 0.019431, 0.006246, 0.004858, 0.002082, 0.000000, 0.000694, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float LTH_all_consonants[] = {
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 
};

static float LTH_consonant_run[] = {
    0.000000, 0.807419, 0.161331, 0.031250, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float LTH_vowel_run[] = {
    0.000000, 0.801170, 0.192426, 0.006405, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float PLK_afters_0[] = { 0.001000, 0.008503, 0.015398, 0.048160, 0.000286, 0.003073, 0.018185, 0.000929, 0.003394, 0.053340, 0.062379, 0.060343, 0.058557, 0.118578, 0.000393, 0.021722, 0, 0.077492, 0.042944, 0.046695, 0.002965, 0.000214, 0.059557, 0, 0.000179, 0.035191, 0, 0, 0, 0, 0, 0, 0.000036, 0, 0, 0, 0, 0.017613, 0, 0.132333, 0.005502, 0.019900, 0, 0.004787, 0.010468, 0.026902, 0.011647, 0.006967, 0.014934, 0.009432};
static float PLK_afters_1[] = { 0.118962, 0, 0.004108, 0.000986, 0.067532, 0.000329, 0, 0, 0.200296, 0.007887, 0.017746, 0.057509, 0.000657, 0.022675, 0.115840, 0, 0, 0.073612, 0.004108, 0.003122, 0.031876, 0, 0.000657, 0, 0.166283, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.011666, 0, 0.016595, 0, 0.015445, 0.020046, 0, 0.000164, 0, 0, 0, 0.000986, 0.000822, 0.000164, 0.034998, 0.004929};
static float PLK_afters_2[] = { 0.106894, 0, 0.001139, 0, 0.126249, 0, 0, 0, 0.448197, 0.000127, 0.023023, 0.000127, 0.002277, 0.014674, 0.093106, 0.000127, 0, 0.001139, 0, 0.002783, 0.024794, 0, 0, 0, 0.110057, 0, 0.000127, 0, 0, 0, 0, 0, 0, 0, 0.016572, 0, 0.021505, 0, 0.006578, 0.000127, 0, 0, 0, 0, 0, 0.000380, 0, 0, 0, 0};
static float PLK_afters_3[] = { 0.140412, 0.009317, 0.003034, 0.013326, 0.064789, 0, 0.003467, 0.000108, 0.002817, 0.003684, 0.015168, 0.024052, 0.005634, 0.117768, 0.230878, 0.021018, 0, 0.054821, 0.008667, 0.002817, 0.048754, 0, 0.028494, 0, 0.074106, 0, 0, 0, 0, 0, 0.000433, 0, 0, 0, 0.006501, 0, 0.016468, 0, 0.010293, 0.046262, 0, 0.000867, 0, 0.011484, 0.005417, 0.001950, 0.004334, 0.001083, 0.019393, 0.002384};
static float PLK_afters_4[] = { 0.001086, 0.022099, 0.032542, 0.050004, 0.000251, 0.000752, 0.097209, 0.000125, 0.003050, 0.113084, 0.037555, 0.034589, 0.151725, 0.078829, 0.002089, 0.019509, 0.000084, 0.079539, 0.035258, 0.023603, 0.002841, 0.000501, 0.025107, 0.000042, 0.000292, 0.029994, 0, 0, 0, 0, 0, 0, 0, 0, 0.000125, 0, 0, 0.005180, 0, 0.012532, 0.009023, 0.020344, 0, 0.003175, 0.016376, 0.017295, 0.018966, 0.013869, 0.008397, 0.032960};
static float PLK_afters_5[] = { 0.289764, 0, 0, 0, 0.089764, 0.001575, 0, 0, 0.192126, 0, 0.003150, 0.053543, 0, 0.017323, 0.196850, 0, 0, 0.100787, 0, 0.001575, 0.044094, 0, 0, 0, 0.007874, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001575, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_6[] = { 0.085749, 0, 0, 0.024929, 0.003904, 0, 0.000451, 0.000300, 0.054663, 0, 0, 0.047004, 0.002253, 0.056615, 0.439255, 0, 0, 0.063823, 0, 0, 0.019673, 0, 0.012765, 0, 0, 0.000601, 0, 0, 0, 0, 0, 0, 0, 0, 0.020574, 0, 0.010662, 0, 0.010812, 0.115783, 0, 0, 0.000300, 0, 0, 0, 0, 0.018772, 0.011113, 0};
static float PLK_afters_7[] = { 0.423592, 0, 0, 0, 0.101877, 0, 0, 0, 0.042895, 0, 0, 0, 0.021448, 0, 0.187668, 0, 0, 0.085791, 0, 0, 0.128686, 0, 0, 0, 0.008043, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_8[] = { 0.085980, 0.002815, 0.013116, 0.006127, 0.437486, 0.000232, 0.003842, 0, 0.001424, 0.003610, 0.012254, 0.035671, 0.016030, 0.025635, 0.022588, 0.001822, 0, 0.015202, 0.017057, 0.006160, 0.011261, 0.000066, 0.021462, 0.000132, 0, 0.003577, 0, 0, 0, 0.000033, 0.000033, 0, 0, 0, 0.002749, 0, 0.028715, 0.007717, 0.113867, 0.042692, 0.001424, 0.007551, 0, 0.000099, 0.007618, 0.021594, 0.011592, 0.005034, 0.000066, 0.005664};
static float PLK_afters_9[] = { 0.207660, 0.001456, 0.010776, 0.009320, 0.307849, 0, 0.002184, 0, 0.000874, 0, 0.003495, 0.000437, 0.020824, 0.025775, 0.017329, 0.002184, 0, 0.000146, 0.016019, 0.000437, 0.083006, 0, 0.002039, 0, 0, 0.000437, 0, 0, 0, 0, 0, 0, 0, 0, 0.000874, 0, 0.184506, 0, 0.030290, 0, 0, 0.010339, 0, 0, 0.002330, 0, 0.002767, 0.001019, 0.034950, 0.020679};
static float PLK_afters_10[] = { 0.149216, 0.015585, 0.004477, 0, 0.000344, 0, 0, 0, 0.253057, 0, 0.007835, 0.013174, 0, 0.042965, 0.214483, 0.000258, 0, 0.073962, 0.008524, 0.035302, 0.060014, 0, 0.011021, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.014896, 0, 0.023248, 0, 0.016962, 0.031169, 0, 0.000086, 0, 0, 0.004391, 0, 0.000344, 0, 0.014121, 0.004563};
static float PLK_afters_11[] = { 0.155673, 0.004958, 0.005535, 0.000577, 0.241351, 0.000346, 0.002422, 0.000115, 0.235009, 0.000115, 0.067343, 0.004151, 0.000577, 0.066305, 0.058810, 0.000923, 0, 0, 0.019142, 0.003921, 0.051430, 0.000115, 0.004958, 0, 0.000692, 0, 0.000346, 0, 0, 0, 0, 0, 0, 0, 0.001268, 0, 0.023178, 0, 0.031365, 0, 0, 0.001038, 0, 0, 0.000346, 0, 0.014530, 0, 0, 0.003459};
static float PLK_afters_12[] = { 0.161659, 0.004116, 0.001338, 0.000823, 0.039617, 0.000720, 0.003602, 0, 0.339267, 0, 0.012040, 0.000823, 0.000617, 0.069150, 0.089319, 0.007306, 0, 0.007409, 0.002470, 0.001646, 0.099403, 0, 0.000823, 0, 0.060918, 0, 0, 0, 0, 0, 0.000309, 0, 0, 0, 0.031900, 0, 0.005351, 0, 0.020889, 0.025931, 0, 0.001646, 0, 0, 0.001441, 0, 0.008644, 0.000103, 0.000617, 0.000103};
static float PLK_afters_13[] = { 0.196820, 0, 0.004975, 0.006174, 0.094701, 0.000844, 0.001244, 0.000089, 0.385333, 0.000355, 0.011504, 0, 0.000044, 0.018123, 0.077289, 0.000044, 0.000044, 0, 0.004175, 0.011949, 0.017590, 0, 0.000355, 0, 0.090614, 0.000222, 0, 0, 0, 0, 0, 0, 0, 0, 0.007151, 0, 0.050504, 0, 0.018123, 0, 0, 0, 0, 0, 0.000355, 0.000444, 0.000089, 0.000622, 0, 0.000222};
static float PLK_afters_14[] = { 0.000280, 0.064084, 0.011288, 0.086859, 0.000240, 0.001441, 0.023376, 0.000680, 0.004683, 0.029220, 0.045071, 0.052836, 0.051835, 0.099107, 0.000640, 0.026978, 0, 0.040227, 0.065364, 0.044270, 0.002562, 0.000120, 0.135732, 0, 0, 0.037385, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.003322, 0, 0.021214, 0.008846, 0.042189, 0, 0.000520, 0.018613, 0.012689, 0.031421, 0.020334, 0.008446, 0.008126};
static float PLK_afters_15[] = { 0.152223, 0, 0.002656, 0, 0.037031, 0, 0, 0.000076, 0.091213, 0.000076, 0.002504, 0.014114, 0, 0.011458, 0.366975, 0.000076, 0, 0.071103, 0.001442, 0.007209, 0.021172, 0, 0, 0, 0.012445, 0, 0, 0.000076, 0, 0.000076, 0.000228, 0, 0, 0, 0.018288, 0, 0.000379, 0, 0.004857, 0.015404, 0, 0, 0, 0, 0, 0.000379, 0.000986, 0, 0.166945, 0.000607};
static float PLK_afters_16[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_17[] = { 0.228721, 0.001972, 0.011758, 0.007814, 0.065847, 0.000303, 0.006069, 0.000228, 0.003490, 0.000076, 0.014034, 0.006372, 0.009483, 0.023820, 0.217873, 0.008269, 0, 0.000683, 0.035958, 0.031103, 0.054620, 0, 0.033531, 0, 0.067213, 0, 0, 0, 0, 0.000152, 0, 0.000076, 0, 0, 0.046351, 0.000076, 0.014945, 0.000607, 0.032469, 0.011758, 0.000152, 0.000759, 0, 0, 0.030724, 0.000228, 0.004096, 0.017220, 0, 0.011152};
static float PLK_afters_18[] = { 0.046677, 0, 0.010560, 0, 0.033652, 0.001830, 0, 0, 0.249859, 0.000070, 0.131301, 0, 0.008800, 0.037032, 0.033582, 0.058645, 0, 0.002042, 0.001619, 0.247254, 0.020206, 0, 0.032808, 0, 0.017249, 0, 0, 0, 0, 0, 0.000141, 0, 0, 0, 0.005914, 0, 0.006688, 0, 0.000915, 0.048719, 0, 0, 0, 0, 0, 0.004435, 0, 0, 0, 0};
static float PLK_afters_19[] = { 0.230757, 0, 0.003377, 0, 0.140204, 0.002609, 0, 0.000077, 0.002302, 0, 0.045277, 0.005372, 0.000921, 0.044433, 0.137979, 0.002916, 0, 0.050418, 0.000307, 0.000460, 0.051876, 0, 0.038984, 0, 0.110813, 0.000077, 0, 0, 0, 0, 0, 0, 0, 0, 0.029929, 0, 0.018878, 0, 0.017497, 0.011204, 0, 0, 0, 0, 0, 0.004528, 0.000460, 0, 0.047272, 0.001074};
static float PLK_afters_20[] = { 0.017073, 0.033859, 0.049354, 0.031133, 0.002152, 0.003874, 0.030703, 0.000143, 0.002296, 0.070875, 0.043615, 0.033286, 0.072023, 0.044333, 0.000143, 0.038164, 0, 0.057389, 0.066284, 0.053372, 0.000143, 0.000574, 0.022669, 0.000430, 0.000287, 0.011908, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.001148, 0, 0.017360, 0.001004, 0.034003, 0, 0.000287, 0.086083, 0.043902, 0.020803, 0.027403, 0.009756, 0.072166};
static float PLK_afters_21[] = { 0.093023, 0, 0, 0, 0.383721, 0, 0, 0, 0.127907, 0, 0, 0, 0, 0, 0.325581, 0, 0, 0.034884, 0, 0, 0, 0, 0, 0, 0.023256, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.011628, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_22[] = { 0.173108, 0.000194, 0.012689, 0.004985, 0.072182, 0, 0, 0, 0.253188, 0.000388, 0.007510, 0.000518, 0, 0.047906, 0.076196, 0.010099, 0, 0.021493, 0.043957, 0.004661, 0.009322, 0, 0, 0, 0.109795, 0.018191, 0, 0, 0, 0, 0, 0, 0, 0, 0.008804, 0, 0.016508, 0, 0.008157, 0.018191, 0, 0.003561, 0, 0, 0.000194, 0.001489, 0.008804, 0.007186, 0.001101, 0.059623};
static float PLK_afters_23[] = { 0, 0, 0, 0, 0.666667, 0, 0, 0, 0.166667, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.166667, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_24[] = { 0.007487, 0.026493, 0.036571, 0.010655, 0.002688, 0.000576, 0.014110, 0.000096, 0.011327, 0.029756, 0.037339, 0.043866, 0.165771, 0.044538, 0.004703, 0.022557, 0, 0.020061, 0.072279, 0.046938, 0.001728, 0, 0.054617, 0, 0, 0.013150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.002592, 0.011998, 0.003744, 0.083701, 0.009983, 0.035132, 0, 0.001344, 0.019581, 0.116913, 0.020637, 0.002880, 0.003935, 0.020253};
static float PLK_afters_25[] = { 0.351757, 0.029783, 0.001073, 0.055943, 0.068285, 0, 0.032063, 0.000134, 0.038771, 0.004427, 0.012477, 0.013281, 0.043332, 0.106520, 0.022404, 0.015294, 0, 0.034881, 0.005098, 0.004561, 0.036625, 0, 0.042662, 0, 0.024819, 0.001878, 0, 0, 0, 0, 0, 0, 0, 0, 0.005232, 0, 0.001878, 0, 0.001610, 0.031795, 0, 0.000805, 0, 0, 0.001744, 0.001610, 0.001878, 0.004830, 0.002281, 0.000268};
static float PLK_afters_27[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_28[] = { 0.500000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.500000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_29[] = { 0, 0, 0.125000, 0, 0, 0, 0.125000, 0, 0, 0, 0, 0.125000, 0, 0, 0, 0, 0, 0.500000, 0.125000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_30[] = { 0, 0, 0, 0, 0, 0.250000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.625000, 0.125000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_31[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.666667, 0, 0.333333, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_32[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_33[] = { 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_34[] = { 0, 0.022023, 0.037853, 0.047832, 0, 0, 0.020991, 0, 0, 0.047488, 0.000688, 0.045079, 0.000344, 0.001032, 0, 0.010323, 0, 0.130076, 0.019959, 0.024088, 0, 0, 0.391259, 0, 0, 0.009291, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.067103, 0, 0, 0, 0.022023, 0.075017, 0.004474, 0.008947, 0, 0.013765, 0.000344};
static float PLK_afters_35[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_36[] = { 0, 0.004470, 0.471470, 0.062582, 0, 0, 0.060216, 0, 0, 0.000263, 0.012885, 0, 0, 0, 0, 0.013148, 0, 0.000263, 0.026032, 0.052590, 0, 0, 0.026032, 0, 0, 0.014725, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.015251, 0, 0.154352, 0, 0.006048, 0, 0, 0.036813, 0.000263, 0.023666, 0.018669, 0, 0.000263};
static float PLK_afters_37[] = { 0, 0.269231, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.192308, 0, 0, 0, 0, 0, 0.153846, 0, 0.038462, 0, 0.153846, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.192308, 0, 0};
static float PLK_afters_38[] = { 0, 0.032781, 0.151297, 0.046470, 0, 0, 0.027738, 0, 0, 0, 0.157421, 0.008646, 0, 0, 0, 0.038184, 0, 0, 0.040346, 0.140490, 0, 0, 0, 0, 0, 0.009366, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.015130, 0, 0.104827, 0, 0.021614, 0, 0.000360, 0.073847, 0.000360, 0.063761, 0.067003, 0, 0.000360};
static float PLK_afters_39[] = { 0.285954, 0.003952, 0.003233, 0.001317, 0.068974, 0, 0.002515, 0, 0.000120, 0, 0.032331, 0.000120, 0.000479, 0.025027, 0.304993, 0.000359, 0, 0, 0.000599, 0.007305, 0.067058, 0, 0.000239, 0, 0.135912, 0.002754, 0, 0, 0, 0, 0, 0, 0, 0, 0.014849, 0, 0.019159, 0, 0.010298, 0.000120, 0, 0.001078, 0, 0, 0.001197, 0, 0.001078, 0.000239, 0, 0.008741};
static float PLK_afters_40[] = { 0, 0.002045, 0.319018, 0, 0, 0, 0, 0, 0, 0, 0.032720, 0, 0.008180, 0, 0, 0, 0, 0, 0.429448, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.208589, 0, 0, 0};
static float PLK_afters_41[] = { 0, 0.002524, 0.299148, 0, 0, 0, 0, 0, 0, 0, 0, 0.156832, 0.098454, 0.167245, 0, 0.048911, 0, 0.022720, 0, 0.000316, 0, 0, 0.106027, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.093720, 0, 0, 0.004102, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_42[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_43[] = { 0, 0.075916, 0.031414, 0.005236, 0, 0, 0, 0, 0, 0, 0.013089, 0.062827, 0.034031, 0.560209, 0, 0, 0, 0.049738, 0, 0, 0, 0, 0.102094, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.023560, 0, 0, 0.005236, 0, 0, 0, 0, 0, 0, 0.036649, 0, 0};
static float PLK_afters_44[] = { 0.121501, 0.008561, 0.000659, 0.024366, 0.241686, 0, 0.000329, 0, 0, 0, 0.015476, 0.006915, 0.000329, 0.073428, 0.072440, 0, 0, 0, 0.000329, 0, 0.013829, 0, 0.002634, 0, 0.325321, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.007244, 0, 0.027659, 0, 0.006585, 0.000659, 0, 0, 0, 0, 0, 0, 0.028317, 0, 0, 0.021732};
static float PLK_afters_45[] = { 0.173502, 0, 0.029180, 0, 0.076893, 0, 0, 0, 0.014590, 0, 0, 0.010252, 0.033517, 0.067035, 0.191640, 0.000394, 0, 0.004732, 0.001183, 0.002760, 0.034306, 0, 0.197161, 0, 0.076893, 0, 0, 0, 0, 0.001183, 0.000394, 0, 0, 0, 0.008281, 0, 0.001972, 0, 0.020505, 0.035883, 0, 0.001183, 0, 0, 0.001972, 0, 0.007886, 0, 0.005126, 0.001577};
static float PLK_afters_46[] = { 0.127511, 0.001092, 0.009825, 0, 0.217467, 0, 0, 0, 0, 0, 0.037555, 0.002620, 0.000437, 0.120961, 0.066812, 0.000218, 0, 0, 0, 0.003930, 0.051528, 0, 0.003712, 0, 0.224236, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.003275, 0, 0.064410, 0, 0.044105, 0.019869, 0, 0, 0, 0, 0, 0, 0.000437, 0, 0, 0};
static float PLK_afters_47[] = { 0.031405, 0.000661, 0.001983, 0, 0.034380, 0, 0, 0, 0.748099, 0, 0.013554, 0.002314, 0.000992, 0.001322, 0.067107, 0, 0, 0, 0, 0.000331, 0.000331, 0, 0.021157, 0, 0.021157, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.000661, 0, 0.046612, 0, 0.007934, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float PLK_afters_48[] = { 0.065587, 0.000810, 0.001822, 0.000202, 0.464170, 0, 0.000202, 0, 0, 0, 0.001012, 0.000810, 0.012551, 0.006073, 0.015385, 0, 0, 0, 0, 0.000607, 0.051822, 0, 0.016802, 0, 0.295951, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.004858, 0, 0.027733, 0, 0.019838, 0.000202, 0, 0, 0, 0, 0, 0.013563, 0, 0, 0, 0};
static float PLK_afters_49[] = { 0.131285, 0, 0.018855, 0, 0.155493, 0, 0, 0, 0, 0, 0.041667, 0.013268, 0.004190, 0.030028, 0.031192, 0.002328, 0, 0, 0, 0.024209, 0.020717, 0, 0.004423, 0, 0.293529, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.005121, 0, 0.032356, 0, 0.023045, 0.019553, 0, 0, 0, 0, 0, 0, 0.148743, 0, 0, 0};
static struct letter_frequencies PLK_counts[] = {
    { "a", 0.096644, 0.007472, 0.103174, 0.121399, PLK_afters_0 },
    { "b", 0.000139, 0.033546, 0.014107, 0.001541, PLK_afters_1 },
    { "c", 0.000000, 0.025642, 0.023000, 0.016154, PLK_afters_2 },
    { "d", 0.000832, 0.048994, 0.021885, 0.015880, PLK_afters_3 },
    { "e", 0.000832, 0.004103, 0.088834, 0.134817, PLK_afters_4 },
    { "f", 0.000139, 0.005298, 0.001003, 0.000158, PLK_afters_5 },
    { "g", 0.000000, 0.028924, 0.017464, 0.001310, PLK_afters_6 },
    { "h", 0.000139, 0.004607, 0.000199, 0.000101, PLK_afters_7 },
    { "i", 0.328064, 0.010237, 0.110726, 0.065724, PLK_afters_8 },
    { "j", 0.000416, 0.045697, 0.013870, 0.034885, PLK_afters_9 },
    { "k", 0.000139, 0.038499, 0.033576, 0.021682, PLK_afters_10 },
    { "l", 0.000000, 0.014167, 0.028874, 0.004132, PLK_afters_11 },
    { "m", 0.000000, 0.052090, 0.022910, 0.065163, PLK_afters_12 },
    { "n", 0.000277, 0.072966, 0.065519, 0.014599, PLK_afters_13 },
    { "o", 0.049639, 0.055516, 0.079347, 0.106958, PLK_afters_14 },
    { "p", 0.000277, 0.130052, 0.015567, 0.000864, PLK_afters_15 },
    { "q", 0.000000, 0.000274, 0.000011, 0.000000, PLK_afters_16 },
    { "r", 0.000971, 0.030623, 0.041520, 0.003239, PLK_afters_17 },
    { "s", 0.000555, 0.098089, 0.027759, 0.004996, PLK_afters_18 },
    { "t", 0.001248, 0.056899, 0.034098, 0.010611, PLK_afters_19 },
    { "u", 0.018719, 0.021654, 0.020529, 0.041796, PLK_afters_20 },
    { "v", 0.000000, 0.000821, 0.000109, 0.000014, PLK_afters_21 },
    { "w", 0.293955, 0.066156, 0.040757, 0.011187, PLK_afters_22 },
    { "x", 0.000277, 0.000014, 0.000019, 0.000043, PLK_afters_23 },
    { "y", 0.000277, 0.000043, 0.039116, 0.080122, PLK_afters_24 },
    { "z", 0.204104, 0.068661, 0.010084, 0.010366, PLK_afters_25 },
    { "à", 0.000832, 0.000000, 0.000000, 0.000072, NULL },
    { "â", 0.000000, 0.000000, 0.000004, 0.000000, PLK_afters_27 },
    { "ç", 0.000000, 0.000029, 0.000000, 0.000000, PLK_afters_28 },
    { "è", 0.000139, 0.000014, 0.000026, 0.000000, PLK_afters_29 },
    { "é", 0.000000, 0.000000, 0.000030, 0.000086, PLK_afters_30 },
    { "ê", 0.000000, 0.000029, 0.000004, 0.000000, PLK_afters_31 },
    { "î", 0.000000, 0.000000, 0.000004, 0.000000, PLK_afters_32 },
    { "ò", 0.000000, 0.000014, 0.000000, 0.000000, PLK_afters_33 },
    { "ó", 0.000000, 0.000432, 0.010801, 0.000029, PLK_afters_34 },
    { "ô", 0.000000, 0.000000, 0.000004, 0.000000, PLK_afters_35 },
    { "ą", 0.000000, 0.000000, 0.014283, 0.032898, PLK_afters_36 },
    { "ć", 0.000000, 0.000058, 0.000083, 0.020991, PLK_afters_37 },
    { "ę", 0.000000, 0.000000, 0.010426, 0.054062, PLK_afters_38 },
    { "ł", 0.000000, 0.004938, 0.030076, 0.054710, PLK_afters_39 },
    { "ń", 0.000000, 0.000000, 0.001837, 0.003930, PLK_afters_40 },
    { "ś", 0.000971, 0.010237, 0.009232, 0.010697, PLK_afters_41 },
    { "ű", 0.000000, 0.000000, 0.000008, 0.000000, PLK_afters_42 },
    { "ź", 0.000000, 0.000418, 0.001326, 0.000864, PLK_afters_43 },
    { "ż", 0.000277, 0.010107, 0.008770, 0.014253, PLK_afters_44 },
    { "ch", 0.000000, 0.011576, 0.006505, 0.031847, PLK_afters_45 },
    { "cz", 0.000000, 0.012195, 0.014020, 0.002822, PLK_afters_46 },
    { "dz", 0.000000, 0.012137, 0.008195, 0.000518, PLK_afters_47 },
    { "rz", 0.000000, 0.004924, 0.017269, 0.001756, PLK_afters_48 },
    { "sz", 0.000000, 0.011849, 0.013044, 0.002721, PLK_afters_49 },
    NULL
};	/* PLK */
static float PLK_word_lens[] = {
    0.000000, 0.094067, 0.087297, 0.125944, 0.095893, 0.139235, 0.113070, 0.112027, 0.087180, 0.058603, 0.038438, 0.022134, 0.012769, 0.007539, 0.002765, 0.001722, 0.001135, 0.000091, 0.000039, 0.000039, 0.000000, 0.000000, 0.000013, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float PLK_all_consonants[] = {
    0.000000, 0.047516, 0.003443, 0.007539, 0.002504, 0.000887, 0.000965, 0.000222, 0.000091, 0.000000, 0.000000, 0.000000, 
};

static float PLK_consonant_run[] = {
    0.000000, 0.675325, 0.231014, 0.058849, 0.024520, 0.005587, 0.003268, 0.001270, 0.000113, 0.000048, 0.000006, 0.000000
};

static float PLK_vowel_run[] = {
    0.000000, 0.847202, 0.152116, 0.000674, 0.000007, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float RUS_afters_0[] = { 0.000522, 0.010960, 0.080376, 0.044624, 0.071764, 0.043841, 0.024791, 0.034186, 0.002349, 0.015136, 0.092641, 0.077244, 0.066284, 0.069415, 0.001305, 0.010960, 0.060282, 0.093163, 0.073591, 0.003392, 0.001566, 0.030793, 0, 0.014875, 0.017223, 0.009656, 0, 0, 0, 0, 0.024269, 0.024791, 0};
static float RUS_afters_1[] = { 0.027715, 0, 0.005090, 0, 0.002262, 0.210973, 0.001131, 0, 0.036199, 0, 0.002828, 0.145362, 0, 0.029977, 0.222851, 0, 0.084276, 0.001697, 0.000566, 0.058258, 0, 0, 0.000566, 0, 0.001131, 0.004525, 0.005656, 0.095023, 0.015271, 0, 0, 0.048643, 0};
static float RUS_afters_2[] = { 0.072487, 0.000539, 0.005659, 0.000269, 0.017515, 0.217192, 0, 0.021558, 0.061439, 0, 0.008084, 0.022366, 0.002425, 0.032336, 0.274859, 0.001617, 0.025600, 0.102398, 0.004042, 0.028833, 0, 0.000539, 0.001078, 0.001078, 0.003503, 0, 0, 0.056589, 0.013473, 0, 0, 0.024522, 0};
static float RUS_afters_3[] = { 0.085714, 0, 0, 0, 0.072143, 0.025000, 0, 0, 0.042857, 0, 0.003571, 0.156429, 0, 0.050714, 0.371429, 0, 0.146429, 0.000714, 0.000714, 0.038571, 0, 0, 0, 0.004286, 0.001429, 0, 0, 0, 0, 0, 0, 0, 0};
static float RUS_afters_4[] = { 0.161030, 0.000384, 0.018832, 0.000769, 0.001153, 0.181783, 0, 0.001153, 0.094543, 0, 0.007302, 0.011145, 0.003459, 0.072636, 0.111837, 0.002306, 0.046118, 0.006533, 0.002690, 0.125673, 0, 0.000384, 0.023059, 0.001537, 0.005765, 0.001153, 0.001153, 0.046503, 0.054958, 0, 0, 0.016141, 0};
static float RUS_afters_5[] = { 0.000934, 0.046056, 0.023495, 0.041232, 0.053057, 0.009491, 0.011981, 0.039054, 0.002334, 0.048545, 0.022250, 0.079664, 0.082465, 0.111561, 0.001245, 0.017426, 0.098024, 0.099891, 0.121519, 0.000311, 0.000311, 0.018516, 0.009802, 0.025984, 0.017738, 0.008558, 0, 0, 0, 0, 0.004201, 0.004357, 0};
static float RUS_afters_6[] = { 0.107417, 0.002558, 0, 0.002558, 0.153453, 0.341432, 0.007673, 0, 0.249361, 0, 0.001279, 0, 0.001279, 0.062660, 0, 0, 0.010230, 0, 0, 0.042199, 0, 0, 0, 0, 0, 0, 0, 0, 0.017903, 0, 0, 0, 0};
static float RUS_afters_7[] = { 0.158443, 0.016678, 0.099375, 0.013204, 0.133426, 0.062543, 0.004864, 0.011119, 0.030577, 0, 0.009034, 0.098680, 0.028492, 0.132731, 0.038221, 0, 0.073662, 0.000695, 0.001390, 0.020848, 0, 0, 0, 0, 0.001390, 0, 0.003475, 0.028492, 0.018763, 0, 0, 0.013899, 0};
static float RUS_afters_8[] = { 0.002260, 0.004520, 0.038983, 0.011299, 0.026836, 0.038701, 0.014689, 0.057627, 0.011299, 0.032486, 0.050565, 0.094068, 0.075141, 0.057062, 0.011582, 0.003390, 0.051130, 0.079379, 0.144633, 0.000565, 0.000282, 0.075706, 0.019774, 0.013842, 0.026554, 0.017232, 0, 0, 0, 0, 0.010169, 0.030226, 0};
static float RUS_afters_9[] = { 0, 0, 0, 0, 0.169643, 0, 0, 0, 0, 0, 0.008929, 0, 0.008929, 0.223214, 0, 0, 0, 0.330357, 0.125000, 0, 0, 0, 0.017857, 0, 0.089286, 0, 0, 0, 0, 0, 0, 0.026786, 0};
static float RUS_afters_10[] = { 0.219178, 0, 0.011912, 0, 0, 0.023228, 0.000596, 0, 0.101251, 0, 0, 0.048243, 0.002978, 0.019655, 0.317451, 0, 0.160214, 0.002382, 0.041096, 0.049434, 0, 0, 0, 0, 0.002382, 0, 0, 0, 0, 0, 0, 0, 0};
static float RUS_afters_11[] = { 0.201395, 0, 0.000951, 0.002537, 0, 0.154456, 0.004123, 0.001269, 0.159213, 0, 0.006660, 0.003489, 0.006343, 0.027910, 0.127815, 0.001586, 0, 0.015541, 0.002220, 0.050428, 0, 0.000634, 0, 0.002537, 0.000634, 0, 0, 0.032350, 0.114494, 0, 0.042499, 0.040913, 0};
static float RUS_afters_12[] = { 0.054320, 0, 0, 0.006725, 0, 0.206932, 0.000517, 0.001552, 0.146922, 0, 0.002587, 0.048112, 0, 0.091050, 0.220900, 0.003104, 0.021728, 0.006725, 0, 0.089498, 0.000517, 0.000517, 0.001035, 0.002069, 0, 0.000517, 0, 0.062597, 0.003104, 0, 0, 0.028971, 0};
static float RUS_afters_13[] = { 0.167133, 0, 0, 0.004302, 0, 0.215100, 0.000430, 0.000215, 0.122392, 0, 0.002151, 0, 0.004517, 0.060228, 0.140030, 0, 0.001936, 0.007959, 0.003227, 0.034631, 0.001291, 0, 0.010970, 0.001506, 0.000645, 0, 0, 0.117445, 0.069262, 0, 0.003872, 0.030759, 0};
static float RUS_afters_14[] = { 0.003260, 0.059929, 0.071883, 0.061792, 0.064120, 0.031362, 0.031672, 0.035864, 0.027014, 0.074367, 0.032914, 0.067226, 0.063655, 0.061326, 0.000466, 0.011644, 0.062878, 0.097966, 0.074523, 0.000776, 0.000621, 0.005434, 0.001708, 0.017389, 0.004658, 0.007452, 0, 0, 0, 0, 0.018320, 0.009781, 0};
static float RUS_afters_15[] = { 0.067901, 0, 0, 0, 0, 0.086420, 0, 0, 0.027778, 0, 0.001543, 0.060700, 0, 0.005658, 0.361626, 0, 0.312757, 0.006687, 0.005144, 0.036523, 0.000514, 0.000514, 0.001029, 0.000514, 0, 0, 0, 0.011831, 0.003086, 0, 0, 0.009774, 0};
static float RUS_afters_16[] = { 0.198518, 0.005029, 0.006088, 0.007941, 0.038380, 0.197724, 0.008735, 0.008470, 0.084966, 0, 0.005558, 0.001853, 0.002912, 0.023293, 0.159873, 0.003970, 0, 0.016146, 0.040498, 0.067496, 0.004764, 0.004235, 0.008999, 0.001059, 0.005029, 0.000265, 0, 0.035998, 0.019058, 0, 0.005294, 0.037851, 0};
static float RUS_afters_17[] = { 0.025490, 0.000588, 0.066863, 0.000392, 0.000588, 0.126471, 0.000196, 0.000196, 0.039216, 0, 0.050980, 0.069608, 0.030000, 0.023529, 0.066863, 0.031373, 0.011569, 0.016667, 0.268039, 0.026863, 0.001373, 0.008039, 0.001373, 0.008431, 0.002353, 0, 0.000196, 0.012941, 0.031569, 0, 0.004118, 0.073529, 0.000588};
static float RUS_afters_18[] = { 0.091819, 0.000222, 0.123833, 0.000222, 0.000222, 0.116052, 0, 0, 0.078257, 0, 0.007114, 0.010227, 0.003557, 0.027345, 0.162961, 0.000222, 0.066919, 0.037350, 0.000667, 0.021120, 0.003335, 0, 0.003557, 0.002890, 0.000222, 0.003112, 0.000222, 0.068697, 0.157403, 0, 0, 0.012450, 0};
static float RUS_afters_19[] = { 0, 0.026053, 0.034922, 0.049889, 0.136364, 0.018847, 0.042683, 0.009424, 0.000554, 0.008869, 0.044900, 0.019956, 0.065965, 0.031596, 0, 0.032705, 0.018847, 0.073725, 0.104213, 0, 0, 0.070399, 0, 0.063193, 0.078160, 0.045455, 0, 0, 0, 0, 0.017738, 0.005543, 0};
static float RUS_afters_20[] = { 0.094340, 0, 0, 0, 0, 0.245283, 0, 0, 0.358491, 0, 0, 0, 0, 0, 0.037736, 0, 0.037736, 0, 0, 0.018868, 0, 0, 0, 0, 0, 0, 0, 0.188679, 0, 0, 0, 0.018868, 0};
static float RUS_afters_21[] = { 0.106965, 0, 0.084577, 0, 0, 0.012438, 0, 0, 0.131841, 0, 0, 0.062189, 0.002488, 0.024876, 0.405473, 0, 0.136816, 0.004975, 0, 0.024876, 0, 0, 0, 0, 0.002488, 0, 0, 0, 0, 0, 0, 0, 0};
static float RUS_afters_22[] = { 0.428941, 0, 0.064599, 0, 0, 0.304910, 0, 0, 0.005168, 0, 0, 0, 0, 0, 0.062016, 0, 0, 0, 0, 0.059432, 0, 0, 0, 0, 0, 0, 0, 0.074935, 0, 0, 0, 0, 0};
static float RUS_afters_23[] = { 0.118821, 0.000951, 0, 0, 0, 0.256654, 0, 0, 0.116920, 0, 0.003802, 0.002852, 0, 0.124525, 0.006654, 0, 0.018061, 0, 0.235741, 0.068441, 0, 0, 0, 0, 0.005703, 0, 0, 0, 0.040875, 0, 0, 0, 0};
static float RUS_afters_24[] = { 0.126645, 0, 0, 0, 0, 0.194079, 0, 0, 0.210526, 0, 0, 0.029605, 0, 0.069079, 0.027961, 0, 0, 0, 0, 0.074013, 0, 0, 0.001645, 0, 0, 0, 0, 0, 0.266447, 0, 0, 0, 0};
static float RUS_afters_25[] = { 0.181598, 0, 0.002421, 0, 0, 0.365617, 0, 0, 0.319613, 0, 0, 0, 0, 0.021792, 0.002421, 0, 0.004843, 0, 0, 0.062954, 0, 0, 0, 0, 0, 0.002421, 0, 0, 0.036320, 0, 0, 0, 0};
static float RUS_afters_26[] = { 0, 0, 0, 0, 0, 0.350000, 0, 0, 0.050000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.600000, 0};
static float RUS_afters_27[] = { 0, 0.012195, 0.028143, 0.000938, 0.010319, 0.036585, 0, 0.001876, 0, 0.217636, 0.016886, 0.078799, 0.111632, 0.039400, 0, 0.009381, 0.007505, 0.097561, 0.060038, 0, 0, 0.196060, 0.000938, 0.001876, 0.061914, 0.004690, 0, 0, 0, 0, 0, 0.005629, 0};
static float RUS_afters_28[] = { 0, 0.026144, 0.004902, 0, 0.001634, 0.303922, 0, 0.014706, 0.084967, 0, 0.024510, 0, 0.063725, 0.094771, 0, 0, 0, 0.112745, 0.006536, 0, 0, 0.003268, 0, 0, 0.013072, 0.006536, 0, 0, 0, 0, 0.098039, 0.140523, 0};
static float RUS_afters_29[] = { 0, 0, 0, 0, 0.187500, 0, 0, 0, 0, 0, 0, 0, 0.062500, 0, 0, 0, 0.187500, 0, 0.062500, 0, 0.437500, 0.062500, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float RUS_afters_30[] = { 0, 0.199262, 0, 0, 0.154982, 0, 0, 0, 0, 0, 0, 0.025830, 0, 0.022140, 0, 0, 0, 0.132841, 0.350554, 0, 0, 0, 0, 0.033210, 0, 0.073801, 0, 0, 0, 0, 0.007380, 0, 0};
static float RUS_afters_31[] = { 0, 0.017591, 0.056834, 0.016238, 0.043302, 0.052774, 0.012179, 0.040595, 0, 0.021651, 0.040595, 0.036536, 0.051421, 0.089310, 0, 0, 0.021651, 0.077131, 0.246279, 0, 0, 0.054127, 0.004060, 0.010825, 0.004060, 0.075778, 0, 0, 0, 0, 0.016238, 0.010825, 0};
static struct letter_frequencies RUS_counts[] = {
    { "а", 0.021195, 0.006671, 0.073814, 0.081327, RUS_afters_0 },
    { "б", 0.009634, 0.057373, 0.017131, 0.004511, RUS_afters_1 },
    { "в", 0.278902, 0.097973, 0.042957, 0.016202, RUS_afters_2 },
    { "г", 0.000963, 0.034183, 0.017072, 0.013533, RUS_afters_3 },
    { "д", 0.000000, 0.045365, 0.037392, 0.014232, RUS_afters_4 },
    { "е", 0.001445, 0.021857, 0.120475, 0.096385, RUS_afters_5 },
    { "ж", 0.014933, 0.015312, 0.010715, 0.002478, RUS_afters_6 },
    { "з", 0.000000, 0.035453, 0.017448, 0.008514, RUS_afters_7 },
    { "и", 0.414258, 0.025160, 0.062267, 0.056802, RUS_afters_8 },
    { "й", 0.000000, 0.000000, 0.002218, 0.070907, RUS_afters_9 },
    { "к", 0.046724, 0.044730, 0.019310, 0.031006, RUS_afters_10 },
    { "л", 0.000000, 0.026558, 0.054167, 0.030497, RUS_afters_11 },
    { "м", 0.000000, 0.055340, 0.021033, 0.063664, RUS_afters_12 },
    { "н", 0.000482, 0.089015, 0.064327, 0.023508, RUS_afters_13 },
    { "о", 0.054913, 0.043141, 0.114117, 0.075227, RUS_afters_14 },
    { "п", 0.000000, 0.091365, 0.010021, 0.000191, RUS_afters_15 },
    { "р", 0.000963, 0.020713, 0.068367, 0.006671, RUS_afters_16 },
    { "с", 0.078998, 0.130377, 0.060366, 0.008450, RUS_afters_17 },
    { "т", 0.000000, 0.078277, 0.064684, 0.078468, RUS_afters_18 },
    { "у", 0.006262, 0.022238, 0.028797, 0.033865, RUS_afters_19 },
    { "ф", 0.000000, 0.000572, 0.000871, 0.001525, RUS_afters_20 },
    { "х", 0.000000, 0.010992, 0.004535, 0.047906, RUS_afters_21 },
    { "ц", 0.000000, 0.007815, 0.005229, 0.004066, RUS_afters_22 },
    { "ч", 0.000000, 0.030180, 0.011428, 0.002224, RUS_afters_23 },
    { "ш", 0.000000, 0.001461, 0.011586, 0.001843, RUS_afters_24 },
    { "щ", 0.000000, 0.001398, 0.007744, 0.000318, RUS_afters_25 },
    { "ъ", 0.000000, 0.000000, 0.000396, 0.000000, RUS_afters_26 },
    { "ы", 0.000000, 0.000000, 0.021112, 0.051846, RUS_afters_27 },
    { "ь", 0.000000, 0.000000, 0.012121, 0.095622, RUS_afters_28 },
    { "э", 0.000000, 0.001017, 0.000000, 0.000000, RUS_afters_29 },
    { "ю", 0.000000, 0.000508, 0.005209, 0.019696, RUS_afters_30 },
    { "я", 0.070328, 0.004956, 0.013091, 0.058326, RUS_afters_31 },
    { "ё", 0.000000, 0.000000, 0.000000, 0.000191, NULL },
    NULL
};	/* RUS */
static float RUS_word_lens[] = {
    0.000000, 0.116531, 0.108897, 0.127926, 0.153691, 0.140387, 0.118552, 0.082739, 0.062307, 0.039686, 0.023688, 0.014651, 0.006231, 0.002750, 0.001347, 0.000505, 0.000112, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float RUS_all_consonants[] = {
    0.000000, 0.050295, 0.025372, 0.003480, 0.003256, 0.001235, 0.000337, 0.000056, 0.000000, 0.000000, 0.000000, 0.000000, 
};

static float RUS_consonant_run[] = {
    0.000000, 0.628283, 0.294164, 0.056766, 0.016981, 0.003209, 0.000454, 0.000114, 0.000028, 0.000000, 0.000000, 0.000000
};

static float RUS_vowel_run[] = {
    0.000000, 0.914516, 0.081923, 0.003560, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float ESP_afters_0[] = { 0, 0.088970, 0.057499, 0.118353, 0.001492, 0.002834, 0.013797, 0.003356, 0.001566, 0.012007, 0, 0.096950, 0.046834, 0.139682, 0.000298, 0.015139, 0.149676, 0.140428, 0.025654, 0.004773, 0.014170, 0.000075, 0.004624, 0.016705, 0, 0, 0.002461, 0.010814, 0, 0.000149, 0, 0.008949, 0.022746};
static float ESP_afters_1[] = { 0.290941, 0, 0, 0, 0.115331, 0, 0, 0, 0.098606, 0.001394, 0, 0.118118, 0, 0, 0.045993, 0, 0.198606, 0.011150, 0.000348, 0.045993, 0.000348, 0, 0.000697, 0, 0.002439, 0.000348, 0.066899, 0, 0.002787, 0, 0, 0, 0};
static float ESP_afters_2[] = { 0.225060, 0, 0.002570, 0, 0.080713, 0, 0, 0.073307, 0.186215, 0, 0, 0.009825, 0, 0.000151, 0.288543, 0, 0.023881, 0, 0.010429, 0.061971, 0, 0, 0, 0, 0.004081, 0.000605, 0.022068, 0, 0.010127, 0.000453, 0, 0, 0};
static float ESP_afters_3[] = { 0.164666, 0, 0, 0, 0.399782, 0, 0, 0.000243, 0.099794, 0, 0, 0, 0.001698, 0, 0.262398, 0, 0.025343, 0, 0, 0.015885, 0.000364, 0, 0, 0, 0.001091, 0.001213, 0.017461, 0, 0.009822, 0, 0, 0, 0.000243};
static float ESP_afters_4[] = { 0.007828, 0.013735, 0.054654, 0.027754, 0.000854, 0.002918, 0.025406, 0.000854, 0.002775, 0.011386, 0, 0.118275, 0.033091, 0.251850, 0.004697, 0.005978, 0.154355, 0.182750, 0.014731, 0.000569, 0.014304, 0.010105, 0.001637, 0.013948, 0.000285, 0, 0.002918, 0.012667, 0.000854, 0, 0, 0.026829, 0.001993};
static float ESP_afters_5[] = { 0.143309, 0, 0, 0, 0.220232, 0, 0, 0, 0.157007, 0, 0, 0.057956, 0, 0, 0.064278, 0, 0.119073, 0, 0, 0.208641, 0, 0, 0, 0, 0.014752, 0.003161, 0.010537, 0, 0, 0.001054, 0, 0, 0};
static float ESP_afters_6[] = { 0.231242, 0, 0, 0, 0.062116, 0, 0, 0, 0.073801, 0, 0, 0.013530, 0.003075, 0.023985, 0.182657, 0, 0.145141, 0, 0, 0.212792, 0, 0, 0, 0, 0.002460, 0.001845, 0.004305, 0, 0.027675, 0.012915, 0.002460, 0, 0};
static float ESP_afters_7[] = { 0.454054, 0, 0, 0, 0.146547, 0, 0, 0, 0.111111, 0, 0, 0, 0, 0, 0.206006, 0, 0, 0, 0, 0.058258, 0, 0, 0, 0, 0.004805, 0.001201, 0.006607, 0, 0.008408, 0.003003, 0, 0, 0};
static float ESP_afters_8[] = { 0.072822, 0.020109, 0.058544, 0.088886, 0.114588, 0.004998, 0.027963, 0, 0.002618, 0.020347, 0, 0.029510, 0.041647, 0.115302, 0.066754, 0.002380, 0.056045, 0.066278, 0.056640, 0.000952, 0.017016, 0.000357, 0, 0.012137, 0.000595, 0.012494, 0, 0.022370, 0.070800, 0, 0, 0.015588, 0.002261};
static float ESP_afters_9[] = { 0.265781, 0, 0, 0, 0.165006, 0, 0, 0, 0.032115, 0, 0, 0, 0, 0, 0.372093, 0, 0, 0, 0, 0.119601, 0, 0, 0, 0, 0.005537, 0, 0.002215, 0, 0.036545, 0.001107, 0, 0, 0};
static float ESP_afters_10[] = { 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ESP_afters_11[] = { 0.435105, 0.002839, 0.008831, 0.007412, 0.160069, 0.001419, 0.018924, 0.000158, 0.075382, 0, 0, 0, 0.012774, 0, 0.143510, 0.003627, 0.000315, 0.003943, 0.019398, 0.039426, 0.040688, 0, 0, 0.005677, 0.006939, 0.001262, 0.005835, 0, 0.005993, 0.000315, 0, 0, 0.000158};
static float ESP_afters_12[] = { 0.201476, 0.059445, 0, 0, 0.228406, 0, 0, 0, 0.145821, 0, 0, 0, 0, 0.002992, 0.138440, 0.074806, 0, 0, 0, 0.090565, 0, 0, 0, 0, 0.025733, 0.010573, 0.014363, 0, 0.007181, 0.000199, 0, 0, 0};
static float ESP_afters_13[] = { 0.151708, 0, 0.076933, 0.135458, 0.047480, 0.018662, 0.020185, 0.002412, 0.061826, 0.002158, 0, 0.000762, 0.005713, 0.000762, 0.122636, 0, 0.008633, 0.039482, 0.233211, 0.027676, 0.006221, 0, 0, 0.009521, 0.001396, 0.002666, 0.013711, 0, 0.003047, 0.000508, 0, 0, 0.007236};
static float ESP_afters_14[] = { 0, 0.031431, 0.038248, 0.038710, 0.000462, 0.005547, 0.007280, 0.001156, 0.000116, 0.017911, 0, 0.056275, 0.063554, 0.213196, 0, 0.007742, 0.186619, 0.250404, 0.022995, 0, 0.009475, 0.000231, 0.010284, 0.019760, 0, 0.000116, 0.002542, 0.013173, 0, 0, 0, 0.001271, 0.001502};
static float ESP_afters_15[] = { 0.235146, 0, 0, 0, 0.175448, 0, 0, 0, 0.055774, 0, 0, 0.047646, 0, 0, 0.221973, 0, 0.151345, 0.001121, 0.002242, 0.088565, 0, 0, 0, 0, 0.010370, 0.001121, 0.005045, 0, 0.003643, 0.000561, 0, 0, 0};
static float ESP_afters_16[] = { 0.230828, 0.005721, 0.017384, 0.024425, 0.213885, 0.001100, 0.009572, 0, 0.108923, 0.000550, 0, 0.013863, 0.056222, 0.015183, 0.114754, 0.007922, 0.032017, 0.018374, 0.042029, 0.019034, 0.008252, 0, 0, 0.003521, 0.012763, 0.009352, 0.014963, 0, 0.013863, 0.000550, 0, 0, 0.004951};
static float ESP_afters_17[] = { 0.172395, 0.001488, 0.029499, 0.006225, 0.184168, 0.003924, 0.003112, 0.000947, 0.120433, 0, 0, 0.000947, 0.009608, 0.001488, 0.123681, 0.048309, 0, 0, 0.139107, 0.124087, 0.001894, 0, 0, 0, 0.000947, 0.002030, 0.012585, 0, 0.009066, 0.002436, 0, 0, 0.001624};
static float ESP_afters_18[] = { 0.245262, 0, 0, 0, 0.253563, 0, 0, 0, 0.097259, 0, 0, 0.000157, 0.000313, 0, 0.191856, 0, 0.109319, 0, 0.000157, 0.048238, 0, 0, 0, 0, 0.013156, 0.002349, 0.009084, 0, 0.016445, 0.012843, 0, 0, 0};
static float ESP_afters_19[] = { 0.049946, 0.023351, 0.054703, 0.037838, 0.126703, 0.003892, 0.008865, 0.000216, 0.020973, 0.011243, 0, 0.046919, 0.034811, 0.264649, 0.004541, 0.011892, 0.084973, 0.091676, 0.015568, 0, 0.011892, 0.000432, 0.031351, 0.026811, 0.002595, 0.020324, 0.004108, 0.003892, 0.001081, 0, 0, 0.004108, 0.000649};
static float ESP_afters_20[] = { 0.231439, 0, 0, 0, 0.267281, 0, 0, 0, 0.273425, 0, 0, 0, 0, 0, 0.166411, 0, 0.002048, 0, 0, 0.013825, 0, 0, 0, 0, 0.006144, 0.005632, 0.028674, 0, 0.005120, 0, 0, 0, 0};
static float ESP_afters_21[] = { 0.047059, 0, 0.088235, 0, 0, 0, 0, 0.005882, 0.147059, 0, 0, 0, 0, 0, 0, 0.317647, 0, 0, 0.341176, 0.005882, 0.023529, 0.011765, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.011765};
static float ESP_afters_22[] = { 0.390306, 0, 0, 0, 0.096939, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.410714, 0, 0, 0, 0, 0.015306, 0, 0, 0, 0, 0.002551, 0.005102, 0, 0, 0.079082, 0, 0, 0, 0};
static float ESP_afters_23[] = { 0.383580, 0, 0.022880, 0, 0.001346, 0, 0.004038, 0, 0, 0, 0, 0, 0.130552, 0.006729, 0.261104, 0, 0, 0, 0.001346, 0.033647, 0, 0, 0, 0, 0.006729, 0, 0, 0, 0.144011, 0, 0, 0, 0.004038};
static float ESP_afters_24[] = { 0, 0.037344, 0.031120, 0.002075, 0.002075, 0.010373, 0.074689, 0, 0, 0.004149, 0, 0.053942, 0.026971, 0.315353, 0, 0.006224, 0.053942, 0.340249, 0.026971, 0.002075, 0.004149, 0.002075, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.004149, 0.002075};
static float ESP_afters_25[] = { 0, 0.009434, 0.002358, 0.117925, 0, 0, 0.002358, 0, 0.002358, 0.002358, 0, 0.212264, 0.016509, 0.259434, 0, 0.011792, 0.054245, 0.242925, 0.040094, 0, 0.009434, 0.011792, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.004717, 0};
static float ESP_afters_26[] = { 0.789570, 0.000915, 0.010979, 0.048490, 0.003660, 0.004575, 0.001830, 0, 0, 0.001830, 0, 0.003660, 0.010979, 0.024703, 0.046661, 0, 0.014639, 0.030192, 0.002745, 0, 0.002745, 0, 0, 0.000915, 0, 0, 0, 0, 0, 0, 0, 0, 0.000915};
static float ESP_afters_27[] = { 0.587869, 0, 0, 0, 0.021773, 0, 0, 0, 0.012442, 0, 0, 0, 0, 0, 0.348367, 0, 0, 0, 0, 0.010886, 0, 0, 0, 0, 0, 0, 0.013997, 0, 0.004666, 0, 0, 0, 0};
static float ESP_afters_28[] = { 0, 0.003527, 0.008818, 0.008818, 0, 0, 0.003527, 0, 0, 0, 0, 0.081129, 0.058201, 0.742504, 0, 0.001764, 0.008818, 0.067019, 0, 0, 0.010582, 0.005291, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ESP_afters_29[] = { 0, 0.035398, 0, 0, 0, 0, 0.017699, 0, 0, 0, 0, 0.168142, 0.061947, 0.433628, 0, 0.053097, 0.035398, 0.168142, 0.026549, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ESP_afters_30[] = { 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float ESP_afters_31[] = { 0.517505, 0, 0, 0, 0.241794, 0, 0, 0, 0.004376, 0, 0, 0, 0, 0, 0.169584, 0, 0, 0, 0, 0.003282, 0, 0, 0, 0, 0.006565, 0.002188, 0.033917, 0, 0.020788, 0, 0, 0, 0};
static float ESP_afters_32[] = { 0, 0, 0, 0, 0.769280, 0, 0, 0, 0.135604, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.066838, 0.028278, 0, 0, 0, 0, 0, 0};
static struct letter_frequencies ESP_counts[] = {
    { "a", 0.313341, 0.067342, 0.119703, 0.248239, ESP_afters_0 },
    { "b", 0.000000, 0.015662, 0.025169, 0.000000, ESP_afters_1 },
    { "c", 0.000000, 0.085619, 0.040123, 0.000000, ESP_afters_2 },
    { "d", 0.000000, 0.112693, 0.047857, 0.008024, ESP_afters_3 },
    { "e", 0.000408, 0.102024, 0.114074, 0.175874, ESP_afters_4 },
    { "f", 0.000000, 0.016553, 0.004211, 0.000000, ESP_afters_5 },
    { "g", 0.000000, 0.009213, 0.014137, 0.000000, ESP_afters_6 },
    { "h", 0.000000, 0.032572, 0.006112, 0.000416, ESP_afters_7 },
    { "i", 0.001224, 0.017504, 0.083952, 0.010847, ESP_afters_8 },
    { "j", 0.000000, 0.005141, 0.007842, 0.000000, ESP_afters_9 },
    { "k", 0.000000, 0.000030, 0.000000, 0.000000, ESP_afters_10 },
    { "l", 0.000000, 0.094891, 0.033817, 0.051770, ESP_afters_11 },
    { "m", 0.000000, 0.058338, 0.032764, 0.000059, ESP_afters_12 },
    { "n", 0.000000, 0.029481, 0.073961, 0.105174, ESP_afters_13 },
    { "o", 0.016320, 0.012244, 0.088539, 0.157687, ESP_afters_14 },
    { "p", 0.000000, 0.066659, 0.014234, 0.000000, ESP_afters_15 },
    { "r", 0.000000, 0.025974, 0.088249, 0.041695, ESP_afters_16 },
    { "s", 0.000000, 0.095189, 0.044978, 0.128028, ESP_afters_17 },
    { "t", 0.000000, 0.045945, 0.051983, 0.000089, ESP_afters_18 },
    { "u", 0.000000, 0.028352, 0.039435, 0.015959, ESP_afters_19 },
    { "v", 0.002448, 0.025796, 0.011656, 0.000149, ESP_afters_20 },
    { "x", 0.000816, 0.000386, 0.001687, 0.000119, ESP_afters_21 },
    { "y", 0.665443, 0.007400, 0.001536, 0.005290, ESP_afters_22 },
    { "z", 0.000000, 0.000743, 0.007713, 0.007222, ESP_afters_23 },
    { "á", 0.000000, 0.000624, 0.004952, 0.002556, ESP_afters_24 },
    { "é", 0.000000, 0.002972, 0.003481, 0.006508, ESP_afters_25 },
    { "í", 0.000000, 0.000297, 0.011634, 0.006122, ESP_afters_26 },
    { "ñ", 0.000000, 0.000000, 0.006907, 0.000000, ESP_afters_27 },
    { "ó", 0.000000, 0.000000, 0.006091, 0.025915, ESP_afters_28 },
    { "ú", 0.000000, 0.001248, 0.000763, 0.002259, ESP_afters_29 },
    { "ü", 0.000000, 0.000000, 0.000043, 0.000000, ESP_afters_30 },
    { "ll", 0.000000, 0.007489, 0.007111, 0.000000, ESP_afters_31 },
    { "qu", 0.000000, 0.031621, 0.005285, 0.000000, ESP_afters_32 },
    NULL
};	/* ESP */
static float ESP_word_lens[] = {
    0.000000, 0.067895, 0.269086, 0.113490, 0.102770, 0.108199, 0.100693, 0.083740, 0.063740, 0.038643, 0.025900, 0.013380, 0.007202, 0.003324, 0.001163, 0.000471, 0.000249, 0.000055, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float ESP_all_consonants[] = {
    0.000000, 0.000222, 0.000055, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 
};

static float ESP_consonant_run[] = {
    0.000000, 0.813276, 0.172621, 0.013920, 0.000183, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float ESP_vowel_run[] = {
    0.000000, 0.908487, 0.079174, 0.011741, 0.000599, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static struct letter_frequencies SAN_counts[] = {
    { "ं", 0.000000, 0.000000, 0.019953, 0.145349 },
    { "अ", 0.002110, 0.075000, 0.000000, 0.000000 },
    { "आ", 0.000000, 0.008721, 0.000000, 0.000000 },
    { "इ", 0.000000, 0.002907, 0.000000, 0.000000 },
    { "उ", 0.000000, 0.005233, 0.000000, 0.000000 },
    { "ए", 0.000000, 0.011047, 0.000000, 0.000000 },
    { "क", 0.010549, 0.036047, 0.042268, 0.055233 },
    { "ग", 0.000000, 0.008140, 0.007088, 0.006977 },
    { "घ", 0.000000, 0.000581, 0.006301, 0.000000 },
    { "च", 0.154008, 0.014535, 0.006301, 0.025000 },
    { "छ", 0.000000, 0.023256, 0.000000, 0.000000 },
    { "ज", 0.002110, 0.029651, 0.007088, 0.006977 },
    { "ञ", 0.014768, 0.005233, 0.000000, 0.003488 },
    { "ट", 0.061181, 0.004070, 0.001313, 0.000581 },
    { "ठ", 0.000000, 0.004651, 0.000525, 0.000000 },
    { "ड", 0.006329, 0.001744, 0.000000, 0.000000 },
    { "ढ", 0.000000, 0.000000, 0.000000, 0.001163, NULL },
    { "ण", 0.008439, 0.005233, 0.020215, 0.013372 },
    { "त", 0.075949, 0.086047, 0.065897, 0.074419 },
    { "थ", 0.000000, 0.015698, 0.001838, 0.000000 },
    { "द", 0.004219, 0.013372, 0.014965, 0.026163 },
    { "ध", 0.002110, 0.018605, 0.029929, 0.008140 },
    { "न", 0.042194, 0.026744, 0.058020, 0.091279 },
    { "प", 0.189873, 0.027326, 0.030717, 0.007558 },
    { "ब", 0.000000, 0.003488, 0.000525, 0.001163 },
    { "भ", 0.000000, 0.013372, 0.011552, 0.004070 },
    { "म", 0.006329, 0.034302, 0.034917, 0.019767 },
    { "य", 0.156118, 0.127907, 0.039643, 0.017442 },
    { "र", 0.050633, 0.103488, 0.051457, 0.090116 },
    { "ल", 0.000000, 0.004651, 0.006038, 0.006977 },
    { "व", 0.037975, 0.134302, 0.032817, 0.018605 },
    { "श", 0.004219, 0.012791, 0.006563, 0.020349 },
    { "ष", 0.014768, 0.025000, 0.013127, 0.035465 },
    { "स", 0.156118, 0.105233, 0.011289, 0.072093 },
    { "ह", 0.000000, 0.010465, 0.007351, 0.003488 },
    { "ऽ", 0.000000, 0.000000, 0.009451, 0.000000 },
    { "ा", 0.000000, 0.000000, 0.199265, 0.066860 },
    { "ि", 0.000000, 0.000000, 0.105014, 0.085465 },
    { "ी", 0.000000, 0.001163, 0.019953, 0.003488 },
    { "ु", 0.000000, 0.000000, 0.042268, 0.014535 },
    { "ू", 0.000000, 0.000000, 0.007351, 0.000000 },
    { "ृ", 0.000000, 0.000000, 0.009189, 0.000000 },
    { "े", 0.000000, 0.000000, 0.050932, 0.054651 },
    { "ै", 0.000000, 0.000000, 0.007876, 0.006977 },
    { "ो", 0.000000, 0.000000, 0.018378, 0.009302 },
    { "ौ", 0.000000, 0.000000, 0.002625, 0.003488 },
    NULL
};	/* SAN */
static float SAN_word_lens[] = {
    0.000000, 0.216044, 0.198268, 0.179125, 0.135825, 0.105743, 0.056518, 0.036463, 0.023701, 0.018232, 0.013218, 0.004558, 0.003191, 0.003191, 0.003646, 0.000912, 0.000456, 0.000456, 0.000456, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float SVE_afters_0[] = { 0.000869, 0.002981, 0.012545, 0.117998, 0.000124, 0.007204, 0.123215, 0.000373, 0, 0.000869, 0.021985, 0.070923, 0.038132, 0.221836, 0.000124, 0.012297, 0.190660, 0.027574, 0.111291, 0.003229, 0.030804, 0, 0.004720, 0, 0, 0, 0, 0.000248, 0, 0, 0};
static float SVE_afters_1[] = { 0.164234, 0.016058, 0, 0, 0.209489, 0, 0, 0, 0.048905, 0.007299, 0, 0.178102, 0, 0, 0.104380, 0, 0.126277, 0.005839, 0.000730, 0.040146, 0, 0, 0, 0.005839, 0, 0, 0.031387, 0.021898, 0, 0, 0.039416};
static float SVE_afters_2[] = { 0.001079, 0, 0, 0, 0.009709, 0, 0, 0.614887, 0.011327, 0, 0.351133, 0.002697, 0.000539, 0, 0.008630, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float SVE_afters_3[] = { 0.113107, 0.002566, 0, 0.026085, 0.567030, 0.000855, 0.001710, 0.007911, 0.031644, 0.011118, 0.000214, 0.005131, 0.000428, 0.010263, 0.021167, 0.001283, 0.039341, 0.015394, 0.001283, 0.033355, 0.003635, 0.000428, 0, 0.002780, 0, 0, 0.039128, 0.053239, 0, 0.001283, 0.009622};
static float SVE_afters_4[] = { 0.000399, 0.001330, 0.002792, 0.061694, 0.000532, 0.012897, 0.009573, 0.006914, 0, 0.007579, 0.015025, 0.060630, 0.019013, 0.368435, 0.001994, 0.001463, 0.176572, 0.041617, 0.191198, 0.000399, 0.014892, 0.000532, 0.001994, 0.002393, 0.000133, 0, 0, 0, 0, 0, 0};
static float SVE_afters_5[] = { 0.097688, 0, 0, 0, 0.028324, 0.030636, 0, 0, 0.065896, 0.005780, 0, 0.037572, 0, 0, 0.042775, 0, 0.132370, 0.001156, 0.075145, 0.013873, 0.002312, 0, 0, 0.005202, 0, 0, 0.035260, 0.065896, 0, 0.006358, 0.353757};
static float SVE_afters_6[] = { 0.167110, 0.001899, 0, 0.015572, 0.230915, 0.003418, 0.034182, 0.006836, 0.050513, 0.015951, 0.001519, 0.026586, 0, 0.028864, 0.115458, 0.000380, 0.063806, 0.017850, 0.072921, 0.022788, 0, 0, 0, 0.001519, 0, 0, 0.012913, 0.085454, 0, 0, 0.023547};
static float SVE_afters_7[] = { 0.370403, 0, 0, 0.000886, 0.149313, 0, 0.000443, 0.000886, 0.026584, 0.015064, 0, 0.000443, 0, 0.003101, 0.168808, 0, 0.000443, 0.001329, 0.000886, 0.089942, 0, 0.001772, 0, 0.006646, 0, 0, 0.076207, 0.027470, 0, 0, 0.059371};
static float SVE_afters_8[] = { 0.008465, 0.002351, 0.056431, 0.065601, 0.007994, 0.005408, 0.196802, 0.002351, 0, 0, 0.030567, 0.141547, 0.005878, 0.264049, 0.015754, 0.003292, 0.015518, 0.078768, 0.081589, 0.000235, 0.016694, 0.000235, 0.000470, 0, 0, 0, 0, 0, 0, 0, 0};
static float SVE_afters_9[] = { 0.608148, 0, 0, 0.017037, 0.031111, 0, 0, 0, 0, 0, 0.009630, 0.014074, 0, 0, 0.054815, 0, 0, 0.000741, 0.002963, 0.162963, 0, 0, 0, 0.001481, 0, 0, 0.076296, 0, 0, 0, 0.020741};
static float SVE_afters_10[] = { 0.239748, 0.002103, 0, 0.001402, 0.153873, 0.001052, 0.000701, 0.010515, 0.023134, 0.001402, 0.001753, 0.045566, 0.001052, 0.020680, 0.134595, 0.000701, 0.055030, 0.028742, 0.111812, 0.061689, 0.026288, 0, 0, 0.014371, 0, 0, 0.034350, 0.006660, 0, 0, 0.022783};
static float SVE_afters_11[] = { 0.145455, 0.008342, 0, 0.032941, 0.107166, 0.011337, 0.001497, 0.002781, 0.129626, 0.017112, 0.011337, 0.253262, 0.022460, 0.014545, 0.026738, 0.004920, 0.002781, 0.035722, 0.034011, 0.028663, 0.017112, 0, 0, 0.016043, 0, 0, 0.034652, 0.023316, 0, 0.000214, 0.017968};
static float SVE_afters_12[] = { 0.184594, 0.024289, 0, 0.005205, 0.236641, 0.004511, 0.004164, 0.004164, 0.154060, 0.003470, 0.004164, 0.014573, 0.092297, 0.018043, 0.055170, 0.012491, 0.002776, 0.014573, 0.015961, 0.006246, 0.002082, 0, 0, 0.027759, 0, 0, 0.043373, 0.037821, 0, 0, 0.031575};
static float SVE_afters_13[] = { 0.141711, 0.005539, 0.001337, 0.123950, 0.086134, 0.005730, 0.125477, 0.003247, 0.050993, 0.003056, 0.029794, 0.008403, 0.000764, 0.108671, 0.044882, 0.000764, 0.000955, 0.079450, 0.093201, 0.019862, 0.002865, 0, 0, 0.009549, 0, 0, 0.021199, 0.027311, 0, 0.000382, 0.004775};
static float SVE_afters_14[] = { 0.003389, 0.006143, 0.255454, 0.033256, 0.000847, 0.007202, 0.025630, 0.002118, 0.000212, 0.004025, 0.008261, 0.042999, 0.214150, 0.145096, 0, 0.017157, 0.147426, 0.026901, 0.046600, 0.001059, 0.010591, 0, 0.001059, 0, 0, 0, 0.000424, 0, 0, 0, 0};
static float SVE_afters_15[] = { 0.132979, 0, 0, 0, 0.089243, 0.010047, 0.001182, 0.001182, 0.018913, 0.000591, 0.000591, 0.039007, 0.010638, 0.014775, 0.058511, 0.162530, 0.060875, 0.005910, 0.034279, 0.012411, 0.001182, 0, 0, 0.000591, 0, 0, 0.017730, 0.325059, 0, 0.000591, 0.001182};
static float SVE_afters_16[] = { 0.175898, 0.014810, 0.000406, 0.054372, 0.139176, 0.011767, 0.024751, 0.003855, 0.073037, 0.006695, 0.023940, 0.026983, 0.017854, 0.064516, 0.068371, 0.007101, 0.030026, 0.047474, 0.059038, 0.035910, 0.008724, 0, 0, 0.008927, 0, 0, 0.045648, 0.025969, 0.000203, 0.000203, 0.024346};
static float SVE_afters_17[] = { 0.110744, 0.004663, 0.005829, 0.000971, 0.072081, 0.002331, 0.001166, 0.001360, 0.067612, 0.010103, 0.109578, 0.025840, 0.016126, 0.009520, 0.099670, 0.021372, 0.001749, 0.062172, 0.212940, 0.006023, 0.021372, 0, 0, 0.006217, 0, 0, 0.026617, 0.088790, 0, 0, 0.015154};
static float SVE_afters_18[] = { 0.149838, 0.004132, 0, 0, 0.213439, 0.002336, 0.000359, 0.001078, 0.122709, 0.005749, 0.001078, 0.007007, 0.005031, 0.005749, 0.061624, 0.000359, 0.062163, 0.028027, 0.237513, 0.012936, 0.012217, 0, 0, 0.016170, 0.000180, 0, 0.026051, 0.014912, 0, 0, 0.009342};
static float SVE_afters_19[] = { 0.004232, 0.006651, 0.004232, 0.048972, 0, 0.010278, 0.034462, 0.000605, 0, 0, 0.021161, 0.116687, 0.035067, 0.201935, 0, 0.086457, 0.140266, 0.094317, 0.175333, 0, 0.019347, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float SVE_afters_20[] = { 0.350614, 0.002274, 0.001364, 0.020464, 0.121419, 0.000455, 0.005457, 0.000455, 0.255571, 0, 0.000455, 0.009550, 0.002274, 0, 0.016371, 0, 0.005002, 0.010459, 0.001819, 0.015007, 0, 0, 0, 0.001819, 0, 0, 0.129604, 0.040018, 0, 0.008186, 0.001364};
static float SVE_afters_21[] = { 0.680000, 0, 0, 0, 0.200000, 0, 0, 0.040000, 0.080000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float SVE_afters_22[] = { 0.250000, 0.025000, 0, 0, 0.275000, 0, 0, 0.050000, 0.050000, 0, 0, 0.025000, 0, 0, 0.050000, 0.175000, 0, 0, 0.100000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float SVE_afters_23[] = { 0.008696, 0.006522, 0.343478, 0.023913, 0, 0.023913, 0.104348, 0.004348, 0, 0, 0.039130, 0.036957, 0.026087, 0.069565, 0.002174, 0.006522, 0.076087, 0.121739, 0.089130, 0, 0.013043, 0, 0.004348, 0, 0, 0, 0, 0, 0, 0, 0};
static float SVE_afters_24[] = { 0, 0, 0, 0, 0.500000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.500000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float SVE_afters_26[] = { 0, 0.001642, 0.012315, 0.027915, 0, 0.006158, 0.053777, 0.000411, 0, 0.000821, 0.019294, 0.101396, 0.028736, 0.225369, 0, 0.007389, 0.371921, 0.037356, 0.089901, 0, 0.012726, 0, 0.002874, 0, 0, 0, 0, 0, 0, 0, 0};
static float SVE_afters_27[] = { 0.001609, 0.001609, 0, 0.059533, 0.006436, 0.001609, 0.219630, 0.004023, 0, 0.001609, 0.016090, 0.077233, 0.002414, 0.182623, 0, 0.003218, 0.250201, 0.037007, 0.131939, 0, 0.002414, 0, 0, 0, 0, 0, 0, 0.000805, 0, 0, 0};
static float SVE_afters_28[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float SVE_afters_29[] = { 0, 0, 0, 0, 0.241379, 0, 0, 0, 0, 0, 0, 0, 0, 0.758621, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float SVE_afters_30[] = { 0.001235, 0.004938, 0.000617, 0.043827, 0, 0.001235, 0.052469, 0, 0, 0.021605, 0.033333, 0.033333, 0.033333, 0.052469, 0, 0.038889, 0.509877, 0.040741, 0.061728, 0, 0.070370, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static struct letter_frequencies SVE_counts[] = {
    { "a", 0.011111, 0.050454, 0.115882, 0.091048, SVE_afters_0 },
    { "b", 0.003704, 0.040139, 0.006801, 0.000373, SVE_afters_1 },
    { "c", 0.000000, 0.001988, 0.030628, 0.000000, SVE_afters_2 },
    { "d", 0.009259, 0.086740, 0.043806, 0.037281, SVE_afters_3 },
    { "e", 0.000000, 0.045441, 0.108946, 0.098587, SVE_afters_4 },
    { "f", 0.011111, 0.051821, 0.008123, 0.001905, SVE_afters_5 },
    { "g", 0.000000, 0.032890, 0.031188, 0.070337, SVE_afters_6 },
    { "h", 0.003704, 0.082474, 0.004511, 0.046601, SVE_afters_7 },
    { "i", 0.885185, 0.025848, 0.061545, 0.011888, SVE_afters_8 },
    { "j", 0.000000, 0.038027, 0.007326, 0.002734, SVE_afters_9 },
    { "k", 0.005556, 0.039684, 0.032138, 0.013380, SVE_afters_10 },
    { "l", 0.000000, 0.025724, 0.068753, 0.025103, SVE_afters_11 },
    { "m", 0.003704, 0.066940, 0.021470, 0.042252, SVE_afters_12 },
    { "n", 0.005556, 0.023694, 0.079098, 0.176215, SVE_afters_13 },
    { "o", 0.009259, 0.061638, 0.054829, 0.007291, SVE_afters_14 },
    { "p", 0.011111, 0.036038, 0.013940, 0.004764, SVE_afters_15 },
    { "r", 0.000000, 0.016486, 0.076842, 0.122572, SVE_afters_16 },
    { "s", 0.005556, 0.120086, 0.038124, 0.030943, SVE_afters_17 },
    { "t", 0.003704, 0.043660, 0.076520, 0.137732, SVE_afters_18 },
    { "u", 0.000000, 0.018475, 0.020487, 0.012883, SVE_afters_19 },
    { "v", 0.001852, 0.056501, 0.014161, 0.010439, SVE_afters_20 },
    { "w", 0.000000, 0.000580, 0.000187, 0.000000, SVE_afters_21 },
    { "x", 0.000000, 0.000000, 0.000678, 0.001201, SVE_afters_22 },
    { "y", 0.000000, 0.000746, 0.007496, 0.001823, SVE_afters_23 },
    { "z", 0.000000, 0.000083, 0.000000, 0.000083, SVE_afters_24 },
    { "à", 0.001852, 0.000000, 0.000000, 0.000000, NULL },
    { "ä", 0.009259, 0.021789, 0.032392, 0.000249, SVE_afters_26 },
    { "å", 0.012963, 0.004557, 0.019215, 0.051199, SVE_afters_27 },
    { "æ", 0.000000, 0.000000, 0.000017, 0.000000, SVE_afters_28 },
    { "é", 0.000000, 0.000000, 0.000492, 0.000456, SVE_afters_29 },
    { "ö", 0.005556, 0.007498, 0.024404, 0.000663, SVE_afters_30 },
    NULL
};	/* SVE */
static float SVE_word_lens[] = {
    0.000000, 0.021879, 0.140675, 0.329484, 0.138892, 0.119728, 0.097605, 0.050444, 0.035290, 0.024594, 0.015477, 0.010048, 0.007496, 0.002998, 0.001904, 0.001378, 0.000972, 0.000324, 0.000203, 0.000284, 0.000041, 0.000000, 0.000203, 0.000000, 0.000041, 0.000041, 0.000000, 0.000000, 0.000000, 0.000000
};

static float SVE_all_consonants[] = {
    0.000000, 0.001459, 0.000527, 0.000041, 0.000041, 0.000122, 0.000000, 0.000081, 0.000122, 0.000000, 0.000000, 0.000000, 
};

static float SVE_consonant_run[] = {
    0.000000, 0.682563, 0.273934, 0.037744, 0.005141, 0.000438, 0.000080, 0.000040, 0.000060, 0.000000, 0.000000, 0.000000
};

static float SVE_vowel_run[] = {
    0.000000, 0.992850, 0.007150, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static struct letter_frequencies TRK_counts[] = {
    { "a", 0.000000, 0.044978, 0.131251, 0.111694 },
    { "b", 0.000000, 0.083208, 0.008639, 0.000000 },
    { "c", 0.000000, 0.008246, 0.008473, 0.000000 },
    { "d", 0.000000, 0.046477, 0.042532, 0.000000 },
    { "e", 0.000000, 0.053223, 0.117129, 0.206147 },
    { "f", 0.000000, 0.015742, 0.002824, 0.001499 },
    { "g", 0.000000, 0.038981, 0.002658, 0.000000 },
    { "h", 0.000000, 0.140930, 0.010965, 0.000000 },
    { "i", 0.000000, 0.068966, 0.094866, 0.083958 },
    { "j", 0.000000, 0.000000, 0.000166, 0.000000 },
    { "k", 0.000000, 0.069715, 0.041037, 0.089205 },
    { "l", 0.000000, 0.000750, 0.091543, 0.027736 },
    { "m", 0.000000, 0.077961, 0.040704, 0.022489 },
    { "n", 0.000000, 0.002249, 0.062635, 0.148426 },
    { "o", 0.000000, 0.030735, 0.009138, 0.000000 },
    { "p", 0.000000, 0.002249, 0.002824, 0.004498 },
    { "r", 0.000000, 0.004498, 0.077255, 0.119190 },
    { "s", 0.000000, 0.053973, 0.037049, 0.012744 },
    { "t", 0.000000, 0.052474, 0.037049, 0.032234 },
    { "u", 0.000000, 0.011244, 0.030736, 0.023988 },
    { "v", 0.000000, 0.111694, 0.006646, 0.000000 },
    { "y", 0.000000, 0.021739, 0.033893, 0.000750 },
    { "z", 0.000000, 0.006747, 0.008141, 0.017991 },
    { "â", 0.000000, 0.000000, 0.000332, 0.000000 },
    { "ç", 0.000000, 0.007496, 0.007975, 0.008996 },
    { "ö", 0.000000, 0.013493, 0.004320, 0.000000 },
    { "ü", 0.000000, 0.008246, 0.016282, 0.008996 },
    { "ğ", 0.000000, 0.000000, 0.009802, 0.000000 },
    { "ı", 0.000000, 0.002249, 0.046353, 0.063718 },
    { "ş", 0.000000, 0.021739, 0.016780, 0.015742 },
    NULL
};	/* TRK */
static float TRK_word_lens[] = {
    0.000000, 0.000000, 0.081709, 0.089955, 0.080960, 0.158921, 0.134933, 0.121439, 0.080960, 0.097451, 0.050975, 0.043478, 0.023988, 0.013493, 0.010495, 0.008246, 0.000750, 0.000750, 0.000000, 0.000750, 0.000750, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float TRK_all_consonants[] = {
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 
};

static float TRK_consonant_run[] = {
    0.000000, 0.761221, 0.234081, 0.004175, 0.000261, 0.000261, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float TRK_vowel_run[] = {
    0.000000, 0.920947, 0.040525, 0.037957, 0.000285, 0.000285, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float WEL_afters_0[] = { 0.000165, 0.004295, 0.038817, 0.070862, 0.104889, 0.024612, 0.016848, 0.001487, 0.126528, 0, 0, 0.031715, 0.050875, 0.152626, 0.000165, 0.000991, 0.160059, 0.013875, 0.018831, 0.081103, 0.000330, 0.086885, 0, 0.000165, 0.013875, 0, 0};
static float WEL_afters_1[] = { 0.118093, 0, 0, 0, 0.197183, 0, 0, 0, 0.031419, 0, 0, 0.086674, 0.002167, 0, 0.130011, 0, 0.203684, 0, 0, 0.047671, 0, 0.027086, 0, 0.156013, 0, 0, 0};
static float WEL_afters_2[] = { 0.121377, 0, 0.000604, 0, 0.039855, 0, 0, 0.530797, 0.021135, 0, 0.000604, 0.048913, 0.000604, 0.002415, 0.029589, 0, 0.031401, 0.001812, 0.002415, 0.021135, 0, 0.030797, 0, 0.116546, 0, 0, 0};
static float WEL_afters_3[] = { 0.122648, 0, 0, 0.370821, 0.073500, 0.004428, 0.006642, 0.000443, 0.102723, 0, 0, 0.013726, 0, 0.000664, 0.056675, 0, 0.063095, 0.004870, 0, 0.024795, 0, 0.066194, 0, 0.088776, 0, 0, 0};
static float WEL_afters_4[] = { 0.008916, 0.026545, 0.010132, 0.179737, 0.000608, 0.042756, 0.016211, 0.001824, 0.153597, 0, 0, 0.102330, 0.003040, 0.132928, 0.001824, 0.000203, 0.103546, 0.049443, 0.050659, 0.051469, 0.001013, 0.040932, 0.000608, 0.010942, 0.010537, 0, 0.000203};
static float WEL_afters_5[] = { 0.159382, 0, 0, 0.000533, 0.166844, 0.094350, 0, 0, 0.054371, 0, 0, 0.040512, 0, 0.033582, 0.125267, 0, 0.077292, 0, 0.000533, 0.037846, 0, 0.020256, 0, 0.189232, 0, 0, 0};
static float WEL_afters_6[] = { 0.220870, 0, 0, 0.001404, 0.072532, 0.000468, 0, 0.038839, 0.023865, 0, 0, 0.086102, 0, 0.002808, 0.105288, 0, 0.046327, 0, 0, 0.024801, 0, 0.258774, 0, 0.117922, 0, 0, 0};
static float WEL_afters_7[] = { 0.205987, 0.003044, 0, 0.004059, 0.149670, 0.000507, 0.004566, 0, 0.085236, 0, 0, 0.014713, 0.003551, 0.006596, 0.125824, 0, 0.040589, 0, 0, 0.057331, 0, 0.132420, 0, 0.165398, 0.000507, 0, 0};
static float WEL_afters_8[] = { 0.154370, 0.010387, 0.014327, 0.084885, 0.017192, 0.018983, 0.041547, 0.000358, 0.006447, 0, 0, 0.044413, 0.030444, 0.105659, 0.119986, 0.002507, 0.136103, 0.081662, 0.053725, 0.000358, 0.001074, 0.045129, 0, 0.001433, 0.029011, 0, 0};
static float WEL_afters_9[] = { 0.066667, 0, 0, 0, 0.100000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.633333, 0.166667, 0, 0, 0, 0.033333, 0, 0, 0, 0, 0, 0, 0};
static float WEL_afters_11[] = { 0.263819, 0.001884, 0.024497, 0.027010, 0.123744, 0.003141, 0.003141, 0, 0.126884, 0, 0, 0, 0.001256, 0, 0.168970, 0.001884, 0, 0.001256, 0, 0.045226, 0, 0.093593, 0, 0.113693, 0, 0, 0};
static float WEL_afters_12[] = { 0.344708, 0.011838, 0, 0.019499, 0.128134, 0.002089, 0.012535, 0.021588, 0.066852, 0, 0, 0.036212, 0.000696, 0.004875, 0.102368, 0.005571, 0.089136, 0.011838, 0.001393, 0.014624, 0, 0.043175, 0, 0.082869, 0, 0, 0};
static float WEL_afters_13[] = { 0.151750, 0.001842, 0.018416, 0.076980, 0.146961, 0.010681, 0.075138, 0.019153, 0.086556, 0, 0, 0.001105, 0.003315, 0.093923, 0.073297, 0, 0.004420, 0.005157, 0.082873, 0.033886, 0, 0.059300, 0, 0.051565, 0.003683, 0, 0};
static float WEL_afters_14[] = { 0.001663, 0.033588, 0.017958, 0.165946, 0.123046, 0.034253, 0.023612, 0.004988, 0.033256, 0, 0, 0.094114, 0.007316, 0.222148, 0.001663, 0.003658, 0.126039, 0.066844, 0.008314, 0.001995, 0.000665, 0.013967, 0, 0.000665, 0.014300, 0, 0};
static float WEL_afters_15[] = { 0.325359, 0, 0, 0, 0.200957, 0, 0.011962, 0, 0.014354, 0, 0, 0.062201, 0, 0, 0.086124, 0.002392, 0.162679, 0, 0.007177, 0.057416, 0, 0.059809, 0, 0.009569, 0, 0, 0};
static float WEL_afters_16[] = { 0.105756, 0.003012, 0.025770, 0.046519, 0.081325, 0.013387, 0.004016, 0.091700, 0.102410, 0, 0, 0.008032, 0.009705, 0.028447, 0.143909, 0.000669, 0.014056, 0.006359, 0.049197, 0.057898, 0, 0.096051, 0, 0.107764, 0.004016, 0, 0};
static float WEL_afters_17[] = { 0.125660, 0.010560, 0.002112, 0.002112, 0.119324, 0.001056, 0.137276, 0.004224, 0.117212, 0, 0, 0.002112, 0.007392, 0.012672, 0.054910, 0.006336, 0, 0.001056, 0.149947, 0.024287, 0, 0.028511, 0, 0.191130, 0.001056, 0, 0.001056};
static float WEL_afters_18[] = { 0.091763, 0, 0.006503, 0, 0.083815, 0, 0, 0.494942, 0.032514, 0, 0, 0.004335, 0.000723, 0.000723, 0.046965, 0, 0.106936, 0.001445, 0.002168, 0.039017, 0, 0.011561, 0, 0.076590, 0, 0, 0};
static float WEL_afters_19[] = { 0.071717, 0.009091, 0.020202, 0.131313, 0.006061, 0.008081, 0.023232, 0, 0.006061, 0, 0, 0.046465, 0.020202, 0.298990, 0.021212, 0, 0.182828, 0.056566, 0.013131, 0.004040, 0, 0.075758, 0, 0, 0.005051, 0, 0};
static float WEL_afters_20[] = { 0.416667, 0, 0, 0, 0.333333, 0, 0, 0, 0.250000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static float WEL_afters_21[] = { 0.076168, 0.006843, 0.058614, 0.012199, 0.173163, 0.002975, 0.005653, 0.000298, 0.057126, 0, 0, 0.043737, 0.008628, 0.122285, 0.010116, 0.000595, 0.136566, 0.010414, 0.005058, 0.005058, 0, 0, 0, 0.262422, 0.002083, 0, 0};
static float WEL_afters_22[] = { 0, 0, 0.250000, 0, 0, 0, 0, 0, 0.250000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.500000, 0, 0, 0, 0, 0, 0, 0, 0};
static float WEL_afters_23[] = { 0.000877, 0.004164, 0.018409, 0.200307, 0, 0.043173, 0.016875, 0.000877, 0.000657, 0, 0, 0.043831, 0.061582, 0.410476, 0, 0, 0.063336, 0.039229, 0.014026, 0, 0, 0.072321, 0, 0, 0.009862, 0, 0};
static float WEL_afters_24[] = { 0.317164, 0, 0, 0.011194, 0.167910, 0.003731, 0.014925, 0, 0.119403, 0, 0, 0, 0, 0, 0.100746, 0, 0, 0.003731, 0.046642, 0.029851, 0, 0.059701, 0, 0.125000, 0, 0, 0};
static float WEL_afters_25[] = { 0.200000, 0, 0, 0, 0.413333, 0, 0, 0, 0, 0, 0, 0.146667, 0, 0.013333, 0.053333, 0, 0.106667, 0, 0, 0.026667, 0, 0.040000, 0, 0, 0, 0, 0};
static float WEL_afters_26[] = { 0, 0, 0, 0, 0, 0, 0, 0, 1.000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static struct letter_frequencies WEL_counts[] = {
    { "a", 0.295305, 0.085526, 0.133654, 0.022534, WEL_afters_0 },
    { "b", 0.001514, 0.047436, 0.005156, 0.013187, WEL_afters_1 },
    { "c", 0.020192, 0.061712, 0.019603, 0.010691, WEL_afters_2 },
    { "d", 0.003029, 0.093336, 0.086657, 0.140324, WEL_afters_3 },
    { "e", 0.001010, 0.052109, 0.116742, 0.021125, WEL_afters_4 },
    { "f", 0.001010, 0.072787, 0.020935, 0.017348, WEL_afters_5 },
    { "g", 0.000000, 0.098329, 0.017025, 0.019717, WEL_afters_6 },
    { "h", 0.000505, 0.050317, 0.033569, 0.055310, WEL_afters_7 },
    { "i", 0.188289, 0.021638, 0.069518, 0.083797, WEL_afters_8 },
    { "j", 0.001010, 0.001920, 0.000000, 0.000000, WEL_afters_9 },
    { "k", 0.000000, 0.000000, 0.000000, 0.000064, NULL },
    { "l", 0.000000, 0.013635, 0.039065, 0.048268, WEL_afters_11 },
    { "m", 0.000505, 0.063312, 0.012663, 0.023814, WEL_afters_12 },
    { "n", 0.004038, 0.034249, 0.061756, 0.217208, WEL_afters_13 },
    { "o", 0.107521, 0.034825, 0.069773, 0.022406, WEL_afters_14 },
    { "p", 0.000000, 0.023942, 0.001246, 0.000512, WEL_afters_15 },
    { "r", 0.002524, 0.033160, 0.069972, 0.125024, WEL_afters_16 },
    { "s", 0.000000, 0.026759, 0.014986, 0.037130, WEL_afters_17 },
    { "t", 0.000505, 0.031240, 0.025382, 0.018629, WEL_afters_18 },
    { "u", 0.000000, 0.009346, 0.023909, 0.053454, WEL_afters_19 },
    { "v", 0.005048, 0.000128, 0.000283, 0.000128, WEL_afters_20 },
    { "w", 0.000505, 0.036489, 0.079065, 0.031944, WEL_afters_21 },
    { "x", 0.000000, 0.000064, 0.000085, 0.000000, WEL_afters_22 },
    { "y", 0.367491, 0.081045, 0.093399, 0.027975, WEL_afters_23 },
    { "ll", 0.000000, 0.021894, 0.005496, 0.009410, WEL_afters_24 },
    { "ph", 0.000000, 0.004801, 0.000000, 0.000000, WEL_afters_25 },
    { "qu", 0.000000, 0.000000, 0.000057, 0.000000, WEL_afters_26 },
    NULL
};	/* WEL */
static float WEL_word_lens[] = {
    0.000000, 0.112544, 0.202761, 0.176855, 0.164640, 0.119191, 0.103909, 0.056982, 0.030508, 0.016873, 0.007954, 0.003579, 0.002954, 0.000966, 0.000057, 0.000000, 0.000114, 0.000057, 0.000057, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float WEL_all_consonants[] = {
    0.000000, 0.004488, 0.000739, 0.000511, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 
};

static float WEL_consonant_run[] = {
    0.000000, 0.746329, 0.224509, 0.028116, 0.000977, 0.000070, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static float WEL_vowel_run[] = {
    0.000000, 0.716191, 0.249052, 0.031673, 0.002959, 0.000125, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000
};

static struct lang_frequencies { uint32 script, lang; char *note; struct letter_frequencies *cnts; float *wordlens; char *vowels; float *consonant_run, *all_consonants, *vowel_run; } lang_frequencies[] = {
    { CHR('l','a','t','n'), CHR('C','S','Y',' '), N_("Czech"),     CSY_counts, CSY_word_lens, "aeiouyráéíóúýěů", CSY_consonant_run, CSY_all_consonants, CSY_vowel_run },
    { CHR('l','a','t','n'), CHR('N','L','D',' '), N_("Dutch"),     NLD_counts, NLD_word_lens, "aeiouyàéêëïòó", NLD_consonant_run, NLD_all_consonants, NLD_vowel_run },
    { CHR('l','a','t','n'), CHR('E','N','G',' '), N_("English"),   ENG_counts, ENG_word_lens, "aeiouy", ENG_consonant_run, ENG_all_consonants, ENG_vowel_run },
    { CHR('l','a','t','n'), CHR('F','R','A',' '), N_("French"),    FRA_counts, FRA_word_lens, "aeiouyàâæèéêëîïôùû", FRA_consonant_run, FRA_all_consonants, FRA_vowel_run },
    { CHR('l','a','t','n'), CHR('D','E','U',' '), N_("German"),    DEU_counts, DEU_word_lens, "aeiouyäöü", DEU_consonant_run, DEU_all_consonants, DEU_vowel_run },
    { CHR('g','r','e','k'), CHR('E','L','L',' '), N_("Lang|Greek"),ELL_counts, ELL_word_lens, "ΐάέήίαειουωϊόύώ", ELL_consonant_run, ELL_all_consonants, ELL_vowel_run },
    { CHR('h','e','b','r'), CHR('I','W','R',' '), N_("Lang|Hebrew"),IWR_counts, IWR_word_lens },
    { CHR('d','e','v','a'), CHR('H','I','N',' '), N_("Hindi"),     HIN_counts, HIN_word_lens },
    { CHR('l','a','t','n'), CHR('H','U','N',' '), N_("Hungarian"), HUN_counts, HUN_word_lens, "aeiouáéëíóöúüőű", HUN_consonant_run, HUN_all_consonants, HUN_vowel_run },
    { CHR('l','a','t','n'), CHR('I','T','A',' '), N_("Italian"),   ITA_counts, ITA_word_lens, "aeiouyàèìíòôùü", ITA_consonant_run, ITA_all_consonants, ITA_vowel_run },
    { CHR('k','a','n','a'), CHR('J','A','P',' '), N_("Hiragana"),  JAP_counts, JAP_word_lens },
    { CHR('k','a','n','a'), CHR('J','A','P',' '), N_("Katakana"),  JAP_kata_counts, JAP_word_lens },
    { CHR('k','a','n','a'), CHR('L','T','H',' '), N_("Lithuanian"),LTH_counts, LTH_word_lens, "aeiouyąęėįųū", LTH_consonant_run, LTH_all_consonants, LTH_vowel_run },
    { CHR('l','a','t','n'), CHR('P','L','K',' '), N_("Polish"),    PLK_counts, PLK_word_lens, "aeiouàâèéêîòóôąęű", PLK_consonant_run, PLK_all_consonants, PLK_vowel_run },
    { CHR('c','y','r','l'), CHR('R','U','S',' '), N_("Russian"),   RUS_counts, RUS_word_lens, "аеийоуэюяё", RUS_consonant_run, RUS_all_consonants, RUS_vowel_run },
    { CHR('l','a','t','n'), CHR('E','S','P',' '), N_("Spanish"),   ESP_counts, ESP_word_lens, "aeiouyáéíñóúü", ESP_consonant_run, ESP_all_consonants, ESP_vowel_run },
    { CHR('d','e','v','a'), CHR('S','A','N',' '), N_("Sanskrit"),  SAN_counts, SAN_word_lens },
    { CHR('l','a','t','n'), CHR('S','V','E',' '), N_("Swedish"),   SVE_counts, SVE_word_lens, "aeiouyäåæéö", SVE_consonant_run, SVE_all_consonants, SVE_vowel_run },
    { CHR('l','a','t','n'), CHR('T','R','K',' '), N_("Turkish"),   TRK_counts, TRK_word_lens, "aeiouyıüö", TRK_consonant_run, TRK_all_consonants, TRK_vowel_run },
    { CHR('l','a','t','n'), CHR('W','E','L',' '), N_("Welsh"),     WEL_counts, WEL_word_lens, "aeiouyw", WEL_consonant_run, WEL_all_consonants, WEL_vowel_run },
    { 0, 0, NULL }
};

/******************************************************************************/
static int isvowel(int index, struct lang_frequencies *lf) {
    const char *pt;
    int ch, ch2;

    if ( lf->vowels==NULL || index==-1 )
return( -1 );
    pt = lf->cnts[index].utf8_letter;
    ch = utf8_ildb(&pt);

    pt = lf->vowels;
    while ( (ch2 = utf8_ildb(&pt))!=0 )
	if ( ch2==ch )
return( true );

return( false );
}

static int RandomWordLength(float *wl) {
    double cum;
    int i;

    if ( wl==NULL )
	wl = word_lengths;

  retry:
    cum = random()/(double) RAND_MAX;
    for ( i=0; i<30; ++i ) {
	if ( cum<wl[i] )
return( i );
	cum -= wl[i];
    }

  goto retry;
}

static int SFHasUtf8Sequence(SplineFont *sf,const char *str) {
    int ch1, ch2;
    ch1 = utf8_ildb(&str);
    ch2 = utf8_ildb(&str);

    if ( ch2!=0 && !SCWorthOutputting(SFGetChar(sf,ch2,NULL)))
return( false );

return( SCWorthOutputting(SFGetChar(sf,ch1,NULL)) );
}

static int RandomChar(struct letter_frequencies *freq, int pos, int last, SplineFont *sf) {
    double cum;
    int i, cnt, tries;
    float *percent;
    float space[100], tot;

    if ( last!=-1 ) {
	if ( freq[last].frequency[MEDI]==0 && freq[last].frequency[INIT]==0 )
return( -1 );		/* Nothing can come after the last character */
	if ( freq[last].afters==NULL )	/* I don't always have these data */
	    last = -1;
    }
    if ( last!=-1 ) {
	percent = freq[last].afters;
	if ( pos==FINA ) {
	    /* Merge probabilities of being after "last" and being at the end */
	    /*  of a word */
	    tot = 0;
	    for ( i=0; freq[i].utf8_letter!=NULL; ++i ) {
		space[i] = percent[i]*freq[i].frequency[FINA];
		tot += space[i];
	    }
	    if ( tot==0 )
return( -1 );
	    for ( i=0; freq[i].utf8_letter!=NULL; ++i )
		space[i]/=tot;
	    percent = space;
	}
	for ( tries=0; tries<5; ++tries ) {
	    cum = random()/(double) RAND_MAX;
	    for ( i=0; freq[i].utf8_letter!=NULL; ++i ) {
		if ( cum<=percent[i] && percent[i]!=0 ) {
		    if ( sf==NULL || SFHasUtf8Sequence(sf,freq[i].utf8_letter))
return( i );
		    else
	    break;
		}
		cum -= percent[i];
	    }
	}
    }

    for ( tries=0; tries<10; ++tries ) {
	cum = random()/(double) RAND_MAX;
	for ( i=0; freq[i].utf8_letter!=NULL; ++i ) {
	    if ( cum<freq[i].frequency[pos] ) {
		if ( sf==NULL || SFHasUtf8Sequence(sf,freq[i].utf8_letter))
return( i );
		else
	break;
	    }
	    cum -= freq[i].frequency[pos];
	}
    }

    /* I suppose it is possible that this font has no characters which can */
    /*  be used in this position.  */
    for ( cnt=0; freq[cnt].utf8_letter!=NULL; ++cnt );
    if ( cnt==0 )
return( -1 );
    for ( tries=0; tries<10; ++tries ) {
	i = random()%cnt;
	if ( sf==NULL || SFHasUtf8Sequence(sf,freq[i].utf8_letter))
return( i );
    }

    /* try to find SOMETHING */
    for ( i=0; freq[i].utf8_letter!=NULL; ++i ) {
	if ( sf==NULL || SFHasUtf8Sequence(sf,freq[i].utf8_letter))
return( i );
    }

return( -1 );
}

static char *RandomWord(struct lang_frequencies *lf, SplineFont *sf) {
    struct letter_frequencies *freq = lf->cnts;
    /* I never generate words longer than 30 characters */
    /* each character is at most 2 unicode chars */
    /* each unicode char takes at most 4 utf8 bytes */
#define WORD_MAX 	(30*8)
    static char word_buf[WORD_MAX+1];
    char *pt;
    int len, i, pos, last, laster, cur;
    int vlen, clen;
    int all_consonant_count=0;

    last = laster = -1;

    len = RandomWordLength(lf->wordlens);
    pt = word_buf;
    vlen = clen = 0;
    for ( i=0; i<len; ++i ) {
	if ( len==1 )
	    pos = ISOL;
	else if ( i==0 )
	    pos = INIT;
	else if ( i==len-1 )
	    pos = FINA;
	else
	    pos = MEDI;
	cur = RandomChar(freq,pos,last,sf);
	if ( cur==last && laster==cur )
	    cur = RandomChar(freq,pos,last,sf);		/* The same letter 3 times in a row is not a good idea */
	if ( cur!=-1 && lf->vowels ) {
	    int isv = isvowel(cur,lf);
	    int retry = false;
	    if ( !isv && i==len-1 && i==clen ) {
		/* If we've got a word with no vowels so far, and we're at the */
		/*  end of it, try hard for a vowel somewhere */
		/* Now there are some "words" with no vowels: "Mrs", "Dr", "PhD" */
		/*  so this is not an absolute */
		if ( len>1 && all_consonant_count++<4 &&
			random()/(double) RAND_MAX >= lf->all_consonants[len] ) {
		    clen = 1;
		    pt = word_buf;
		    for ( last=0; freq[last].utf8_letter!=NULL; ++last )
			if ( strncmp(word_buf,freq[last].utf8_letter,strlen(freq[last].utf8_letter))==0 )
		    break;
		    (void) utf8_ildb((const char **) &pt); laster = -1;
		    i = 0;
    continue;
		}
	    }
	    /* If this is a vowel and we last had a consonant, then that's ok */
	    /* (and vice versa) */
	    /* If this is a vowel, and we have had exactly one vowel before */
	    /*  then the bigram table will have the correct probability for */
	    /*  that so it's ok */
	    /* But if we've already processed two vowels, and now have a third */
	    /*  (or fourth, ...) then take the prob. of an n-vowel sequence */
	    /*  and divide by the prob of a two vowel sequence, and check */
	    if (( vlen>1 && isv ) || ( clen>1 && !isv )) {
		if ( isv )
		    retry = random()/(double) RAND_MAX >= lf->vowel_run[vlen+1]/lf->vowel_run[2];
		else
		    retry = random()/(double) RAND_MAX >= lf->consonant_run[vlen+1]/lf->consonant_run[2];
		if ( retry ) {
		    int j;
		    for ( j=0; j<10; ++j ) {
			cur = RandomChar(freq,pos,last,sf);
			if ( isvowel(cur,lf)!=isv )
		    break;
		    }
		    if ( cur==last && laster==cur )
			cur = RandomChar(freq,pos,last,sf);
		    isv = isvowel(cur,lf);
		}
	    }
	    if ( isv ) {
		clen = 0;
		++vlen;
	    } else {
		vlen = 0;
		++clen;
	    }
	}
	if ( cur!=-1 ) {
	    strcpy(pt,freq[cur].utf8_letter);
	    pt += strlen(pt);
	    laster = last;
	    last = cur;
	}
    }
    *pt = '\0';
return( word_buf );
}

/* ************************************************************************** */

struct script_chars {
    int cnt;
    unichar_t *chars;
};

static void ScriptCharInit(SplineFont *sf,uint32 script, struct script_chars *chrs) {
    int gid, cnt, k;
    SplineFont *subsf;
    SplineChar *sc;
    struct altuni *alt;
    PST *pst;

    cnt = 0;
    k=0;
    do {
	subsf = sf->subfontcnt==0 ? sf : sf->subfonts[k];
	for ( gid = 0 ; gid<subsf->glyphcnt; ++gid ) if ( SCWorthOutputting(sc=subsf->glyphs[gid]) ) {
	    if ( sc->unicodeenc==-1 )
	continue;
	    for ( pst=sc->possub; pst!=NULL; pst=pst->next )
		if ( pst->type == pst_ligature )
	    break;
	    if ( pst!=NULL )		/* Ligatures don't count */
	continue;
	    if ( sc->unicodeenc<0x10000 && isupper(sc->unicodeenc))
	continue;
	    if ( isideoalpha(sc->unicodeenc) &&
		    ScriptFromUnicode(sc->unicodeenc,subsf)==script )
		++cnt;
	    else {
		for ( alt = sc->altuni; alt!=NULL; alt=alt->next ) {
		    if ( alt->vs==-1 && isideoalpha(alt->unienc) &&
			    ScriptFromUnicode(alt->unienc,subsf)==script ) {
			++cnt;
		break;
		    }
		}
	    }
	}
	++k;
    } while ( k<sf->subfontcnt );

    chrs->cnt = cnt;
    chrs->chars = galloc(cnt*sizeof(unichar_t));
    cnt = 0;
    k=0;
    do {
	subsf = sf->subfontcnt==0 ? sf : sf->subfonts[k];
	for ( gid = 0 ; gid<subsf->glyphcnt; ++gid ) if ( SCWorthOutputting(sc=subsf->glyphs[gid]) ) {
	    if ( sc->unicodeenc==-1 )
	continue;
	    if ( sc->unicodeenc<0x10000 && isupper(sc->unicodeenc))
	continue;
	    for ( pst=sc->possub; pst!=NULL; pst=pst->next )
		if ( pst->type == pst_ligature )
	    break;
	    if ( pst!=NULL )		/* Ligatures don't count */
	continue;
	    if ( isideoalpha(sc->unicodeenc) &&
		    ScriptFromUnicode(sc->unicodeenc,subsf)==script )
		chrs->chars[cnt++] = sc->unicodeenc;
	    else {
		for ( alt = sc->altuni; alt!=NULL; alt=alt->next ) {
		    if ( alt->vs==-1 && isideoalpha(alt->unienc) &&
			    ScriptFromUnicode(alt->unienc,subsf)==script ) {
			chrs->chars[cnt++] = alt->unienc;
		break;
		    }
		}
	    }
	}
	++k;
    } while ( k<sf->subfontcnt );
}

static char *ch2utf8( int ch ) {
    static char buffer[8];
    char *pt = buffer;

    if ( ch<=127 )
	*pt++ = ch;
    else if ( ch<=0x7ff ) {
	*pt++ = 0xc0 | (ch>>6);
	*pt++ = 0x80 | (ch&0x3f);
    } else if ( ch<=0xffff ) {
	*pt++ = 0xe0 | (ch>>12);
	*pt++ = 0x80 | ((ch>>6)&0x3f);
	*pt++ = 0x80 | (ch&0x3f);
    }
    *pt = '\0';
return( buffer );
}

static char *ScriptRandomChar(struct script_chars *chrs) {
    int i;

    i = random()%chrs->cnt;
return( ch2utf8( chrs->chars[i]) );
}

static char *ScriptRandomWord(struct script_chars *chrs) {
    static char word_buf[WORD_MAX+1];
    char *pt;
    int len, i;

    len = RandomWordLength(word_lengths);
    pt = word_buf;
    for ( i=0; i<len; ++i ) {
	strcpy(pt,ScriptRandomChar(chrs));
	pt += strlen(pt);
    }
    *pt = '\0';
return( word_buf );
}

static char *RandomPara(struct lang_frequencies *lf,
	struct script_chars *chrs, SplineFont *sf) {
    /* paragraphs will be somewhere between 20 and 84 words */
    int i, len = 20 + (random()/(RAND_MAX>>6));
#define PARA_MAX	(84*(WORD_MAX+1)+10)
    char parabuf[PARA_MAX];
    char *pt = parabuf;

    for ( i=0 ; i<len; ++i ) {
	if ( lf!=NULL )
	    strcpy(pt,RandomWord(lf,sf));
	else
	    strcpy(pt,ScriptRandomWord(chrs));
	pt += strlen(pt);
	*pt++ = ' ';
    }
    if ( pt>parabuf )
	--pt;
    *pt++ = '\n';
    *pt = '\0';
return( copy( parabuf ));
}

char *RandomParaFromScriptLang(uint32 script, uint32 lang, SplineFont *sf,
	struct lang_frequencies *lf) {
    int i;
    struct script_chars chrs;
    char *ret;

    memset(&chrs,0,sizeof(chrs));

    if ( lf==NULL ) {
	for ( i=0; lang_frequencies[i].script!=0; ++i ) {
	    if ( lang_frequencies[i].script==script &&
		    lang_frequencies[i].lang==lang ) {
		lf = &lang_frequencies[i];
	    break;
	    }
	}
    }

    if ( lf==NULL )
	ScriptCharInit(sf,script,&chrs);
    ret = RandomPara(lf,&chrs,sf);
    free(chrs.chars);
return( ret );
}

char *RandomParaFromScript(uint32 script, uint32 *lang, SplineFont *sf) {
    int i;
    struct lang_frequencies *lf;
    struct script_chars chrs;
    char *ret;
    int cnt = 0;

    memset(&chrs,0,sizeof(chrs));
    lf = NULL;

    /* Pick a language at random for this script, (one "language" choice */
    /*  is default which means we fall through without picking anything) */
    for ( i=0; lang_frequencies[i].script!=0; ++i ) {
	if ( lang_frequencies[i].script==script )
	    ++cnt;
    }
    if ( cnt!=0 ) {
	int pos = random()%(cnt+1);
	if ( pos<cnt ) {
	    cnt = 0;
	    for ( i=0; lang_frequencies[i].script!=0; ++i ) {
		if ( lang_frequencies[i].script==script ) {
		    if ( cnt==pos ) {
			lf = &lang_frequencies[i];
			*lang = lang_frequencies[i].lang;
	    break;
		    }
		    ++cnt;
		}
	    }
	}
    }

    if ( lf==NULL ) {
	ScriptCharInit(sf,script,&chrs);
	*lang = CHR('d','f','l','t');
    }
    ret = RandomPara(lf,&chrs,sf);
    free(chrs.chars);
return( ret );
}

static int tag_compare(const void *tag1, const void *tag2) {
    if ( *(uint32 *) tag1 > *(uint32 *) tag2 )
return( 1 );
    else if ( *(uint32 *) tag1 < *(uint32 *) tag2 )
return( -1 );
    else
return( 0 );
}

int SF2Scripts(SplineFont *sf,uint32 scripts[100]) {
    int scnt, s, gid, k;
    SplineFont *subsf;
    SplineChar *sc;
    uint32 script;
    PST *pst;

    scnt=0;
    k=0;
    do {
	subsf = sf->subfontcnt==0 ? sf : sf->subfonts[k];
	for ( gid = 0 ; gid<subsf->glyphcnt; ++gid ) if ( SCWorthOutputting(sc=subsf->glyphs[gid]) ) {
	    if ( sc->unicodeenc==-1 || !isideoalpha(sc->unicodeenc ) )
	continue;
	    for ( pst=sc->possub; pst!=NULL; pst=pst->next )
		if ( pst->type == pst_ligature )
	    break;
	    if ( pst!=NULL )		/* Ligatures don't count */
	continue;
	    script = ScriptFromUnicode(sc->unicodeenc,subsf);
	    if ( script == DEFAULT_SCRIPT )
	continue;
	    for ( s=0; s<scnt; ++s)
		if (scripts[s] == script )
	    break;
	    if ( s==scnt )
		scripts[scnt++] = script;
	}
	++k;
    } while ( k<sf->subfontcnt );
    qsort(scripts,scnt,sizeof(uint32),tag_compare);
    scripts[scnt] = 0;
return( scnt );
}

char **SFScriptLangs(SplineFont *sf,struct lang_frequencies ***_freq) {
    uint32 scripts[100];
    int scnt, s, extras, i, pos;
    char **sl;
    char buffer[100];
    struct lang_frequencies **freq;

    scnt = SF2Scripts(sf,scripts);

    extras = 0;
    for ( s=0 ; s<scnt; ++s ) {
	for ( i=0; lang_frequencies[i].script!=0; ++i )
	    if ( lang_frequencies[i].script==scripts[s] )
		++extras;
    }

    sl = galloc( (scnt+extras+1) * sizeof(char *));
    freq = galloc( (scnt+extras+1) * sizeof(struct lang_frequencies *));
    pos = 0;
    for ( s=0 ; s<scnt; ++s ) {
	for ( i=0; lang_frequencies[i].script!=0; ++i ) {
	    if ( lang_frequencies[i].script==scripts[s] ) {
		sprintf( buffer, "%.70s %c%c%c%c{%c%c%c%c}",
			S_(lang_frequencies[i].note),
			scripts[s]>>24, scripts[s]>>16, scripts[s]>>8, scripts[s],
			lang_frequencies[i].lang>>24, lang_frequencies[i].lang>>16, lang_frequencies[i].lang>>8, lang_frequencies[i].lang );
		freq[pos] = &lang_frequencies[i];
		sl[pos++] = copy( buffer );
	    }
	}
	sprintf( buffer, "%c%c%c%c{dflt}",
		scripts[s]>>24, scripts[s]>>16, scripts[s]>>8, scripts[s] );
	freq[pos] = NULL;
	sl[pos++] = copy( buffer );
    }
    sl[pos] = NULL;
    if ( _freq!=NULL ) *_freq = freq;
    else free(freq);
return( sl );
}
