/*
 * Copyright (c) 2005 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * The contents of this file constitute Original Code as defined in and
 * are subject to the Apple Public Source License Version 1.1 (the
 * "License").  You may not use this file except in compliance with the
 * License.  Please obtain a copy of the License at
 * http://www.apple.com/publicsource and read it before using this file.
 * 
 * This Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */

#ifndef _IO80211INTERFACE_H
#define _IO80211INTERFACE_H

/*
 * Kernel
 */
#if defined(KERNEL) && defined(__cplusplus)

#include <IOKit/network/IOEthernetInterface.h>

enum IO80211LinkState
{
	kIO80211NetworkLinkUndefined,			// Starting link state when an interface is created
	kIO80211NetworkLinkDown,				// Interface not capable of transmitting packets
	kIO80211NetworkLinkUp,					// Interface capable of transmitting packets
};
typedef enum IO80211LinkState IO80211LinkState;

/*!	@defined kIO80211InterfaceClass
	@abstract The name of the IO80211Interface class. 
	*/
#define kIO80211InterfaceClass     "IO80211Interface"

/*!	@class IO80211Interface
	@abstract The 80211 interface object. 
	@discussion An 80211 controller driver, that is a subclass of IOEthernetInterface, will instantiate an object of this class when the driver calls the attachInterface() method. This interface object will then vend an Ethernet interface to DLIL, and manage the connection between the controller driver and the upper networking layers. Drivers will seldom need to subclass IO80211Interface. 
	*/
class IO80211Interface : public IOEthernetInterface
{
	OSDeclareDefaultStructors( IO80211Interface );
	
public:
	
	/*!	@function init
		@abstract Initializes an IO80211Interface instance.
		@discussion Instance variables are initialized, and ???.
		@param controller A network controller object that will service the interface object being initialized.
		@result Returns true on success, false otherwise. 
		*/
	virtual bool		init(IONetworkController *controller );
	
	/*! @function attachToDataLinkLayer
		@abstract ???.
		@discussion ???. 
		*/
	virtual IOReturn	attachToDataLinkLayer(IOOptionBits	options,
											  void			*parameter);
											
	/*! @function detachFromDataLinkLayer
		@abstract ???.
		@discussion ???. 
		*/
	virtual void		detachFromDataLinkLayer(IOOptionBits options,
												void		*parameter);
												
	/*! @function inputEvent
		@abstract Sends an event to the data link layer.
		@discussion This method can be used by the network controller to
		send an event to the data link layer.
		@param type A constant describing the event type.
		@param data Data associated with the event.
		@result Returns true if the event was delivered, false if the event type
		specified is invalid, or if the event delivery was unsuccesful. 
		*/
	virtual bool inputEvent(UInt32 type, void * data);
										  
	/*! @function setPoweredOnByUser
		@abstract ???.
		@discussion ???. 
		*/
	virtual void		setPoweredOnByUser(bool poweredOn);

	/*! @function poweredOnByUser
		@abstract ???.
		@discussion ???. 
		*/
	bool		poweredOnByUser();
	
	/*! @function setEnabledBySystem
		@abstract ???.
		@discussion ???. 
		*/
	virtual void		setEnabledBySystem(bool systemEnabled);

	/*! @function enabledBySystem
		@abstract ???.
		@discussion ???. 
		*/
	bool		enabledBySystem();

	/*! @function createStatusDevice
		@abstract ???.
		@discussion ???. 
		*/
	IOReturn			createStatusDevice(char *name );

	/*! @function postMessage
		@abstract <abstract>.
		@discussion ???. 
		*/
	void				postMessage(int msg);
	
	/*! @function setScanTimeout
		@abstract <abstract>.
		@discussion ???. 
		*/
	void				setScanTimeout( unsigned int timeout );
	
	/*! @function scanTimout
		@abstract <abstract>.
		@discussion ???. 
		*/
	unsigned int		scanTimout();
	
	/*! @function setScanPid
		@abstract <abstract>.
		@discussion ???. 
		*/
	void				setScanPid( pid_t pid );
	
	/*! @function scanPid
		@abstract <abstract>.
		@discussion ???. 
		*/
	pid_t				scanPid();
	
	/*! @function setLinkState
		@abstract <abstract>.
		@discussion ???. 
		*/
	virtual bool		setLinkState( IO80211LinkState linkState );
	
	/*! @function linkState
		@abstract <abstract>.
		@discussion ???. 
		*/
	IO80211LinkState	linkState();
	
#if defined( _MODERN_BPF )
	
	/*! @function bpfAttach
		@abstract <abstract>.
		@discussion ???. 
		*/
	errno_t		bpfAttach( UInt32 dlt, UInt32 headerLen );
	
	/*! @function bpfTap
		@abstract <abstract>.
		@discussion ???. 
		*/
	errno_t		monitorModeInputPacket( mbuf_t m, UInt32 dlt, void * header, size_t header_len );
	
#else
	
	/*! @function monitorModeInputPacket
		@abstract ???.
		@discussion ???. 
		*/
	void		monitorModeInputPacket(mbuf_t m);
	
#endif
	
protected:
	
	/*!	@function performCommand
		@abstract Handles an ioctl command sent to the Ethernet interface.
		@discussion This method handles socket ioctl commands sent to the Ethernet interface from DLIL. Commands recognized and processed by this method are SIOCSIFADDR, SIOCSIFFLAGS, SIOCADDMULTI, and SIOCDELMULTI. Other commands are passed to the superclass.
		@param controller The controller object.
		@param cmd The ioctl command code.
		@param arg0 Command argument 0. Generally a pointer to an ifnet structure associated with the interface.
		@param arg1 Command argument 1.
		@result Returns a BSD return value defined in bsd/sys/errno.h. 
		*/
	virtual SInt32		performCommand(IONetworkController	*controller,
									   UInt32				cmd,
									   void					*arg0,
									   void					*arg1);
									   
	/*! @function bpfTap
		@abstract <abstract>.
		@discussion ???. 
		*/
	static int performGatedCommand( void *, void *, void *, void *, void * );
									   
	/*! @function free
		@abstract Frees the IO80211Interface instance.
		@discussion ???, followed by a call to super::free().
		*/
	virtual void		free();
	
private:
		
	ifnet_t		wlt_if;
	int			_minor;
	
	bool		_poweredOnByUser;
	bool		_enabledBySystem;	// IFF_UP flag set?
	
	// Scan data
	unsigned int	_scanTimeout;
	pid_t			_scanPid;
	
	IO80211LinkState _linkState;
	bool			 _pendingLinkEvent;
	
public:

	// Virtual function padding
	
	OSMetaClassDeclareReservedUnused( IO80211Interface,  0);
	OSMetaClassDeclareReservedUnused( IO80211Interface,  1);
	OSMetaClassDeclareReservedUnused( IO80211Interface,  2);
	OSMetaClassDeclareReservedUnused( IO80211Interface,  3);
	OSMetaClassDeclareReservedUnused( IO80211Interface,  4);
	OSMetaClassDeclareReservedUnused( IO80211Interface,  5);
	OSMetaClassDeclareReservedUnused( IO80211Interface,  6);
	OSMetaClassDeclareReservedUnused( IO80211Interface,  7);
	OSMetaClassDeclareReservedUnused( IO80211Interface,  8);
	OSMetaClassDeclareReservedUnused( IO80211Interface,  9);
	OSMetaClassDeclareReservedUnused( IO80211Interface, 10);
	OSMetaClassDeclareReservedUnused( IO80211Interface, 11);
	OSMetaClassDeclareReservedUnused( IO80211Interface, 12);
	OSMetaClassDeclareReservedUnused( IO80211Interface, 13);
	OSMetaClassDeclareReservedUnused( IO80211Interface, 14);
	OSMetaClassDeclareReservedUnused( IO80211Interface, 15);
};

#endif /* defined(KERNEL) && defined(__cplusplus) */

#endif /* ! _IO80211INTERFACE_H */

